package freenet.client;

import freenet.support.Bucket;
import freenet.support.BucketSink;
import freenet.support.BucketFactory;
import freenet.client.metadata.SplitFile;
import java.io.IOException;

/*
  This code is distributed under the GNU Public Licence (GPL)
  version 2.  See http://www.gnu.org/ for further details of the GPL.
*/

/**
 * Interface for plugging Forward Error Correction (FEC) decoder
 * implementaions into fproxy. 
 * <p>
 * Implementations should stream if they can.
 * Clients <em>must</em> call decode() after isDecodable()
 * returns true for each segment, to insure that all data blocks
 * are streamed to the sink.
 * <p>
 * Implementations are not guaranteed to be 
 * thread safe. The client code is responsible
 * for serializing access.
 * <p>
 * Implementations must have a no args constructor in order
 * to be loadable by FECFactory.  All initialization should
 * be done in init().
 *
 * @author giannij 
 **/
public interface FECDecoder extends BucketSink {

    /**
     * Create a FECDecoder capable of decoding the
     * data in the SplitFile. 
     * <p>
     * init() leaves the decoder defaulted to segment 0.
     * If there are additional segments the client should 
     * call setSegement() to reset the decoder for each one.
     * <p>
     * @param sf  the SplitFile to decode
     * @param sink the sink to write the data blocks of the
     *             SplitFile to. Note that there is no
     *             guarantee that blocks will be written
     *             in order.
     * @param factory a BucketFactory used to make temporary 
     *                Buckets.
     *
     * @return true on success, false if a decoder can't be
     *         allocated for the SplitFile.
     **/
    boolean init(SplitFile sf, BucketSink sink,
		 BucketFactory factory);

    /**
     * Releases all resources (memory, file handles, temp files, etc.)
     * used by the encoder instance.
     * <p>
     * Implementations must support calling this function 
     * irregardless of the state of the decoder instance. i.e.
     * it isn't an error to call release on a decoder that
     * has already been released or has never been initialized.
     **/
    void release();
    
    /** 
     * @return the block size for all data and check blocks.
     **/
    int getBlockSize();

    /**
     * @return the number of data and check blocks in the
     *         current segment.
     **/
    int getN();

    /**
     * @return the number of data blocks in the
     *         current segment.
     **/
    int getK();
    
    /**
     * @return the number of the current segment.
     **/
    int getCurrentSegment();

    /**
     * Sets the current segement.
     * <p>
     * @return true on success, false otherwise.
     **/
    boolean setSegment(int segment);

    /**
     * @return the number of segments used to encode the file.
     **/
    int getSegmentCount();

    /**
     * @return the sum of getN() over all segments.
     **/
    int getTotalN();

    /**
     * @return the sum of getK() over all segments.
     **/
    int getTotalK();

    /**
     * Called by clients to submit blocks to the decoder.
     * <p>
     * The implementation owns all buckets submitted
     * via this function and is responsible for 
     * deleting them (if they are not given away to
     * the target BucketSink).
     * <p>
     * Block numbers are absolute. Data blocks are numbered from
     * 0 to getTotalK() - 1. Check blocks are numbered from
     * getTotalK() to getTotalN() - 1.  It is an error for client
     * code to submit a block from a segment other than the 
     * current segment as denoted by getCurrentSegment().
     * <p>
     * @param bucket the block.
     * @param number the block number.
     **/
    void putBucket(Bucket bucket, int number) throws IOException;

    /**
     * @ true as soon as the decoder has enough blocks
     * to fully decode the the current segment.
     **/
    public boolean isDecodable();

    /**
     * Clients must call this after isDecodable() 
     * returns true to ensure that all data blocks
     * for the current segment are written to the sink.
     * <p>
     * This is a blocking call that may take
     * a long time. Polite implementations
     * should abort as quickly as possible and
     * throw InterruptedIOException if the thread
     * is interrupted.
     * <p>
     * This call transfers ownership of the data Buckets
     * to the target BucketSink which is responible for
     * deleting them.
     *
     * @return true on success, false otherwise.
     **/
    public boolean decode() throws IOException;
}










