package freenet;
import freenet.crypt.*;
import freenet.support.Fields;
import java.math.BigInteger;
import java.util.Random;

public class DSAAuthentity extends DSAPrivateKey implements Authentity {

    private DSAGroup grp;

    public DSAAuthentity(BigInteger x, DSAGroup grp) {
        super(x);
        this.grp = grp;
    }
    
    public DSAAuthentity(DSAGroup g, Random r) {
        super(g, r);
        this.grp = g;
    }

    // FIXME: get rid of this constructor
    public DSAAuthentity(String value, String grp) throws NumberFormatException {
        this(new BigInteger(value, 16),
             (grp == null ? Global.DSAgroupC
                          : (DSAGroup) DSAGroup.readFromField(grp))
        );
    }
    
    /**
     * Construct the authentity from hex string values.
     * @param fs  a FieldSet containing x, p, q, and g
     */
    public DSAAuthentity(FieldSet fs) throws NumberFormatException {
        this(getX(fs), getGroup(fs));
    }

    private static final DSAGroup getGroup(FieldSet fs) throws NumberFormatException {
        BigInteger p, q, g;
        try {
            p = new BigInteger(fs.get("p"), 16);
            q = new BigInteger(fs.get("q"), 16);
            g = new BigInteger(fs.get("g"), 16);
        }
        catch (NullPointerException e) {
            // yea, i know, don't catch NPEs .. but _some_ JVMs don't
            // throw the NFE like they are supposed to (*cough* kaffe)
            throw new NumberFormatException(""+e);
        }
        return new DSAGroup(p, q, g);
    }

    private static final BigInteger getX(FieldSet fs) throws NumberFormatException {
        try {
            return new BigInteger(fs.get("x"), 16);
        }
        catch (NullPointerException e) {
            // yea, i know, don't catch NPEs .. but _some_ JVMs don't
            // throw the NFE like they are supposed to (*cough* kaffe)
            throw new NumberFormatException(""+e);
        }
    }
    

    /**
     * Returns the signature of the byte digest.
     */
    public CryptoElement sign(byte[] digest) {
        return sign(Util.byteArrayToMPI(digest));
    }

    public CryptoElement sign(BigInteger b) {
        return DSA.sign(grp, this, b, Core.randSource);
    }

    /** @return  a FieldSet containing the private key and DSA group primitives
      *          as hex strings (x, p, q, g)
      */
    public final FieldSet getFieldSet() {
        FieldSet fs = new FieldSet();
        fs.put("x", getX().toString(16));
        fs.put("p", grp.getP().toString(16));
        fs.put("q", grp.getQ().toString(16));
        fs.put("g", grp.getG().toString(16));
        return fs;
    }
    
    /**
     * Returns the Cryptographic private key of this authentity 
     */
    public final CryptoKey getKey() {
        return this;
    }

    /** @return a DSAIdentity made from this
      */
    public final Identity getIdentity() {
        return new DSAIdentity(grp, this);
    }
    
}


