//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Fourrier.hpp,v 1.2 2004/07/11 22:37:06 delpinux Exp $

// this class allows to define Fourrier Boundary Conditions

#ifndef _FOURRIER_HPP_
#define _FOURRIER_HPP_

#include <TinyVector.hpp>
#include <TinyMatrix.hpp>

#include <PDECondition.hpp>

/*! \class Fourrier
   This class sets Fourrier or Robin boundary conditions:
   \f$\alpha u + A\nabla u \cdot n = g \f$ on \f$\gamma\f$.

   The \f$ A\nabla u \cdot n \f$ is the \b conormal derivative arising from
   the operator \f$ \nabla \cdot A\nabla \f$ when applying the Green formula
   on the \em variationnal problem associated to the \em PDE problem.

   \author Stphane Del Pino.
 */
class Fourrier
  : public PDECondition
{
private:
  //! The \f$\alpha\f$ coeficient
  ReferenceCounting<UserFunction> __Alpha;

  //! The function to impose as a Fourrier Boundary Condition
  ReferenceCounting<UserFunction> __g;

public:

  inline const UserFunction& alpha() const
  {
    return *__Alpha;
  }

  inline const UserFunction& g() const
  {
    return *__g;
  }

  inline ReferenceCounting<UserFunction> alphaReference() const
  {
    return __Alpha;
  }

  inline ReferenceCounting<UserFunction> gReference() const
  {
    return __g;
  }

  inline const real_t alpha(const TinyVector<3,real_t>& X) const
  {
    return (*__Alpha)(X);
  }

  inline const real_t g(const TinyVector<3,real_t>& X) const
  {
    return (*__g)(X);
  }

  const std::string typeName() const
  {
    return std::string("fourrier");
  }

  //! Constructs a Fourrier boundary condition.
  Fourrier(ReferenceCounting<UserFunction> ualpha,
	   ReferenceCounting<UserFunction> ug,
	   const size_t unknownNumber)
    : PDECondition(PDECondition::fourrier,
		   unknownNumber),
      __Alpha(ualpha),
      __g(ug)
  {
    ;
  }

  //! Copy constructor.
  Fourrier(const Fourrier& F)
    : PDECondition(F),
      __Alpha(F.__Alpha),
      __g(F.__g)
  {
    ;
  }

  //! Destrutor
  ~Fourrier()
  {
    ;
  }
};


#endif // _FOURRIER_HPP_

