//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: FEMDiscretization.hpp,v 1.11 2004/05/02 17:34:54 delpinux Exp $

#ifndef FEM_DISCRETIZATION_HPP
#define FEM_DISCRETIZATION_HPP

/**
 * @file   FEMDiscretization.hpp
 * @author Stephane Del Pino
 * @date   Sat Apr 12 18:17:18 2003
 * 
 * @brief  PDE discretization using finite element method.
 * 
 * Partial Differential Equation discretization using finite element
 * method. The trick here is that elementary matrices assembling is
 * factorized in order to optimize the process.
 *
 * @warning variational formula is not complete for right hand side
 * containing partial differential operators.
 * @todo re-implement rhs discretization using the same design than
 * for left part.
 */

#include <ElementaryMatrixSet.hpp>

#include <BaseFEMDiscretization.hpp>

#include <Mesh.hpp>
#include <Structured3DMesh.hpp>

#include <Timer.hpp>
#include <Q1FiniteElement.hpp>

#include <DoubleHashedMatrix.hpp>
#include <UnAssembledMatrix.hpp>

#warning Should not use language classes here
#include <FunctionExpression.hpp>
#include <MeshExpression.hpp>


template <typename GivenMeshType>
class FEMDiscretization
  : public BaseFEMDiscretization<GivenMeshType>
{
private:
  /// The type of mesh used for discretization
  typedef GivenMeshType MeshType;

  /// The geometry of finite elements
  typedef typename MeshType::ElementGeometry ElementGeometry;

  /// The finite element
  typedef typename FiniteElementTraits<ElementGeometry>::Type FiniteElement;

  /// Elementary matrices type
  typedef typename FiniteElement::ElementaryMatrix ElementaryMatrixType;

  /// Elementary vector type
  typedef typename FiniteElement::ElementaryVector ElementaryVectorType;

  /// type of transformation from reference element
  typedef typename FiniteElementTraits<ElementGeometry>::Transformation ConformTransformation;

  /// Associated jacobian
  typedef typename FiniteElementTraits<ElementGeometry>::JacobianTransformation JacobianTransformation;

public:
  /** 
   * Assembles the matrix associated to the PDE operators of the PDE
   * problem.
   * 
   */
  void assembleMatrix()
  {
    Timer t;

    ffout(2) << "Assembling matrix\n";
    t.start();
  
    switch ((this->__A).type()) {
    case BaseMatrix::doubleHashedMatrix: {

      DoubleHashedMatrix& A = dynamic_cast<DoubleHashedMatrix&>(this->__A);

      for(typename MeshType::const_iterator icell(this->__mesh);
	  not(icell.end()); ++icell) {
	const ElementGeometry& K = *icell;

	size_t indices[FiniteElement::numberOfDegreesOfFreedom];
	for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	  indices[l] = this->__mesh.vertexNumber(K(l));
	}

	ConformTransformation T(K);
	JacobianTransformation J(T);
	generatesElementaryMatrix(this->__eSet,J);

	TinyVector<3> massCenter;
	T.value(FiniteElement::massCenter(), massCenter);

	for(typename DiscretizedOperators<ElementaryMatrixType>
	      ::iterator iOperator = this->__discretizedOperators.begin();
	    iOperator != this->__discretizedOperators.end(); ++iOperator) {
	  const real_t coef = ((*iOperator).second)(massCenter);

	  const ElementaryMatrixType& elementaryMatrix
	    = (*(*iOperator).first);

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	    const size_t I // Index of Ith value
	      = __degreeOfFreedomSet(((*iOperator).second).i(), indices[l]);
	    for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; m++) {
	      const size_t J // Index of Jth value
		= __degreeOfFreedomSet(((*iOperator).second).j(), indices[m]);
	      A(I,J) += coef*elementaryMatrix(l,m);
	    }
	  }
	}
      }
      break;
    }
    case BaseMatrix::unAssembled: {
      UnAssembledMatrix& A = dynamic_cast<UnAssembledMatrix&>(this->__A);
      A.setDiscretization(this);
      break;
    }

    default: {
      fferr(2) << '\n' << __FILE__ << ':' << __LINE__
	       << ':' << "Not implemented\n";
      std::exit(1);
    }
    }
    ffout(2) << "done\n";
    t.stop();
    ffout(3) << "Assembling cost: " << t << '\n';
  }

  /** 
   *  Applies directly the operator discretization to the vector X.
   * 
   * @param X input vector
   * @param Z \f$ z=Ax \f$
   */
  void timesX(const BaseVector& X, BaseVector& Z) const
  {
    const Vector<real_t>& x = dynamic_cast<const Vector<real_t>&>(X);
    Vector<real_t>& z = dynamic_cast<Vector<real_t>&>(Z);
    z=0;

    for(typename MeshType::const_iterator icell(this->__mesh);
	not(icell.end()); ++icell) {
      const ElementGeometry& K = *icell;

      ConformTransformation T(K);
      JacobianTransformation J(T);
      generatesElementaryMatrix(this->__eSet,J);

      size_t indices[FiniteElement::numberOfDegreesOfFreedom];
      for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	indices[l] = this->__mesh.vertexNumber(K(l));
      }

      TinyVector<3> massCenter;
      T.value(FiniteElement::massCenter(), massCenter);

      for (size_t i=0; i<this->problem().numberOfUnknown(); ++i) {

	for(typename DiscretizedOperators<ElementaryMatrixType>
	      ::iterator iOperator = this->__discretizedOperators.begin();
	    iOperator != this->__discretizedOperators.end(); ++iOperator) {
	  const real_t coef = ((*iOperator).second)(massCenter);
	  const ElementaryMatrixType& elementaryMatrix
	    = (*(*iOperator).first);

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	    const size_t I1 // Index of Ith value
	      = __degreeOfFreedomSet(((*iOperator).second).i(), indices[l]);
	    for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; m++) {
	      const size_t I2 // Index of Jth value
		= __degreeOfFreedomSet(((*iOperator).second).j(), indices[m]);
	      z[I2] += coef*elementaryMatrix(l,m)*x[I1];
	    }
	  }
	}
      }
    }
  }

  /** 
   * Computes diagonal of the operator
   * 
   * @param Z diagonal of the operator
   */
  void getDiagonal(BaseVector& Z) const
  {
    Vector<real_t>& z = dynamic_cast<Vector<real_t>&>(Z);
    z=0;

    for(typename MeshType::const_iterator icell(this->__mesh);
	not(icell.end()); ++icell) {
      const ElementGeometry& K = *icell;

      ConformTransformation T(K);
      JacobianTransformation J(T);
      generatesElementaryMatrix(this->__eSet,J);

      size_t indices[FiniteElement::numberOfDegreesOfFreedom];
      for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	indices[l] = this->__mesh.vertexNumber(K(l));
      }

      TinyVector<3> massCenter;
      T.value(FiniteElement::massCenter(), massCenter);

      for (size_t i=0; i<this->problem().numberOfUnknown(); ++i) {

	for(typename DiscretizedOperators<ElementaryMatrixType>
	      ::iterator iOperator = this->__discretizedOperators.begin();
	    iOperator != this->__discretizedOperators.end(); ++iOperator) {
	  const real_t coef = ((*iOperator).second)(massCenter);

	  const ElementaryMatrixType& elementaryMatrix
	    = (*(*iOperator).first);

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	    const size_t I // Index of Ith value
	      = this->__degreeOfFreedomSet(((*iOperator).second).i(),
					   indices[l]);
	    z[I] += coef*elementaryMatrix(l,l);
	  }
	}
      }
    }
  }

  /** 
   * Second member assembling
   * 
   */
  void assembleSecondMember()
  {
    ffout(2) << "Assembling Second Member\n";
    //! The elementary vector
    ElementaryVectorType eVect;
    Vector<real_t>& b = (static_cast<Vector<real_t>&>(this->__b));
    b = 0;

    switch(this->problem().type()) {
    case Problem::pdeProblem: {
      const PDESystem& pdeSystem
	= static_cast<const PDESystem&>(this->problem());

      for(typename MeshType::const_iterator icell(this->__mesh);
	  not(icell.end()); ++icell) {
	const ElementGeometry& K = *icell;
	eVect = 0;
	ConformTransformation T(K);
	JacobianTransformation J(T);
	TinyVector<FiniteElement::numberOfDegreesOfFreedom, int> index;
	TinyVector<FiniteElement::numberOfDegreesOfFreedom, TinyVector<3> > quadrature;

	for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	  quadrature[l] = T.quadratureVertex(l);
	  index[l] = this->__mesh.vertexNumber(K(l));
	}

	for (size_t i=0; i<this->problem().numberOfUnknown(); ++i) {
	  ElementaryVectorType f;
	  const UserFunction& F = pdeSystem.secondMember(i);

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l)
	    f[l] = F(quadrature[l]);

	  generatesElementaryVector(eVect,J,f);

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	    const size_t dof = __degreeOfFreedomSet(i,index[l]);
	    b[dof] += eVect[l]*J.jacobianDet();
	  }
	}
      }
      break;
    }
    case Problem::variationalProblem: {
      const VariationalProblem& variationalProblem
	=  dynamic_cast<const VariationalProblem&>(this->problem());

      for (VariationalProblem::linearOperatorConst_iterator i
	     = variationalProblem.beginLinearOperator();
	   i != variationalProblem.endLinearOperator(); ++i) {
	switch ((*(*i)).type()) {
	case VariationalLinearOperator::FV: {
	  const VariationalOperatorFV& fv
	    = dynamic_cast<const VariationalOperatorFV&>(*(*i));

	  const UserFunction& F = fv.f();
	  const size_t testFunctionNumber = fv.testFunctionNumber();

	  for(typename MeshType::const_iterator icell(this->__mesh);
	      not(icell.end()); ++icell) {
	    const ElementGeometry& K = *icell;
	    eVect = 0;
	    ConformTransformation T(K);
	    JacobianTransformation J(T);
	    TinyVector<FiniteElement::numberOfDegreesOfFreedom,int> index;
	    TinyVector<FiniteElement::numberOfDegreesOfFreedom,TinyVector<3> > quadrature;

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	      quadrature[l] = T.quadratureVertex(l);
	      index[l] = this->__mesh.vertexNumber(K(l));
	    }

	    ElementaryVectorType f;

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l)
	      f[l] = F(quadrature[l]);

	    generatesElementaryVector(eVect,J,f);

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	      const size_t dof = __degreeOfFreedomSet(testFunctionNumber,index[l]);
	      b[dof] += eVect[l]*J.jacobianDet();
	    }
	  }
	  break;
	}
	case VariationalLinearOperator::FdxGV: {
	  const VariationalOperatorFdxGV& fv
	    = dynamic_cast<const VariationalOperatorFdxGV&>(*(*i));

	  const UserFunction& F = fv.f();
	  const UserFunction& G = fv.g();
	  const size_t testFunctionNumber = fv.testFunctionNumber();

	  Vector<real_t> gValues (this->__mesh.numberOfVertices());

	  for (size_t i=0; i<this->__mesh.numberOfVertices(); ++i) {
	    gValues[i] = G(this->__mesh.vertex(i));
	  }

	  typename MeshType::const_iterator icell0(this->__mesh);
	  ConformTransformation T0(*icell0);
	  JacobianTransformation J(T0);

	  ElementaryMatrixType edxUV = 0;
	  generatesElementaryMatrix(PDEOperator::firstorderop,
				    J, edxUV, fv.number());

	  for (typename MeshType::const_iterator icell(this->__mesh);
	       not(icell.end()); ++icell) {
	    const ElementGeometry& K = *icell;

	    TinyVector<FiniteElement::numberOfDegreesOfFreedom, int> index;
	    TinyVector<FiniteElement::numberOfDegreesOfFreedom, TinyVector<3> > quadrature;

	    ConformTransformation T(K);

	    TinyVector<3> massCenter;
	    T.value(FiniteElement::massCenter(), massCenter);

	    const real_t fValue = F(massCenter);

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	      quadrature[l] = T.quadratureVertex(l);
	      index[l] = this->__mesh.vertexNumber(K(l));
	    }

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	      const size_t dof
		= __degreeOfFreedomSet(testFunctionNumber,index[l]);

	      for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; ++m) {
		b[dof] += edxUV(l,m)*gValues[index[m]]*fValue;
	      }
	    }
	  }
	  break;
	}
	default: {
	  fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
	  std::exit(1);
	}
	}
      }
      break;
    }
    default: {
      fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
      std::exit(1);
    }
    }
  }

public:

  /** 
   * Constructor of the discretization
   * 
   * @param p the problem
   * @param m the mesh used for discretization
   * @param a matrix storing discretization
   * @param bb vector that stores second member discretization
   * @param dof degrees of freedom set
   * 
   */
  FEMDiscretization(const Problem& p,
		    MeshType& m,
		    BaseMatrix& a,
		    BaseVector& bb,
		    const DegreeOfFreedomSet& dof)
    : BaseFEMDiscretization<MeshType>(p, m, a, bb, dof)
  {
    ;
  }

  /** 
   * Virtual destructor
   * 
   */
  ~FEMDiscretization()
  {
    ;
  }
};


/**
 * This is specialisation in the particular case of Cartesian grids.
 * It is an important specialization since the elementary matrices are
 * computed \bf once for all!
 */
template <>
class FEMDiscretization<Structured3DMesh>
  : public BaseFEMDiscretization<Structured3DMesh>
{
public:
  /// The type of mesh used for discretization
  typedef Structured3DMesh MeshType;

  /// The geometry of finite elements
  typedef MeshType::ElementGeometry ElementGeometry;

  /// The finite element
  typedef FiniteElementTraits<ElementGeometry>::Type FiniteElement; /**< The finite element */

  /// Elementary matrices type
  typedef FiniteElement::ElementaryMatrix ElementaryMatrixType;

  /// Elementary vector type
  typedef FiniteElement::ElementaryVector ElementaryVectorType;

  /// type of transformation from reference element
  typedef FiniteElementTraits<ElementGeometry>::Transformation ConformTransformation;

  /// Associated jacobian
  typedef FiniteElementTraits<ElementGeometry>::JacobianTransformation JacobianTransformation;

public:
  /** 
   * Assembles the matrix associated to the PDE operators of the PDE
   * problem.
   * 
   */
  void assembleMatrix()
  {
    Timer t;

    ffout(3) << "Assembling matrix\n";
    t.start();
    
    switch ((__A).type()) {
    case BaseMatrix::doubleHashedMatrix: {

      DoubleHashedMatrix& A = dynamic_cast<DoubleHashedMatrix&>(__A);

      MeshType::iterator icell0(__mesh);
      ConformTransformation T0(*icell0);
      JacobianTransformation J(T0);
	generatesElementaryMatrix(__eSet,J);

	for(MeshType::const_iterator icell(__mesh);
	    not(icell.end()); ++icell) {
	  const ElementGeometry& K = *icell;
	  ConformTransformation T(K);
	  TinyVector<3> massCenter;
	  T.value(FiniteElement::massCenter(), massCenter);

	  size_t indices[FiniteElement::numberOfDegreesOfFreedom];
	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	    indices[l] = __mesh.vertexNumber(K(l));
	  }
	  for(DiscretizedOperators<ElementaryMatrixType>
		::iterator iOperator = __discretizedOperators.begin();
	      iOperator != __discretizedOperators.end(); ++iOperator) {

	    const real_t coef = ((*iOperator).second)(massCenter);

	    const ElementaryMatrixType& elementaryMatrix
	      = (*(*iOperator).first);

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	      const size_t I1 // Index of Ith value
		= __degreeOfFreedomSet(((*iOperator).second).i(), indices[l]);
	      for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; m++) {
		const size_t I2 // Index of Jth value
		  = __degreeOfFreedomSet(((*iOperator).second).j(), indices[m]);

		A(I1,I2) += coef*elementaryMatrix(l,m);
	      }
	    }
	  }
	}
	break;
    }
    case BaseMatrix::unAssembled: {
      UnAssembledMatrix& A = dynamic_cast<UnAssembledMatrix&>(__A);
      A.setDiscretization(this);
      break;
    }

    default: {
      fferr(2) << '\n' << __FILE__ << ':' << __LINE__
	       << ':' << "Not implemented\n";
      std::exit(1);
    }
    }
    ffout(2) << "done\n";
    t.stop();
    ffout(3) << "Assembling cost: " << t << '\n';
  }

  /** 
   *  Applies directly the operator discretization to the vector X.
   * 
   * @param X input vector
   * @param Z \f$ z=Ax \f$
   */
  void timesX(const BaseVector& X, BaseVector& Z) const
  {
    const Vector<real_t>& x = dynamic_cast<const Vector<real_t>&>(X);

    Vector<real_t>& z = dynamic_cast<Vector<real_t>&>(Z);
    z=0;
    MeshType::const_iterator icell(__mesh);
    ConformTransformation T0(*icell);
    JacobianTransformation J(T0);
    generatesElementaryMatrix(__eSet,J);

    for(MeshType::const_iterator icell(__mesh);
	not(icell.end()); ++icell) {
      const ElementGeometry& K = *icell;
      ConformTransformation T(K);

      size_t indices[FiniteElement::numberOfDegreesOfFreedom];
      for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	indices[l] = __mesh.vertexNumber(K(l));
      }

      TinyVector<3> massCenter;
      T.value(FiniteElement::massCenter(), massCenter);

      for(DiscretizedOperators<ElementaryMatrixType>
	    ::iterator iOperator = __discretizedOperators.begin();
	  iOperator != __discretizedOperators.end(); ++iOperator) {
	const real_t coef = ((*iOperator).second)(massCenter);

	const ElementaryMatrixType& elementaryMatrix = (*(*iOperator).first);

#warning can optimize this loops by precomputing (i and) j
	for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	  // Number of the degree of freedom
	  const size_t i = __degreeOfFreedomSet(((*iOperator).second).i(), indices[l]);
	  for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; m++) {
	    // Number of the degree of freedom
	    const size_t j = __degreeOfFreedomSet(((*iOperator).second).j(), indices[m]);
	    z[i] += coef*elementaryMatrix(l,m)*x[j];
	  }
	}
      }
    }
  }

  /** 
   * Computes diagonal of the operator
   * 
   * @param Z diagonal of the operator
   */
  void getDiagonal(BaseVector& Z) const
  {
    Vector<real_t>& z = dynamic_cast<Vector<real_t>&>(Z);
    z=0;
    MeshType::const_iterator icell0(__mesh);
    ConformTransformation T0(*icell0);
    JacobianTransformation J(T0);
    generatesElementaryMatrix(__eSet,J);

    for(MeshType::const_iterator icell(__mesh);
	not(icell.end()); ++icell) {
      const ElementGeometry& K = *icell;
      ConformTransformation T(K);

      size_t indices[FiniteElement::numberOfDegreesOfFreedom];
      for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	indices[l] = __mesh.vertexNumber(K(l));
      }

      TinyVector<3> massCenter;
      T.value(FiniteElement::massCenter(), massCenter);

      for (size_t i=0; i<problem().numberOfUnknown(); ++i) {

	for(DiscretizedOperators<ElementaryMatrixType>
	      ::iterator iOperator = __discretizedOperators.begin();
	    iOperator != __discretizedOperators.end(); ++iOperator) {
	  const real_t coef = ((*iOperator).second)(massCenter);

	  const ElementaryMatrixType& elementaryMatrix = (*(*iOperator).first);
	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; l++) {
	    const size_t I = // Index of Ith value
	      __degreeOfFreedomSet(((*iOperator).second).i(), indices[l]);
	    z[I] += coef*elementaryMatrix(l,l);
	  }
	}
      }
    }
  }

  /** 
   * Second member assembling
   * 
   */
  void assembleSecondMember()
  {
    ffout(2) << "Assembling Second Member\n";
    //! The elementary vector
    ElementaryVectorType eVect;
    Vector<real_t>& b = (static_cast<Vector<real_t>&>(__b));
    b = 0;

    switch (problem().type()) {
    case Problem::pdeProblem: {
      const PDESystem& pdeSystem
	= static_cast<const PDESystem&>(problem());

      for(MeshType::const_iterator icell(__mesh);
	    not(icell.end()); ++icell) {
	const ElementGeometry& K = *icell;
	eVect = 0;
	ConformTransformation T(K);
	JacobianTransformation J(T);
	TinyVector<FiniteElement::numberOfDegreesOfFreedom,int> index;
	TinyVector<FiniteElement::numberOfDegreesOfFreedom,TinyVector<3> > quadrature;

	for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	  quadrature[l] = T.quadratureVertex(l);
	  index[l] = __mesh.vertexNumber(K(l));
	}

	for (size_t i=0; i<pdeSystem.numberOfUnknown(); ++i) {
	  const UserFunction& F = pdeSystem.secondMember(i);

	  TinyVector<FiniteElement::numberOfDegreesOfFreedom> f;

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l)
	    f[l] = F(quadrature[l]);

	  generatesElementaryVector(eVect,J,f);

	  for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	    const size_t dof = __degreeOfFreedomSet(i,index[l]);
	    b[dof] += eVect[l]*J.jacobianDet();
	  }
	}
      }
      break;
    }
    case Problem::variationalProblem: {
      const VariationalProblem& variationalProblem
	=  dynamic_cast<const VariationalProblem&>(problem());

      for (VariationalProblem::linearOperatorConst_iterator i
	     = variationalProblem.beginLinearOperator();
	   i != variationalProblem.endLinearOperator(); ++i) {
	switch ((*(*i)).type()) {
	case VariationalLinearOperator::FV: {
	  const VariationalOperatorFV& fv
	    = dynamic_cast<const VariationalOperatorFV&>(*(*i));

	  const UserFunction& F = fv.f();
	  const size_t testFunctionNumber = fv.testFunctionNumber();

	  for(MeshType::const_iterator icell(__mesh);
	      not(icell.end()); ++icell) {
	    const ElementGeometry& K = *icell;
	    eVect = 0;
	    ConformTransformation T(K);
	    JacobianTransformation J(T);
	    TinyVector<FiniteElement::numberOfDegreesOfFreedom,int> index;
	    TinyVector<FiniteElement::numberOfDegreesOfFreedom,TinyVector<3> > quadrature;

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	      quadrature[l] = T.quadratureVertex(l);
	      index[l] = __mesh.vertexNumber(K(l));
	    }

	    ElementaryVectorType f;

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l)
	      f[l] = F(quadrature[l]);

	    generatesElementaryVector(eVect,J,f);

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	      const size_t dof = __degreeOfFreedomSet(testFunctionNumber,index[l]);
	      b[dof] += eVect[l]*J.jacobianDet();
	    }
	  }
	  break;
	}
	case VariationalLinearOperator::FdxGV: {
	  const VariationalOperatorFdxGV& fv
	    = dynamic_cast<const VariationalOperatorFdxGV&>(*(*i));

	  const UserFunction& F = fv.f();
	  const UserFunction& G = fv.g();
	  const size_t testFunctionNumber = fv.testFunctionNumber();

	  Vector<real_t> gValues (__mesh.numberOfVertices());

	  for (size_t i=0; i<__mesh.numberOfVertices(); ++i) {
	    gValues[i] = G(__mesh.vertex(i));
	  }

	  MeshType::const_iterator icell0(__mesh);
	  ConformTransformation T0(*icell0);
	  JacobianTransformation J(T0);

	  ElementaryMatrixType edxUV = 0;
	  generatesElementaryMatrix(PDEOperator::firstorderop,
				    J, edxUV, fv.number());

	  for(MeshType::const_iterator icell(__mesh);
	      not(icell.end()); ++icell) {
	    const ElementGeometry& K = *icell;

	    TinyVector<FiniteElement::numberOfDegreesOfFreedom, int> index;
	    TinyVector<FiniteElement::numberOfDegreesOfFreedom, TinyVector<3> > quadrature;

	    ConformTransformation T(K);

	    TinyVector<3> massCenter;
	    T.value(FiniteElement::massCenter(), massCenter);

	    const real_t fValue = F(massCenter);

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	      quadrature[l] = T.quadratureVertex(l);
	      index[l] = __mesh.vertexNumber(K(l));
	    }

	    for (size_t l=0; l<FiniteElement::numberOfDegreesOfFreedom; ++l) {
	      const size_t dof
		= __degreeOfFreedomSet(testFunctionNumber,index[l]);

	      for (size_t m=0; m<FiniteElement::numberOfDegreesOfFreedom; ++m) {
		b[dof] += edxUV(l,m)*gValues[index[m]]*fValue;
	      }
	    }
	  }
	  break;
	}
	default: {
	  fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
	  std::exit(1);
	}
	}
      }
      break;
    }
    default: {
      fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
      std::exit(1);
    }
    }
  }

public:
  /** 
   * Constructor of the discretization
   * 
   * @param p the problem
   * @param m the mesh used for discretization
   * @param a matrix storing discretization
   * @param bb vector that stores second member discretization
   * @param dof degrees of freedom set
   * 
   */
  FEMDiscretization(const Problem& p,
		    Structured3DMesh& m,
		    BaseMatrix& a,
		    BaseVector& bb,
		    const DegreeOfFreedomSet& dof)
    : BaseFEMDiscretization<Structured3DMesh>(p, m, a, bb, dof)
  {
    ;
  }

  /** 
   * destructor
   * 
   */
  ~FEMDiscretization()
  {
    ;
  }
};


#endif // FEM_DISCRETIZATION_HPP
