/* FIGARO'S PASSWORD MANAGER 2 (FPM2)
 * Copyright (C) 2000 John Conneely
 * Copyright (C) 2008 Aleš Koval
 * 
 * FPM is open source / free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * FPM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * fpm.c - general routines
 */


#include "fpm.h"
#include "fpm_gpw.h"
#include "passfile.h"
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "fpm_crypt.h"
#include <fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <X11/Xatom.h>

/* GLOBAL VARIABLES */
GtkWidget* glb_win_app;   /* The main window for the application   */
GtkTreeView* glb_clist_main; /* Password CList on main window      */

GtkWidget* glb_win_misc;  /* Misc window, used for dialogs         */

GtkWidget* glb_win_edit;  /* The edit window for a password item   */
GtkWidget* glb_win_import; /* The import window                    */
fpm_data* glb_edit_data;  /* The password item being edited now    */
gint glb_num_row;         /* The total number of rows in the CList */
gint glb_cur_row;         /* The last item clicked on in the CList */
gchar* glb_filename;      /* The location of the pasword file.     */
gboolean glb_dirty;       /* Have changes been made?               */
GTimer* glb_timer_click;  /* Timer used to check for double clicks */
gint glb_click_count;	  /* Click count used for double clicks    */
gchar* new_salt;	  /* Salt to use on next save.             */
gchar* old_salt;	  /* Salt to used on last save.            */
gboolean glb_click_btn;	  /* Right button clicked -> context menu  */
gint glb_click_row;       /* Store row for first click on dblclick */
gboolean glb_minimized;	  /* Is program minimized?		   */
gboolean glb_locked;	  /* Is program locked?			   */
gboolean glb_timeout; /* Running idle check?		   */
gboolean glb_lock_timeout; /* Running lock check?		   */
gint glb_idle_min;

gboolean glb_events;
gboolean glb_test;

GtkStatusIcon *glb_tray_icon; /* Tray icon */

fpm_ini* glb_fpm_ini;	  /* Store FPM settings. */

static gchar* glb_filename_lock; /* The location of the pasword loc kfile. */

#ifdef FPM_LINK_LOCK
static gchar* glb_filename_lock_pid; /* The location of the pasword loc kfile. */
#endif

static int lock_fd = -1;
#if defined(sun) && defined(__SVR4)
struct flock lck;
#endif

void* old_context;
void* new_context;

char* vstring;
char* file_version;

GList *glb_pass_list;	 /* A GList of fpm_data containing all items */
GList *glb_cat_string_list;	 /* A GList of strings, used to populate combos */
GList *glb_launcher_list;
GList *glb_launcher_string_list;

GCompareFunc glb_cmp;


/* Current versions of FPM encrypt all fields.  Fields other than the
 * password are decrypted when the program reads the password file,
 * and the password is decrypted as it is needed (to try to prevent it
 * from showing up in coredumps, etc.)  The global variable  glb_need_decrypt
 * is set by the passfile loading routines to specify that the passfile
 * was created with a version of FPM that requires this decryption.  
 * This allows backward compatibility with previous versions of FPM which
 * only encrypted passwords.
 */
gboolean glb_need_decrypt;

/* glb_using_defaults is true if the user has made at least one
 * password item a default.  If the user has done this, the
 * default category should be the opening category.  Otherwise,
 * the opening category will be <ALL CATEGORIES>.
 */
gboolean glb_using_defaults;
gboolean glb_quitting=FALSE;
gchar* glb_last_category;

void
lock_fpm_file(gchar* glb_filename)
{
	gchar *dia_str;
	gchar *cmd;
	gchar* pwd_dir_name = NULL;
	struct stat fs_stat;

        /* iif g_dirname goes away in one of the next Glib releases
         * we'll need to replace it by g_path_get_dirname */
        pwd_dir_name = g_dirname(glb_filename);

	/* Need the directory of the password file to create a lock.
         * This is called when fpm is being used for the first time.
	 */

	if ( stat(pwd_dir_name, &fs_stat) )
	{
		cmd = g_strdup_printf("mkdir %s", pwd_dir_name);
		fpm_execute_shell(cmd);
		g_free(cmd);
		cmd = g_strdup_printf("chmod 0700 %s", pwd_dir_name);
		fpm_execute_shell(cmd);
		g_free(cmd);
		/* Wait for the shell commands to finish */
		sleep(1);
	}

	/* Create lock file if one does not exist. */

	if ( ( lock_fd = open( glb_filename_lock, O_CREAT | O_RDONLY, 0600 )) < 0 )
	{
		dia_str = g_strdup_printf(_("Could not open %s"), 
			glb_filename_lock);
		fpm_message(GTK_WINDOW(glb_win_app), dia_str, GTK_MESSAGE_ERROR);

#ifdef FPM_LINK_LOCK
		unlink(glb_filename_lock_pid);
#endif
		perror(glb_filename_lock);
		exit(1);
	}

	/* Create an advisory lock. This will not work across nfs on multiple
	   system. This will work on nfs file on the same system. 
	*/

#if defined(sun) && defined(__SVR4)
	lck.l_type = F_WRLCK;	/* setting a write lock */
	lck.l_whence = 0;	/* offset l_start from beginning of file */
	lck.l_start = (off_t)0;	
	lck.l_len = (off_t)0;	/* until the end of the file */

	if ( (fcntl(lock_fd, F_SETLK ) <0) && (errno == EAGAIN || errno == EACCES) )
#else
	if ( flock(lock_fd, LOCK_EX | LOCK_NB) )
#endif
	{
		dia_str = g_strdup_printf(_("Could not lock %s\nFile already locked."), 
			glb_filename_lock);
		fpm_message(GTK_WINDOW(glb_win_app), dia_str, GTK_MESSAGE_ERROR);
		printf("%s\n", dia_str);
		exit(1);
	}

/*
 * Locking the file with hard links works across systems, but if the process
 * or system crash, the link will be left and the user will have to manually
 * remove the lock files to fix the problem. Don't think this is a good
 * option at this time.
*/

#ifdef FPM_LINK_LOCK
	if ( link(glb_filename_lock, glb_filename_lock_pid) )
	{
		dia_str = g_strdup_printf(_("Could not create a link to %s"), 
			glb_filename_lock);
		fpm_message(GTK_WINDOW(glb_win_app), dia_str, GTK_MESSAGE_ERROR);
		unlink(glb_filename_lock_pid);
		exit(1);
	}

	if ( fstat(lock_fd, &fs_stat) )
	{
		dia_str = g_strdup_printf(_("Could not fstat %s"), 
			glb_filename_lock);
		fpm_message(GTK_WINDOW(glb_win_app), dia_str, GTK_MESSAGE_ERROR);
		unlink(glb_filename_lock_pid);
		printf("Lock already exists\n");
		exit(1);
	}

	if ( fs_stat.st_nlink > 2 )
	{
		dia_str = g_strdup_printf(_("Lock file %s exists.\nIf you know "
			"this to be an error, clean up the\nlock files in"
			"the directory %s."), glb_filename_lock, pwd_dir_name);
		fpm_message(GTK_WINDOW(glb_win_app), dia_str, GTK_MESSAGE_ERROR);
		unlink(glb_filename_lock_pid);
		exit(1);
	}
#endif /* FPM_LINK_LOCK */

}

void
unlock_fpm_file(void)
{
	if ( lock_fd > -1 )
	{
		close(lock_fd);
	}
#ifdef FPM_LINK_LOCK
	unlink(glb_filename_lock_pid);
#else
	unlink(glb_filename_lock);
#endif /* FPM_LINK_LOCK */
}


void
fpm_new_passitem(GtkWidget** win_edit_ptr, fpm_data** data_ptr)
{
    fpm_data* new_data;
    gchar plaintext[FPM_PASSWORD_LEN+1] = {0};

    new_data = g_malloc0(sizeof(fpm_data));

    strncpy(plaintext, "", FPM_PASSWORD_LEN);
    fpm_encrypt_field(	old_context, new_data->password,
			plaintext, FPM_PASSWORD_LEN);
    new_data->title=g_strdup("");
    new_data->arg=g_strdup("");
    new_data->user=g_strdup("");
    new_data->notes=g_strdup("");

    if (g_utf8_collate(glb_last_category, FPM_ALL_CAT_MSG) == 0)
	new_data->category=g_strdup("");
    else
	new_data->category=g_strdup(glb_last_category);

    new_data->launcher=g_strdup("");
    *data_ptr = new_data;


    fpm_edit_passitem(win_edit_ptr, new_data);
}




void
fpm_edit_passitem(GtkWidget** win_edit_ptr, fpm_data* data)
{

  GtkCombo* combo;
  gchar cleartext[FPM_PASSWORD_LEN+1] = {0};
  GtkTextView* notes;
  GtkTextBuffer*  buffer;
  int ix;

  /* Create new dialog box */
  *win_edit_ptr = create_dialog_edit_passitem();

  /* Populate drop down boxes on edit screen */
  fpm_create_category_list(1);
  combo = GTK_COMBO(lookup_widget(*win_edit_ptr, "combo_item_category"));
  g_assert(glb_cat_string_list!=NULL);
  gtk_combo_set_popdown_strings(combo, glb_cat_string_list);

  fpm_create_launcher_string_list();
  g_assert(glb_launcher_string_list!=NULL);
  combo = GTK_COMBO(lookup_widget(*win_edit_ptr, "combo_edit_launcher"));
  if(glb_launcher_string_list!=NULL)
    gtk_combo_set_popdown_strings(combo, glb_launcher_string_list);

  /* Load the data into the dialog box */
  fpm_set_entry(*win_edit_ptr, "entry_title", data->title);
  fpm_set_entry(*win_edit_ptr, "entry_arg", data->arg);
  fpm_set_entry(*win_edit_ptr, "entry_user", data->user);
  fpm_set_entry(*win_edit_ptr, "entry_edit_category", data->category);
  fpm_set_entry(*win_edit_ptr, "entry_edit_launcher", data->launcher);
  gtk_toggle_button_set_active(
	GTK_TOGGLE_BUTTON(lookup_widget(*win_edit_ptr, "checkbutton_default")),
	data->default_list);

  ix = strlen(data->password) / 2;
  fpm_decrypt_field(old_context, cleartext, data->password, ix);
  fpm_set_entry(*win_edit_ptr, "entry_password", cleartext);
  memset(cleartext, 0, FPM_PASSWORD_LEN);

  notes = GTK_TEXT_VIEW(lookup_widget(*win_edit_ptr, "text_notes"));

  buffer = gtk_text_view_get_buffer(notes);
  gtk_text_buffer_insert_at_cursor(buffer, data->notes ,strlen(data->notes));

  gtk_window_set_transient_for (GTK_WINDOW(*win_edit_ptr), GTK_WINDOW(glb_win_app));
  gtk_widget_show(*win_edit_ptr);
  gtk_widget_set_sensitive(glb_win_app, FALSE);

}


void
fpm_save_passitem(GtkWidget* win_edit, fpm_data* data)
{
  gint row = -1;
  gchar* orig_plaintext;
  gchar plaintext[FPM_PASSWORD_LEN+1] = {0};
  GtkEntry* entry;
  GtkTextView* notes;
  gchar *dia_str;

  fpm_data* data1;

  GtkTextIter start, end;
  GtkTextBuffer *buffer;
  GtkTreeIter iter;
  GtkTreeModel *model;
  gboolean valid;
  
  /* Set dirty flag so we know we need to save later. */
  glb_dirty = TRUE;

  /* First update data structures */
  data->title = fpm_get_entry(win_edit, "entry_title");
  data->arg = fpm_get_entry(win_edit, "entry_arg");
  data->user = fpm_get_entry(win_edit, "entry_user");
  data->category = fpm_get_entry(win_edit, "entry_edit_category");
  data->launcher = fpm_get_entry(win_edit, "entry_edit_launcher");

  /* Update default check box */
  data->default_list = (GTK_TOGGLE_BUTTON(lookup_widget(win_edit, "checkbutton_default"))->active);

  /* Update password */
  entry = GTK_ENTRY(lookup_widget(win_edit, "entry_password"));

  orig_plaintext = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);

  if ( strlen(orig_plaintext) > (FPM_PASSWORD_LEN -1 ))
  {
  	dia_str = g_malloc0(256);
	sprintf(dia_str, _("Password exceeded limit of %d characters"
		"\nYour password of %d characters was truncated."), 
		FPM_PASSWORD_LEN-1, (int) strlen(orig_plaintext));

	fpm_message(GTK_WINDOW(win_edit), dia_str, GTK_MESSAGE_INFO);
	g_free(dia_str);

  }
  memset(plaintext, 0, FPM_PASSWORD_LEN);
  strncpy(plaintext, orig_plaintext, FPM_PASSWORD_LEN-1);
  fpm_encrypt_field(old_context, data->password, plaintext, FPM_PASSWORD_LEN);
  memset(orig_plaintext, 0, strlen(orig_plaintext));
  memset(plaintext, 0, FPM_PASSWORD_LEN);

  g_free(orig_plaintext);

  /* Update notes */
  notes = GTK_TEXT_VIEW(lookup_widget(win_edit, "text_notes"));
  gtk_text_view_set_editable (GTK_TEXT_VIEW (notes), TRUE);
  g_free(data->notes);
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (notes));
  gtk_text_buffer_get_bounds (buffer, &start, &end);
  data->notes = gtk_text_iter_get_text (&start, &end);
     
  model = gtk_tree_view_get_model(glb_clist_main);
  valid = gtk_tree_model_get_iter_first(model, &iter);

  while (valid) {
    gtk_tree_model_get(model, &iter, FPM_COL_DATA_POINTER, &data1, -1);
    if( data == data1 ) {
	row = atoi(gtk_tree_model_get_string_from_iter(model, &iter));
    }
    valid = gtk_tree_model_iter_next (model, &iter);
  }

  if (row<0)
  {
    /* Keep track of the number of rows in the CList */
    glb_num_row++;

    /* Update the current row */
    row = glb_num_row-1;

    /* Also, add this row to our main GList */
    glb_pass_list=g_list_append(glb_pass_list, data);

    g_print("inserting...\n");

    if (glb_fpm_ini->save_on_add)
	passfile_save(glb_filename);
  }
  else
  { 
    g_print("updating...\n");
    if (glb_fpm_ini->save_on_change)
	passfile_save(glb_filename);
  }

  /* Update Category drop down list on main screen */
  fpm_clist_populate_cat_list();
}


char*
fpm_get_entry(GtkWidget* win, gchar* entry_name)
{
  gchar* tmp;
  GtkEntry* entry_field;

  entry_field = GTK_ENTRY(lookup_widget (win, entry_name));
  tmp = gtk_editable_get_chars (GTK_EDITABLE(entry_field), 0, -1);
  return (g_strdup(tmp));
}


void
fpm_set_entry(GtkWidget* win, gchar* entry_name, char* text)
{

  GtkEntry* entry_field;

  entry_field = GTK_ENTRY(lookup_widget (win, entry_name));
  gtk_entry_set_text(entry_field, text);
}

void
fpm_init(char* opt_file_name)
{
#ifdef FPM_LINK_LOCK
	char hostname[256];
#endif

  glb_launcher_string_list=NULL;

  fpm_ini_load();

  if( glb_win_app == NULL)
  {
	/* First run */
	glb_win_app = create_app_safe();
	gtk_widget_hide (GTK_WIDGET(lookup_widget (glb_win_app , "hide_to_tray")));	
  }
  else
	gtk_widget_hide(glb_win_app);

  if( opt_file_name )
    glb_filename = opt_file_name;
  else
    glb_filename=g_strdup_printf("%s/.fpm/fpm", g_get_home_dir());

  /* Lock filename. Should be the same directory as the password file. */

  glb_filename_lock = g_strdup_printf("%s.lock", glb_filename);

/* LINK_LOCK is for hard links. Creates a cleanup problem. */

#ifdef FPM_LINK_LOCK
  gethostname(&hostname, (size_t) 255);
  glb_filename_lock_pid = g_strdup_printf("%s.lock.%s.%d", glb_filename,
  	hostname, getpid());
#endif

//  glb_edit_data = g_malloc0(sizeof(fpm_data));
  glb_timer_click = g_timer_new();
  g_timer_start(glb_timer_click);
  glb_click_count=-1;
  glb_need_decrypt=FALSE;
  fpm_gpw_set_options(8, TRUE, TRUE, TRUE, FALSE, TRUE);

  glb_clist_main = GTK_TREE_VIEW(lookup_widget (glb_win_app, "clist_main"));
  if(gtk_tree_view_get_model(glb_clist_main) == NULL) fpm_clist_init();
  
  /* Lock the password file */

    lock_fpm_file(glb_filename);

 /* Now done in load passfile 
  * fpm_init_launchers();
  */

  if(passfile_load(glb_filename))
  {
    glb_win_misc = create_dialog_cpw();
    gtk_widget_show(glb_win_misc);
  }
  else
  {
    glb_win_misc = create_dialog_password();
    gtk_widget_show(glb_win_misc);
  }
}

void
fpm_quit(void)
{
  glb_quitting=TRUE;
  if (glb_dirty) {
    if (glb_fpm_ini->save_on_quit)
	passfile_save(glb_filename);		
    else
	fpm_question(GTK_WINDOW(glb_win_app), _("Do you want to save changes?"),
		 G_CALLBACK(fpm_dialog_answer_save));
  }

  if (glb_fpm_ini->enable_tray_icon)
    glb_tray_icon = (g_object_unref (glb_tray_icon), NULL);

  fpm_ini_save ();

  gtk_main_quit();

}

void
fpm_dialog_answer_edit(GtkWidget *widget, gint reply)
{
    if (reply == GTK_RESPONSE_YES) {
	gtk_widget_destroy(glb_win_edit);
	gtk_widget_set_sensitive(glb_win_app, TRUE);
    }
}

void
fpm_dialog_answer_save(GtkWidget *widget, gint reply)
{
   if (reply == GTK_RESPONSE_YES)
   {
     passfile_save(glb_filename);
   }

   if (glb_quitting) gtk_main_quit();
}

void
fpm_dialog_answer_delete(GtkWidget *widget, gint reply)
{
  GtkTreeIter iter;
  GtkTreeModel *model;

  if (reply == GTK_RESPONSE_YES)
  {
    glb_pass_list=g_list_remove(glb_pass_list, glb_edit_data);
    model = gtk_tree_view_get_model(glb_clist_main);
    gtk_tree_model_get_iter_from_string(model, &iter, g_strdup_printf("%i",glb_cur_row));
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    glb_num_row--;

    glb_edit_data = NULL;
    glb_dirty = TRUE;
    if (glb_fpm_ini->save_on_delete)
	passfile_save(glb_filename);
  }
}


void
fpm_select(gchar* text, gboolean use_clipboard)
{

  g_assert(text != NULL);

  gtk_clipboard_set_text(gtk_clipboard_get(GDK_SELECTION_PRIMARY), text, -1);
  if (use_clipboard)
  gtk_clipboard_set_text(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD), text, -1);

}

void
fpm_select_password(fpm_data* data, gboolean use_clipboard)
{
  gchar plaintext[FPM_PASSWORD_LEN+1] = {0};
  int ix;
  g_assert(data != NULL);

  ix = strlen(data->password)/2;
  fpm_decrypt_field(old_context, plaintext, data->password, ix);
  fpm_select(plaintext, use_clipboard);
  memset(plaintext, 0, FPM_PASSWORD_LEN);
}

void
fpm_double_click(fpm_data* data)
{
  if (data != NULL)
  {
    fpm_jump(data);
  }


}

void
fpm_jump(fpm_data* data)
{
  gchar* cmd;
  fpm_launcher *launcher;
  fpm_launcher *tmp;
  GList *list;

  launcher=NULL;
  list=g_list_first(glb_launcher_list);
  while (list!=NULL)
  {
    tmp=list->data;
    if (!strcmp(tmp->title, data->launcher)) launcher=tmp;
    list=g_list_next(list);
  }

  if (launcher )
  {

  /* Check for items that should be on clipboard or primary selection */
    if(launcher->copy_password == 2)
    {
      fpm_select_password(data, TRUE);
    }
    if(launcher->copy_user == 2) 
    {
      fpm_select(data->user, TRUE);
    }
    if(launcher->copy_user == 1)
    {
      fpm_select(data->user, FALSE);
    }
    if(launcher->copy_password == 1)
    {
      fpm_select_password(data, FALSE);
    }

    if ( strlen(launcher->cmdline) > 0 )
    {
        cmd = fpm_create_cmd_line(launcher->cmdline, data->arg,
		 data->user, data->password);
	fpm_execute_shell(cmd);
    }
    else
    {
    fpm_message(GTK_WINDOW(glb_win_app), _("This password's launcher command is undefined.\nPlease define it in the edit launcher preference screen."), GTK_MESSAGE_WARNING);
    }
  }
  else
    fpm_message(GTK_WINDOW(glb_win_app), _("This password's launcher is undefined.\nPlease define it in the edit password item screen."), GTK_MESSAGE_WARNING);

}

void
fpm_check_password()
{
  GtkEntry* entry;
  GtkLabel* label;

  char vstring_plain[9];

  entry=GTK_ENTRY(lookup_widget(glb_win_misc, "entry_password"));

  fpm_crypt_init(gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1));

  fpm_decrypt_field(old_context, vstring_plain, vstring, 8);
  if (strcmp(vstring_plain, "FIGARO"))
  {
    gtk_editable_select_region(GTK_EDITABLE(entry), 0, -1);
    label=GTK_LABEL(lookup_widget(glb_win_misc, "label_password_message"));
    gtk_widget_show(GTK_WIDGET(label));
    gtk_label_set_text(label, _("Password incorrect.  Please try again."));
    return;
  }

  if (glb_need_decrypt)
  {
    fpm_decrypt_all();
    glb_need_decrypt=FALSE;
  }
/*
  fpm_clist_create_view(NULL);
 */

  if (glb_using_defaults)
  {
    glb_last_category=g_strdup(FPM_DEFAULT_CAT_MSG);
  }
  else
  {
    glb_last_category=g_strdup(FPM_ALL_CAT_MSG);
  }

  glb_last_category = glb_fpm_ini->last_category;

  fpm_clist_populate_cat_list();

  gtk_window_resize (GTK_WINDOW(glb_win_app), glb_fpm_ini->main_width, glb_fpm_ini->main_height);
  if ((glb_fpm_ini->main_x != -1) && (glb_fpm_ini->main_y != -1))
    gtk_window_move(GTK_WINDOW(glb_win_app), glb_fpm_ini->main_x, glb_fpm_ini->main_y);

  if (glb_fpm_ini->enable_tray_icon)
    fpm_tray_icon();

  gtk_window_set_skip_taskbar_hint(GTK_WINDOW(glb_win_app), FALSE);
  gtk_window_set_skip_pager_hint(GTK_WINDOW(glb_win_app), FALSE);
  gtk_window_present(GTK_WINDOW(glb_win_app));
  gtk_window_deiconify(GTK_WINDOW(glb_win_app));

  gtk_widget_destroy(glb_win_misc);

  glb_minimized = FALSE;
}


void fpm_set_password()
{
  GtkEntry *entry;
  gchar *pw1, *pw2;
  byte *md5val, *prehash;

  entry=GTK_ENTRY(lookup_widget(glb_win_misc, "entry_cpw1"));
  pw1 = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
  entry=GTK_ENTRY(lookup_widget(glb_win_misc, "entry_cpw2"));
  pw2 = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
  
  if (strcmp(pw1, pw2))
  {
    fpm_message(GTK_WINDOW(glb_win_misc), _("Sorry, the passwords do not match."), GTK_MESSAGE_ERROR);
    return;
  }
  if (strlen(pw1)<4)
  {
    fpm_message(GTK_WINDOW(glb_win_misc), _("Sorry, your password is too short."), GTK_MESSAGE_ERROR);
    return;
  }

  if(old_context == NULL)
  {
    /* This is the first time we are running the app.*/
    glb_last_category=g_strdup(FPM_ALL_CAT_MSG);
 /*
    fpm_init_launchers();
  */

    fpm_crypt_init(pw1);
    glb_dirty = TRUE;
    gtk_widget_destroy(glb_win_misc);
    gtk_widget_show(glb_win_app);

    fpm_clist_populate_cat_list();
  }
  else
  {
    /* We are chaning the password.*/
    new_salt=get_new_salt();
    prehash=(guchar *) g_strdup_printf("%s%s", new_salt, pw1);
    md5val=md5(prehash, strlen((gchar *) prehash));
    fpm_setkey(new_context, md5val, 16);
    g_free(md5val);
    gtk_widget_destroy(glb_win_misc);

    passfile_save(glb_filename);
  }
}

void fpm_clear_list()
{
  GList * list;

  list=g_list_first(glb_pass_list);
  while (list!= NULL)
  {
    g_free(list->data);
    list=g_list_next(list);
  }
  g_list_free(glb_pass_list);
  glb_pass_list=NULL;
}

void fpm_init_launchers()
{
  GList * list;
  fpm_launcher* launcher;

  list = NULL;

  printf("Initiating default launchers...\n");

  launcher = g_malloc0(sizeof(fpm_launcher));
  launcher->title=g_strdup("Web");
  launcher->cmdline=g_strdup("gnome-moz-remote \"$a\"");
  launcher->copy_user = 2;
  launcher->copy_password = 1;
  list = g_list_append(list, launcher);

  launcher = g_malloc0(sizeof(fpm_launcher));
  launcher->title=g_strdup("ssh");
  launcher->cmdline=g_strdup("gnome-terminal -e 'ssh $a -l $u'");
  launcher->copy_user = 0;
  launcher->copy_password = 1;
  list = g_list_append(list, launcher);

  launcher = g_malloc0(sizeof(fpm_launcher));
  launcher->title=g_strdup("Generic Command");
  launcher->cmdline=g_strdup("$a");
  launcher->copy_user = 0;
  launcher->copy_password = 0;
  list = g_list_append(list, launcher);

  glb_launcher_list=list;

  fpm_create_launcher_string_list();
}

void fpm_create_launcher_string_list()
{
  GList * list;
  fpm_launcher* launcher;

  g_assert(glb_launcher_list!=NULL);
  /* FIXME:  I think you need the following free statement, but it seg faults
   * when I enable it.  Why?  If we need this statement, then the current
   * app will have a mem leak.  How big a deal is that?
   * g_free(glb_launcher_string_list);
   */
  glb_launcher_string_list=NULL;
  list=g_list_first(glb_launcher_list);
  while (list!= NULL)
  {
    launcher=list->data;
    glb_launcher_string_list=g_list_append(glb_launcher_string_list, g_strdup(launcher->title));
    list=g_list_next(list);
  }
}

gchar* fpm_create_cmd_line(gchar* cmd, gchar* arg, gchar* user, gchar* pass)
{
  gchar* result;
  gchar* tmp;
  gchar** cmd_arr;
  gint num, i;
  gchar* cleartext;

  cmd_arr = g_strsplit(cmd, "$", 0);
  num=sizeof(cmd_arr)/sizeof(char);
  result=g_strdup(cmd_arr[0]);
  i=1;
  while (cmd_arr[i]!=NULL)
  {
    tmp=result;
    if(cmd_arr[i][0]=='a')
      result=g_strconcat(tmp, arg, cmd_arr[i]+1, NULL);
    else if(cmd_arr[i][0]=='u')
      result=g_strconcat(tmp, user, cmd_arr[i]+1, NULL);
    else if(cmd_arr[i][0]=='p')
    {
      cleartext=fpm_decrypt_field_var(old_context, pass);
      result=g_strconcat(tmp, cleartext, cmd_arr[i]+1, NULL);
      memset(cleartext, 0, strlen(cleartext));
      g_free(cleartext);
    }
    else
      result=g_strconcat(tmp, "$", cmd_arr[i], NULL);
    memset(tmp, 0, strlen(tmp));
    g_free(tmp);
    i++;
  }
  g_strfreev(cmd_arr);

  return(result);
}

void fpm_debug(gchar* msg)
{
  printf("Debug %s:%d:%d\n", msg, (glb_launcher_list==NULL), (glb_launcher_string_list==NULL));

}
                            
void fpm_message(GtkWindow* win, gchar* message, GtkMessageType message_type)
{
    GtkWidget *dw;
    dw = gtk_message_dialog_new (GTK_WINDOW(win),
                              GTK_DIALOG_DESTROY_WITH_PARENT,
                              message_type,
                              GTK_BUTTONS_CLOSE, 
                              message);
    gtk_window_set_title (GTK_WINDOW(dw), _("Information"));
    gtk_dialog_run (GTK_DIALOG (dw));
    gtk_widget_destroy (dw);
}

void fpm_question(GtkWindow* win, gchar* message, GCallback callback)
{
    GtkWidget *dw;
    dw = gtk_message_dialog_new (GTK_WINDOW(win),
                        GTK_DIALOG_DESTROY_WITH_PARENT,
                        GTK_MESSAGE_QUESTION,
			GTK_BUTTONS_YES_NO,
                        message);
    gtk_window_set_title (GTK_WINDOW(dw),_("Question"));

    g_signal_connect (dw,
            "response", 
	     G_CALLBACK (callback),
    	     NULL);

    gint result = gtk_dialog_run (GTK_DIALOG (dw));
    if(result != GTK_RESPONSE_NONE) gtk_widget_destroy (dw);
}

void fpm_ini_load()
{
    GKeyFile *keyfile;
    GError *error = NULL;
    gsize count;
    gint i;
    gint *columns;

    glb_fpm_ini = g_malloc0(sizeof(fpm_ini));

    gchar *fpm_ini_file = g_build_filename (g_get_home_dir(), ".fpm", "fpm.ini", NULL);

    keyfile = g_key_file_new ();
    if (!g_key_file_load_from_file (keyfile, fpm_ini_file, G_KEY_FILE_NONE, NULL))
    {
	/* No ini file found -> set defaults and save */
	glb_fpm_ini->save_on_add = TRUE;
	glb_fpm_ini->save_on_change = TRUE;
	glb_fpm_ini->save_on_delete = FALSE;
	glb_fpm_ini->save_on_quit = FALSE;
	glb_fpm_ini->create_backup = TRUE;
	glb_fpm_ini->number_backup_files = 5;
	glb_fpm_ini->main_x = -1;
	glb_fpm_ini->main_y = -1;
	glb_fpm_ini->main_width = 500;
	glb_fpm_ini->main_height = 350;
	glb_fpm_ini->last_category = FPM_ALL_CAT_MSG;
	glb_fpm_ini->search_in_title = TRUE;
	glb_fpm_ini->search_in_url = FALSE;
	glb_fpm_ini->search_in_user = FALSE;
	glb_fpm_ini->search_in_notes = FALSE;
	glb_fpm_ini->search_match_case = FALSE;
	glb_fpm_ini->search_limit_category = TRUE;
	glb_fpm_ini->enable_tray_icon = FALSE;
	glb_fpm_ini->tr_always_visible = FALSE;
	glb_fpm_ini->tr_auto_hide = FALSE;
	glb_fpm_ini->tr_auto_hide_minutes = 15;
	glb_fpm_ini->tr_auto_lock = FALSE;
	glb_fpm_ini->tr_auto_lock_minutes = 15;

	glb_last_category = FPM_ALL_CAT_MSG;

	fpm_ini_save();
    } else {
	glb_fpm_ini->save_on_add = g_key_file_get_boolean (keyfile, "general", "save_on_add", NULL);
        glb_fpm_ini->save_on_delete = g_key_file_get_boolean (keyfile, "general", "save_on_delete", NULL);
        glb_fpm_ini->save_on_quit = g_key_file_get_boolean (keyfile, "general", "save_on_quit", NULL);
        glb_fpm_ini->save_on_change = g_key_file_get_boolean (keyfile, "general", "save_on_change", NULL);
        glb_fpm_ini->create_backup = g_key_file_get_boolean (keyfile, "general", "create_backup", NULL);
        glb_fpm_ini->number_backup_files = g_key_file_get_integer (keyfile, "general", "number_backup_files", NULL);

        glb_fpm_ini->last_category = g_key_file_get_string (keyfile, "general", "last_category", &error);
	if (error != NULL)
	    glb_fpm_ini->last_category = FPM_ALL_CAT_MSG;

	error = NULL;
	glb_fpm_ini->main_x = g_key_file_get_integer (keyfile, "general", "main_x", &error);
	if (error != NULL)
	    glb_fpm_ini->main_x = -1;
	error = NULL;
	glb_fpm_ini->main_y = g_key_file_get_integer (keyfile, "general", "main_y", &error);
	if (error != NULL)
	    glb_fpm_ini->main_y = -1;

	error = NULL;
	glb_fpm_ini->main_width = g_key_file_get_integer (keyfile, "general", "main_width", &error);
	if (error != NULL)
	    glb_fpm_ini->main_width = 500;

	error = NULL;
	glb_fpm_ini->main_height = g_key_file_get_integer (keyfile, "general", "main_height", &error);
	if (error != NULL)
	    glb_fpm_ini->main_height = 350;

	error = NULL;
	columns = g_key_file_get_integer_list (keyfile, "general", "columns_width", &count, &error);
	if (error != NULL || count != FPM_NUM_COL || columns == NULL) {
	    for (i=0; i< FPM_NUM_COL; i++) 
		glb_fpm_ini->columns_width[i] = 150;
	} else {
	    memcpy(glb_fpm_ini->columns_width, columns, sizeof(glb_fpm_ini->columns_width));
	    g_free(columns);
	}

	error = NULL;
	glb_fpm_ini->search_in_title = g_key_file_get_boolean (keyfile, "search", "search_in_title", &error);
	if (error != NULL)
	    glb_fpm_ini->search_in_title = TRUE;

	glb_fpm_ini->search_in_url = g_key_file_get_boolean (keyfile, "search", "search_in_url", NULL);
	glb_fpm_ini->search_in_user = g_key_file_get_boolean (keyfile, "search", "search_in_user", NULL);
	glb_fpm_ini->search_in_notes = g_key_file_get_boolean (keyfile, "search", "search_in_notes", NULL);
	glb_fpm_ini->search_match_case = g_key_file_get_boolean (keyfile, "search", "search_match_case", NULL);

	error = NULL;
	glb_fpm_ini->search_limit_category = g_key_file_get_boolean (keyfile, "search", "search_limit_category", &error);
	if (error != NULL)
	    glb_fpm_ini->search_limit_category = TRUE;

	glb_last_category = glb_fpm_ini->last_category;

	error = NULL;
	glb_fpm_ini->enable_tray_icon = g_key_file_get_boolean (keyfile, "tray_icon", "enable_tray_icon", &error);
	if (error != NULL)
	    glb_fpm_ini->enable_tray_icon = FALSE;

	glb_fpm_ini->tr_always_visible = g_key_file_get_boolean (keyfile, "tray_icon", "tr_always_visible", NULL);
	glb_fpm_ini->tr_auto_hide = g_key_file_get_boolean (keyfile, "tray_icon", "tr_auto_hide", NULL);

	error = NULL;
	glb_fpm_ini->tr_auto_hide_minutes = g_key_file_get_integer (keyfile, "tray_icon", "tr_auto_hide_minutes", &error);
	if (error != NULL)
	    glb_fpm_ini->tr_auto_hide_minutes = 15;

	glb_fpm_ini->tr_auto_lock = g_key_file_get_boolean (keyfile, "tray_icon", "tr_auto_lock", NULL);

	error = NULL;
	glb_fpm_ini->tr_auto_lock_minutes = g_key_file_get_integer (keyfile, "tray_icon", "tr_auto_lock_minutes", &error);
	if (error != NULL)
	    glb_fpm_ini->tr_auto_lock_minutes = 15;

    }

    g_key_file_free (keyfile);
}

void fpm_ini_save()
{
    GKeyFile *keyfile;
    GtkTreeViewColumn *column;
    gint x, y, width, height;
    gint i;

    if (glb_win_app == NULL) {
	x = -1;
	y = -1;
	width = 500;
	height = 350;
	for (i=0; i< FPM_NUM_COL; i++) 
	    glb_fpm_ini->columns_width[i] = 150;
    } else {
	gtk_window_get_size (GTK_WINDOW(glb_win_app), &width, &height);
	gtk_window_get_position(GTK_WINDOW(glb_win_app), &x, &y);

	for (i=0; i< FPM_NUM_COL; i++) {
	    column = gtk_tree_view_get_column (glb_clist_main, i);
	    glb_fpm_ini->columns_width[i] = gtk_tree_view_column_get_width (column);
	}
    }

    gchar *fpm_ini_file = g_build_filename (g_get_home_dir(), ".fpm", "fpm.ini", NULL);

    keyfile = g_key_file_new ();

    g_key_file_set_boolean (keyfile, "general", "save_on_add", glb_fpm_ini->save_on_add);
    g_key_file_set_boolean (keyfile, "general", "save_on_change", glb_fpm_ini->save_on_change);
    g_key_file_set_boolean (keyfile, "general", "save_on_delete", glb_fpm_ini->save_on_delete);
    g_key_file_set_boolean (keyfile, "general", "save_on_quit", glb_fpm_ini->save_on_quit);
    g_key_file_set_boolean (keyfile, "general", "create_backup", glb_fpm_ini->create_backup);
    g_key_file_set_integer (keyfile, "general", "number_backup_files", glb_fpm_ini->number_backup_files);
    g_key_file_set_string (keyfile, "general", "last_category", glb_last_category);
    g_key_file_set_integer (keyfile, "general", "main_x", x);
    g_key_file_set_integer (keyfile, "general", "main_y", y);
    g_key_file_set_integer (keyfile, "general", "main_width", width);
    g_key_file_set_integer (keyfile, "general", "main_height", height);
    g_key_file_set_integer_list (keyfile, "general", "columns_width", glb_fpm_ini->columns_width, 3);

    g_key_file_set_boolean (keyfile, "search", "search_in_title", glb_fpm_ini->search_in_title);
    g_key_file_set_boolean (keyfile, "search", "search_in_url", glb_fpm_ini->search_in_url);
    g_key_file_set_boolean (keyfile, "search", "search_in_user", glb_fpm_ini->search_in_user);
    g_key_file_set_boolean (keyfile, "search", "search_in_notes", glb_fpm_ini->search_in_notes);
    g_key_file_set_boolean (keyfile, "search", "search_match_case", glb_fpm_ini->search_match_case);
    g_key_file_set_boolean (keyfile, "search", "search_limit_category", glb_fpm_ini->search_limit_category);

    g_key_file_set_boolean (keyfile, "tray_icon", "enable_tray_icon", glb_fpm_ini->enable_tray_icon);
    g_key_file_set_boolean (keyfile, "tray_icon", "tr_always_visible", glb_fpm_ini->tr_always_visible);
    g_key_file_set_boolean (keyfile, "tray_icon", "tr_auto_hide", glb_fpm_ini->tr_auto_hide);
    g_key_file_set_integer (keyfile, "tray_icon", "tr_auto_hide_minutes", glb_fpm_ini->tr_auto_hide_minutes);
    g_key_file_set_boolean (keyfile, "tray_icon", "tr_auto_lock", glb_fpm_ini->tr_auto_lock);
    g_key_file_set_integer (keyfile, "tray_icon", "tr_auto_lock_minutes", glb_fpm_ini->tr_auto_lock_minutes);

    gchar *save_data = g_key_file_to_data (keyfile, NULL, NULL);

    g_file_set_contents (fpm_ini_file, save_data, -1, NULL);
    
    g_key_file_free (keyfile);
}

void fpm_search(gchar *search_text, gboolean select_first)
{
    fpm_data *data;
    GList *list;
    GtkTreeIter iter;
    GtkTreeModel *list_store;
    gchar *title, *url, *user, *notes;

    if (!glb_fpm_ini->search_match_case)
	search_text = g_utf8_casefold(search_text, -1);

    list_store = gtk_tree_view_get_model (glb_clist_main);
    gtk_list_store_clear(GTK_LIST_STORE(list_store));

    list = g_list_first(glb_pass_list);
    while (list != NULL)
    {
	data = list->data;
        title = data->title;
        url = data->arg;
	user = data->user;
	notes = data->notes;

        if ((!strcmp(glb_last_category, data->category) && glb_fpm_ini->search_limit_category)
	    || ((!strcmp(glb_last_category, FPM_ALL_CAT_MSG)
	    || !glb_fpm_ini->search_limit_category)))
	{

	    if (!glb_fpm_ini->search_match_case) {
		title = g_utf8_casefold(title, -1);
		url = g_utf8_casefold(url, -1);
		user = g_utf8_casefold(user, -1);
		notes = g_utf8_casefold(notes, -1);
	    }

	    if ((strstr(title, search_text) && glb_fpm_ini->search_in_title)
		|| (strstr(url, search_text) && glb_fpm_ini->search_in_url)
		|| (strstr(user, search_text) && glb_fpm_ini->search_in_user)
		|| (strstr(notes, search_text) && glb_fpm_ini->search_in_notes))
	    {

        	gtk_list_store_append (GTK_LIST_STORE(list_store), &iter);

		gtk_list_store_set(GTK_LIST_STORE(list_store), &iter,
	    	    FPM_COL_TITLE, data->title,
    		    FPM_COL_URL, data->arg,
	    	    FPM_COL_USER, data->user,
	    	    FPM_COL_DATA_POINTER, data,
	    	    -1);
	    }
	}
	list = g_list_next(list);
    }
    
    if (select_first)
	gtk_widget_grab_focus(GTK_WIDGET(glb_clist_main));
}

void fpm_execute_shell(gchar *cmd)
{
    gchar *execute;
    int ret;

    execute = g_strjoin(NULL, cmd, " &", NULL);
    ret = system(execute);
}

void fpm_statusbar_push(gchar *message)
{
    GtkStatusbar *statusbar;

    statusbar = GTK_STATUSBAR(lookup_widget (glb_win_app, "statusbar"));
    gtk_statusbar_pop(statusbar, 1);
    gtk_statusbar_push(statusbar, 1, message);
}

gboolean fpm_auto_hide() {

    fpm_tr_toggle_win_app(TRUE);

    return FALSE;
}

void fpm_tray_icon() {

    if (glb_tray_icon == NULL) {
	glb_tray_icon = gtk_status_icon_new();

	gtk_status_icon_set_visible(glb_tray_icon, FALSE);
	g_signal_connect(G_OBJECT(glb_tray_icon), "activate", 
			G_CALLBACK(tray_icon_on_click), NULL);
	g_signal_connect(G_OBJECT(glb_tray_icon), "popup-menu",
			G_CALLBACK(tray_icon_on_menu), NULL);
	gtk_status_icon_set_from_file (glb_tray_icon, PACKAGE_DATA_DIR "/" PACKAGE "/pixmaps/logo.png");
    }

    gtk_status_icon_set_tooltip(glb_tray_icon, _("Figaro's Password Manager 2"));
    gtk_status_icon_set_visible(glb_tray_icon, glb_fpm_ini->tr_always_visible);
    gtk_widget_show (GTK_WIDGET(lookup_widget (glb_win_app , "hide_to_tray")));
}

gboolean fpm_hide_win_app() {
    gtk_widget_hide(glb_win_app);
    return FALSE;
}

void fpm_tr_toggle_win_app(gboolean force_hide) {

    static int x, y;
    static GdkScreen *screen;

    GdkRectangle  bounds;
    gulong        data[4];
    Display      *dpy;
    GdkWindow    *gdk_window;


    if (glb_minimized && !force_hide) {
	gtk_window_set_screen(GTK_WINDOW(glb_win_app), screen);
	gtk_window_move(GTK_WINDOW(glb_win_app), x, y);

	gtk_widget_show(glb_win_app);
	gtk_window_deiconify(GTK_WINDOW(glb_win_app));
	gtk_window_present(GTK_WINDOW(glb_win_app));

	gtk_window_set_skip_taskbar_hint(GTK_WINDOW(glb_win_app), FALSE);
	gtk_window_set_skip_pager_hint(GTK_WINDOW(glb_win_app), FALSE);

	if (!glb_fpm_ini->tr_always_visible)
	    gtk_status_icon_set_visible(glb_tray_icon, FALSE);

	glb_minimized = FALSE;

    } else {

	gtk_status_icon_set_visible(glb_tray_icon, TRUE);
	gtk_window_set_skip_taskbar_hint(GTK_WINDOW(glb_win_app), TRUE);
	gtk_window_set_skip_pager_hint(GTK_WINDOW(glb_win_app), TRUE);

	if(!glb_fpm_ini->tr_always_visible)
	    usleep(50000);

	g_timeout_add(50, fpm_hide_win_app, NULL);

	gtk_window_get_position(GTK_WINDOW(glb_win_app), &x, &y);
	screen = gtk_window_get_screen(GTK_WINDOW(glb_win_app));

	gtk_status_icon_get_geometry(glb_tray_icon, NULL, &bounds, NULL );
	gdk_window = glb_win_app->window;
	dpy = gdk_x11_drawable_get_xdisplay (gdk_window);

	data[0] = bounds.x;
	data[1] = bounds.y;
	data[2] = bounds.width;
	data[3] = bounds.height;

	XChangeProperty (dpy,
			GDK_WINDOW_XID (gdk_window),
			gdk_x11_get_xatom_by_name_for_display (gdk_drawable_get_display (gdk_window),
			"_NET_WM_ICON_GEOMETRY"),
			XA_CARDINAL, 32, PropModeReplace,
			(guchar*)&data, 4);

	gtk_window_iconify(GTK_WINDOW(glb_win_app));

	if (glb_fpm_ini->tr_auto_lock && !glb_locked) {
	    g_timeout_add_seconds(glb_fpm_ini->tr_auto_lock_minutes*60, (GSourceFunc) fpm_lock, NULL);
	}
	glb_minimized = TRUE;
    }
}


void fpm_tr_cleanup() {
    if (glb_tray_icon) {
	gtk_status_icon_set_visible (glb_tray_icon, FALSE);
	glb_tray_icon = (g_object_unref (glb_tray_icon), NULL);
    }
}

gboolean fpm_lock() {

    glb_locked = TRUE;

    if (!glb_minimized)
	fpm_tr_toggle_win_app(TRUE);

//    g_signal_handlers_disconnect_by_func(glb_tray_icon, G_CALLBACK (tray_icon_on_menu), NULL);

    gtk_status_icon_set_tooltip(glb_tray_icon, _("Figaro's Password Manager 2 - locked"));
    fpm_crypt_init("");
    fpm_clear_list();
    fpm_ini_save();

    return FALSE;
}

gboolean fpm_window_check() {
    GList* list;
    list = gtk_window_list_toplevels();
    while (list) {
	if ((gtk_window_get_title(list->data) != NULL) && (list->data != glb_win_app))
	    return TRUE;
	list = g_list_next(list);
    }
    return FALSE;
}
