# FIAIF is an Intelligent firewall
#
# description: Setup Traffic Shaping.
#
# Script Author:	Anders Fugmann <afu at fugmann dot net>
#                       
# FIAIF is an Intelligent firewall
# Copyright (C) 2002-2011 Anders Peter Fugmann
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

###############################################################################
# Stop TC for an interface
###############################################################################
function tc_stop_zone ()
{
    local DEVICE=$1
    # clean existing down- and uplink qdiscs, hide any errors
    TC qdisc del dev ${DEVICE} root    > /dev/null 2>&1
    TC qdisc del dev ${DEVICE} ingress > /dev/null 2>&1
}

###############################################################################
# Setup HTB TC for an interface
###############################################################################
function tc_setup_zone_htb ()
{
    local DEVICE=$1
    local SPEED_UPLINK=$2
    local SPEED_DOWNLINK=$3
    
    SPEED_UPLINK=$((85*SPEED_UPLINK/100))
    SPEED_DOWNLINK=$((85*SPEED_DOWNLINK/100))
    CEIL=$((95*SPEED_UPLINK/100))
    ###### uplink
    
    # shape everything at $UPLINK speed - this prevents huge queues in your
    # DSL modem which destroy latency:

    TC qdisc add dev $DEVICE root handle 1: htb default 20
    BURST="burst 300b cburst 1500b"
    
    TC class add dev $DEVICE parent 1: classid 1:1 htb rate $((SPEED_UPLINK))kbit \
	mpu 64 mtu 1500 overhead 0 quantum 1500 ${BURST}
	
    #rate     rate allocated to this class (class can still borrow)
    #burst    max bytes burst which can be accumulated during idle period {computed}
    #mpu      minimum packet size used in rate computations
    #overhead per-packet size overhead used in rate computations
    #ceil     definite upper class rate (no borrows) {rate}
    #cburst   burst but for ceil {computed}
    #mtu      max packet size we create rate map for {1600}
    #prio     priority of leaf; lower are served first {0}
    #quantum  how much bytes to serve from leaf at once {use r2q}
	

    TC class add dev $DEVICE parent 1:1 classid 1:10 htb rate $((50*SPEED_UPLINK/100))kbit \
        ceil $((SPEED_UPLINK))kbit prio 2 mpu 64 mtu 1500 overhead 0 quantum 1500 ${BURST}

    TC class add dev $DEVICE parent 1:1 classid 1:20 htb rate $((30*SPEED_UPLINK/100))kbit \
        ceil $((CEIL))kbit prio 3 mpu 64 mtu 1500 overhead 0 quantum 1500 ${BURST}

    TC class add dev $DEVICE parent 1:1 classid 1:30 htb rate $((20*SPEED_UPLINK/100))kbit \
        ceil $((CEIL))kbit prio 4 mpu 64 mtu 1500 overhead 0 quantum 1500 ${BURST}

    TC qdisc add dev $DEVICE parent 1:10 handle 10: pfifo
    TC qdisc add dev $DEVICE parent 1:20 handle 20: sfq perturb 10
    TC qdisc add dev $DEVICE parent 1:30 handle 30: sfq perturb 10


    # VoIP traffic always get first in line.
    TC filter add dev ${DEVICE} parent 1: prio 3 protocol ip u32 \
	match ip tos 0x68 0xff \
	flowid 1:10
    
    TC filter add dev ${DEVICE} parent 1: prio 4 protocol ip u32 \
	match ip tos 0xb8 0xff \
	flowid 1:10

    # TOS Minimum Delay (ssh, NOT scp) in 1:10:

    TC filter add dev $DEVICE parent 1: protocol ip prio 10 u32 \
        match ip tos 0x10 0xff flowid 1:10

    # ICMP (ip protocol 1) in the interactive class 1:10 so we
    # can do measurements & impress our friends:

    #TC filter add dev $DEVICE parent 1: protocol ip prio 10 u32 \
    #    match ip protocol 1 0xff flowid 1:10

    # To speed up downloads while an upload is going on, put ACK packets in
    # the interactive class:

    TC filter add dev $DEVICE parent 1: protocol ip prio 20 u32 \
        match ip protocol 6 0xff \
        match u8 0x05 0x0f at 0 \
        match u16 0x0000 0xffc0 at 2 \
        flowid 1:10

    # Place bulk traffic in low prio
    
    TC filter add dev $DEVICE parent 1: protocol ip prio 30 u32 \
        match ip tos 0x08 0xff flowid 1:30    

    # default 

    TC filter add dev $DEVICE parent 1: protocol ip prio 40 u32 \
         match ip dst 0.0.0.0/0 flowid 1:20


    ########## downlink #############
    # slow downloads down to somewhat less than the real speed  to prevent
    # queuing at our ISP. Tune to see how high you can set it.
    # ISPs tend to have *huge* queues to make sure big downloads are fast
    #
    # attach ingress policer:

    TC qdisc add dev ${DEVICE} handle ffff: ingress

    # filter *everything* to it (0.0.0.0/0), drop everything that's
    # coming in too fast:

    TC filter add dev ${DEVICE} parent ffff: protocol ip prio 50 u32 \
       match ip src 0.0.0.0/0 police rate $((SPEED_DOWNLINK))kbit \
       burst 5k drop flowid :1
}

###############################################################################
# Setup HFSC TC for an interface
###############################################################################
function tc_setup_zone_hfsc ()
{
    local DEVICE=$1
    local SPEED_UPLINK=$2
    local SPEED_DOWNLINK=$3
    local VOIP=$4
    
    SPEED_UPLINK=$((80*SPEED_UPLINK/100))
    SPEED_DOWNLINK=$((80*SPEED_DOWNLINK/100))
    
    # Traffic classes:
    # 1:2 Interactive (SSH, DNS, ACK)
    # 1:3 Low latency (VoIP)
    # 1:4 Browsing (HTTP, HTTPs)
    # 1:5 Default
    # 1:6 Low priority (p2p, pop3, smtp, etc)

    # add HFSC root qdisc
    TC qdisc add dev ${DEVICE} root handle 1: hfsc default 5

    # add main rate limit class
    TC class add dev ${DEVICE} parent 1: classid 1:1 hfsc \
	sc rate $((SPEED_UPLINK))kbit \
        ul rate $((SPEED_UPLINK))kbit

    # VoIP: guarantee full uplink for 200ms, then 3/10
    # rt m1   $((SPEED_UPLINK))kbit d  200ms m2 128kbit \
    # sc m1   $((SPEED_UPLINK))kbit d  200ms m2 $((30*SPEED_UPLINK/100))kbit \
    tc class add dev ${DEVICE} parent 1:1  classid 1:2 hfsc \
	sc umax 1514b dmax 5ms rate $((SPEED_UPLINK))kbit \
	ul rate $((SPEED_UPLINK))kbit

    # Interactive traffic: guarantee realtime full uplink for 50ms, then
    # 5/10 of the uplink
    TC class add dev ${DEVICE} parent 1:1  classid 1:3 hfsc \
	rt m1   $((SPEED_UPLINK/2))kbit d  50ms m2 $((50*SPEED_UPLINK/100))kbit \
	ls m1   $((SPEED_UPLINK/2))kbit d  50ms m2 $((70*SPEED_UPLINK/100))kbit \
	ul rate $((SPEED_UPLINK))kbit

    # Browsing: Don't guarantee anything for the first second, then
    # guarantee 1/10
    tc class add dev ${DEVICE} parent 1:1  classid 1:4 hfsc \
	sc m1         0 d    1s m2 $((10*SPEED_UPLINK/100))kbit \
	ul rate $((SPEED_UPLINK))kbit
    
    # Default traffic: don't guarantee anything for the first two seconds,
    # then guarantee 1/20   
    tc class add dev ${DEVICE} parent 1:1  classid 1:5 hfsc \
	sc m1         0 d    2s m2 $((5*SPEED_UPLINK/100))kbit \
	ul rate $((100*SPEED_UPLINK/100))kbit
    
    # Low prio traffic: don't guarantee anything for the first 10 seconds,
    # then guarantee 1/20
    tc class add dev ${DEVICE} parent 1:1  classid 1:6 hfsc \
	sc m1         0 d   10s m2 $((5*SPEED_UPLINK/100))kbit \
	ul rate $((80*SPEED_UPLINK/100))kbit

    #################################################
    # Filters 
    #################################################
    TC filter add dev ${DEVICE} parent 1: prio 1 protocol ip u32 \
        match ip tos 0x68 0xff \
        flowid 1:2

    TC filter add dev ${DEVICE} parent 1: prio 1 protocol ip u32 \
        match ip tos 0xb8 0xff \
        flowid 1:2

    TC filter add dev ${DEVICE} parent 1: prio 2 protocol ip u32 \
        match ip protocol 6 0xff \
        match u8 0x10 0xff at nexthdr+13 \
        match u16 0x0000 0xffc0 at 2 \
        flowid 1:3

    TC filter add dev ${DEVICE} parent 1: prio 2 protocol ip u32 \
        match ip tos 0x10 0x10 \
        flowid 1:3

    TC filter add dev ${DEVICE} parent 1: prio 3  protocol ip u32 \
        match ip tos 0x2 0x02 \
        flowid 1:6

    TC filter add dev ${DEVICE} parent 1: prio 4  protocol ip u32 \
        match ip tos 0x08 0x08 \
        flowid 1:6

    #########
    # Add qdisc for leaves.
    TC qdisc add dev ${DEVICE} parent 1:3 handle 20: pfifo limit 2
    for n in 2 4 5 6; do
	TC qdisc add dev ${DEVICE} parent 1:${n} handle $((n-1))0: sfq limit 20 perturb 10
    done

 

    ########## downlink #############
    # slow downloads down to somewhat less than the real speed  to prevent
    # queuing at our ISP. Tune to see how high you can set it.
    # ISPs tend to have *huge* queues to make sure big downloads are fast
    #
    # attach ingress policer:

    TC qdisc add dev ${DEVICE} handle ffff: ingress

    # Add filters. the police continue, just passes the on the packet,
    # And only the last filter actually drops packets. 
    
    # As we cannot do overall traffic shaping on 
    # incoming packets, this can make it worse for 

    # Allow all voip traffic (Never drop it, as there is no retransmit.)
    TC filter add dev ${DEVICE} parent ffff: protocol ip prio 100 u32 \
	match ip tos 0x68 0xff \
	police rate 128kbit buffer 10k continue \
	flowid :1
    
    TC filter add dev ${DEVICE} parent ffff: protocol ip prio 105 u32 \
	match ip tos 0xb8 0xff \
	police rate 128kbit buffer 10k continue \
	flowid :1
    
    # Dont drop icmp packets, 
    # but only allow a very small rate to pass through.
    TC filter add dev ${DEVICE} parent ffff: protocol ip prio 110 u32 \
        match ip protocol 1 0xff \
	police rate 2kbit buffer 1k continue \
        flowid :1

    
    # Allow all ACK'a, to avoid retransmits
    # take up that much traffic
    TC filter add dev ${DEVICE} parent ffff: protocol ip prio 115 u32 \
        match ip protocol 6 0xff \
        match u8 0x05 0x0f at 0 \
        match u16 0x0000 0xffc0 at 2 \
	police rate 2kbit buffer 1k continue \
	flowid :1

    # filter *everything* to it (0.0.0.0/0), drop everything that's
    # coming in too fast:

    TC filter add dev ${DEVICE} parent ffff: protocol ip prio 120 u32 \
	match ip src 0.0.0.0/0 \
	police rate $((SPEED_DOWNLINK))kbit \
	mtu 1500 burst 10k drop flowid :1
}

###############################################################################
# Status 
###############################################################################
function tc_status ()
{
    for ZONE in ${ZONES}; do
	# Retrieve each zone.
	read_zone ${ZONE}
	if (( $? != 0 )); then
	    continue
	fi
	if (( TC_ENABLE == 1 )); then
	    echo "---- qdisc parameters ----"
	    TC -s -d qdisc ls dev ${DEV}
	    echo "---- Class parameters ----"
	    TC -s -d class ls dev ${DEV}
	    echo "---- filter parameters ----"
	    TC -s -d filter ls dev ${DEV}
	fi
    done 
}

###############################################################################
# Stats 
###############################################################################
function tc_stat()
{
    for ZONE in ${ZONES}; do
	# Retrieve each zone.
	read_zone ${ZONE}
	if (( $? != 0 )); then
	    continue
	fi
	if (( TC_ENABLE == 1 )); then
	    TC -s -d qdisc list dev ${DEV}
	    TC -s -d class show dev ${DEV}
	fi
    done	    
}

###############################################################################
# Setup Traffic Shaping
###############################################################################
function tc_setup ()
{
    # Load the configuration file:
    if (( ENABLE_TC == 1 )); then
    echo -n "Setting up Traffic Shaping: "
	for ZONE in ${ZONES}; do
	    # Retrieve each zone.
	    read_zone ${ZONE}
	    if (( $? != 0 )); then
		continue
	    fi
	    if (( TC_ENABLE == 1 )); then
		echo -n "${ZONE} "
		debug_out "Traffic shaping for zone: ${ZONE}"
		debug_out "Traffic shaping type: ${TC_TYPE}"
		case "${TC_TYPE:="htb"}" in
                    "htb"|"HTB") 
			tc_setup_zone_htb ${DEV%%+} ${TC_UPLINK} ${TC_DOWNLINK} ${VOIP}
			;;
		    "hfsc"|"HFSC")
			tc_setup_zone_hfsc ${DEV%%+} ${TC_UPLINK} ${TC_DOWNLINK} ${VOIP}
			;;
		    *)
			debug_out "Traffic shaping: Unknown type"
		esac
		debug_out "Traffic shaping done"
	    fi
	done
	echo 
    fi
}

###############################################################################
# Stop Traffic Shaping
###############################################################################
function tc_stop ()
{
    if (( ENABLE_TC == 1 )); then  
	echo -n "Stopping Traffic Shaping: "
	debug_out "Stopping Traffic Shaping"
	for ZONE in ${ZONES}; do
	    # Retrieve each zone.
	    read_zone ${ZONE}
	    if (( $? != 0 )); then
		continue
	    fi            
	    tc_stop_zone ${DEV%%+}
	done
	echo "Done."
    fi
}
