/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *   
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity 
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *    
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
/* a heap allocator based on jedi mind powers. and the algorithm
 * described in perl's malloc.c
 * (c) chris ross */

#include "ferite.h"
#include <math.h>

#define BLOCK_COUNT     32     /* number of chunks to allocate each time */
#define NBUCKETS        8*4-2  /* number of chains to have */
#define MAGIC           42     /* the answer to life, the universe, everything and whether a pointer
								* is correct */

/* macors to keep real world stats correct */
#define rmalloc( size )         malloc( size ); real_stats.malloc_c++;
#define rcalloc( size, block )  calloc( size, block ); real_stats.calloc_c++;
#define rrealloc( ptr, size )   realloc( ptr, size ); real_stats.realloc_c++;
#define rfree( ptr )            free( ptr ); real_stats.free_c++;

#define PTR_GET_HEADER( ptr ) (FeriteMemoryChunkHeader*)((char *)ptr - sizeof(FeriteMemoryChunkHeader))
#define PTR_GET_BODY( ptr )   (void*)((char *)ptr + sizeof(FeriteMemoryChunkHeader))

#ifdef THREAD_SAFE_JEDI
#include <pthread.h>

pthread_mutex_t         __ferite_jedi_memory_lock = PTHREAD_MUTEX_INITIALIZER;
# define LOCK_MEMORY     pthread_mutex_lock( &__ferite_jedi_memory_lock )
# define UNLOCK_MEMORY   pthread_mutex_unlock( &__ferite_jedi_memory_lock )
#else
# define LOCK_MEMORY
# define UNLOCK_MEMORY
#endif 

typedef union __ferite_memory_chunk_header FeriteMemoryChunkHeader;
union __ferite_memory_chunk_header
{
   FeriteMemoryChunkHeader *next;
   struct
   {
      unsigned char index; /* bucket for index, < NBUCKETS */
      unsigned char magic; /* magic -> so we can check that the pointer is allocated and
			    * we haven't gone painfully wrong somewhere */
   }
   assigned_info;
   unsigned int alignment; /* force word alignment */
};

long                     pow_lookup[NBUCKETS];
FeriteMemoryChunkHeader *free_chunks[NBUCKETS];
FeriteMemoryChunkHeader *big_chunks = NULL;

typedef struct
{
   long malloc_c;
   long calloc_c;
   long realloc_c;
   long free_c;
}
FeriteMemoryStats;

FeriteMemoryStats       real_stats; /* how many times we hit the OS */
FeriteMemoryStats       vrtl_stats; /* how many times we hit the local heap */

void __ferite_jedi_memory_init()
{
   int i;
   
   real_stats.malloc_c = 0;
   real_stats.calloc_c = 0;
   real_stats.realloc_c = 0;
   real_stats.free_c = 0;

   vrtl_stats.malloc_c = 0;
   vrtl_stats.calloc_c = 0;
   vrtl_stats.realloc_c = 0;
   vrtl_stats.free_c = 0;

   if( !hide_mem_use )
   {
#ifdef FERITE_MEM_DEBUG
      ferite_warning( NULL, "Jedi Memory System Active\n" );
#endif
   }

   for( i = 0; i < NBUCKETS; i++ )
   {
      pow_lookup[i] = (long)pow( (double)2, (double)i );
      FUD(( "pow: 2^%d: %ld\n", i, pow_lookup[i] ));
      free_chunks[i] = NULL;
   }
}

void __ferite_jedi_memory_deinit()
{
   void *ptr;

   while( big_chunks )
   {
      ptr = big_chunks->next;
      rfree( big_chunks );
      big_chunks = ptr;
   }
   if( !hide_mem_use ) /* 2/3's of all statistics are made up. unfortunatly not here */
   {
      printf( "Ferite Memory Usage Statistics (jedi)\n" );
      printf( " |- Virtual.. %ld mallocs, %ld callocs, %ld reallocs, %ld frees",
			 vrtl_stats.malloc_c, vrtl_stats.calloc_c, vrtl_stats.realloc_c, vrtl_stats.free_c );
      printf( " [%ld block%s still allocated]\n", (vrtl_stats.malloc_c + vrtl_stats.calloc_c) - vrtl_stats.free_c,
			 (((vrtl_stats.malloc_c + vrtl_stats.calloc_c) - vrtl_stats.free_c) == 1 ? "" : "s"));
      printf( " `- Real..... %ld mallocs, %ld callocs, %ld reallocs, %ld frees",
			 real_stats.malloc_c, real_stats.calloc_c, real_stats.realloc_c, real_stats.free_c );
      printf( " [%ld block%s still allocated]\n", (real_stats.malloc_c + real_stats.calloc_c) - real_stats.free_c,
			 (((real_stats.malloc_c + real_stats.calloc_c) - real_stats.free_c) == 1 ? "" : "s"));
   }
}

void __ferite_jedi_dump_memory( int bucket )
{
#ifdef DEBUG
   int i = 0;
#endif
   FeriteMemoryChunkHeader *hdr;

   /* dump a free bucket chain... */
   FUD(( "================================ Table Dump ==================================\n" ));
   hdr = free_chunks[bucket];
   while( hdr != NULL ) /* keep looping until we hit the end... */
   {
      FUD(( " [%d]> hdr=%p, hdr->next=%p, %ld, sizeof(Header): %d, pow[%d]=%ld\n", ++i, hdr, hdr->next,
		 (long)((void*)hdr->next) - (long)((void*)hdr), sizeof(FeriteMemoryChunkHeader), bucket, pow_lookup[bucket] ));
      hdr = hdr->next;
   }
}

void *__ferite_jedi_malloc( size_t size, char *file, int line )
{
   char *return_ptr;
   int target_bucket;
   FeriteMemoryChunkHeader *ptr;

   LOCK_MEMORY;
  /* get the correct bucket size -> we should check the wastage and have an odd's and
   * sods chain we allocate random chunks from -> but thats for another day... */
   target_bucket = 1;
   if( size < 5 ) /* stops some issues */
	 size = 5;
   while( size > pow_lookup[target_bucket] )
     target_bucket++;

   FUD(( "Target bucket for data of %d is %d(%ld)\n", size, target_bucket, pow_lookup[target_bucket] ));

   /* check to see if we have memory :), if not go a eat some :) */
   if( free_chunks[target_bucket] == NULL )
     __ferite_jedi_morecore( target_bucket );

   /* oooh we are up the creek, so to say */
   if( (ptr = free_chunks[target_bucket]) == NULL )
   {
#ifdef FERITE_MEM_DEBUG
      ferite_warning( NULL, "JEDI: Out of memory. Oh dear. Oh dear. Go out and buy some more :)\n" );
#endif
      UNLOCK_MEMORY;
      return NULL;
   }

   /* rebuild the chain */
   FUD(( "free_chunks[target_bucket]: %p\n", free_chunks[target_bucket] ));
   FUD(( "new free_chunks:            %p\n", ptr->next ));
   free_chunks[target_bucket] = ptr->next;

   /* setup the information for the wild goose chase :) */
   ptr->assigned_info.index = target_bucket;
   ptr->assigned_info.magic = MAGIC;
   /* get the memory chunk */
   return_ptr = PTR_GET_BODY(ptr);
   FUD(( "returning: %p, %d\n", return_ptr, (int)((void *)return_ptr) - (int)((void *)ptr) ));
   vrtl_stats.malloc_c++;
   UNLOCK_MEMORY;
   return return_ptr;
}

void *__ferite_jedi_calloc( size_t size, size_t blk_size, char *file, int line )
{ /* surely the easist calloc *ever*? ;) */
   void *ptr = __ferite_jedi_malloc( size, __FILE__, __LINE__ );

   vrtl_stats.malloc_c--;
   vrtl_stats.calloc_c++;
   memset( ptr, 0, size );
   return ptr;
}

void *__ferite_jedi_realloc( void *ptr, size_t size )
{
   FeriteMemoryChunkHeader *hdr = NULL;
   long old_size, old_index;
   void *new_ptr = __ferite_jedi_malloc( size, __FILE__, __LINE__ );

   LOCK_MEMORY;
   if( ptr != NULL )
   { /* we want to copy the old memory to the new memory */
      hdr = PTR_GET_HEADER( ptr );
      old_size = pow_lookup[hdr->assigned_info.index];
      old_index = hdr->assigned_info.index;
      /* this is an evil slow method of realloc - but necessary */
      memcpy( new_ptr, ptr, old_size );
      /* now we move the older ptr onto it's old block */
      hdr->next = free_chunks[old_index];
      free_chunks[old_index] = hdr;
      vrtl_stats.malloc_c--;
      vrtl_stats.realloc_c++;
   }
   UNLOCK_MEMORY;
   return new_ptr;
}

void __ferite_jedi_free( void *ptr, char *file, int line )
{
   FeriteMemoryChunkHeader *hdr = NULL;
   int bucket = 0;

   LOCK_MEMORY;
   if( ptr != NULL )
   {
      hdr = PTR_GET_HEADER(ptr);
      FUD(( "freeing %p (hdr=%p,diff=%ld)\n", ptr, hdr, (long)((void*)ptr) - (long)((void*)hdr) ));
      if( hdr->assigned_info.magic == MAGIC )
      {
	 /*	 int length = pow_lookup[hdr->assigned_info.index];
	  *	 memset( ptr, '\0', length );*/
	 bucket = hdr->assigned_info.index;
	 /* relink the chain */
	 FUD(( "Setting next as %p\n", free_chunks[bucket] ));
	 hdr->next = free_chunks[bucket];
	 FUD(( "Setting new header as %p\n", hdr ));
	 free_chunks[bucket] = hdr;
	 vrtl_stats.free_c++;
      }
      else /* fubar'd pointer -> we must know about these */
      {
	 ferite_warning( NULL, "JEDI: expecting %d for magic, but got %d (culprit %s, line %d)\n", MAGIC, hdr->assigned_info.magic, file, line );
	 ferite_warning( NULL, "JEDI: MEM DATA: `%s'\n", (char *)hdr );
      }
   }
#ifdef FERITE_MEM_DEBUG /* only if we are debugging shouyld we hear these */
   else
     ferite_warning( NULL, "JEDI: trying to free a NULL pointer in %s on line %d\n", file, line );
#endif
   UNLOCK_MEMORY;
}

void __ferite_jedi_morecore( int bucket )
{
   int i = 0;
   void *new_block = NULL;
   int chunk_size = pow_lookup[bucket] + sizeof(FeriteMemoryChunkHeader);
   FeriteMemoryChunkHeader *hdr;

   FUD(( "in more core -> allocating for %d\n", bucket ));

   /* check to see if we have actually run out of space on bucket list */
   if( free_chunks[bucket] )
     return;

   /* this is where the buckets will be */
   new_block = rmalloc( (chunk_size * BLOCK_COUNT) +       /* blocks of memory */
					   sizeof(FeriteMemoryChunkHeader) ); /* initial header   */
   ((FeriteMemoryChunkHeader*)new_block)->next = big_chunks; /* hook up  header */
   big_chunks = new_block; /* this now becomes the header */

   /* the memory on this chunk needs to be setup as follows:
    *       /---------------------\
    *   [header][memory         ][header][memory         ]
    *   |<-a ->||<-p_l[bucket]->||<-a ->||<-p_l[bucket]->|  */
   hdr = (FeriteMemoryChunkHeader*)(((char*)new_block) + sizeof(FeriteMemoryChunkHeader)); /* get the top of our memory */
   FUD(( "================================  Morecore %ld ================================\n", pow_lookup[bucket] ));
   for( i = 0; i < BLOCK_COUNT-1; i++ ) /* go through the block linking up the blocks */
   {
      hdr->next =  (FeriteMemoryChunkHeader*)(((char*)hdr) + pow_lookup[bucket] + sizeof(FeriteMemoryChunkHeader) );
      FUD(( " [%d]> hdr=%p, hdr->next=%p, %ld, sizeof(Header): %d, pow[%d]=%ld\n", i, hdr, hdr->next,
		 (long)((void*)hdr->next) - (long)((void*)hdr), sizeof(FeriteMemoryChunkHeader), bucket, pow_lookup[bucket] ));
      hdr = hdr->next;
   }
   hdr->next = NULL; /* stop the chain -> so we know when to allocate more */
   FUD(( " [%d]> hdr=%p, hdr->next=%p, %ld, sizeof(Header): %d, pow[%d]=%ld\n", i, hdr, hdr->next,
      (long)((void*)hdr->next) - (long)((void*)hdr), sizeof(FeriteMemoryChunkHeader), bucket, pow_lookup[bucket] ));

   /* link the memory in */
   free_chunks[bucket] =  (FeriteMemoryChunkHeader*)(((char*)new_block)+sizeof(FeriteMemoryChunkHeader));
   /* dump the table so that we can check that is correlates with the above table */
#ifdef FERITE_MEM_DEBUG
   __ferite_jedi_dump_memory( bucket );
#endif
   FUD(( "morecore: free_chunks[bucket]: %p\n", free_chunks[bucket] ));
}
