/*
  fblogo.c
  Generate linux_logo.h header file for use with the framebuffer boot logo
  Written by Daniel Vedy <daniel@kongsberg.online.no> 1999
  Adapted by Gordon Fraser <gordon@freakzone.net> 2001/2002
  Released under the GPL
  
  11/12/2001: 	Fixed missing includes
  		Fixed ImageMagick-Bug
  		Updated outputted defines to match current kernel
	  	Updated low_color.h to match current kernel (2.4.15-pre3)
		Use ImageMagick ReadImage instead of ReadTIFFImage
		(Gordon Fraser <gordon@freakzone.net>)
  11/25/2001:   cast signed color info to unsigned char :)
		(Gordon Fraser <gordon@freakzone.net>)
  11/27/2001:	Dither & Reduce colors to get rid of artefacts
                Add 2.2/2.4 Kernel-Option
		(Gordon Fraser <gordon@freakzone.net>)
  03/07/2002:   Get rid of imagemagick, use libpng instead
                Reason: new libmagick versions kept makeing fblogo segfault :(
		(Gordon Fraser <gordon@freakzone.net>)        
                
*/

#include "fblogo.h"

/*------------------------------------------------------------------*/
int main(int argc, char** argv)
{
  FILE* inputfile = NULL;
  FILE* outputfile = NULL;
  char version = LINUX_24;
  char sourcefile[255];
  int i;
  
  verbose = 0;
      
  if(argc>5)
    {
      usage();
      exit(EXIT_FAILURE);
    }

  for ( i=1; i<argc; i++ ) 
    {
      if ( *argv[i] == '-' ) 
	{
	  switch ( *(argv[i]+1) ) 
	    {
	    case '2':
	      version = LINUX_22;
	      break;
	    case 'v':
	      verbose = 1;
	      break;
	    case 'h':
	      usage();
	      exit(EXIT_SUCCESS);
	      break;
	    default:
	      usage();
	      exit(EXIT_FAILURE);
	      break;
	    }
	} 
      else
	{
	  strcpy(sourcefile, argv[i]);
	  if ((inputfile = fopen(argv[i], "rb")) == (FILE *)NULL) 
	    {
	      fprintf(stderr, "fblogo error: cannot open file %s\n", argv[i]);
	      exit(EXIT_FAILURE);
	    }
	  if(++i<argc) 
	    {
	      strcpy(sourcefile, argv[i]);
	      if ((outputfile = fopen(argv[i], "w")) == (FILE *)NULL) 
		{
		  fprintf(stderr, "fblogo error: cannot open file %s\n", argv[i]);
		  exit(EXIT_FAILURE);
		}
	    }
	}
    }

  if (inputfile == (FILE *)NULL) 
    {
      if ((inputfile = fdopen(0, "rb")) == (FILE *)NULL) 
	{
	  fprintf(stderr, "fblogo error: cannot open stdin\n");
	  exit(EXIT_FAILURE);
	}
    }
      
  if(outputfile == (FILE *)NULL) 
    {
      verbose = 0;		/* No messages if output goes to stdout */
      if ((outputfile = fdopen(1, "w")) == (FILE *)NULL) 
	{
	  fprintf(stderr, "fblogo error: cannot open stdout\n");
	  exit(EXIT_FAILURE);
	}
    }

  // Now read the source image
  if(read_image(inputfile))
    {
      fprintf(stderr, "fblogo error: unable to read image - %s\n", strerror(errno));
      exit(EXIT_FAILURE);
    }
  
  // Write the resulting header file
  if(write_header(outputfile, version))
    {
      fprintf(stderr, "fblogo error: unable to write header - %s\n", strerror(errno));
      exit(EXIT_FAILURE);
    }

  if(verbose) 
    {
      printf("Generated %s successfully\n\n", sourcefile);
      printf("Remember to modify /usr/src/linux/drivers/video/fbcon.c:\n");
      printf(" Change \"#define LOGO_H 80\" to \"#define LOGO_H %li\"\n", height);
      printf(" Change \"#define LOGO_W 80\" to \"#define LOGO_W %li\"\n\n", width);
      printf("Then copy %s to /usr/src/linux/include/linux/linux_logo.h,\n", sourcefile);
      printf("and recompile the kernel.\n");
    }
  
  fclose(inputfile);
  fclose(outputfile);

  free(image_data);

  exit(EXIT_SUCCESS);
}

/*------------------------------------------------------------------*/
void usage(void)
{
    printf("Usage:\n");
    printf("fblogo [-2] [-v] [-?] [inputfile [outputfile]]\n");
}


/*------------------------------------------------------------------*/
/* Based on libpng example.c and                                    */
/*          png2linuxlogo by Greg Roelofs <newt@pobox.com>          */
int read_image(FILE *fp)
{
   char buf[PNG_SIG_BYTES];
   int  i, bit_depth, color_type;

   png_structp png_ptr;
   png_infop info_ptr;
   png_infop end_info;

   png_uint_32 rowbytes;
   png_bytepp row_pointers = NULL;

   // First check if file really is a png
   if (fread(buf, 1, PNG_SIG_BYTES, fp) != PNG_SIG_BYTES)
     {
       return ERR_NOPNG;
     }

   if(png_sig_cmp(buf, 0, PNG_SIG_BYTES)) 
     {
       return ERR_NOPNG;
     }

   // Now create png_struct and png_info
   png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, (png_voidp)NULL, NULL, NULL);
   if (!png_ptr)
     {
       return (ERR_PNGPTR);
     }
   
   info_ptr = png_create_info_struct(png_ptr);
   if (!info_ptr)
     {
       png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
       return (ERR_PNGINFO);
     }
   
   end_info = png_create_info_struct(png_ptr);
   if (!end_info)
     {
       png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
       return (ERR_PNGINFO);
     }


   png_init_io(png_ptr, fp);
   png_set_sig_bytes(png_ptr, PNG_SIG_BYTES);
   png_read_info(png_ptr, info_ptr);
   png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth, &color_type, NULL, NULL, NULL);
   
   png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette);
   
   if (num_palette > 223) 
     {
       fprintf(stderr, "fblogo error: too many colors (%d); must be less than 224\n", num_palette);
       png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
       return ERR_COLNUM;
     }
   
   /* allocate space for the PNG image data */
   rowbytes = png_get_rowbytes(png_ptr, info_ptr);

   if ((image_data = (png_bytep)malloc(rowbytes*height)) == NULL) 
     {
       fprintf(stderr, "fblogo error: can't allocate image data\n");
       png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
       return ERR_PNGPTR;
     }
   if ((row_pointers = (png_bytepp)malloc(height*sizeof(png_bytep))) == NULL) 
     {
       fprintf(stderr, "fblogo error: can't allocate row pointers\n");
       png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
       free(image_data);
       return ERR_PNGPTR;
     }

    /* set the individual row_pointers to point at the correct offsets */
    for (i = 0;  i < height;  ++i)
      row_pointers[i] = image_data + i * rowbytes;
    
    png_read_image(png_ptr, row_pointers);
    png_read_end(png_ptr, NULL);

    return 0;
}

/*------------------------------------------------------------------*/
int write_header(FILE* fd, char version)
{
  int i;
  time_t timestamp;
  char datestr[32];
  png_bytep pixel;
  
  if(version==LINUX_22 && verbose)
    printf("Creating for Linux kernel version 2.2.x\n");
  else if(version==LINUX_24 && verbose)
    printf("Creating for Linux kernel version 2.4.x/2.5.x\n");
  
  timestamp = time((time_t *)NULL);
  strftime(datestr, 32, "%Y/%m/%d %H:%M:%S", localtime(&timestamp));
  
  fprintf(fd, "/* $Id: linux_logo.h,v 1.5 %s jj Exp $
 * include/linux/linux_logo.h: This is a linux logo
 *                             to be displayed on boot.
 *
 * Copyright (C) 1996 Larry Ewing (lewing@isc.tamu.edu)
 * Copyright (C) 1996,1998 Jakub Jelinek (jj@sunsite.mff.cuni.cz)
 *
 * You can put anything here, but:
 * LINUX_LOGO_COLORS has to be less than 224\n", datestr);

    
  fprintf(fd, " * Generated by fblogo version %s\n *\n", VERSION);
  fprintf(fd, " *\n * Remember to modify drivers/video/fbcon.c:\n");
  fprintf(fd, " * Change \"#define LOGO_H 80\" to \"#define LOGO_H %li\"\n", width);
  fprintf(fd, " * Change \"#define LOGO_W 80\" to \"#define LOGO_W %li\"\n */\n\n", height);
  
  if(version == LINUX_24) 
    {
      fprintf(fd, "#ifndef __HAVE_ARCH_LINUX_LOGO\n");
      fprintf(fd, "#define LINUX_LOGO_COLORS %d\n", num_palette);
      fprintf(fd, "#endif\n");
      
      fprintf(fd, "#ifdef INCLUDE_LINUX_LOGO_DATA\n");
      fprintf(fd, "#ifndef __HAVE_ARCH_LINUX_LOGO\n");
      
    } 
  else
    {
      fprintf(fd, "#if LINUX_LOGO_COLORS == %d\n", num_palette);
    }
  
  
  fprintf(fd, "unsigned char linux_logo_red[] __initdata = {");
  for(i=0;i<num_palette;i++)
    {
      if(i%8==0)
	fprintf(fd, "\n  ");
      fprintf(fd, "0x%2.2X",palette[i].red);
      if(i!=num_palette-1)
	fprintf(fd, ", ");
    }
  fprintf(fd,"\n};\n\n");
  
  
  
  fprintf(fd, "unsigned char linux_logo_green[] __initdata = {");
  for(i=0;i<num_palette;i++)
    {
      if(i%8==0)
	fprintf(fd, "\n  ");
      fprintf(fd, "0x%2.2X",palette[i].green);
      if(i!=num_palette-1)
	fprintf(fd, ", ");
    }
  fprintf(fd,"\n};\n\n");
  
  fprintf(fd, "unsigned char linux_logo_blue[] __initdata = {");
  for(i=0;i<num_palette;i++)
    {
      if(i % 8 == 0)
	fprintf(fd, "\n ");
      fprintf(fd, "0x%2.2X",palette[i].blue);
      if(i!=num_palette-1)
	fprintf(fd, ", ");
    }
  fprintf(fd,"\n};\n\n");
  
  fprintf(fd, "unsigned char linux_logo[] __initdata = {");

  pixel = image_data;
  for (i = 0;  i < width*height;  ++i) 
    {
      if (i > 0)
	fprintf(fd, ",");
      if (i % 8 == 0)
	fprintf(fd, "\n ");
      fprintf(fd, " 0x%2.2X", (*pixel++) + 0x20);
    }
  
  if(version == LINUX_22) 
    {
      fprintf(fd,"\n};\n\n#endif\n\n");
      fprintf(fd, low_color_logo_22);
    } 
  else
    {
      fprintf(fd,"\n};\n\n#endif /* !__HAVE_ARCH_LINUX_LOGO */\n");
      fprintf(fd, low_color_logo_24);
    }
  
  return 0;
}




