/*
 *  Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "potentiometer.h"

#define COMP_(x)	potentiometer_ ## x

struct cpssp {
	unsigned int state_left;
	struct sig_std_logic *port_mid;
	unsigned int state_right;
	unsigned int state_adj;
};

static void
COMP_(update)(struct cpssp *cpssp)
{
	int mV;
	int mA;

	mV = (int) cpssp->state_left
		+ ((int) cpssp->state_right - (int) cpssp->state_left) * (int) cpssp->state_adj / 100;
	mA = -1;

	sig_std_logic_set(cpssp->port_mid, cpssp, SIG_COMB(mV, mA));
}

static void
COMP_(left_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_left = SIG_mV(val);
	COMP_(update)(cpssp);
}

static void
COMP_(right_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_right = SIG_mV(val);
	COMP_(update)(cpssp);
}

static void
COMP_(adj_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_adj = val;
	COMP_(update)(cpssp);
}

void *
COMP_(create)(
	const char *name,
        struct sig_manage *port_manage,
	struct sig_std_logic *port_left,
	struct sig_std_logic *port_mid,
	struct sig_std_logic *port_right,
	struct sig_integer *port_adj
)
{
	static const struct sig_std_logic_funcs left_funcs = {
		.std_logic_set = COMP_(left_set),
	};
	static const struct sig_std_logic_funcs right_funcs = {
		.std_logic_set = COMP_(right_set),
	};
	static const struct sig_integer_funcs adj_funcs = {
		.set = COMP_(adj_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* Out */
	cpssp->port_mid = port_mid;
	sig_std_logic_connect_out(port_mid, cpssp, SIG_STD_LOGIC_0);

	/* In */
	cpssp->state_left = 0;
	sig_std_logic_connect_in(port_left, cpssp, &left_funcs);
	cpssp->state_right = 0;
	sig_std_logic_connect_in(port_right, cpssp, &right_funcs);

	cpssp->state_adj = 0;
	sig_integer_connect_in(port_adj, cpssp, &adj_funcs);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
