/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <semaphore.h>
#include <pthread.h>
#include <linux/ioctl.h>
#include <stdint.h>
#include <linux/types.h>
#include <stdio.h>

#include "gencrc.h"

#include "glue.h"
#include "lib_usb.h"
#include "bridge_usb.h"

#define VIRTUALDEV_PDEV_TYPE_DUMMY 0x0
#define VIRTUALDEV_PDEV_TYPE_USB 0x1
#define VIRTUALDEV_PDEV_TYPE_NET 0x2

#define VIRTUALDEV_IOC_MAGIC 'v'

#define VIRTUALDEV_IOC_REGISTER _IOW(VIRTUALDEV_IOC_MAGIC, 0, int)
#define VIRTUALDEV_IOC_UNREGISTER _IO(VIRTUALDEV_IOC_MAGIC, 1)

#define VIRTUALDEV_IOC_HCI_MAGIC 'h'

#define VIRTUALDEV_IOC_HCI_WHATEVER _IO(VIRTUALDEV_IOC_HCI_MAGIC, 0)
#define VIRTUALDEV_IOC_HCI_CONNECT _IO(VIRTUALDEV_IOC_HCI_MAGIC, 1)
#define VIRTUALDEV_IOC_HCI_DISCONNECT _IO(VIRTUALDEV_IOC_HCI_MAGIC, 2)
#define VIRTUALDEV_IOC_PENDING_RESET _IO(VIRTUALDEV_IOC_HCI_MAGIC, 3)

struct virtualdev_hci_interact {
  // OUT
	__u32 seqnum;
	__u32 direction;
  __u32 device;
	__u32 endpoint;
	__u32 type;

	__u32 transfer_flags;
	__u32 transfer_buffer_length;

	__s32 number_of_packets;
	__s32 interval;

  int setup_packet_available;

  unsigned char setup_packet[8];

  // IN
  int status;

  int error_count;

  __u32 actual_length;

  // MODIFY
  __s32 start_frame;

	// TODO: Split in 2 different reads/writes
	unsigned char transfer_buffer[65536];
} __packed;

// TODO: Toggle DATA0/DATA1

#define PIPE_ISOCHRONOUS                0
#define PIPE_INTERRUPT                  1
#define PIPE_CONTROL                    2
#define PIPE_BULK                       3

#define USB_IN 1
#define USB_OUT 0

#define TRANSACTION_IDLE 0
#define TRANSACTION_CONTROL_IN 1
#define TRANSACTION_CONTROL_OUT 2
#define TRANSACTION_BULK_IN 3
#define TRANSACTION_BULK_OUT 4

#define STAGE_IDLE 0
#define STAGE_SETUP 1
#define STAGE_DATA 2
#define STAGE_STATUS 3

#define RECV_NOTHING 0
#define RECV_HANDSHAKE_ACK 1
#define RECV_HANDSHAKE_NAK 2
#define RECV_HANDSHAKE_STALL 3
#define RECV_DATA 4
#define RECV_DATA_0 5

#define USB_TIMER_FREQ	1000

struct cpssp {
	struct sig_usb_bus *bus;

	struct virtualdev_hci_interact virtualdev_hci_interact;

	int fd;

	int state_transaction;
  int state_stage;
	int state_recv;
	uint8_t *state_recv_data;
	unsigned int state_recv_data_length;

	unsigned long long tsc_passed;
	unsigned long long tsc_step;
	unsigned int timer_scheduled;
};

static void
bridge_usb_print(struct cpssp *cpssp)
{
	fprintf(stderr, "####################\n");
	fprintf(stderr, "PULL SEQNR: %d\n", cpssp->virtualdev_hci_interact.seqnum);
	fprintf(stderr, "PULL DIRECTION: %s\n", (cpssp->virtualdev_hci_interact.direction == 1)?("IN"):("OUT"));
	fprintf(stderr, "PULL DEVICE: %d\n", cpssp->virtualdev_hci_interact.device);
	fprintf(stderr, "PULL ENDPOINT: %d\n", cpssp->virtualdev_hci_interact.endpoint);
	fprintf(stderr, "PULL TRANSFER_BUFFER_LENGTH: %d\n", cpssp->virtualdev_hci_interact.transfer_buffer_length);
	fprintf(stderr, "####################\n");
}

static void
bridge_usb_recv_copy(
	struct cpssp *cpssp)
{
	if ((cpssp->virtualdev_hci_interact).actual_length + cpssp->state_recv_data_length > (cpssp->virtualdev_hci_interact).transfer_buffer_length) {
		assert(0);
	}
	memcpy((cpssp->virtualdev_hci_interact).transfer_buffer + (cpssp->virtualdev_hci_interact).actual_length, cpssp->state_recv_data, cpssp->state_recv_data_length);
	(cpssp->virtualdev_hci_interact).actual_length += cpssp->state_recv_data_length;
}

static void
bridge_usb_recv_reset(
	struct cpssp *cpssp)
{
	free(cpssp->state_recv_data);

	cpssp->state_recv_data = NULL;
	cpssp->state_recv_data_length = 0;
	cpssp->state_recv = RECV_NOTHING;
}

static void
bridge_usb_transaction_start(
	struct cpssp *cpssp)
{
	if (read(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact)) == sizeof(struct virtualdev_hci_interact)) {
		(cpssp->virtualdev_hci_interact).actual_length = 0;

		bridge_usb_print(cpssp);

		if ((cpssp->virtualdev_hci_interact).type == PIPE_CONTROL) {
			if (cpssp->virtualdev_hci_interact.direction == USB_IN) {
				fprintf(stderr, "### TRANSACTION_CONTROL_IN ###\n");
				cpssp->state_transaction = TRANSACTION_CONTROL_IN;
			} else {
				fprintf(stderr, "### TRANSACTION_CONTROL_OUT ###\n");
				cpssp->state_transaction = TRANSACTION_CONTROL_OUT;
			}
		} else if ((cpssp->virtualdev_hci_interact).type == PIPE_BULK) {
			if (cpssp->virtualdev_hci_interact.direction == USB_IN) {
				fprintf(stderr, "### TRANSACTION_BULK_IN ###\n");
				cpssp->state_transaction = TRANSACTION_BULK_IN;
			} else {
				fprintf(stderr, "### TRANSACTION_BULK_OUT ###\n");
				cpssp->state_transaction = TRANSACTION_BULK_OUT;
			}
		} else if ((cpssp->virtualdev_hci_interact).type == PIPE_ISOCHRONOUS) {
			assert(0);
		} else if ((cpssp->virtualdev_hci_interact).type == PIPE_INTERRUPT) {
			if (cpssp->virtualdev_hci_interact.direction == USB_IN) {
				fprintf(stderr, "### TRANSACTION_BULK_IN ###\n");
				cpssp->state_transaction = TRANSACTION_BULK_IN;
			} else {
				fprintf(stderr, "### TRANSACTION_BULK_OUT ###\n");
				cpssp->state_transaction = TRANSACTION_BULK_OUT;
			}
		} else {
			assert(0);
		}
		cpssp->state_stage = STAGE_IDLE;

		bridge_usb_recv_reset(cpssp);
	} else {
		sleep(10);
	}
}

static void bridge_usb_transaction_control_in(
	struct cpssp *cpssp) {
	if (cpssp->state_stage == STAGE_IDLE) {
		fprintf(stderr, " = STAGE_SETUP =\n");
		cpssp->state_stage = STAGE_SETUP;

		fprintf(stderr, " -> SETUP\n");
		sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_SETUP, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);

		fprintf(stderr, " -> DATA0 8\n");
		sig_usb_bus_send_data(cpssp->bus, cpssp, USB_PID_DATA0, 8, (cpssp->virtualdev_hci_interact).setup_packet, gen_crc16((cpssp->virtualdev_hci_interact).setup_packet, 8));
	} else if (cpssp->state_stage == STAGE_SETUP) {
		if (cpssp->state_recv == RECV_HANDSHAKE_ACK) {
			bridge_usb_recv_reset(cpssp);

			fprintf(stderr, " = STAGE_DATA =\n");
			cpssp->state_stage = STAGE_DATA;

			fprintf(stderr, " -> IN\n");
			sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_IN, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);
		} else if (cpssp->state_recv == RECV_HANDSHAKE_NAK) {
			bridge_usb_recv_reset(cpssp);

			// TODO: Send again

			assert(0);
		} else if (cpssp->state_recv == RECV_HANDSHAKE_STALL) {
			bridge_usb_recv_reset(cpssp);

			// TODO: Push with error

			assert(0);
		} else if (cpssp->state_recv == RECV_DATA) {
			bridge_usb_recv_reset(cpssp);

			// TODO: Invalid, send NAK and try again

			assert(0);
		} else {
			fprintf(stderr, "RECV: %d\n", cpssp->state_recv);

			// TODO: WAIT

			assert(0);
		}
	} else if (cpssp->state_stage == STAGE_DATA) {
		if (cpssp->state_recv == RECV_DATA) {
			bridge_usb_recv_copy(cpssp);

			bridge_usb_recv_reset(cpssp);

			fprintf(stderr, " -> ACK\n");
			sig_usb_bus_send_handshake(cpssp->bus, cpssp, USB_PID_ACK);

			fprintf(stderr, " -> IN\n");
			sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_IN, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);

		} else if (cpssp->state_recv == RECV_DATA_0 || cpssp->state_recv == RECV_HANDSHAKE_NAK) {
			bridge_usb_recv_reset(cpssp);

			fprintf(stderr, " = STAGE_STATUS =\n");
			cpssp->state_stage = STAGE_STATUS;

			fprintf(stderr, " -> ACK\n");
			sig_usb_bus_send_handshake(cpssp->bus, cpssp, USB_PID_ACK);

			fprintf(stderr, " -> OUT\n");
			sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_OUT, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);

			fprintf(stderr, " -> DATA_0\n");
			sig_usb_bus_send_data(cpssp->bus, cpssp, USB_PID_DATA1, 0, (cpssp->virtualdev_hci_interact).transfer_buffer, 0);
		} else {
			// TODO: Invalid, send again

			assert(0);
		}
	} else if (cpssp->state_stage == STAGE_STATUS) {
		if (cpssp->state_recv == RECV_HANDSHAKE_ACK) {
			bridge_usb_recv_reset(cpssp);

			(cpssp->virtualdev_hci_interact).status = 0;
			(cpssp->virtualdev_hci_interact).error_count = 0;

			write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));
			fprintf(stderr, "### TRANSACTION END OK ###\n");
			cpssp->state_transaction = TRANSACTION_IDLE;
			cpssp->state_stage = STAGE_IDLE;

			bridge_usb_recv_reset(cpssp);
		} else if (cpssp->state_recv == RECV_HANDSHAKE_NAK) {
			bridge_usb_recv_reset(cpssp);

			(cpssp->virtualdev_hci_interact).status = 1;
			(cpssp->virtualdev_hci_interact).error_count = 0;

			write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));
			fprintf(stderr, "### TRANSACTION END ERROR ###\n");
			cpssp->state_transaction = TRANSACTION_IDLE;
			cpssp->state_stage = STAGE_IDLE;

			bridge_usb_recv_reset(cpssp);
		} else if (cpssp->state_recv == RECV_HANDSHAKE_STALL) {
			assert(0);
		} else {
			assert(0);
		}
	} else {
		assert(0);
	}
}

static void bridge_usb_transaction_control_out(
	struct cpssp *cpssp) {
	if (cpssp->state_stage == STAGE_IDLE) {
		fprintf(stderr, " = STAGE_SETUP =\n");
		cpssp->state_stage = STAGE_SETUP;

		fprintf(stderr, " -> SETUP\n");
		sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_SETUP, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);

		fprintf(stderr, " -> DATA0 8\n");
		sig_usb_bus_send_data(cpssp->bus, cpssp, USB_PID_DATA0, 8, (cpssp->virtualdev_hci_interact).setup_packet, gen_crc16((cpssp->virtualdev_hci_interact).setup_packet, 8));
	} else if (cpssp->state_stage == STAGE_SETUP) {
		if (cpssp->state_recv == RECV_HANDSHAKE_ACK) {
			bridge_usb_recv_reset(cpssp);

			if ((cpssp->virtualdev_hci_interact).transfer_buffer_length > 0) {
				fprintf(stderr, " = STAGE_DATA =\n");
				cpssp->state_stage = STAGE_DATA;

				fprintf(stderr, " -> OUT\n");
				sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_OUT, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);

				fprintf(stderr, " -> DATA\n");
				sig_usb_bus_send_data(cpssp->bus, cpssp, USB_PID_DATA0, (cpssp->virtualdev_hci_interact).transfer_buffer_length, (cpssp->virtualdev_hci_interact).transfer_buffer, gen_crc16((cpssp->virtualdev_hci_interact).transfer_buffer, (cpssp->virtualdev_hci_interact).transfer_buffer_length));

				(cpssp->virtualdev_hci_interact).actual_length = (cpssp->virtualdev_hci_interact).transfer_buffer_length;
			} else {
				fprintf(stderr, " = STAGE_STATUS =\n");
				cpssp->state_stage = STAGE_STATUS;

				fprintf(stderr, " -> IN\n");
				sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_IN, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);
			}
		} else if (cpssp->state_recv == RECV_HANDSHAKE_NAK) {
			bridge_usb_recv_reset(cpssp);

			// TODO: Send again

			assert(0);
		} else if (cpssp->state_recv == RECV_HANDSHAKE_STALL) {
			bridge_usb_recv_reset(cpssp);

			// TODO: Push with error

			assert(0);
		} else if (cpssp->state_recv == RECV_DATA) {
			bridge_usb_recv_reset(cpssp);

			// TODO: Invalid, send NAK and try again

			assert(0);
		} else {
			// TODO: WAIT

			assert(0);
		}
	} else if (cpssp->state_stage == STAGE_DATA) {
		if (cpssp->state_recv == RECV_HANDSHAKE_ACK) {
			bridge_usb_recv_reset(cpssp);

			fprintf(stderr, " = STAGE_STATUS =\n");
			cpssp->state_stage = STAGE_STATUS;

			fprintf(stderr, " -> IN\n");
			sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_IN, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);
		} else if (cpssp->state_recv == RECV_HANDSHAKE_NAK) {
			bridge_usb_recv_reset(cpssp);

			// TODO: Send again

			assert(0);
		} else if (cpssp->state_recv == RECV_HANDSHAKE_STALL) {
			bridge_usb_recv_reset(cpssp);

			// TODO: Push with error

			assert(0);
		} else if (cpssp->state_recv == RECV_DATA) {
			bridge_usb_recv_reset(cpssp);

			assert(0);
		} else {
			// TODO: WAIT

			assert(0);
		}
	} else if (cpssp->state_stage == STAGE_STATUS) {
		if (cpssp->state_recv == RECV_DATA) {
			bridge_usb_recv_reset(cpssp);

			fprintf(stderr, " -> ACK\n");
			sig_usb_bus_send_handshake(cpssp->bus, cpssp, USB_PID_ACK);

			(cpssp->virtualdev_hci_interact).status = 0;
			(cpssp->virtualdev_hci_interact).error_count = 0;

			write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));

			fprintf(stderr, "### TRANSACTION END OK ###\n");
			cpssp->state_transaction = TRANSACTION_IDLE;
			cpssp->state_stage = STAGE_IDLE;

			bridge_usb_recv_reset(cpssp);
		} else if (cpssp->state_recv == RECV_HANDSHAKE_NAK) {
			bridge_usb_recv_reset(cpssp);

			(cpssp->virtualdev_hci_interact).status = 1;
			(cpssp->virtualdev_hci_interact).error_count = 0;

			write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));
			fprintf(stderr, "### TRANSACTION END ERROR ###\n");
			cpssp->state_transaction = TRANSACTION_IDLE;
			cpssp->state_stage = STAGE_IDLE;

			bridge_usb_recv_reset(cpssp);
		} else if (cpssp->state_recv == RECV_HANDSHAKE_STALL) {
			bridge_usb_recv_reset(cpssp);

			// TODO: Push with error

			assert(0);
		} else {
			assert(0);
		}
	} else {
		assert(0);
	}
}

static void bridge_usb_transaction_bulk_in(
	struct cpssp *cpssp)
{
	if (cpssp->state_stage == STAGE_IDLE) {
		fprintf(stderr, " = STAGE_DATA =\n");
		cpssp->state_stage = STAGE_DATA;

		fprintf(stderr, " -> IN\n");
		sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_IN, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);
	} else if (cpssp->state_stage == STAGE_DATA) {
		if (cpssp->state_recv == RECV_DATA) {
			unsigned int length = cpssp->state_recv_data_length;

			bridge_usb_recv_copy(cpssp);

			fprintf(stderr, " DATA: %ld / %ld\n", (cpssp->virtualdev_hci_interact).actual_length, (cpssp->virtualdev_hci_interact).transfer_buffer_length);

			bridge_usb_recv_reset(cpssp);

			fprintf(stderr, " -> ACK\n");
			sig_usb_bus_send_handshake(cpssp->bus, cpssp, USB_PID_ACK);

			if ((cpssp->virtualdev_hci_interact).actual_length < (cpssp->virtualdev_hci_interact).transfer_buffer_length && length >= 64) {
				fprintf(stderr, " -> IN\n");
				sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_IN, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);
			} else {
				(cpssp->virtualdev_hci_interact).status = 0;
				(cpssp->virtualdev_hci_interact).error_count = 0;

				write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));
				fprintf(stderr, "### TRANSACTION END OK ###\n");
				cpssp->state_transaction = TRANSACTION_IDLE;
				cpssp->state_stage = STAGE_IDLE;
			}
		} else if (cpssp->state_recv == RECV_DATA_0) {
			fprintf(stderr, " -> ACK\n");
			sig_usb_bus_send_handshake(cpssp->bus, cpssp, USB_PID_ACK);

			bridge_usb_recv_reset(cpssp);

			(cpssp->virtualdev_hci_interact).status = 0;
			(cpssp->virtualdev_hci_interact).error_count = 0;

			write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));
			fprintf(stderr, "### TRANSACTION END OK ###\n");
			cpssp->state_transaction = TRANSACTION_IDLE;
			cpssp->state_stage = STAGE_IDLE;
		} else if (cpssp->state_recv == RECV_HANDSHAKE_NAK) {
			bridge_usb_recv_reset(cpssp);

			(cpssp->virtualdev_hci_interact).status = 1;
			(cpssp->virtualdev_hci_interact).error_count = 0;

			write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));
			fprintf(stderr, "### TRANSACTION END OK ###\n");
			cpssp->state_transaction = TRANSACTION_IDLE;
			cpssp->state_stage = STAGE_IDLE;

		} else if (cpssp->state_recv == RECV_HANDSHAKE_STALL) {
			bridge_usb_recv_reset(cpssp);

			(cpssp->virtualdev_hci_interact).status = 1;
			(cpssp->virtualdev_hci_interact).error_count = 0;

			write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));
			fprintf(stderr, "### TRANSACTION END ERROR ###\n");
			cpssp->state_transaction = TRANSACTION_IDLE;
			cpssp->state_stage = STAGE_IDLE;

		} else {
			assert(0);
		}
	}
}

static void
bridge_usb_send_data(
	struct cpssp *cpssp, unsigned int length)
{
	if (length + (cpssp->virtualdev_hci_interact).actual_length > (cpssp->virtualdev_hci_interact).transfer_buffer_length) {
		length = (cpssp->virtualdev_hci_interact).transfer_buffer_length - (cpssp->virtualdev_hci_interact).actual_length;
	}

	fprintf(stderr, " -> DATA\n");
	sig_usb_bus_send_data(cpssp->bus, cpssp, USB_PID_DATA0,
		length, (cpssp->virtualdev_hci_interact).transfer_buffer + (cpssp->virtualdev_hci_interact).actual_length,
		gen_crc16((cpssp->virtualdev_hci_interact).transfer_buffer + (cpssp->virtualdev_hci_interact).actual_length, length));

	(cpssp->virtualdev_hci_interact).actual_length += length;
}

static void bridge_usb_transaction_bulk_out(
	struct cpssp *cpssp)
{
	if (cpssp->state_stage == STAGE_IDLE) {
		fprintf(stderr, " = STAGE_DATA =\n");
		cpssp->state_stage = STAGE_DATA;

		fprintf(stderr, " -> OUT\n");
		sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_OUT, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);

		bridge_usb_send_data(cpssp, 64);
	} else if (cpssp->state_stage == STAGE_DATA) {
		if (cpssp->state_recv == RECV_HANDSHAKE_ACK) {
			bridge_usb_recv_reset(cpssp);

			if ((cpssp->virtualdev_hci_interact).actual_length < (cpssp->virtualdev_hci_interact).transfer_buffer_length) {
				fprintf(stderr, " -> OUT\n");
				sig_usb_bus_send_token(cpssp->bus, cpssp, USB_PID_OUT, (cpssp->virtualdev_hci_interact).device, (cpssp->virtualdev_hci_interact).endpoint);

				bridge_usb_send_data(cpssp, 64);

			} else {
				(cpssp->virtualdev_hci_interact).status = 0;
				(cpssp->virtualdev_hci_interact).error_count = 0;

				write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));
				fprintf(stderr, "### TRANSACTION END OK ###\n");
				cpssp->state_transaction = TRANSACTION_IDLE;
				cpssp->state_stage = STAGE_IDLE;
			}
		} else if (cpssp->state_recv == RECV_HANDSHAKE_STALL) {
			bridge_usb_recv_reset(cpssp);

			(cpssp->virtualdev_hci_interact).status = 1;
			(cpssp->virtualdev_hci_interact).error_count = 0;

			write(cpssp->fd, &(cpssp->virtualdev_hci_interact), sizeof(struct virtualdev_hci_interact));
			fprintf(stderr, "### TRANSACTION END ERROR ###\n");
			cpssp->state_transaction = TRANSACTION_IDLE;
			cpssp->state_stage = STAGE_IDLE;
		} else {
			assert(0);
		}
	}
}

static void
bridge_usb_transaction_work(
	struct cpssp *cpssp)
{
	if (cpssp->state_transaction == TRANSACTION_CONTROL_IN) {
		bridge_usb_transaction_control_in(cpssp);
	} else if (cpssp->state_transaction == TRANSACTION_CONTROL_OUT) {
		bridge_usb_transaction_control_out(cpssp);
	} else if (cpssp->state_transaction == TRANSACTION_BULK_IN) {
		bridge_usb_transaction_bulk_in(cpssp);
	} else if (cpssp->state_transaction == TRANSACTION_BULK_OUT) {
		bridge_usb_transaction_bulk_out(cpssp);
	} else {
		assert(0);
	}
}

static void
bridge_usb_recv_data(
        void *_cpssp,
        int pid,
        unsigned int length,
        uint8_t *data,
        uint16_t crc16
)
{
	struct cpssp *cpssp = _cpssp;

	assert(cpssp->state_recv == RECV_NOTHING);

	if (length != 0) {
		fprintf(stderr, " <- DATA\n");
		cpssp->state_recv = RECV_DATA;
		cpssp->state_recv_data_length = length;
		cpssp->state_recv_data = malloc(cpssp->state_recv_data_length * sizeof(uint8_t));
		assert(cpssp->state_recv_data != NULL);
		memcpy(cpssp->state_recv_data, data, cpssp->state_recv_data_length);
	} else {
		fprintf(stderr, " <- DATA_0\n");
		cpssp->state_recv = RECV_DATA_0;
	}
}

static void
bridge_usb_recv_handshake(
	void *_cpssp,
	int pid)
{
	struct cpssp *cpssp = _cpssp;

	assert(cpssp->state_recv == RECV_NOTHING);

	switch(pid) {
		case USB_PID_ACK:
			cpssp->state_recv = RECV_HANDSHAKE_ACK;
			fprintf(stderr, " <- ACK\n");
			break;
		case USB_PID_NAK:
			fprintf(stderr, " <- NAK\n");
			cpssp->state_recv = RECV_HANDSHAKE_NAK;
			break;
		case USB_PID_STALL:
			fprintf(stderr, " <- STALL\n");
			cpssp->state_recv = RECV_HANDSHAKE_STALL;
			break;
	}
}

static int
bridge_usb_reset(struct cpssp *cpssp)
{
	// if (ioctl(cpssp->fd, VIRTUALDEV_IOC_PENDING_RESET, VIRTUALDEV_PDEV_TYPE_USB) == 0) {
	// 	sig_usb_bus_reset_set(cpssp->bus, cpssp, 1);
	// 	bridge_usb_recv_reset(cpssp);
	// 	cpssp->state_transaction = TRANSACTION_IDLE;
	// 	cpssp->state_stage = STAGE_IDLE;
	// 	return 1;
	// } else {
	// 	return 0;
	// }
	return 0;
}

static void
bridge_usb_timer_event(void *datap)
{
	struct cpssp *cpssp = datap;

	cpssp->timer_scheduled--;

	if(cpssp->state_transaction == TRANSACTION_IDLE) {
		if (!bridge_usb_reset(cpssp)) {
			bridge_usb_transaction_start(cpssp);
		}
	} else {
		bridge_usb_transaction_work(cpssp);
	}

	if (!cpssp->timer_scheduled) {
		cpssp->timer_scheduled++;
		cpssp->tsc_passed += cpssp->tsc_step;
		time_call_at(cpssp->tsc_passed + cpssp->tsc_step, bridge_usb_timer_event, cpssp);
	}
}

static void
bridge_usb_recv_token(
        void *_cpssp,
        int pid,
        int addr,
        int endp
)
{
	assert(0);
}

static void
bridge_usb_speed_set(
	void *_cpssp, int val)
{
	assert(0);
}

static void
bridge_usb_recv_sof(
        void *_cpssp,
        int frame_num
)
{
	assert(0);
}

void *
bridge_usb_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_usb_conn *port_usb
)
{
	static const struct sig_usb_bus_funcs main_funcs = {
		/* .reset_set = bridge_usb_reset_set, */
		.speed_set = bridge_usb_speed_set,
		.recv_token = bridge_usb_recv_token,
		.recv_sof = bridge_usb_recv_sof,
		.recv_data = bridge_usb_recv_data,
		.recv_handshake = bridge_usb_recv_handshake,
	};
	struct cpssp *cpssp;

	cpssp = malloc(sizeof(struct cpssp));
	assert(cpssp != NULL);

	cpssp->bus = port_usb->bus;
	sig_usb_bus_connect(port_usb->bus, cpssp, &main_funcs);

	cpssp->state_transaction = TRANSACTION_IDLE;
	cpssp->state_stage = STAGE_IDLE;
	cpssp->state_recv = RECV_NOTHING;

	sig_usb_bus_reset_set(cpssp->bus, cpssp, 0);

	cpssp->fd = open("/dev/virtualdev", O_RDWR);
	if (cpssp->fd == -1) {
		assert(0);
	}
	if (ioctl(cpssp->fd, VIRTUALDEV_IOC_REGISTER, VIRTUALDEV_PDEV_TYPE_USB) == -1) {
		assert(0);
	}
	if (ioctl(cpssp->fd, VIRTUALDEV_IOC_HCI_CONNECT) == -1) {
		assert(0);
	}

	/* schedule timer event handler */
	cpssp->tsc_passed = time_virt();
	cpssp->tsc_step = TIME_HZ / USB_TIMER_FREQ;
	time_call_at(cpssp->tsc_passed + cpssp->tsc_step, bridge_usb_timer_event, cpssp);
	cpssp->timer_scheduled++;

	return cpssp;
}

void
bridge_usb_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	if (ioctl(cpssp->fd, VIRTUALDEV_IOC_HCI_DISCONNECT) == -1) {
		assert(0);
	}

	close(cpssp->fd);

	free(cpssp);
}

void
bridge_usb_suspend(
	void *_cpssp,
	FILE *fp)
{
}

void
bridge_usb_resume(
	void *_cpssp,
	FILE *fp)
{
}

/* ####################################################################################################### */
