/* GStreamer
 *
 * unit test for FarsightStream object
 *
 * Copyright (C) <2007> Nokia Corporation.
 *     Contact: Zeeshan Ali <first.last@nokia.com>
 * Copyright (C) 2005,2006 Collabora Ltd.
 *     Contact: Philippe Khalaf <philippe.khalaf@collabora.co.uk>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <unistd.h>
#include <check.h>
#include <farsight/farsight-session.h>
#include <farsight/farsight-stream.h>
#include <farsight/farsight-transport.h>

#define TEST_HOST "localhost"
#define TEST_PORT 4958

GMainLoop *main_loop;
gboolean session_error, stream_error, sending_dtmf;

FarsightSession *setup_rtp_session ();
FarsightStream *setup_rtp_stream (FarsightSession *session);
void prepare_candidate (FarsightStream *stream, gchar * candidate_id);
void setup_transmitters (FarsightStream *stream);
void setup_source_and_sink (FarsightStream *stream);
void run_stream_for_n_seconds (FarsightStream *stream, guint seconds);
void on_session_error (FarsightSession *stream, FarsightSessionError error,
        const gchar *debug);
void on_stream_error (FarsightSession *stream, FarsightSessionError error,
        const gchar *debug);
gboolean normal_timeout (gpointer user_data);
gboolean dtmf_event (gpointer user_data);

START_TEST (test_creation)
{
  FarsightSession *session;
  FarsightStream *stream;

  main_loop = NULL;
  session = setup_rtp_session ();
  fail_if (session == NULL);
  stream = setup_rtp_stream (session);
  fail_if (stream == NULL);
}
END_TEST;

START_TEST (test_destruction)
{
  FarsightSession *session;
  FarsightStream *stream;

  main_loop = NULL;
  session = setup_rtp_session ();
  stream = setup_rtp_stream (session);
  
  farsight_session_destroy (session);
}
END_TEST;

START_TEST (test_transmitter_preparation)
{
  FarsightSession *session;
  FarsightStream *stream;

  main_loop = NULL;
  /*main_loop = g_main_loop_new (NULL, FALSE);
  fail_if (main_loop == NULL);*/
  
  session = setup_rtp_session ();
  stream = setup_rtp_stream (session);
  
  setup_transmitters (stream);

  g_print ("unrefing session\n");
  farsight_session_destroy (session);
  g_print ("session unrefed\n");
  
  fail_if (stream_error || session_error);
}
END_TEST;

START_TEST (test_candidate_preparation)
{
  FarsightSession *session;
  FarsightStream *stream;

  main_loop = NULL;
  session = setup_rtp_session ();
  stream = setup_rtp_stream (session);
  
  setup_transmitters (stream);
  prepare_candidate (stream, "L1");
  farsight_stream_set_active_candidate_pair (stream, "L1", "L1");
  
  farsight_session_destroy (session);
  fail_if (stream_error || session_error);
}
END_TEST;

START_TEST (test_source_sink_setup)
{
  FarsightSession *session;
  FarsightStream *stream;

  main_loop = NULL;
  session = setup_rtp_session ();
  stream = setup_rtp_stream (session);
  
  setup_transmitters (stream);
  prepare_candidate (stream, "L1");
  farsight_stream_set_active_candidate_pair (stream, "L1", "L1");
  setup_source_and_sink (stream);
  
  farsight_session_destroy (session);
  fail_if (stream_error || session_error);
}
END_TEST;

START_TEST (test_stream)
{
  FarsightSession *session;
  FarsightStream *stream;

  main_loop = g_main_loop_new (NULL, FALSE);
  fail_if (main_loop == NULL);
  
  session = setup_rtp_session ();
  stream = setup_rtp_stream (session);
  
  setup_transmitters (stream);
  prepare_candidate (stream, "L1");
  farsight_stream_set_active_candidate_pair (stream, "L1", "L1");
  setup_source_and_sink (stream);

  run_stream_for_n_seconds (stream, 3);

  farsight_session_destroy (session);
  fail_if (stream_error || session_error);
}
END_TEST;

START_TEST (test_dtmf)
{
  FarsightSession *session;
  FarsightStream *stream;

  main_loop = g_main_loop_new (NULL, FALSE);
  fail_if (main_loop == NULL);
  
  session = setup_rtp_session ();
  stream = setup_rtp_stream (session);
  
  setup_transmitters (stream);
  prepare_candidate (stream, "L1");
  farsight_stream_set_active_candidate_pair (stream, "L1", "L1");
  setup_source_and_sink (stream);

  sending_dtmf = FALSE;
  /* Start and stop DTMF event within 3 seconds */
  g_timeout_add (1500, dtmf_event, stream);
  run_stream_for_n_seconds (stream, 0);

  farsight_session_destroy (session);
  fail_if (stream_error || session_error);
}
END_TEST;

Suite *
stream_suite (void)
{
  Suite *s = suite_create ("stream");
  TCase *tc_general = tcase_create ("general");
  TCase *tc_api = tcase_create ("api");

  suite_add_tcase (s, tc_general);
  tcase_add_test (tc_general, test_creation);
  tcase_add_test (tc_general, test_destruction);
  
  suite_add_tcase (s, tc_api);
  tcase_add_test (tc_api, test_candidate_preparation);
  tcase_add_test (tc_api, test_transmitter_preparation);
  tcase_add_test (tc_api, test_source_sink_setup);
  tcase_add_test (tc_api, test_stream);
  tcase_add_test (tc_api, test_dtmf);
  tcase_set_timeout (tc_api, 15);

  return s;
}

int
main (int argc, char **argv)
{
  int nf;

  gst_init (&argc, &argv);

  Suite *s = stream_suite ();
  SRunner *sr = srunner_create (s);

  srunner_run_all (sr, CK_NORMAL);
  nf = srunner_ntests_failed (sr);
  srunner_free (sr);

  return nf;
}

FarsightSession *setup_rtp_session ()
{
    FarsightSession *session;

    session = farsight_session_factory_make ("rtp");

    if (!session) {
      g_error("RTP plugin not found");
      return NULL;
    }
    session_error = FALSE;
    g_signal_connect (G_OBJECT (session), "error", 
                      G_CALLBACK (on_session_error), NULL);

    g_print ("protocol details:\n name: %s\n description: %s\n author: %s\n",
             farsight_plugin_get_name (session->plugin),
             farsight_plugin_get_description (session->plugin),
             farsight_plugin_get_author (session->plugin));
 
    return session;
}

FarsightStream *setup_rtp_stream (FarsightSession *session)
{
    FarsightStream *stream;
    const GList *possible_codecs, *lp;
    FarsightCodec *codec;

    stream = farsight_session_create_stream (session,
            FARSIGHT_MEDIA_TYPE_AUDIO, FARSIGHT_STREAM_DIRECTION_BOTH);
    stream_error = FALSE;
    g_signal_connect (G_OBJECT (stream), "error", 
                      G_CALLBACK (on_stream_error), NULL);
    
    return stream;
}

void
setup_transmitters (FarsightStream *stream)
{
  g_object_set (G_OBJECT (stream), "transmitter", "rawudp", NULL);
  /* stunserver.org */
  /* automated tests shouldn't depend on net access */
  // g_object_set (G_OBJECT (stream), "stun_ip", "192.245.12.229", NULL);
  farsight_stream_prepare_transports (stream);
}

void
prepare_candidate (FarsightStream *stream, gchar * candidate_id)
{
  FarsightTransportInfo *trans = NULL;
  GList *candidate_glist = NULL;
  GList *local_codecs;
  FarsightCodec *codec;
  
  /* this assumes both hosts have the same codecs */
  local_codecs = farsight_codec_list_copy (
                  farsight_stream_get_local_codecs(stream));
  codec = g_new0 (FarsightCodec, 1);

  codec->media_type = FARSIGHT_MEDIA_TYPE_AUDIO;
  codec->encoding_name = g_strdup ("telephone-event");
  codec->clock_rate = 8000;
  codec->channels = 1;
  codec->id = 96;

  local_codecs = g_list_append (local_codecs, codec);
  farsight_stream_set_remote_codecs(stream, local_codecs);

  farsight_codec_list_destroy (local_codecs);

  trans = g_new0 (FarsightTransportInfo,1);
  trans->candidate_id = g_strdup_printf (candidate_id);
  trans->component = 1;
  trans->ip = g_strdup (TEST_HOST);
  trans->port = TEST_PORT;
  trans->proto = FARSIGHT_NETWORK_PROTOCOL_UDP;
  trans->proto_subtype = "RTP";
  trans->proto_profile = "AVP";
  trans->preference = 1.0;
  trans->type = FARSIGHT_CANDIDATE_TYPE_LOCAL;

  candidate_glist = g_list_append (candidate_glist, trans);
  farsight_stream_set_remote_candidate_list (stream, candidate_glist);
  
  g_free(trans);
  g_list_free (candidate_glist);
}

void
run_stream_for_n_seconds (FarsightStream *stream, guint seconds)
{
  if (seconds)
    g_timeout_add (seconds * 1000, normal_timeout, NULL);
  farsight_stream_start (stream);
  g_main_loop_run (main_loop);
}

void
setup_source_and_sink (FarsightStream *stream)
{
  GstElement *audiosrc, *audiosink;

  audiosrc = gst_element_factory_make("audiotestsrc", "audiosrc"); 
  audiosink = gst_element_factory_make("fakesink", "audiosink");

  g_object_set (G_OBJECT(audiosrc), "is-live", TRUE, NULL);
  g_object_set (G_OBJECT(audiosink), "sync", FALSE, NULL);
  g_object_set (G_OBJECT(audiosrc), "blocksize", 320, NULL);
    
  farsight_stream_set_source (stream, audiosrc);
  farsight_stream_set_sink (stream, audiosink);
}

void
on_session_error (FarsightSession *stream,
       FarsightSessionError error,
       const gchar *debug)
{
  g_print ("%s: session error: session=%p error=%s\n", __FUNCTION__, stream, debug);
  session_error = TRUE;
  if (main_loop)
    g_main_loop_quit (main_loop);
}

void
on_stream_error (FarsightSession *stream,
       FarsightSessionError error,
       const gchar *debug)
{
  g_print ("%s: session error: session=%p error=%s\n", __FUNCTION__, stream, debug);
  stream_error = TRUE;
  if (main_loop)
    g_main_loop_quit (main_loop);
}

gboolean
normal_timeout (gpointer user_data)
{
  if (main_loop)
    g_main_loop_quit (main_loop);

  return FALSE;
}

gboolean
dtmf_event (gpointer user_data)
{
  FarsightStream *stream = FARSIGHT_STREAM (user_data);

  if (sending_dtmf) {
    stream_error = !farsight_stream_stop_telephony_event (stream);
    
    sending_dtmf = FALSE;
    g_main_loop_quit (main_loop);
    return FALSE;
  } else {
    stream_error = !farsight_stream_start_telephony_event (stream, 1, 5);
    sending_dtmf = TRUE;
 
    if (stream_error)
      return FALSE;
    else
      return TRUE;
  }
} 
