//  Copyright (C) 2001 Red Hat, Inc.  All Rights Reserved.
//  Copyright (C) 1999 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify it
//  under the terms of version 2 of the GNU General Public License as
//  published by the Free Software Foundation.
//
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  Further, any
//  license provided herein, whether implied or otherwise, is limited to
//  this program in accordance with the express provisions of the GNU
//  General Public License.  Patent licenses, if any, provided herein do not
//  apply to combinations of this program with other product or programs, or
//  any other product whatsoever.  This program is distributed without any
//  warranty that the program is delivered free of the rightful claim of any
//  third person by way of infringement or the like.  See the GNU General
//  Public License for more details.
//
//  You should have received a copy of the GNU General Public License along
//  with this program; if not, write the Free Software Foundation, Inc., 59
//  Temple Place - Suite 330, Boston MA 02111-1307, USA.

#define _GNU_SOURCE  
#include <fcntl.h>

#include <string.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <libgen.h>

#include "DNotify.h"

#include "Interest.h"
#include "Log.h"
#include "Scheduler.h"
#include "alloc.h"


int DNotify::pipe_write_fd = -2;
int DNotify::pipe_read_fd = -2;
volatile sig_atomic_t DNotify::queue_overflowed = 0;
volatile sig_atomic_t DNotify::queue_changed = 0;
int DNotify::change_queue[QUEUESIZE];
volatile int DNotify::queue_head = 0; // Only modified by read handler
volatile int DNotify::queue_tail = 0; // Only modified by signal handler
DNotify::EventHandler DNotify::ehandler;

DNotify::DirWatch *DNotify::dir_hash[DIR_HASHSIZE];
DNotify::FileWatch *DNotify::file_hash[FILE_HASHSIZE];

struct DNotify::FileWatch
{
    DirWatch *dir_watch;
    dev_t file_dev;
    ino_t file_ino;
    FileWatch *next; // The DirWatch.watches list
    FileWatch *hash_link;
};

struct DNotify::DirWatch
{
    int fd;
    dev_t dir_dev;
    ino_t dir_ino;
    
    DirWatch *hash_link;
    FileWatch *watches;
};

DNotify::DNotify(EventHandler h)
{
    assert(ehandler == NULL);
    ehandler = h;
}

DNotify::~DNotify()
{
    if (pipe_read_fd >= 0)
    {
	//  Tell the scheduler.

	(void) Scheduler::remove_read_handler(pipe_read_fd);

	//  Close the pipe.

	if (close(pipe_read_fd) < 0)
	    Log::perror("can't pipe read end");
	else
	    Log::debug("closed pipe read end");
	
	if (close(pipe_write_fd) < 0)
	    Log::perror("can't pipe write end");
	else
	    Log::debug("closed pipe write end");
	pipe_read_fd = -1;
    }
    ehandler = NULL;
}

void
DNotify::overflow_signal_handler(int sig, siginfo_t *si, void *data)
{
  char c = 'x';
  
  {
    char *str = "*************** overflow sigqueue ***********************\n";
    write (0, str, strlen(str));
  }

  if (!queue_overflowed)
  {
      queue_overflowed = 1;
      // Trigger the read handler
      write(pipe_write_fd, &c, 1);
  }
}

void
DNotify::signal_handler(int sig, siginfo_t *si, void *data)
{
  int left;
  char c = 'x';

  if (queue_head <= queue_tail)
    left = (QUEUESIZE + queue_head) - queue_tail;
  else 
    left = queue_head - queue_tail;
  
  // Must leave at least one item unused to see difference
  // Betweeen empty and full
  if (left <= 1)
  {
      queue_overflowed = 1;
      {
	char *str = "*************** overflow famqueue ****************\n";
	write (0, str, strlen(str));
      }
  }
  else
  {
      change_queue[queue_tail] = si->si_fd;
      queue_tail = (queue_tail + 1) % QUEUESIZE;
  }
  
  if (!queue_changed)
  {
      queue_changed = 1;
      // Trigger the read handler
      write(pipe_write_fd, &c, 1);
  }
}

bool
DNotify::is_active()
{
    if (pipe_read_fd == -2)
    {
        int filedes[2];
	int res;
	
	res = pipe (filedes);
	if (res >= 0)
	{   Log::debug("opened pipe");
   	    pipe_read_fd = filedes[0];
   	    pipe_write_fd = filedes[1];

	    // Setup signal handler:
	    struct sigaction act;
	    
	    act.sa_sigaction = signal_handler;
	    sigemptyset(&act.sa_mask);
	    act.sa_flags = SA_SIGINFO;
	    sigaction(SIGRTMIN, &act, NULL);

	    // When the RT queue overflows we get a SIGIO
	    act.sa_sigaction = overflow_signal_handler;
	    sigemptyset(&act.sa_mask);
	    sigaction(SIGIO, &act, NULL);

	    (void) Scheduler::install_read_handler(pipe_read_fd, read_handler, NULL);
	}
    }
    return pipe_read_fd >= 0;
}

DNotify::DirWatch *
DNotify::lookup_dirwatch (int fd)
{
  DirWatch **p;
  DirWatch *w;

  p = dir_hashchain (fd);

  while (*p)
    {
      w = *p;

      if (w->fd == fd)
	return w;

      p = &w->hash_link;
    }
  
  return *p;
}

// This colud be made faster by using another hash table.
// But it's not that bad, since it is only used by express/revoke
DNotify::DirWatch *
DNotify::lookup_dirwatch (dev_t dir_dev, ino_t dir_ino)
{
  DirWatch *p;
  int i;

  for (i=0;i<DIR_HASHSIZE;i++)
    {
      p = dir_hash[i];
      
      while (p)
	{
	  if (p->dir_dev == dir_dev && p->dir_ino == dir_ino)
	    return p;
	  
	  p = p->hash_link;
	}
    }
  
  return NULL;
}

DNotify::FileWatch *
DNotify::lookup_filewatch (dev_t dev, ino_t ino)
{
  FileWatch **p;
  FileWatch *w;

  p = file_hashchain (dev, ino);

  while (*p)
    {
      w = *p;

      if (w->file_dev == dev && w->file_ino == ino)
	return w;

      p = &w->hash_link;
    }
  
  return *p;
}

// Make sure w is not already in the hash table before calling
// this function.
void
DNotify::hash_dirwatch(DirWatch *w)
{
  DirWatch **p;
  p = dir_hashchain (w->fd);
  w->hash_link = *p;
  *p = w;
}

// Make sure w is not already in the hash table before calling
// this function.
void
DNotify::hash_filewatch(FileWatch *w)
{
  FileWatch **p;
  p = file_hashchain (w->file_dev, w->file_ino);
  w->hash_link = *p;
  *p = w;
}

void
DNotify::unhash_dirwatch(DirWatch *w)
{
  DirWatch **p;
  
  p = dir_hashchain (w->fd);
  
  while (*p)
    {
      if (*p == w)
	{
	  *p = w->hash_link;
	  break;
	}
      p = &(*p)->hash_link;
    }
  w->hash_link = NULL;
}

void
DNotify::unhash_filewatch(FileWatch *w)
{
  FileWatch **p;
  
  p = file_hashchain (w->file_dev, w->file_ino);
  
  while (*p)
    {
      if (*p == w)
	{
	  *p = w->hash_link;
	  break;
	}
      p = &(*p)->hash_link;
    }
  w->hash_link = NULL;
}

DNotify::Status
DNotify::watch_dir(const char *notify_dir, dev_t file_dev, ino_t file_ino)
{
  struct stat stat;
  dev_t dir_dev;
  ino_t dir_ino;
  DirWatch *dwatch;
  FileWatch **p;
  FileWatch *fw;
    
  if (lstat (notify_dir, &stat) == -1)
      return BAD;
  
  dwatch = lookup_dirwatch(stat.st_dev, stat.st_ino);
  if (!dwatch)
    {
      Log::debug ("New DirWatch for %s (%x %x)\n",
		  notify_dir, (int)stat.st_dev, (int)stat.st_ino);
      dwatch = new DirWatch;
      dwatch->watches = NULL;
      dwatch->hash_link = NULL;
      dwatch->dir_dev = stat.st_dev;
      dwatch->dir_ino = stat.st_ino;
      
      dwatch->fd = open(notify_dir, O_RDONLY);
      fcntl (dwatch->fd, F_SETSIG, SIGRTMIN);
      fcntl (dwatch->fd, F_NOTIFY,
	     (DN_MODIFY|DN_CREATE|DN_DELETE|DN_RENAME|DN_ATTRIB) | DN_MULTISHOT);
      hash_dirwatch (dwatch);
    }

  for (p=&dwatch->watches; *p; p=&(*p)->next)
    {
      fw = *p;
      if (fw->file_dev == file_dev && fw->file_ino == file_ino)
	return OK;
    }
  
  // No old FileWatch, need to add one:
  Log::debug("New FileWatch for %x %x\n", (int)file_dev, (int)file_ino);
  *p = new FileWatch;
  fw = *p;
  fw->next = NULL;
  fw->file_dev = file_dev;
  fw->file_ino = file_ino;
  fw->dir_watch = dwatch;
  hash_filewatch(fw);
  return OK;
}

char *
dirname_dup (const char *name)
{
  char *copy = strdup(name);
  char *res = dirname(copy);
  res = strdup(res);
  free (copy);
  return res;
}

DNotify::Status
DNotify::express(const char *name, struct stat *status)
{
    struct stat stat;
    char *notify_dir;
    int res;
    Status s;
    dev_t dev;
    ino_t ino;

    Log::debug("express() name: %s\n", name);

    if (!is_active())
	return BAD;

    if (::lstat (name, &stat) == -1)
      return BAD;

    dev = stat.st_dev;
    ino = stat.st_ino;
    
    if ((stat.st_mode & S_IFMT) != S_IFDIR)
	notify_dir = dirname_dup (name);
    else
	notify_dir = (char *)name;

    s = watch_dir (notify_dir, dev, ino);
    if (notify_dir != name)
        free (notify_dir);
    if (s)
      return s;

    // Check for a race condition; if someone removed or changed the
    // file at the same time that we are expressing interest in it,
    // revoke the interest so we don't get notifications about changes
    // to a recycled inode that we don't otherwise care about.
    //
    struct stat st;
    if (status == NULL) {
	status = &st;
    }
    if (::lstat(name, status) == -1) {
	Log::perror("stat on \"%s\" failed", name);
	revoke(name, stat.st_dev, stat.st_ino);
	return BAD;
    }
    if (status->st_dev != stat.st_dev
	|| status->st_ino != stat.st_ino) {
	Log::error("File \"%s\" changed between express and stat",
		   name);
	revoke(name, stat.st_dev, stat.st_ino);
	return BAD;
    }	

    Log::debug("told dnotify to monitor \"%s\" = dev %d/%d, ino %d", name,
	       major(status->st_dev), minor(status->st_dev),
	       status->st_ino);
    return OK;
}

DNotify::Status
DNotify::revoke(const char *name, dev_t dev, ino_t ino)
{
    FileWatch *fwatch;
    DirWatch *dwatch;
    
    Log::debug("revoke() name: %s, dev: %x, ino: %x\n", name, dev, ino);

    if (!is_active())
	return BAD;

    // Lookup FileWatch by dev:ino, and its DirWatch.
    fwatch = lookup_filewatch (dev, ino);
    if (fwatch == NULL)
	return BAD;
    
    dwatch = fwatch->dir_watch;
    
    // delete FileWatch, if last FileWatch: close fd, delete DirWatch
    Log::debug ("Destroying FileWatch for (%x %x)\n",
		(int)fwatch->file_dev, (int)fwatch->file_ino);
    FileWatch **p;
    for (p=&dwatch->watches; *p; p=&(*p)->next)
    {
      if (*p == fwatch)
	{
	  *p = (*p)->next;
	  break;
	}
    }
    unhash_filewatch(fwatch);
    delete fwatch;
    if (dwatch->watches == NULL)
      {
	Log::debug ("Destroying DirWatch for (%x %x)\n",
		    (int)dwatch->dir_dev, (int)dwatch->dir_ino);
	close(dwatch->fd);
	unhash_dirwatch(dwatch);
	delete dwatch;
      }
  
    return OK;
}


void
DNotify::all_watches_changed(void)
{
  int i;
  FileWatch *fw;

  for (i=0; i<FILE_HASHSIZE; i++)
  {
      fw = file_hash[i];
      while (fw)
      {
	  (*ehandler)(fw->file_dev, fw->file_ino, CHANGE);

	  fw = fw->hash_link;
      }
  }
}


void
DNotify::read_handler(int fd, void *)
{
    static char readbuf[5000];
    DirWatch *dw;
    FileWatch *fw;
    int snap_queue_tail;
    int last_fd;

    int rc = read(fd, readbuf, sizeof readbuf);
    queue_changed = 0;
    if (rc < 0)
        Log::perror("pipe read");
    else if (queue_overflowed)
    {
	  // There is a *slight* race condition here. Between reading
	  // the queue_overflow flag and resetting it. But it doesn't
	  // matter, since I'm gonna handle the overflow after reseting
	  // anyway.
	  queue_overflowed = false;

	  // We're soon gonna check all watches anyway, so
	  // get rid of the current queue
	  queue_head = queue_tail;
	  
	  all_watches_changed ();
    }
    else
    {
	// Don't read events that happen later than
	// the initial read. (Otherwise skipping fd's
	// might miss some changes).
	snap_queue_tail = queue_tail;
	last_fd = -1;
	while (queue_head != snap_queue_tail)
	{
	    fd = change_queue[queue_head];
	    queue_head = (queue_head + 1) % QUEUESIZE;

	    // Skip multiple changes to the same fd
	    if (fd != last_fd)
	    {
		dw = lookup_dirwatch (fd);
		if (dw)
		{
		    Log::debug("dnotify said dev %d/%d, ino %ld changed",
			       major(dw->dir_dev), minor(dw->dir_dev), dw->dir_ino);
		    for (fw=dw->watches; fw; fw=fw->next)
		    {
			(*ehandler)(fw->file_dev, fw->file_ino, CHANGE);
		    }
		}
	    }
	    last_fd = fd;
	}
    }
}

