/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: snapshot.c
 *
 * This module provides support for snapshotting volumes. This function is
 * implemented as an associative feature so that it can be applied at any time
 * to an existing volume. The 'snapshot' volume is created on top of an
 * existing, unused storage object, and becomes an EVMS volume.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "snapshot.h"

engine_functions_t * EngFncs;
plugin_record_t * my_plugin_record;

/**
 * snap_setup_evms_plugin
 *
 * Make a local copy of the engine functions.
 **/
static int snap_setup_evms_plugin(engine_functions_t * functions)
{
	EngFncs = functions;
	my_plugin_record = &Snapshot_Plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * snap_cleanup_evms_plugin
 *
 * Get the list of all objects owned by snapshotting, and delete the snapshot
 * private data. Do not actually free the objects - the engine will handle that.
 **/
static void snap_cleanup_evms_plugin(void)
{
	list_anchor_t tmp_list;
	storage_object_t * object;
	snapshot_volume_t * volume;
	list_element_t itr1, itr2;
	int rc = 0;

	LOG_ENTRY();

	rc = EngFncs->get_object_list(EVMS_OBJECT, DATA_TYPE,
				      my_plugin_record, NULL, 0, &tmp_list);
	if (!rc) {
		LIST_FOR_EACH_SAFE(tmp_list, itr1, itr2, object) {
			EngFncs->delete_element(itr1);
			volume = object->private_data;
			if (is_origin(volume)) {
				deallocate_origin(volume);
			} else {
				deallocate_snapshot(volume);
			}
		}
		EngFncs->destroy_list(tmp_list);
	}

	LOG_EXIT_VOID();
}

/**
 * snap_can_delete
 *
 * Only allow deletes of snapshot objects, not origins. The origins will be
 * automatically deleted when their last snapshot is deleted.
 **/
static int snap_can_delete(storage_object_t * object)
{
	snapshot_volume_t * volume = object->private_data;
	int rc = 0;

	LOG_ENTRY();
	if (is_origin(volume)) {
		rc = EINVAL;
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_can_expand
 *
 * A snapshot volume can be expanded by allowing its child object to expand.
 * Origin volumes cannot expand at this time. There aren't any restrictions
 * on when you can expand a snapshot, because the snapshot device itself isn't
 * expanded. Instead, the underlying device is expanded, and the snapshot
 * device just happens to use the new space that's available for storing
 * copy-on-writes.
 **/
static int snap_can_expand(storage_object_t * object,
			   sector_count_t expand_limit,
			   list_anchor_t expand_points)
{
	snapshot_volume_t * volume = object->private_data;
	int rc = EINVAL;

	LOG_ENTRY();

	if (!is_origin(volume)) {
		rc = volume->child->plugin->functions.plugin->can_expand(volume->child,
									 expand_limit,
									 expand_points);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_can_expand_by
 *
 * If a snapshot's child object can expand, the snapshot object can expand by
 * the same amount. Origin volumes cannot expand at this time.
 **/
static int snap_can_expand_by(storage_object_t * object,
			      sector_count_t * size)
{
	snapshot_volume_t * volume = object->private_data;
	int rc = EINVAL;

	LOG_ENTRY();

	if (!is_origin(volume)) {
		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_can_set_volume
 *
 * Origins cannot have their volume removed. Origins may have their volume
 * set. This will only happen if they are being renamed. Snapshots no longer
 * care whether they have a volume.
 **/
static int snap_can_set_volume(storage_object_t * object,
			       boolean creating)
{
	snapshot_volume_t * volume = object->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (is_origin(volume) && !creating) {
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_discover
 * @input_objects:	List of one object, which will be the snapshot child.
 * @output_objects:	List to add the newly created snapshot object to.
 * @final_call:		Ignored in snapshotting.
 *
 * Read the snapshot metadata from the child object. Find the origin volume.
 * Create a new snapshot object. Create a new origin object (if this is the
 * first snapshot of that origin) and insert it in the origin volume's
 * stack.
 **/
static int snap_discover(list_anchor_t input_objects,
			 list_anchor_t output_objects,
			 boolean final_call)
{
	storage_object_t * org_child, * snap_child;
	snapshot_volume_t * org_volume, * snap_volume = NULL;
	snapshot_metadata_t * metadata = NULL;
	int rc = EINVAL;

	LOG_ENTRY();

	/* Get the snapshot child object from the input list. */
	snap_child = EngFncs->first_thing(input_objects, NULL);
	if (!snap_child) {
		LOG_ERROR("Discovery input list empty or corrupt.\n");
		goto error;
	}

	/* Read and verify the snapshot metadata. */
	rc = get_snapshot_metadata(snap_child, &metadata);
	if (rc) {
		LOG_ERROR("Error getting snapshot metadata from %s.\n",
			  snap_child->name);
		goto error;
	}

	/* Look up the origin volume. */
	org_child = find_origin((char *)metadata->origin_volume);
	if (!org_child) {
		goto error;
	}

	/* Verify the origin volume has the correct size. */
	if (org_child->volume->vol_size != metadata->origin_size) {
		goto error;
	}

	/* Allocate a new snapshot. */
	snap_volume = allocate_snapshot(snap_child, metadata);
	if (!snap_volume) {
		goto error;
	}

	if (org_child->plugin == my_plugin_record) {
		/* Already have at least one snapshot on this origin. */
		org_volume = org_child->private_data;
		org_child = org_volume->child;
	} else {
		/* Need a new origin. */
		org_volume = allocate_origin(org_child);
		if (!org_volume) {
			goto error;
		}
	}

	/* Update the status of the snapshot and origin from the kernel. */
	get_snapshot_state(snap_volume);
	get_origin_state(org_volume);

	/* Link the snapshot and the origin together. */
	add_snapshot_to_origin(snap_volume, org_volume);

	if (is_invalid(snap_volume)) {
		/* Display a warning if the snapshot is full/disabled. */
		MESSAGE(_("Snapshot object %s discovered in full/disabled state.  "
			  "Use the \"Reset\" function to reset the snapshot to "
			  "the current state of %s"),
			snap_volume->parent->name,
			org_volume->parent->volume->name);
	} else if (rollback_is_pending(snap_volume)) {
		/* Display a warning if the snapshot was interrupted in the
		 * middle of a rollback.
		 */
		MESSAGE(_("Snapshot object %s needs to complete a roll-back "
			  "to %s. The operation was interrupted while in-"
			  "progress. Save changes to continue this roll-back "
			  "from the point it was interrupted.\n"),
			snap_volume->parent->name,
			org_volume->parent->volume->name);
		schedule_for_commit(snap_volume);
	}

	/* Put the snapshot parent on the output list. */
	EngFncs->insert_thing(output_objects, snap_volume->parent,
			      INSERT_BEFORE, NULL);

	LOG_EXIT_INT(0);
	return 0;

error:
	deallocate_snapshot(snap_volume);
	EngFncs->engine_free(metadata);
	LOG_EXIT_INT(EVMS_FEATURE_FATAL_ERROR);
	return EVMS_FEATURE_FATAL_ERROR;
}

/**
 * snap_create
 * @objects:		List of one object, which will be the new snapshot child.
 * @options:		Array of creation options
 *			- object name
 *			- volume to snapshot
 *			- chunk size
 *			- read-only or writeable
 * @new_objects:	List to output the new snapshot and origin objects to.
 *
 * Create a new snapshot object.
 **/
static int snap_create(list_anchor_t objects,
		       option_array_t * options,
		       list_anchor_t new_objects)
{
	storage_object_t * org_child, * snap_child;
	snapshot_volume_t * org_volume, * snap_volume = NULL;
	evms_feature_header_t * feature_header = NULL;
	snapshot_metadata_t * metadata = NULL;
	char * org_vol_name, * snap_name;
	u_int32_t chunk_size;
	int writeable, new_origin = FALSE, rc = EINVAL;

	LOG_ENTRY();

	/* Get the snapshot child object from the input list. */
	snap_child = EngFncs->first_thing(objects, NULL);
	if (!snap_child) {
		LOG_ERROR("Input list corrupt.\n");
		goto error;
	}

	/* Parse and verify the options for creating the new snapshot. */
	parse_creation_options(options, &org_vol_name, &snap_name,
			       &chunk_size, &writeable);

	rc = verify_creation_options(org_vol_name, snap_name,
				     chunk_size, snap_child);
	if (rc) {
		goto error;
	}

	/* Find the specified origin volume, and check that
	 * it is valid for using as a snapshot origin.
	 */
	org_child = find_origin(skip_dev_path(org_vol_name));
	if (!org_child) {
		rc = EINVAL;
		goto error;
	}

	rc = verify_origin(org_child, snap_child);
	if (rc) {
		goto error;
	}

	/* Allocate a new feature header and metadata for the snapshot. */
	feature_header = allocate_feature_header(snap_child, snap_name);
	if (!feature_header) {
		rc = ENOMEM;
		goto error;
	}
	snap_child->feature_header = feature_header;

	metadata = allocate_metadata(org_vol_name, org_child->volume->vol_size,
				     snap_child->size, chunk_size, writeable);
	if (!metadata) {
		rc = ENOMEM;
		goto error;
	}

	/* Allocate the new snapshot */
	snap_volume = allocate_snapshot(snap_child, metadata);
	if (!snap_volume) {
		rc = ENOMEM;
		goto error;
	}

	if (org_child->plugin == my_plugin_record) {
		/* Already have at least one snapshot on this origin. */
		org_volume = org_child->private_data;
		org_child = org_volume->child;
	} else {
		/* Need a new origin. */
		org_volume = allocate_origin(org_child);
		if (!org_volume) {
			LOG_ERROR("Memory error allocating origin %s.\n",
				  org_vol_name);
			goto error;
		}
		new_origin = TRUE;
	}

	/* Link the snapshot and the origin together. */
	add_snapshot_to_origin(snap_volume, org_volume);

	/* This snapshot's metadata needs to be written during commit. */
	schedule_for_commit(snap_volume);

	/* Erase the snapshot header so it looks like a new snapshot to DM. */
	KILL_SECTORS(snap_child, 0, 1);

	/* Put newly created objects on the new_objects list. The engine will
	 * use this list to decide which objects should be activated.
	 */
	if (new_origin) {
		EngFncs->insert_thing(new_objects, org_volume->parent,
				      INSERT_AFTER, NULL);
	}
	EngFncs->insert_thing(new_objects, snap_volume->parent,
			      INSERT_AFTER, NULL);

	LOG_EXIT_INT(0);
	return 0;

error:
	if (snap_child) {
		snap_child->feature_header = NULL;
	}
	deallocate_snapshot(snap_volume);
	EngFncs->engine_free(feature_header);
	EngFncs->engine_free(metadata);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_discard
 *
 * Discard the specified snapshot or origin object. Unlike delete, the snapshot
 * and origin objects are handled independently. However, the engine will
 * ensure that all snapshots are discarded before their origin is discarded.
 **/
static int snap_discard(list_anchor_t objects)
{
	storage_object_t * object, * child;
	snapshot_volume_t * volume;
	int rc = 0;

	LOG_ENTRY();

	/* The way the engine calls this routine, there should only be
	 * one item on this list for the snapshot plugin.
	 */
	object = EngFncs->first_thing(objects, NULL);
	if (!object) {
		rc = EINVAL;
		goto out;
	}

	volume = object->private_data;
	child = volume->child;

	if (is_origin(volume)) {
		/* The origin must have no snapshots left. */
		if (volume->count != 0) {
			rc = EBUSY;
			goto out;
		}

		unmake_parent_and_child(object, child);
		child->volume->object = child;
		EngFncs->free_evms_object(object);
		if (is_active(volume)) {
			schedule_for_delete(volume);
		} else {
			EngFncs->engine_free(volume);
		}
	} else {
		unmake_parent_and_child(object, child);
		remove_snapshot_from_origin(volume);
		EngFncs->free_evms_object(object);

		if (is_active(volume)) {
			/* Wait until deactivation to delete all private data. */
			schedule_for_delete(volume);
		} else {
			/* Delete all private data now. */
			EngFncs->engine_free(volume->metadata);
			EngFncs->engine_free(volume->sibling);
			EngFncs->engine_free(volume);
		}
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_delete
 *
 * Delete the specified snapshot object.
 *
 * If the object is currently active, all of the private data must be left
 * intact so it can be deactivated later. At deactivation time, all of the
 * private data will be deleted.
 *
 * If the object is currently inactive, everything can be deleted now.
 *
 * Either way, queue up kill-sectors to erase the metadata from disk.
 **/
static int snap_delete(storage_object_t * snap_parent,
		       list_anchor_t child_objects)
{
	snapshot_volume_t * snap_volume = snap_parent->private_data;
	snapshot_volume_t * org_volume = snap_volume->origin;
	storage_object_t * snap_child = snap_volume->child;
	int snap_active = is_active(snap_volume);
	int org_active = is_active(org_volume);
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Deleting snapshot %s.\n", snap_parent->name);

	/* Don't allow origins to be deleted. */
	rc = snap_can_delete(snap_parent);
	if (rc) {
		goto out;
	}

	unmake_parent_and_child(snap_parent, snap_child);
	remove_snapshot_from_origin(snap_volume);
	EngFncs->free_evms_object(snap_parent);

	if (snap_active) {
		/* Wait until deactivation to delete all private data. */
		schedule_for_delete(snap_volume);
	} else {
		/* Delete all private data now. */
		EngFncs->engine_free(snap_volume->metadata);
		EngFncs->engine_free(snap_volume->sibling);
		EngFncs->engine_free(snap_volume);
	}

	/* If the origin has no more snapshots, it can be deleted. */
	if (org_volume->count == 0) {
		LOG_DEBUG("Deleting origin %s.\n", org_volume->parent->name);
		unmake_parent_and_child(org_volume->parent, org_volume->child);
		org_volume->child->volume->object = org_volume->child;
		EngFncs->free_evms_object(org_volume->parent);
		if (org_active) {
			schedule_for_delete(org_volume);
		} else {
			EngFncs->engine_free(org_volume);
		}
	}

	erase_snapshot_metadata(snap_child);

	EngFncs->insert_thing(child_objects, snap_child, INSERT_BEFORE, NULL);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_expand
 *
 * Start by sending the expand down to the snapshot child. When that is
 * complete, adjust the sizes in the snapshot metadata, the feature-header,
 * and the snapshot sibling. The snapshot parent object will not actually
 * be expanded.
 **/
static int snap_expand(storage_object_t * object,
		       storage_object_t * expand_object,
		       list_anchor_t input_objects,
		       option_array_t * options)
{
	snapshot_volume_t * volume = object->private_data;
	int rc;

	LOG_ENTRY();

	if (is_origin(volume)) {
		/* Can't expand origin volumes yet. */
		rc = EINVAL;
		goto out;
	}

	rc = volume->child->plugin->functions.plugin->expand(volume->child,
							     expand_object,
							     input_objects,
							     options);
	if (rc) {
		goto out;
	}

	volume->child->feature_header->feature_data1_start_lsn = volume->child->size - 3;
	volume->sibling->size = volume->child->feature_header->feature_data1_start_lsn;
	volume->metadata->total_chunks = calculate_data_chunks(volume->child->size,
							       volume->metadata->chunk_size);

	schedule_for_commit(volume);
	volume->child->flags |= SOFLAG_FEATURE_HEADER_DIRTY;

	/* Reactivate the snapshot only if it is already active
	 * and not waiting to be deactivated.
	 */
	if (is_active(volume) &&
	    !deactivate_is_pending(volume)) {
		schedule_for_activate(volume);
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_set_volume
 *
 * When a volume is added to an origin, it means the origin volume is being
 * renamed. The metadata for all of its snapshots must be updated to reflect
 * the new name.
 *
 * Snapshots no longer care whether they have EVMS volumes.
 **/
static void snap_set_volume(storage_object_t * object,
			    boolean creating)
{
	snapshot_volume_t * volume = object->private_data;
	snapshot_volume_t * snap_volume;

	LOG_ENTRY();
	LOG_DEBUG("%s volume on %s.\n",
		  creating ? "Adding" : "Removing", object->name);

	if (is_origin(volume) && creating) {
		for (snap_volume = volume->next;
		     snap_volume; snap_volume = snap_volume->next) {
			strncpy((char *)snap_volume->metadata->origin_volume,
				skip_dev_path(object->volume->name),
				EVMS_VOLUME_NAME_SIZE);
			schedule_for_commit(snap_volume);
		}
	}

	LOG_EXIT_VOID();
}

/**
 * snap_add_sectors_to_kill_list
 *
 * We don't really expect to get called on this API.
 **/
static int snap_add_sectors_to_kill_list(storage_object_t * object,
					 lsn_t lsn,
					 sector_count_t count)
{
	LOG_ENTRY();
	LOG_WARNING("WARNING!!! Called for Kill-Sectors!\n");
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * snap_commit_changes
 *
 * FIRST_METADATA_WRITE:
 *   Write out the metadata sector for the specified snapshot object. Clearing
 *   the snapshot header sector is handled in set_volume.
 *
 * MOVE:
 *   Snapshot rollback is handled during the MOVE phase. After the rollback is
 *   complete, the snapshot is reset and reactivated, giving a fresh snapshot
 *   of its origin.
 *
 * All other commit-phases are ignored.
 **/
static int snap_commit_changes(storage_object_t * object,
			       commit_phase_t phase)
{
	snapshot_volume_t * volume = object->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (is_origin(volume)) {
		/* Origin volumes are never committed. */
		commit_complete(volume);
		goto out;
	}

	switch (phase) {

	case FIRST_METADATA_WRITE:
		/* Need to check for rollback so we don't write the rollback
		 * flag to the metadata if the rollback can't proceed.
		 */
		if (rollback_is_pending(volume)) {
			rc = can_rollback(volume, TRUE);
			if (rc) {
				break;
			}
		}

		rc = write_snapshot_metadata(volume, FALSE);
		break;

	case MOVE:
		rc = rollback(volume);
		break;

	default:
		goto out;
	}

	if (!rc && !rollback_is_pending(volume)) {
		commit_complete(volume);
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

static int snap_can_activate(storage_object_t * object)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

static int snap_can_deactivate(storage_object_t * object)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * snap_activate
 *
 * Activate a snapshot or an origin. Origins are expected to be activated
 * independently, and prior to activating any of their snapshots.
 **/
static int snap_activate(storage_object_t * object)
{
	snapshot_volume_t * volume = object->private_data;
	snapshot_volume_t * snap_volume;
	int rc = 0;

	LOG_ENTRY();

	if (is_active(volume)) {
		if (!is_origin(volume)) {
			/* Reactivating a snapshot means the snapshot was
			 * expanded, and only the snapshot sibling needs
			 * to be reactivated.
			 */
			rc = activate_snapshot_sibling(volume);
		}
		goto out;
	}

	LOG_DEBUG("Activating %s.\n", object->name);

	if (is_origin(volume)) {
		/* Since we're activating the origin, we know none of the
		 * snapshots can be active yet. We need to check that all the
		 * snapshots will be activated. Any that are not scheduled for
		 * activation need to be reset, since they will no longer be
		 * valid.
		 *
		 * FIXME: Instead of always resetting the snapshot, we might
		 * want to prompt the user whether to reset the snapshot, or
		 * mark it for activation, or fail the activation of the origin.
		 */
		for (snap_volume = volume->next; snap_volume;
		     snap_volume = snap_volume->next) {
			if (!activate_is_pending(snap_volume)) {
				rc = erase_snapshot_header(snap_volume, FALSE);
				if (rc) {
					goto out;
				}
			}
		}

		rc = activate_origin(volume);

	} else {
		/* Only activate the snapshot if its origin volume is active. */
		if (is_active(volume->origin)) {
			rc = activate_snapshot(volume);
		} else {
			LOG_ERROR("Cannot activate snapshot %s with an "
				  "inactive origin.\n", object->name);
			rc = EINVAL;
		}
	}

out:
	if (!rc) {
		activate_complete(volume);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_deactivate
 *
 * Deactivate a snapshot or an origin. If it is marked for a pending delete,
 * finish freeing all the private data. All snapshots must be deactivated
 * before the origin can be deactivated.
 **/
static int snap_deactivate(storage_object_t * object)
{
	snapshot_volume_t * volume = object->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (!is_active(volume)) {
		deactivate_complete(volume);
		goto out;
	}

	LOG_DEBUG("Deactivating %s.\n", object->name);

	if (is_origin(volume)) {
		if (volume->active_count) {
			LOG_ERROR("Cannot deactivate origin %s with %d active "
				  "snapshots.\n", object->name,
				  volume->active_count);
			rc = EINVAL;
			goto out;
		}
		deactivate_origin(volume);
	} else {
		deactivate_snapshot(volume);
		/* If we deactivate the snapshot without deactivating the
		 * origin, the snapshot must be reset.
		 */
		if (!deactivate_is_pending(volume->origin)) {
			LOG_WARNING("Deactivating snapshot %s while origin %s "
				    "remains active. The snapshot header must "
				    "be erased to force a reset the next time "
				    "this snapshot is activated.\n",
				    volume->parent->name,
				    volume->origin->parent->volume->name);
			erase_snapshot_header(volume, FALSE);
		}
	}

	deactivate_complete(volume);

	if (delete_is_pending(volume)) {
		if (!is_origin(volume)) {
			EngFncs->engine_free(volume->metadata);
			EngFncs->engine_free(volume->sibling);
		}
		EngFncs->engine_free(volume);
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_get_option_count
 *
 * Return the number of available options for the specified task.
 **/
static int snap_get_option_count(task_context_t * context)
{
	int count = 0;

	LOG_ENTRY();

	switch (context->action) {

	case EVMS_Task_Create:
		count = SNAP_OPTION_WRITEABLE_INDEX + 1;
		break;

	case SNAPSHOT_FUNCTION_RESET:
	case SNAPSHOT_FUNCTION_ROLLBACK:
		count = 0;
		break;

	default:
		count = -1;
	}

	LOG_EXIT_INT(count);
	return count;
}

/**
 * snap_init_task
 *
 * Prepare an option descriptor for the specified task.
 **/
static int snap_init_task(task_context_t * context)
{
	int rc;

	LOG_ENTRY();

	switch (context->action) {

	case EVMS_Task_Create:
		rc = init_task_create(context);
		break;

	case SNAPSHOT_FUNCTION_RESET:
	case SNAPSHOT_FUNCTION_ROLLBACK:
		rc = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_set_option
 *
 * Set an option for the specified task.
 **/
static int snap_set_option(task_context_t * context,
			   u_int32_t index,
			   value_t * value,
			   task_effect_t * effect)
{
	int rc;

	LOG_ENTRY();

	switch (context->action) {

	case EVMS_Task_Create:
		rc = set_option_create(context, index, value, effect);
		break;

	case SNAPSHOT_FUNCTION_RESET:
	case SNAPSHOT_FUNCTION_ROLLBACK:
		rc = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_set_objects
 *
 * Verify a selected object for the specified task.
 **/
static int snap_set_objects(task_context_t * context,
			    list_anchor_t declined_objects,
			    task_effect_t * effect)
{
	int rc;

	LOG_ENTRY();

	switch (context->action) {

	case EVMS_Task_Create:
		rc = set_objects_create(context, declined_objects, effect);
		break;

	case SNAPSHOT_FUNCTION_RESET:
	case SNAPSHOT_FUNCTION_ROLLBACK:
		rc = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_get_info
 **/
static int snap_get_info(storage_object_t * object,
			 char * name,
			 extended_info_array_t ** info_array)
{
	extended_info_array_t * info;
	snapshot_volume_t * volume = object->private_data;
	snapshot_volume_t * snap_volume;
	int i = 0, rc = 0;

	LOG_ENTRY();

	if (name) {
		LOG_ERROR("No support for extra information about "
			  "\"%s\" for object %s.\n", name, object->name);
		rc = EINVAL;
		goto out;
	}

	if (is_origin(volume)) {
		/* Origin */
		info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
					     sizeof(extended_info_t) * volume->count);
		if (!info) {
			rc = ENOMEM;
			goto out;
		}

		for (i = 0, snap_volume = volume->next;
		     snap_volume; snap_volume = snap_volume->next, i++) {
			info->info[i].name = EngFncs->engine_strdup("SnapShot");
			info->info[i].title = EngFncs->engine_strdup(_("SnapShotted on"));
			info->info[i].desc = EngFncs->engine_strdup(_("Snapshots of this volume"));
			info->info[i].type = EVMS_Type_String;
			if (snap_volume->parent->volume) {
				info->info[i].value.s = EngFncs->engine_strdup(snap_volume->parent->volume->name);
			} else {
				info->info[i].value.s = EngFncs->engine_strdup(snap_volume->parent->name);
			}
		}
	} else {
		/* Snapshot */
		info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
					     sizeof(extended_info_t) * 5);
		if (!info) {
			rc = ENOMEM;
			goto out;
		}

		get_snapshot_state(volume);

		if (is_active(volume)) {
			info->info[i].name = EngFncs->engine_strdup("Original");
			info->info[i].title = EngFncs->engine_strdup(_("SnapShot of"));
		} else {
			info->info[i].name = EngFncs->engine_strdup("iOriginal");
			info->info[i].title = EngFncs->engine_strdup(_("Inactive SnapShot of"));
		}
		info->info[i].desc = EngFncs->engine_strdup(_("Indicates which volume this volume is a snapshot of."));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(volume->origin->parent->volume->name);
		i++;

		info->info[i].name = EngFncs->engine_strdup("ChunkSize");
		info->info[i].title = EngFncs->engine_strdup(_("Chunk Size"));
		info->info[i].desc = EngFncs->engine_strdup(_("The size of the chunks which are copied to this snapshot"));
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_Sectors;
		info->info[i].value.ui32 = volume->metadata->chunk_size;
		i++;

		info->info[i].name = EngFncs->engine_strdup("Writeable");
		info->info[i].title = EngFncs->engine_strdup(_("Writeable"));
		info->info[i].desc = EngFncs->engine_strdup(_("Whether snapshot is writeable or read-only"));
		info->info[i].type = EVMS_Type_Boolean;
		info->info[i].value.b = (volume->metadata->flags & SNAPSHOT_WRITEABLE);
		i++;

		info->info[i].name = EngFncs->engine_strdup("State");
		info->info[i].title = EngFncs->engine_strdup(_("State"));
		info->info[i].desc = EngFncs->engine_strdup(_("Current state of the snapshot."));
		info->info[i].type = EVMS_Type_String;
		if (!is_active(volume)) {
			info->info[i].value.s = EngFncs->engine_strdup(_("Inactive"));
		} else if (is_invalid(volume)) {
			info->info[i].value.s = EngFncs->engine_strdup(_("Disabled/Full"));
		} else if (rollback_is_pending(volume)) {
			info->info[i].value.s = EngFncs->engine_strdup(_("Pending Rollback"));
		} else {
			info->info[i].value.s = EngFncs->engine_strdup(_("Active"));
		}
		i++;

		info->info[i].name = EngFncs->engine_strdup("PercentFull");
		info->info[i].title = EngFncs->engine_strdup(_("Percent Full"));
		info->info[i].desc = EngFncs->engine_strdup(_("Percentage of the snapshot device that has been used to save data from the origin."));
		if (volume->percent_full < 0) {
			info->info[i].type = EVMS_Type_String;
			info->info[i].value.s = EngFncs->engine_strdup(_("Unknown"));
		} else {
			info->info[i].type = EVMS_Type_Unsigned_Int;
			info->info[i].unit = EVMS_Unit_Percent;
			info->info[i].value.ui = volume->percent_full;
		}
		i++;
	}

	info->count = i;
	*info_array = info;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_get_plugin_info
 *
 * Return information about the Snapshot plugin. There is no "extra"
 * information about Snapshot, so "name" should always be NULL.
 **/
static int snap_get_plugin_info(char * name,
				extended_info_array_t ** info_array)
{
	extended_info_array_t * info = NULL;
	char buffer[50] = {0};
	int  i = 0, rc = 0;

	LOG_ENTRY();

	if (name) {
		LOG_ERROR("No support for extra plugin information about "
			  "\"%s\"\n", name);
		rc = EINVAL;
		goto out;
	}

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t)*6);
	if (!info) {
		LOG_ERROR("Error allocating memory for plugin info array\n");
		rc = ENOMEM;
		goto out;
	}

	/* Short Name */
	info->info[i].name = EngFncs->engine_strdup("ShortName");
	info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->short_name);
	i++;

	/* Long Name */
	info->info[i].name = EngFncs->engine_strdup("LongName");
	info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->long_name);
	i++;

	/* Plugin Type */
	info->info[i].name = EngFncs->engine_strdup("Type");
	info->info[i].title = EngFncs->engine_strdup(_("Plug-in Type"));
	info->info[i].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(_("Associative Feature"));
	i++;

	/* Plugin Version */
	info->info[i].name = EngFncs->engine_strdup("Version");
	info->info[i].title = EngFncs->engine_strdup(_("Plug-in Version"));
	info->info[i].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
	info->info[i].type = EVMS_Type_String;
	snprintf(buffer, 50, "%d.%d.%d",
		 my_plugin_record->version.major,
		 my_plugin_record->version.minor,
		 my_plugin_record->version.patchlevel);
	info->info[i].value.s = EngFncs->engine_strdup(buffer);
	i++;

	/* Required Engine Services Version */
	info->info[i].name = EngFncs->engine_strdup("Required_Engine_Version");
	info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
	info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
						      "It will not run on older versions of the Engine services."));
	info->info[i].type = EVMS_Type_String;
	snprintf(buffer, 50, "%d.%d.%d",
		 my_plugin_record->required_engine_api_version.major,
		 my_plugin_record->required_engine_api_version.minor,
		 my_plugin_record->required_engine_api_version.patchlevel);
	info->info[i].value.s = EngFncs->engine_strdup(buffer);
	i++;

	/* Required Plug-in API Version */
	info->info[i].name = EngFncs->engine_strdup("Required_Plugin_Version");
	info->info[i].title = EngFncs->engine_strdup(_("Required Plug-in API Version"));
	info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine plug-in API that this plug-in requires.  "
						      "It will not run on older versions of the Engine plug-in API."));
	info->info[i].type = EVMS_Type_String;
	snprintf(buffer, 50, "%d.%d.%d",
		 my_plugin_record->required_plugin_api_version.plugin.major,
		 my_plugin_record->required_plugin_api_version.plugin.minor,
		 my_plugin_record->required_plugin_api_version.plugin.patchlevel);
	info->info[i].value.s = EngFncs->engine_strdup(buffer);
	i++;

	info->count = i;
	*info_array = info;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_read
 *
 * Reads are only allowed to the origin volume. Reads from the snapshot
 * would essentially be "around" the snapshot kernel code, and hence would
 * not reflect the correct state of the snapshot.
 **/
static int snap_read(storage_object_t * object,
		     lsn_t lsn,
		     sector_count_t count,
		     void * buffer)
{
	snapshot_volume_t * volume = object->private_data;
	int rc = ENOSYS;

	LOG_ENTRY();

	if (is_origin(volume)) {
		rc = READ(volume->child, lsn, count, buffer);
	} else {
		LOG_ERROR("No reading from snapshot volumes within the engine!\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_write
 *
 * No writes are allowed to snapshot or origin objects within the engine.
 * These writes would essentially be "around" the snapshot kernel code, and
 * thus not actually be snapshotted.
 **/
static int snap_write(storage_object_t * object,
		      lsn_t lsn,
		      sector_count_t count,
		      void * buffer)
{
	LOG_ENTRY();
	LOG_ERROR("No writing to snapshot or origin volumes within the engine!\n");
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * can_reset
 *
 * To reset the snapshot to the current state of the origin, the snapshot must
 * - be active.
 * - not be pending a rollback.
 * - not be mounted (can't change the state of the snapshot out from under the
 *   filesystem).
 **/
static int can_reset(snapshot_volume_t * snap_volume)
{
	logical_volume_t * lvol;
	int rc = 0;

	LOG_ENTRY();

	if (!is_active(snap_volume)) {
		LOG_DETAILS("Snapshot %s is not active. Nothing to reset.\n",
			    snap_volume->parent->name);
		rc = EINVAL;
		goto out;
	}

	if (rollback_is_pending(snap_volume)) {
		LOG_DETAILS("Snapshot %s is pending a rollback operation.\n",
			    snap_volume->parent->name);
		rc = EINVAL;
		goto out;
	}

	if (!EngFncs->is_offline(snap_volume->parent, &lvol)) {
		LOG_DETAILS("Snapshot %s is mounted. Please unmount before "
			    "resetting.\n", snap_volume->parent->name);
		rc = EINVAL;
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_get_plugin_functions
 *
 * Set up an array to describe the Snapshot-private-functions supported for
 * the specified object.
 **/
static int snap_get_plugin_functions(storage_object_t * object,
				     function_info_array_t ** functions)
{
	snapshot_volume_t * volume;
	function_info_array_t * fia;
	int rc;

	LOG_ENTRY();

	/* No Snapshot-global functions. */
	if (!object) {
		rc = ENOSYS;
		goto out;
	}
	volume = object->private_data;

	/* No private functions for origin volumes. */
	if (is_origin(volume)) {
		rc = EINVAL;
		goto out;
	}

	/* Allocate the function info array. */
	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) +
				    sizeof(function_info_t) *
				    SNAPSHOT_FUNCTION_COUNT);
	if (!fia) {
		LOG_CRITICAL("Error allocating memory for function info array "
			     "for %s.\n", object->name);
		rc = ENOMEM;
		goto out;
	}

	*functions = fia;
	fia->count = SNAPSHOT_FUNCTION_COUNT;

	/* Can the snapshot be reset? */
	fia->info[0].function = SNAPSHOT_FUNCTION_RESET;
	fia->info[0].name = EngFncs->engine_strdup("reset");
	fia->info[0].title = EngFncs->engine_strdup(_("Reset"));
	fia->info[0].verb = EngFncs->engine_strdup(_("Reset"));
	fia->info[0].help = EngFncs->engine_strdup(_("Reset the snapshot to the "
						     "current state of the "
						     "origin volume."));

	rc = can_reset(volume);
	if (rc) {
		fia->info[0].flags = EVMS_FUNCTION_FLAGS_INACTIVE;
	}

	/* Can the snapshot be rolled-back? */
	fia->info[1].function = SNAPSHOT_FUNCTION_ROLLBACK;
	fia->info[1].name = EngFncs->engine_strdup("rollback");
	fia->info[1].title = EngFncs->engine_strdup(_("Roll Back"));
	fia->info[1].verb = EngFncs->engine_strdup(_("Roll Back"));
	fia->info[1].help = EngFncs->engine_strdup(_("Roll back the contents of "
                                                     "the snapshot to the "
						     "original volume."));

	rc = can_rollback(volume, FALSE);
	if (rc) {
		fia->info[1].flags = EVMS_FUNCTION_FLAGS_INACTIVE;
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_plugin_function
 **/
static int snap_plugin_function(storage_object_t * object,
				task_action_t action,
				list_anchor_t objects,
				option_array_t * options)
{
	snapshot_volume_t * volume = object->private_data;
	char * reset_choices[3] = { _("Reset"), _("Don't Reset"), NULL };
	char * rollback_choices[3] = { _("Rollback"), _("Don't Rollback"), NULL };
	int answer = 0, rc = EINVAL;

	LOG_ENTRY();

	/* No private functions for origin volumes. */
	if (is_origin(volume)) {
		goto out;
	}

	switch (action) {

	case SNAPSHOT_FUNCTION_RESET:
		rc = can_reset(volume);
		if (rc) {
			LOG_ERROR("Snapshot %s cannot be reset at this time.\n",
				  volume->parent->name);
			break;
		}

		QUESTION(&answer, reset_choices,
			 _("WARNING: Selecting \"Reset\" will cause "
			   "all data saved in the snapshot \"%s\" "
			   "to be lost, and the snapshot will be "
			   "reset to the current state of the "
			   "origin volume \"%s\".\n\n"
			   "Do not mount the snapshot volume until "
			   "after saving changes."),
			 object->name,
			 volume->origin->parent->volume->name);
		if (answer != 0) {
			rc = E_CANCELED;
			break;
		}

		/* Erase the snapshot header, mark the snapshot valid and dirty,
		 * and schedule for reactivation.
		 */
		KILL_SECTORS(volume->child, 0, 1);
		schedule_for_commit(volume);
		mark_valid(volume);
		schedule_for_activate(volume);
		schedule_for_deactivate(volume);

		break;

	case SNAPSHOT_FUNCTION_ROLLBACK:
		rc = can_rollback(volume, FALSE);
		if (rc) {
			LOG_ERROR("Snapshot %s cannot be rolled-back at this "
				  "time.\n", object->name);
			break;
		}

		QUESTION(&answer, rollback_choices,
			 _("WARNING: Selecting \"Rollback\" will cause "
			   "all data saved in the snapshot \"%s\" "
			   "to be copied back to the origin volume "
			   "\"%s\". The current state of the origin "
			   "volume will be lost. Both the snapshot "
			   "and the origin volumes must remain un-"
			   "mounted until the rollback is complete.\n"),
			 object->name,
			 volume->origin->parent->volume->name);
		if (answer != 0) {
			rc = E_CANCELED;
			break;
		}

		schedule_for_rollback(volume);
		schedule_for_commit(volume);

		break;

	default:
		LOG_ERROR("Action %d is not allowed for object %s\n",
			  action, object->name);
		rc = EINVAL;
		break;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * snap_backup_metadata
 *
 * Save the snapshot metadata to the engine's metadata-backup database. We
 * only save data for the snapshot, since there's no metadata on the origin
 * side. We also save a zeroed header, so if the snapshot volume is restored,
 * it will appear as a new, valid snapshot.
 **/
static int snap_backup_metadata(storage_object_t *object)
{
	snapshot_volume_t * volume = object->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (is_origin(volume)) {
		goto out;
	}

	rc = write_snapshot_metadata(volume, TRUE);
	if (rc) {
		goto out;
	}

	rc = erase_snapshot_header(volume, TRUE);
	if (rc) {
		goto out;
	}

	rc = EngFncs->save_metadata(volume->parent->name,
				    volume->origin->parent->volume->name,
				    0, 0, NULL);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/* Snapshot function table and plugin record. */
static plugin_functions_t snapshot_functions = {
	.setup_evms_plugin		= snap_setup_evms_plugin,
	.cleanup_evms_plugin		= snap_cleanup_evms_plugin,
	.can_delete			= snap_can_delete,
	.can_expand			= snap_can_expand,
	.can_expand_by			= snap_can_expand_by,
	.can_set_volume			= snap_can_set_volume,
	.discover			= snap_discover,
	.create				= snap_create,
	.discard			= snap_discard,
	.delete				= snap_delete,
	.expand				= snap_expand,
	.set_volume			= snap_set_volume,
	.add_sectors_to_kill_list	= snap_add_sectors_to_kill_list,
	.commit_changes			= snap_commit_changes,
	.activate			= snap_activate,
	.deactivate			= snap_deactivate,
	.can_activate			= snap_can_activate,
	.can_deactivate			= snap_can_deactivate,
	.get_option_count		= snap_get_option_count,
	.init_task			= snap_init_task,
	.set_option			= snap_set_option,
	.set_objects			= snap_set_objects,
	.get_info			= snap_get_info,
	.get_plugin_info		= snap_get_plugin_info,
	.read				= snap_read,
	.write				= snap_write,
	.get_plugin_functions		= snap_get_plugin_functions,
	.plugin_function		= snap_plugin_function,
	.backup_metadata		= snap_backup_metadata,
};

plugin_record_t Snapshot_Plugin = {
	.id = EVMS_SNAPSHOT_PLUGIN_ID,
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.plugin = {
			.major		= 13,
			.minor		= 1,
			.patchlevel	= 0
		}
	},
	.short_name = EVMS_SNAPSHOT_PLUGIN_SHORT_NAME,
	.long_name = EVMS_SNAPSHOT_PLUGIN_LONG_NAME,
	.oem_name = EVMS_IBM_OEM_NAME,
	.functions = {
		.plugin = &snapshot_functions
	},
	.container_functions = NULL
};

plugin_record_t * evms_plugin_records[] = {
	&Snapshot_Plugin,
	NULL
};

