/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid5_funcs.c
 *
 * Description: This file contains all MD RAID5's plugin functions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <sys/ioctl.h>

#include "md.h"
#include "raid5_mgr.h"

#define my_plugin_record raid5_plugin

int raid5_add_spare_disk(md_volume_t * volume, storage_object_t * spare)
{
	if (md_is_region_active(volume->region))
		return md_volume_add_spare_to_active_region(volume, spare);
	else
		return md_volume_add_spare_to_inactive_region(volume, spare);
}

int raid5_remove_spare_disk(md_volume_t *vol, storage_object_t *spare)
{
	if (md_is_region_active(vol->region)) {
		return md_volume_remove_spare_from_active_region(vol, spare);
	} else {
		return md_volume_remove_spare_from_inactive_region(vol, spare);
	}
}

int raid5_remove_faulty_disk(md_volume_t *vol, storage_object_t *faulty)
{
	int rc=0;
	LOG_ENTRY();

	if (!vol || !faulty) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	if (md_is_region_active(vol->region)) {
		rc = md_volume_remove_faulty_from_active_region(vol, faulty);
	} else {
		rc = md_volume_remove_faulty_from_inactive_region(vol, faulty);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int raid5_remove_stale_disk(md_volume_t *vol, storage_object_t * stale_disk)
{
	int rc=0;
	
	LOG_ENTRY();

	if (!vol || !stale_disk) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (md_is_region_active(vol->region)) {
		rc = md_volume_remove_stale_from_active_region(vol, stale_disk);
	} else {
		rc = md_volume_remove_stale_from_inactive_region(vol, stale_disk);
	}
	LOG_EXIT_INT(rc);
	return rc;
}


static int mark_disk_faulty_post_ioctl(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	if (pkg->parm.disk_info) {
		EngFncs->engine_free(pkg->parm.disk_info);
		pkg->parm.disk_info = NULL;
	}
	volume->region_mgr_flags &= MD_RAID5_CONFIG_CHANGE_PENDING;
	return 0;
}


int raid5_mark_faulty_disk(md_volume_t *vol, storage_object_t * active)
{
	int rc=0;
	evms_md_disk_info_t *disk_info = NULL;
	evms_md_ioctl_parm_t parm;
	mdu_disk_info_t info;
	md_member_t *member;

	LOG_ENTRY();

	if (!vol || !active) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	member = md_volume_find_object(vol, active);
	if (!member) {
		LOG_WARNING("%s is not in region %s.\n", active->name, vol->name);
		rc = EINVAL;
		goto out;
	}
	if (!md_member_is_raid_disk(member)) {
		LOG_WARNING("%s is not active.\n", active->name);
		rc = EINVAL;
		goto out;
	}

	info.number = member->dev_number;
	rc = md_ioctl_get_disk_info(vol->region, &info);
	if (rc) {
		LOG_MD_BUG();
		rc = ENODEV;
		goto out;
	} else {
		if ((info.major != active->dev_major) || (info.minor != active->dev_minor)) {
			LOG_WARNING("(%s) mismatch major/minor, kernel(%d:%d), EVMS(%d:%d)\n",
				    active->name, info.major, info.minor, 
				    active->dev_major, active->dev_minor);
		}
	}

	rc = md_volume_mark_faulty(member, FALSE);
	if (rc) {
		goto out;
	}

	disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
	if ( !disk_info ) {
		rc = ENOMEM;
		goto out;
	}
	
	disk_info->number = member->dev_number;
	disk_info->major = info.major;
	disk_info->minor = info.minor;
	disk_info->object = active;
	parm.disk_info = disk_info;
	rc = schedule_md_ioctl_pkg(vol, EVMS_MD_DEACTIVATE, &parm, mark_disk_faulty_post_ioctl);
	if (!rc) {
		vol->region_mgr_flags |= MD_RAID5_CONFIG_CHANGE_PENDING;
	}

out:
	if (rc && disk_info) {
		EngFncs->engine_free(disk_info);
	}
	LOG_EXIT_INT(rc);
	return rc;
}


