/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid1_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Raid1 MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <time.h>
#include <sys/ioctl.h>
#define MY_PLUGIN raid1_plugin
#include "md.h"
#include "raid1_mgr.h"
#include "raid1_discover.h"



// Global variables
static engine_mode_t        open_mode;	    // Still don't know what this is for. :)



/* Function: raid1_setup_evms_plugin
 *
 *  This function gets called shortly after the plugin is loaded by the
 *  Engine. It performs all tasks that are necessary before the initial
 *  discovery pass.
 */
static int raid1_setup_evms_plugin( engine_mode_t        mode,
				    engine_functions_t * functions)
{
	int rc = 0;

	// Parameter check
	if (!functions) {
		return EINVAL;
	}

	open_mode = mode;
	EngFncs = functions;

	LOG_ENTRY;
	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	RETURN(rc);
}


/****** Region Checking Functions ******/


/* All of the following md_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: raid1_can_delete
 *
 *  Can we remove the specified MD logical volume
 */
static int raid1_can_delete( storage_object_t * region )
{
	LOG_ENTRY;

	// Parameter check
	if (! region) {
		RETURN(EFAULT);
	}

	// The region must be owned by MD,
	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s is not owned by MD.\n", region->name);
		RETURN(EINVAL);
	}
	RETURN(0);
}


/* Function: raid1_can_expand
 *
 *	Can this region be expanded? If so,
 *	add the region to the expansion_points output list. else, just
 *      return 0 to allow those above us to do as they please
 */
static int raid1_can_expand( storage_object_t * region,
			     u_int64_t        * expand_limit,
			     dlist_t            expansion_points )
{
	LOG_ENTRY;
	RETURN(0);
}


/* Function: raid1_can_expand_by
 *
 *  No plans for raid1 expand.
 *
 */

static int raid1_can_expand_by( storage_object_t * child_object,
				u_int64_t        * size )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: raid1_can_shrink
 *
 *	Just like can_expand, but in the opposite direction.
 */
static int raid1_can_shrink( storage_object_t * region,
			     u_int64_t        * shrink_limit,
			     dlist_t            shrink_points )
{
	LOG_ENTRY;
	RETURN(0);
}


/* Function: raid1_can_shrink_by
 *
 *  Can we shrink the specified region? If it can be shrunk, but not to
 *  exactly new_size, reset new_size to the next lower possible size.
 */
static int raid1_can_shrink_by( storage_object_t * region,
				u_int64_t        * size )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: raid1_can_move
 *
 */
static int raid1_can_move( storage_object_t * region )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: raid1_can_set_volume
 *
 *  Any non-freespace MD region can be converted into an EVMS
 *  compatibility volume. Snapshots and snapshot originals
 *  cannot have their volumes removed.
 */
static int raid1_can_set_volume( storage_object_t * region,
				 boolean            flag )
{
	LOG_ENTRY;

	// Parameter check
	if (! region) {
		RETURN(EFAULT);
	}

	// The region must be owned by MD
	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s is not owned by MD.\n", region->name);
		RETURN(EINVAL);
	}
	RETURN(0);
}


/* Function: raid1_discover
 *
 *  Examine all disk segments and find MD PVs. Assemble volume groups
 *  and export all MD logical volumes as EVMS regions.
 *
 *  All newly created regions must be added to the output list, and all
 *  segments from the input list must either be claimed or moved to the
 *  output list.
 */
static int raid1_discover( dlist_t input_list,
			   dlist_t output_list,
			   boolean final_call )
{
	int count = 0;

	LOG_ENTRY;

	// Parameter check
	if (!input_list || !output_list) {
		RETURN(EFAULT);
	}

	md_discover_volumes(input_list, output_list);
	LOG_DETAILS("PV discovery complete.\n");

	// LV discovery and exporting
	raid1_discover_regions(output_list, &count, final_call);
	LOG_DETAILS("Volume discovery complete.\n");

	RETURN(count);
}



/****** Region Functions ******/


static int raid1_get_create_options( option_array_t * options,
				     char          ** spare_disk)
{
	int i;
	int rc = 0;

	LOG_ENTRY;

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			
			case MD_OPTION_SPARE_DISK_INDEX:
				// Not worth validation, will catch when we try to find the original
				*spare_disk = options->option[i].value.s;
				break;

			default:
				break;

			}

		} else {

			if (strcmp(options->option[i].name, MD_OPTION_SPARE_DISK_NAME) == 0) {
				*spare_disk = options->option[i].value.s;
			}
		}						
	}

	RETURN(rc);

}


/* get the list of objects, search for the one requested.  If found, RETURN the top object
 otherwise RETURN NULL */
static storage_object_t * find_object( char * name )
{

	int rc = 0;
	storage_object_t * object = NULL;
	dlist_t objects;
	int size, tag;

	LOG_ENTRY;
	if (!name) {
		RETURN(NULL);
	}

	// get the list of objects
	rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,		// non-evms objects
				 DATA_TYPE,
				 NULL,
				 VALID_INPUT_OBJECT,
				 &objects);
	if (!rc) {
		while (BlindExtractObject(objects, &size, (TAG *) &tag, NULL, (void**)&object)==0) {
			if (!strncmp(object->name ,name,128)) {
				break;
			}
		}
		DestroyList(&objects, FALSE);
	} else {
		LOG_ERROR("Error getting object list = %d....\n",rc);
	}
	RETURN(object);
}

/* Function: raid1_create
 *
 *  Create a new MD volume using the specified freespace.
 */
static int raid1_create( dlist_t          objects,
			 option_array_t * options,
			 dlist_t          new_region_list )
{
	md_volume_t * volume = NULL;
	storage_object_t * object;
	int nr_disks;
	unsigned long size = -1;
	int tag, waste;
	int i, spare_disks=0, spare_index = 0, index = 0;
	int rc = 0;
	mdp_disk_t disk;
	storage_object_t * spare=NULL;
	char * spare_disk = NULL;
	int chunk_size = MD_DEFAULT_CHUNK_SIZE;

	LOG_ENTRY;

	// Parameter check
	if (!objects || !options || !new_region_list) {
		RETURN(EFAULT);
	}

	rc = GetListSize(objects, &nr_disks);
	if (rc) {
		LOG_CRITICAL("DLIST error getting number of objects rc = %d.\n",rc);
		RETURN(EINVAL);
	}

	// allow 1 or more disks
	if (nr_disks < 1) {
		LOG_CRITICAL("Error, must have at least 2 objects selected for RAID 1 array. %d selected.\n",nr_disks);
		RETURN(EINVAL);
	}

	if (md_allocate_memory((void**)&volume, sizeof(md_volume_t) )) {
		LOG_CRITICAL("Memory error new volume structure.\n");
		RETURN(ENOMEM);
	}

	while (!(rc = BlindExtractObject(objects, &waste, (TAG *)&tag, NULL, (void *)&object))) {
		size = min(size, object->size);	 // track smallest object for super block
		volume->child_object[index] = object;
		index ++;
	}

	raid1_get_create_options(options, &spare_disk);

	if (spare_disk) {
		spare = find_object( spare_disk);
	}
	if (spare) {
		size = min(size, spare->size);	// track smallest object for super block
		volume->child_object[index] = spare;
		nr_disks++;
		spare_disks = 1;
		spare_index = index;
	}
	disk.number = 0;
	disk.raid_disk = 0;
	disk.state = (1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC);

	size = MD_NEW_SIZE_BLOCKS(size/2); // first convert sectors to blocks

	md_create_first_superblock(volume, disk, pers_to_level(RAID1), chunk_size, size, nr_disks, spare_disks, 0);

	if (spare) {
		volume->super_block->disks[spare_index].state = 0;  // set the state to inactive for the spare disk.
	}

	for (i = 0; i < nr_disks; i++) {
		rc = md_clone_superblock(volume, i);
	}

	volume->personality = RAID1;
	volume->nr_disks = nr_disks;
	volume->next = volume_list_head;
	volume_list_head = volume;
	volume->removed_disks = CreateList();
	volume->added_disks = CreateList();
	volume->activated_disks = CreateList();
	volume->deactivated_disks = CreateList();

	rc = raid1_create_region(volume, new_region_list, TRUE);
	RETURN(rc);
}


/* Function: raid1_delete
 *
 *  Remove the specified region and consolidate all of its space into
 *  the appropriate freespace region.
 */
static int raid1_delete( storage_object_t * region,
			 dlist_t            children )
{
	md_volume_t   * volume;
	int     rc;

	LOG_ENTRY;

	// Check that this region can be removed.
	if ((rc = raid1_can_delete(region))) {
		RETURN(rc);
	}
	volume = region->private_data;

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	// Delete the volume.
	md_delete_volume(volume);
	EngFncs->free_region(region);

	RETURN(rc);
}


/* Function: raid1_expand
 */

static int raid1_expand( storage_object_t    * region,
			 storage_object_t    * expand_object,
			 dlist_t               input_objects,
			 option_array_t      * options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* Function: raid1_shrink
 */
static int raid1_shrink( storage_object_t * region,
			 storage_object_t * shrink_object,
			 dlist_t            input_objects,
			 option_array_t   * options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* Function: raid1_move
 */
static int raid1_move( storage_object_t * source,
		       storage_object_t * target,
		       option_array_t   * options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* Function: raid1_set_volume
 *
 *  MD doesn't really care if its regions are made into volumes.
 */
static void raid1_set_volume( storage_object_t    * region,
			      boolean               flag )
{
	LOG_ENTRY;
	LOG("That's nice. :)\n");
	LOG_EXIT(0);
}


/* Function: raid1_add_sectors_to_kill_list
 *
 *  The kill_sectors list contains a list of sectors that need to be zeroed
 *  during the next commit. This function is very similar to read/write.
 */
static int raid1_add_sectors_to_kill_list( storage_object_t * region,
					   lsn_t              lsn,
					   sector_count_t     count )
{

	md_volume_t   * volume = (md_volume_t *)region->private_data;
	int     rc = 0;
	int i, found=0;

	LOG_ENTRY;

	// Parameter check.
	if (!region) {
		RETURN(EFAULT);
	}

	if (volume->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, writing data is not allowed\n ",volume->name);
		MESSAGE("MD Object %s is corrupt, writing data is not allowed\n ",volume->name);
		RETURN(EIO);
	}
	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%d\n ",volume->name,lsn+count);
		RETURN(EIO);
	}

	for (i = 0; (i < MAX_MD_DEVICES) && (found < volume->nr_disks); i++ ) {
		// check for null object, if missing, skip
		if (volume->child_object[i]) {
			found++;
			// only kill sectors on Active disks, not the spares.
			if (volume->super_block->disks[i].state & (1 << MD_DISK_ACTIVE)) {
				rc = KILL_SECTORS(volume->child_object[i], lsn, count);
				if (rc) {
					//BUGBUG disable mirror
				}
			}
		}
	}

	RETURN(rc);
}


/* Function: raid1_commit_changes
 *
 *  All other commit operations are done in commit_container_changes.
 */
static int raid1_commit_changes( storage_object_t * region,
				 uint               phase )
{
	md_volume_t * volume = (md_volume_t *)region->private_data;
	int         rc = 0;

	LOG_ENTRY;

	// Parameter check
	if (! region) {
		RETURN(EFAULT);
	}

	// Make sure this region belongs to MD, and is dirty
	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s does not belong to MD.\n", region->name);
		RETURN(EINVAL);
	}
	if (! (region->flags & SOFLAG_DIRTY)) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		RETURN(0);
	}

	switch (phase) {
	case 0:
		rc = md_process_modify_list(volume, EVMS_MD_REMOVE);
		if (!rc) {
			// don't really need to do the adds anyway
			rc = md_process_modify_list(volume, EVMS_MD_ADD);
		}
		break;
	case 1:
		rc = md_write_sbs_to_disk(volume);   // write super blocks
		break;
	case 2:
		break;
	case 3:
//		rc = md_process_modify_list(volume, EVMS_MD_ACTIVATE);
		region->flags &= ~SOFLAG_DIRTY;	 // mark clean only after all phases done
		break;
	default	:
		break;
	}

	RETURN(rc);
}


/* Function: raid1_get_option_count
 *
 *  Determine the type of Task that is being performed, and return
 *  the number of options that are available for that Task.
 */
static int raid1_get_option_count( task_context_t * task )
{
	int count = 0;

	LOG_ENTRY;

	switch (task->action) {
	case EVMS_Task_Create:
		count = 1;
		break;
	case EVMS_Task_Set_Info:
		count = 11;
		break;
	default:
		count = -1;
		break;
	}

	RETURN(count);
}


// get list spare disks to put in the activate list
static int get_spare_list( value_list_t ** value_list, storage_object_t * region){
	int rc = 0;
	int count, i, j;
	md_volume_t * volume = (md_volume_t *)region->private_data;

	LOG_ENTRY;

	count = volume->super_block->spare_disks + 1; // plus 1 for 'None' selection
	*value_list = EngFncs->engine_alloc(count * sizeof(value_t) + sizeof(value_list_t));  // yeah it's too big, but so what
	if (*value_list) {
		(*value_list)->count = count;
		i=0;
		SET_STRING((*value_list)->value[i].s,MD_NO_SELECTION);
		i++;
		for (j = 0; j < MAX_MD_DEVICES; j++ ) {
			// check for null object, if missing, skip
			if (volume->child_object[j]) {
				// look for spares.
				if ((volume->super_block->disks[j].state == 0) ||
					volume->super_block->disks[j].state == (1<<MD_DISK_NEW)) {
					(*value_list)->value[i].s = EngFncs->engine_alloc(strlen(volume->child_object[j]->name) + 1);
					strcpy((*value_list)->value[i].s, volume->child_object[j]->name);
					i++;
				}
			}
		}
	}

	RETURN(rc);
}

// get list active disks to put in the activeate list
static int get_active_list( value_list_t ** value_list, storage_object_t * region){
	int rc = 0;
	int count, i, j;
	md_volume_t * volume = (md_volume_t *)region->private_data;

	LOG_ENTRY;

	count = volume->super_block->raid_disks + 1; //+ 1 for 'None' selection
	*value_list = EngFncs->engine_alloc(count * sizeof(value_t) + sizeof(value_list_t));  // yeah it's too big, but so what
	if (*value_list) {
		(*value_list)->count = count;
		i=0;
		SET_STRING((*value_list)->value[i].s,MD_NO_SELECTION);
		i++;
		for (j = 0; j < MAX_MD_DEVICES; j++ ) {
			// check for null object, if missing, skip
			if (volume->child_object[j]) {
				// look for Active disks.
				if (volume->super_block->disks[j].state & (1 << MD_DISK_ACTIVE)) {
					(*value_list)->value[i].s = EngFncs->engine_alloc(strlen(volume->child_object[j]->name) + 1);
					strcpy((*value_list)->value[i].s, volume->child_object[j]->name);
					i++;
				}
			}
		}
	}

	RETURN(rc);
}

// get list faulty disks to put in the faulty list
// This includes those disks which are active but not sync since the kernel
// treats these just like faulty.
static int get_faulty_list( value_list_t ** value_list, storage_object_t * region){
	int rc = 0;
	int count, i, j;
	md_volume_t * volume = (md_volume_t *)region->private_data;

	LOG_ENTRY;

	count = volume->super_block->failed_disks + 1; //+ 1 for 'None' selection
	*value_list = EngFncs->engine_alloc(count * sizeof(value_t) + sizeof(value_list_t));  // yeah it's too big, but so what
	if (*value_list) {
		(*value_list)->count = count;
		i=0;
		SET_STRING((*value_list)->value[i].s,MD_NO_SELECTION);
		i++;
		for (j = 0; j < MAX_MD_DEVICES; j++ ) {
			// check for null object, if missing, skip
			if (volume->child_object[j]) {
				// look for faulty disks.
				if ( (volume->super_block->disks[j].state & (1 << MD_DISK_FAULTY)) ||
				    (volume->super_block->disks[j].state & (1 << MD_DISK_REMOVED))){
					(*value_list)->value[i].s = EngFncs->engine_alloc(strlen(volume->child_object[j]->name) + 1);
					strcpy((*value_list)->value[i].s, volume->child_object[j]->name);
					i++;
				}
			}
		}
	}

	RETURN(rc);
}

// Get the list of volumes on the system that we can use as spares
static int get_object_list( value_list_t ** 	value_list,
			    dlist_t         	selected_objects,
			    u_int64_t		min_size)
{

	int rc = 0;
	storage_object_t * object;
	dlist_t tmp_list, selected_tmp_list;
	int count, i, tag, size;

	LOG_ENTRY;
	// get the list of objects to search for original volumes

	rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,		// all types
				      DATA_TYPE,
				      NULL,
				      VALID_INPUT_OBJECT,
				      &tmp_list);

	// loop through selected object, removing objects from tmp_list
	// must copy list to avoid DLIST foreach
	selected_tmp_list = CreateList();
	if (!selected_tmp_list) {
		LOG_ERROR("Error DLIST Create failed, give up \n");
		RETURN(ENOMEM);
	}
	rc = CopyList(selected_tmp_list, selected_objects,InsertAtStart);
	if (rc) {
		LOG_ERROR("Error copying list\n");
	}

	GoToStartOfList(selected_tmp_list);
	while (!BlindExtractObject(selected_tmp_list,&size, (TAG *)&tag, NULL, (void **)&object)) {
		LOG_DETAILS("Object %s selected, removing from spare list\n",object->name);
		rc = DeleteObject(tmp_list, object);
		if (rc) {
			LOG_ERROR("Error removimg object %s form list of top objects rc = %d\n",object->name,rc);
			rc = 0;   // needed to fudge list on modify
		}
	}
	DestroyList(&selected_tmp_list, FALSE);

	if (*value_list) {
		for (i = 0; i < (*value_list)->count; i++) {
			if ((*value_list)->value[i].s) {
				EngFncs->engine_free((*value_list)->value[i].s);
			}
		}
		EngFncs->engine_free(*value_list);
	}

	GetListSize(tmp_list, &count);
	count++; // increment count to holed the 'None' selection
	*value_list = EngFncs->engine_alloc(count * sizeof(value_t) + sizeof(value_list_t));  // yeah it's too big, but so what
	if (!rc) {
		(*value_list)->count = count;
		i = 0;
		SET_STRING((*value_list)->value[i].s,MD_NO_SELECTION);
		i++;
		while (BlindExtractObject(tmp_list, &size,(TAG *) &tag, NULL, (void**)&object)==0) {
			if (object->size >= min_size) {
				(*value_list)->value[i].s = EngFncs->engine_alloc(strlen(object->name) + 1);
				strcpy((*value_list)->value[i].s, object->name);
				i++;
			} else {
				(*value_list)->count--;
			}
		}
	}
	rc = DestroyList(&tmp_list, FALSE);

	RETURN(rc);
}
/* Function: raid1_init_task
 *
 *  Determine the type of Task that is being performed, and set up the
 *  context structure with the appropriate initial values.
 */
static int raid1_init_task( task_context_t * context )
{
	int rc = 0;
	dlist_t tmp_list;
	void * waste;
	md_volume_t * volume;
	evms_md_array_info_t    md_info = {0};

	LOG_ENTRY;

	// Parameter check.
	if (! context) {
		RETURN(EFAULT);
	}

	switch (context->action) {
	
	case EVMS_Task_Create:

		context->option_descriptors->count = 1;

		// option 0 is the spare disk
		context->option_descriptors->option[0].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		// get the list of disks that can be spares.
		get_object_list((value_list_t **)&context->option_descriptors->option[0].constraint.list,context->selected_objects,0);
		context->option_descriptors->option[0].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[0].help = NULL;
		SET_STRING(context->option_descriptors->option[0].name, MD_OPTION_SPARE_DISK_NAME );
		context->option_descriptors->option[0].size = EVMS_VOLUME_NAME_SIZE;
		SET_STRING(context->option_descriptors->option[0].tip, "Object to use as a spare disk in the array" );
		SET_STRING(context->option_descriptors->option[0].title, "Spare Disk" );
		context->option_descriptors->option[0].type = EVMS_Type_String;
		context->option_descriptors->option[0].unit = EVMS_Unit_None;
		context->option_descriptors->option[0].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
		strcpy(context->option_descriptors->option[0].value.s, MD_NO_SELECTION);

		// get a list of all valid input  disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					 DATA_TYPE,
					 NULL,
					 VALID_INPUT_OBJECT,
					 &tmp_list);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		DestroyList(&tmp_list, FALSE);

		context->min_selected_objects = 1;
		context->max_selected_objects = 27;
		break;

	case EVMS_Task_Set_Info:

		volume = (md_volume_t *) context->object->private_data;

		if (!md_get_kernel_info(volume,&md_info) ){
			if (md_info.state & EVMS_MD_ARRAY_SYNCING) {
				int    answer = 0;
				char * choice_text[2] = { "OK", NULL };
				EngFncs->user_message(my_plugin, &answer,choice_text,
					"Object %s is currently being synced.  "
					"It can not be modified until the sync is complete.\n",
					volume->name);
				rc = EBUSY;
				break;
			}

			if (!(context->object->flags & SOFLAG_DIRTY)) {
				mdp_disk_t * new_disks = md_info.sb->disks;
				mdp_disk_t * old_disks = volume->super_block->disks;
				mdp_super_t * tmp_super;
				storage_object_t * tmp_object;
				mdp_disk_t    tmp_disk;
				int i,j;
				LOG_DETAILS("object %s active in kernel and not dirty, syncing with kernel\n",volume->name);
				for (i = 0; i< MAX_MD_DEVICES; i++) {
					for (j = i; j < MAX_MD_DEVICES; j++) {
						if (old_disks[j].major == new_disks[i].major &&
						    old_disks[j].minor == new_disks[i].minor) {
							if (i==j) {
								break;
							}
							LOG_DETAILS("Swapping new index %d with old %d\n",j,i);
							tmp_super = volume->super_array[j];
							tmp_object = volume->child_object[j];
							tmp_disk = old_disks[j];
							volume->super_array[j] = volume->super_array[i];
							volume->child_object[j] = volume->child_object[i];
							old_disks[j] = old_disks[i];
							volume->super_array[i] = tmp_super;
							volume->child_object[i] = tmp_object;
							old_disks[i] = tmp_disk;
							break;
						}
					}
				}
				md_deallocate_memory(volume->super_block);
				volume->super_block = md_info.sb;
			}
		}

		if ((volume->flags & MD_CORRUPT) || (volume->flags & MD_DEGRADED)){
			int    answer = 0;
			char * choice_text[3] = { "Don't Fix", "Fix", NULL };
			EngFncs->user_message(my_plugin, &answer,choice_text,
				"Errors have been detected on MD region %s. Check the message "
				"display for messages detailing the errors requiring fixing.  "
				"Selecting \"Fix\" will permanently change the array.  If you "
				"elect to not fix the region at this time, you may not modify "
				"the region. \n\n",
				volume->name);
			if (answer) {
				raid1_verify_and_fix_array(volume, 1);
			} else{
				RETURN(EINVAL);
			}
		}

		context->min_selected_objects = 1;
		context->max_selected_objects = 1;
		context->option_descriptors->count = 11;

		// add the reference object (the region we are modifying) to the
		// acceptable and selected objects lists to be able to parse this object
		// out of option lists.
		InsertObject(context->acceptable_objects, sizeof(storage_object_t),context->object,
			     REGION_TAG, NULL, InsertAtStart, TRUE, &waste);
		InsertObject(context->selected_objects, sizeof(storage_object_t),context->object,
			     REGION_TAG, NULL, InsertAtStart, TRUE, &waste);


		context->option_descriptors->option[0].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		// get the list of disks that can be spares.
		get_object_list((value_list_t **)&context->option_descriptors->option[0].constraint.list,context->selected_objects,
				volume->super_block->size * 2);
		context->option_descriptors->option[0].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[0].help = NULL;
		SET_STRING(context->option_descriptors->option[0].name, MD_OPTION_AVAIL_DISKS_NAME );
		context->option_descriptors->option[0].size = EVMS_VOLUME_NAME_SIZE;
		SET_STRING(context->option_descriptors->option[0].tip, "List of Objects eligible to be added to the array" );
		SET_STRING(context->option_descriptors->option[0].title, "Available Objects" );
		context->option_descriptors->option[0].type = EVMS_Type_String;
		context->option_descriptors->option[0].unit = EVMS_Unit_None;
		context->option_descriptors->option[0].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
		strcpy(context->option_descriptors->option[0].value.s, MD_NO_SELECTION);
		context->option_descriptors->option[0].group.group_number=1;
		context->option_descriptors->option[0].group.group_level=2;

		context->option_descriptors->option[1].flags = EVMS_OPTION_FLAGS_INACTIVE;
		context->option_descriptors->option[1].constraint.list = NULL;
		context->option_descriptors->option[1].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[1].help = NULL;
		SET_STRING(context->option_descriptors->option[1].name,MD_OPTION_ADD_SPARE_NAME);
		context->option_descriptors->option[1].size = sizeof(u_int8_t);
		SET_STRING(context->option_descriptors->option[1].tip, "Add the selected Available Object into the array as a spare object" );
		SET_STRING(context->option_descriptors->option[1].title, "        Add selected Available Object as Spare" );
		context->option_descriptors->option[1].type = EVMS_Type_Boolean;
		context->option_descriptors->option[1].unit = EVMS_Unit_None;
		context->option_descriptors->option[1].value.bool = TRUE;
		context->option_descriptors->option[1].group.group_number=1;
		context->option_descriptors->option[1].group.group_level=1;

		context->option_descriptors->option[2].flags = EVMS_OPTION_FLAGS_INACTIVE;
		context->option_descriptors->option[2].constraint.list = NULL;
		context->option_descriptors->option[2].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[2].help = NULL;
		SET_STRING(context->option_descriptors->option[2].name,MD_OPTION_ADD_ACTIVE_NAME);
		context->option_descriptors->option[2].size = sizeof(u_int8_t);
		SET_STRING(context->option_descriptors->option[2].tip, "Add the selected Available Object into the array as an active object" );
		SET_STRING(context->option_descriptors->option[2].title, "        Add selected Available Object as Active" );
		context->option_descriptors->option[2].type = EVMS_Type_Boolean;
		context->option_descriptors->option[2].unit = EVMS_Unit_None;
		context->option_descriptors->option[2].value.bool = FALSE;
		context->option_descriptors->option[2].group.group_number=1;
		context->option_descriptors->option[2].group.group_level=1;

		context->option_descriptors->option[3].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		// get the list of disks that can be spares.
		get_spare_list((value_list_t **)&context->option_descriptors->option[3].constraint.list,context->object);
		context->option_descriptors->option[3].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[3].help = NULL;
		SET_STRING(context->option_descriptors->option[3].name, MD_OPTION_SPARE_DISKS_NAME );
		context->option_descriptors->option[3].size = EVMS_VOLUME_NAME_SIZE;
		SET_STRING(context->option_descriptors->option[3].tip, "Spare Objects in the array to be made active or removed from the array" );
		SET_STRING(context->option_descriptors->option[3].title, "Spare Objects" );
		context->option_descriptors->option[3].type = EVMS_Type_String;
		context->option_descriptors->option[3].unit = EVMS_Unit_None;
		context->option_descriptors->option[3].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
		strcpy(context->option_descriptors->option[3].value.s, MD_NO_SELECTION);
		context->option_descriptors->option[3].group.group_number=2;
		context->option_descriptors->option[3].group.group_level=2;

		context->option_descriptors->option[4].flags = EVMS_OPTION_FLAGS_INACTIVE;
		context->option_descriptors->option[4].constraint.list = NULL;
		context->option_descriptors->option[4].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[4].help = NULL;
		SET_STRING(context->option_descriptors->option[4].name,MD_OPTION_ACTIVATE_SPARE_NAME);
		context->option_descriptors->option[4].size = sizeof(u_int8_t);
		SET_STRING(context->option_descriptors->option[4].tip, "Make the selected Spare Object an active member of the array" );
		SET_STRING(context->option_descriptors->option[4].title, "        Activate selected Spare Object" );
		context->option_descriptors->option[4].type = EVMS_Type_Boolean;
		context->option_descriptors->option[4].unit = EVMS_Unit_None;
		context->option_descriptors->option[4].value.bool = TRUE;
		context->option_descriptors->option[4].group.group_number=2;
		context->option_descriptors->option[4].group.group_level=1;

		context->option_descriptors->option[5].flags = EVMS_OPTION_FLAGS_INACTIVE;
		context->option_descriptors->option[5].constraint.list = NULL;
		context->option_descriptors->option[5].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[5].help = NULL;
		SET_STRING(context->option_descriptors->option[5].name,MD_OPTION_REMOVE_SPARE_NAME);
		context->option_descriptors->option[5].size = sizeof(u_int8_t);
		SET_STRING(context->option_descriptors->option[5].tip, "Remove the selected Spare Object from the array" );
		SET_STRING(context->option_descriptors->option[5].title, "        Remove selected Spare Object" );
		context->option_descriptors->option[5].type = EVMS_Type_Boolean;
		context->option_descriptors->option[5].unit = EVMS_Unit_None;
		context->option_descriptors->option[5].value.bool = FALSE;
		context->option_descriptors->option[5].group.group_number=2;
		context->option_descriptors->option[5].group.group_level=1;


		context->option_descriptors->option[6].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		// get the list of disks that are active.
		get_active_list((value_list_t **)&context->option_descriptors->option[6].constraint.list,context->object);
		// if only 1 active disk, do not allow removal, include in count the 'none' selection
		if (context->option_descriptors->option[6].constraint.list->count == 2){
			context->option_descriptors->option[6].flags = EVMS_OPTION_FLAGS_INACTIVE;
		}
		context->option_descriptors->option[6].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[6].help = NULL;
		SET_STRING(context->option_descriptors->option[6].name,MD_OPTION_ACTIVE_DISKS_NAME);
		context->option_descriptors->option[6].size = EVMS_VOLUME_NAME_SIZE;
		SET_STRING(context->option_descriptors->option[6].tip, "Active bjects in the array to be removed or deactivated" );
		SET_STRING(context->option_descriptors->option[6].title, "Active Objects" );
		context->option_descriptors->option[6].type = EVMS_Type_String;
		context->option_descriptors->option[6].unit = EVMS_Unit_None;
		context->option_descriptors->option[6].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
		strcpy(context->option_descriptors->option[6].value.s, MD_NO_SELECTION);
		context->option_descriptors->option[6].group.group_number=3;
		context->option_descriptors->option[6].group.group_level=2;

		context->option_descriptors->option[7].flags = EVMS_OPTION_FLAGS_INACTIVE;
		context->option_descriptors->option[7].constraint.list = NULL;
		context->option_descriptors->option[7].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[7].help = NULL;
		SET_STRING(context->option_descriptors->option[7].name,MD_OPTION_DEACTIVATE_ACTIVE_NAME);
		context->option_descriptors->option[7].size = sizeof(u_int8_t);
		SET_STRING(context->option_descriptors->option[7].tip, "Make selected Active Object a spare member of the array" );
		SET_STRING(context->option_descriptors->option[7].title, "        Deactivate selected Active Object" );
		context->option_descriptors->option[7].type = EVMS_Type_Boolean;
		context->option_descriptors->option[7].unit = EVMS_Unit_None;
		context->option_descriptors->option[7].value.bool = TRUE;
		context->option_descriptors->option[7].group.group_number=3;
		context->option_descriptors->option[7].group.group_level=1;

		context->option_descriptors->option[8].flags = EVMS_OPTION_FLAGS_INACTIVE;
		context->option_descriptors->option[8].constraint.list = NULL;
		context->option_descriptors->option[8].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[8].help = NULL;
		SET_STRING(context->option_descriptors->option[8].name,MD_OPTION_REMOVE_ACTIVE_NAME);
		context->option_descriptors->option[8].size = sizeof(u_int8_t);
		SET_STRING(context->option_descriptors->option[8].tip, "Remove selected Active Object from the array" );
		SET_STRING(context->option_descriptors->option[8].title, "        Remove selected Active Object" );
		context->option_descriptors->option[8].type = EVMS_Type_Boolean;
		context->option_descriptors->option[8].unit = EVMS_Unit_None;
		context->option_descriptors->option[8].value.bool = FALSE;
		context->option_descriptors->option[8].group.group_number=3;
		context->option_descriptors->option[8].group.group_level=1;


		context->option_descriptors->option[9].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		// get the list of disks that are active.
		get_faulty_list((value_list_t **)&context->option_descriptors->option[9].constraint.list,context->object);
		context->option_descriptors->option[9].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[9].help = NULL;
		SET_STRING(context->option_descriptors->option[9].name,MD_OPTION_FAULTY_DISKS_NAME);
		context->option_descriptors->option[9].size = EVMS_VOLUME_NAME_SIZE;
		SET_STRING(context->option_descriptors->option[9].tip, "Faulty Objects in the array to be removed" );
		SET_STRING(context->option_descriptors->option[9].title, "Faulty Objects" );
		context->option_descriptors->option[9].type = EVMS_Type_String;
		context->option_descriptors->option[9].unit = EVMS_Unit_None;
		context->option_descriptors->option[9].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
		strcpy(context->option_descriptors->option[9].value.s, MD_NO_SELECTION);
		context->option_descriptors->option[9].group.group_number=4;
		context->option_descriptors->option[9].group.group_level=2;

		context->option_descriptors->option[10].flags = EVMS_OPTION_FLAGS_INACTIVE;
		context->option_descriptors->option[10].constraint.list = NULL;
		context->option_descriptors->option[10].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[10].help = NULL;
		SET_STRING(context->option_descriptors->option[10].name,MD_OPTION_REMOVE_FAULTY_NAME);
		context->option_descriptors->option[10].size = sizeof(u_int8_t);
		SET_STRING(context->option_descriptors->option[10].tip, "Remove selected Faulty Object from the array" );
		SET_STRING(context->option_descriptors->option[10].title, "        Remove selected Faulty Object" );
		context->option_descriptors->option[10].type = EVMS_Type_Boolean;
		context->option_descriptors->option[10].unit = EVMS_Unit_None;
		context->option_descriptors->option[10].value.bool = TRUE;
		context->option_descriptors->option[10].group.group_number=4;
		context->option_descriptors->option[10].group.group_level=1;

		break;

	default:
		rc = EINVAL;
		break;
	}

	RETURN(rc);
}


#define PERCENT_WARN_THRESHOLD  5

static void warn_if_big_objects( task_context_t * context )
{

	int rc;
	storage_object_t * obj = NULL;
	storage_object_t * spare = NULL;
	u_int64_t smallest_size = 0;
	uint size;
	TAG tag;

	LOG_ENTRY;

	/* Find the smallest object. */
	GoToStartOfList(context->selected_objects);

	rc = BlindGetObject(context->selected_objects,
			    &size,
			    &tag,
			    NULL,
			    TRUE,
			    (ADDRESS *)&obj);

	if (rc == DLIST_SUCCESS) {
		smallest_size = MD_NEW_SIZE_SECTORS(obj->size);

		rc = NextItem(context->selected_objects);
	}

	while (rc == DLIST_SUCCESS) {

		rc = BlindGetObject(context->selected_objects,
				    &size,
				    &tag,
				    NULL,
				    TRUE,
				    (ADDRESS *)&obj);

		if (rc == DLIST_SUCCESS) {
			smallest_size = min(smallest_size, MD_NEW_SIZE_SECTORS(obj->size));

			rc = NextItem(context->selected_objects);
		}
	}

	/*
	 * If we got a smallest size, then check the size of the spare, if one
	 * is specified and see if it is the smallest.
	 */
	if (smallest_size != 0) {
		if (context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].value.s != NULL) {
			spare = find_object(context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].value.s);

			if (spare != NULL) {
				smallest_size = min(smallest_size, MD_NEW_SIZE_SECTORS(spare->size));
			}
		}
	}

	/*
	 * Now go through the objects again and issue a warnign message for
	 * any object whose size exceeds the threshold over the smallest
	 * object size.
	 */
	if (smallest_size != 0) {
		u_int64_t diffsize;

		GoToStartOfList(context->selected_objects);

		do {
			rc = BlindGetObject(context->selected_objects,
					    &size,
					    &tag,
					    NULL,
					    TRUE,
					    (ADDRESS *) &obj);

			if (rc == DLIST_SUCCESS) {
				diffsize = MD_NEW_SIZE_SECTORS(obj->size) - smallest_size;

				if (diffsize > (smallest_size * PERCENT_WARN_THRESHOLD) / 100) {
					EngFncs->user_message(&raid1_plugin_record,
							      NULL,
							      NULL,
							      "The %s object is %lld MB larger than the smallest object in the RAID array.  "
							      "The extra space will not be used.",
								obj->name, diffsize * EVMS_VSECTOR_SIZE / (1024*1024) );
				}

				rc = NextItem(context->selected_objects);
			}

		} while (rc == DLIST_SUCCESS);

		/*
		 * If we have a spare, check its size too.
		 */
		if (spare != NULL) {
			diffsize = MD_NEW_SIZE_SECTORS(spare->size) - smallest_size;

			if (diffsize > (smallest_size * PERCENT_WARN_THRESHOLD) / 100) {
				EngFncs->user_message(&raid1_plugin_record,
						      NULL,
						      NULL,
						      "The %s object is %lld MB larger than the smallest object in the RAID array.  "
						      "The extra space will not be used.",
						      spare->name, diffsize * EVMS_VSECTOR_SIZE / (1024*1024));
			}
		}
	}

	LOG_EXIT(0);
}

/* Function: raid1_set_option
 *
 *  Determine the type of Task that is being performed. Then examine the
 *  desired option (using the index), and verify that the given value is
 *  appropriate. Reset the value if necessary and possible. Adjust other
 *  options as appropriate.
 */
static int raid1_set_option( task_context_t * context,
			     u_int32_t        index,
			     value_t        * value,
			     task_effect_t  * effect )
{
	int rc = 0;

	LOG_ENTRY;

	// Parameter check.
	if (!context || !value || !effect) {
		RETURN(EFAULT);
	}

	*effect = 0;

	switch (context->action) {
	
	case EVMS_Task_Create:
		switch (index) {
		
		case MD_OPTION_SPARE_DISK_INDEX:
			// Not worth validation, will catch when we try to find the original
			strcpy(context->option_descriptors->option[index].value.s, value->s);
			warn_if_big_objects(context);
			break;

		default:
			break;

		}
		break;

	case EVMS_Task_Set_Info:
		*effect = EVMS_Effect_Reload_Options;
		switch (index) {
		
		case MD_OPTION_SPARE_DISKS_INDEX:
			strcpy(context->option_descriptors->option[index].value.s, value->s);
			if (!strcmp(value->s, MD_NO_SELECTION)) {
				context->option_descriptors->option[MD_OPTION_REMOVE_SPARE_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
				context->option_descriptors->option[MD_OPTION_ACTIVATE_SPARE_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
			} else {
				context->option_descriptors->option[MD_OPTION_REMOVE_SPARE_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
				// only allow spares to be activated if the volume is active in the
				// kernel.  This is to prevent problems with syncing SBs and
				// beacuse the object will not be synced right away if the volume is not active
			  //      if (((md_volume_t *)(context->object->private_data))->flags & MD_ACTIVE) {
			       	context->option_descriptors->option[MD_OPTION_ACTIVATE_SPARE_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
			//	}
			}
			break;
		case MD_OPTION_FAULTY_DISKS_INDEX:
			strcpy(context->option_descriptors->option[index].value.s, value->s);
			break;
		case MD_OPTION_AVAIL_DISKS_INDEX:
			strcpy(context->option_descriptors->option[index].value.s, value->s);
			if (!strcmp(value->s, MD_NO_SELECTION)) {
				context->option_descriptors->option[MD_OPTION_ADD_SPARE_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
				context->option_descriptors->option[MD_OPTION_ADD_ACTIVE_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
			} else {
				context->option_descriptors->option[MD_OPTION_ADD_SPARE_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
				// only allow spares to be activated if the volume is active in the
				// kernel.  This is to prevent problems with syncing SBs and
				// beacuse the object will not be synced right away if the volume is not active
			//	if (((md_volume_t *)(context->object->private_data))->flags & MD_ACTIVE) {
			       	context->option_descriptors->option[MD_OPTION_ADD_ACTIVE_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
			//	}
			}
//			warn_if_big_objects(context);
			break;
		case MD_OPTION_ACTIVE_DISKS_INDEX:
			strcpy(context->option_descriptors->option[index].value.s, value->s);
			if (!strcmp(value->s, MD_NO_SELECTION)) {
				context->option_descriptors->option[MD_OPTION_DEACTIVATE_ACTIVE_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
				context->option_descriptors->option[MD_OPTION_REMOVE_ACTIVE_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
			} else {
				context->option_descriptors->option[MD_OPTION_DEACTIVATE_ACTIVE_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
				context->option_descriptors->option[MD_OPTION_REMOVE_ACTIVE_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
			}
			break;
		case MD_OPTION_ADD_SPARE_INDEX:
			if ( value->bool ) {
				context->option_descriptors->option[index].value.bool = TRUE;
				context->option_descriptors->option[MD_OPTION_ADD_ACTIVE_INDEX].value.bool = FALSE;
			}else{
				context->option_descriptors->option[index].value.bool = FALSE;
				context->option_descriptors->option[MD_OPTION_ADD_ACTIVE_INDEX].value.bool = TRUE;
			}
			*effect = EVMS_Effect_Reload_Options;
			break;
		case MD_OPTION_ADD_ACTIVE_INDEX:
			if ( value->bool ) {
				context->option_descriptors->option[index].value.bool = TRUE;
				context->option_descriptors->option[MD_OPTION_ADD_SPARE_INDEX].value.bool = FALSE;
			}else{
				context->option_descriptors->option[index].value.bool = FALSE;
				context->option_descriptors->option[MD_OPTION_ADD_SPARE_INDEX].value.bool = TRUE;
			}
			*effect = EVMS_Effect_Reload_Options;
			break;
		case MD_OPTION_ACTIVATE_SPARE_INDEX:
			if ( value->bool ) {
				context->option_descriptors->option[index].value.bool = TRUE;
				context->option_descriptors->option[MD_OPTION_REMOVE_SPARE_INDEX].value.bool = FALSE;
			}else{
				context->option_descriptors->option[index].value.bool = FALSE;
				context->option_descriptors->option[MD_OPTION_REMOVE_SPARE_INDEX].value.bool = TRUE;
			}
			*effect = EVMS_Effect_Reload_Options;
			break;
		case MD_OPTION_REMOVE_SPARE_INDEX:
			if ( value->bool ) {
				context->option_descriptors->option[index].value.bool = TRUE;
				context->option_descriptors->option[MD_OPTION_ACTIVATE_SPARE_INDEX].value.bool = FALSE;
			}else{
				context->option_descriptors->option[index].value.bool = FALSE;
				context->option_descriptors->option[MD_OPTION_ACTIVATE_SPARE_INDEX].value.bool = TRUE;
			}
			*effect = EVMS_Effect_Reload_Options;
			break;
		case MD_OPTION_DEACTIVATE_ACTIVE_INDEX:
			if ( value->bool ) {
				context->option_descriptors->option[index].value.bool = TRUE;
				context->option_descriptors->option[MD_OPTION_REMOVE_ACTIVE_INDEX].value.bool = FALSE;
			}else{
				context->option_descriptors->option[index].value.bool = FALSE;
				context->option_descriptors->option[MD_OPTION_REMOVE_ACTIVE_INDEX].value.bool = TRUE;
			}
			*effect = EVMS_Effect_Reload_Options;
			break;
		case MD_OPTION_REMOVE_ACTIVE_INDEX:
			if ( value->bool ) {
				context->option_descriptors->option[index].value.bool = TRUE;
				context->option_descriptors->option[MD_OPTION_DEACTIVATE_ACTIVE_INDEX].value.bool = FALSE;
			}else{
				context->option_descriptors->option[index].value.bool = FALSE;
				context->option_descriptors->option[MD_OPTION_DEACTIVATE_ACTIVE_INDEX].value.bool = TRUE;
			}
			*effect = EVMS_Effect_Reload_Options;
			break;

		default:
			break;

		}
		break;

	default:
		break;
	}
	RETURN(rc);
}


/* Function: raid1_set_objects
 *
 *  Determine the type of task, and then validate that the objects on the
 *  "selected" list are valid for that task. If so, adjust the option
 *  descriptor as appropriate.
 */
static int raid1_set_objects( task_context_t * context,
			      dlist_t          declined_objects,
			      task_effect_t  * effect )
{
	int rc = 0;

	LOG_ENTRY;

	// Parameter check.
	if (!context || !declined_objects || !effect) {
		RETURN(EFAULT);
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		get_object_list((value_list_t **)&context->option_descriptors->option[0].constraint.list,context->selected_objects,0);
		warn_if_big_objects(context);
		*effect = EVMS_Effect_Reload_Options;
		break;

	case EVMS_Task_Set_Info:
		// should be nothing.
		break;

	default:
		break;
	}
	RETURN(rc);
}

static int get_set_info_options(option_array_t * options, char ** active_disk, int * active_flag, char ** spare_disk,
				int *spare_flag, char ** avail_disk, int * avail_flag,char ** faulty_disk, int * faulty_flag){

	int i, rc = 0;

	LOG_ENTRY;

	for (i = 0; i < options->count; i++) {
		switch (options->option[i].number) {
			
			case MD_OPTION_SPARE_DISKS_INDEX:
				if (strcmp(MD_NO_SELECTION,options->option[i].value.s)) {
					SET_STRING(*spare_disk,options->option[i].value.s);
				}
				break;
			case MD_OPTION_AVAIL_DISKS_INDEX:
				if (strcmp(MD_NO_SELECTION,options->option[i].value.s)) {
					SET_STRING(*avail_disk,options->option[i].value.s);
				}
				break;
			case MD_OPTION_ACTIVE_DISKS_INDEX:
				if (strcmp(MD_NO_SELECTION,options->option[i].value.s)) {
					SET_STRING(*active_disk,options->option[i].value.s);
				}
				break;
        		case MD_OPTION_FAULTY_DISKS_INDEX:
        			if (strcmp(MD_NO_SELECTION,options->option[i].value.s)) {
        				SET_STRING(*faulty_disk,options->option[i].value.s);
        			}
        			break;
			case MD_OPTION_ADD_SPARE_INDEX:
				if (options->option[i].value.bool) {
					*avail_flag = MD_OPTION_ADD_SPARE_INDEX;
				}
				break;
			case MD_OPTION_ADD_ACTIVE_INDEX:
				if (options->option[i].value.bool) {
					*avail_flag = MD_OPTION_ADD_ACTIVE_INDEX;
				}
				break;
			case MD_OPTION_ACTIVATE_SPARE_INDEX:
				if (options->option[i].value.bool) {
					*spare_flag = MD_OPTION_ACTIVATE_SPARE_INDEX;
				}
				break;
			case MD_OPTION_REMOVE_SPARE_INDEX:
				if (options->option[i].value.bool) {
					*spare_flag = MD_OPTION_REMOVE_SPARE_INDEX;
				}
				break;
			case MD_OPTION_DEACTIVATE_ACTIVE_INDEX:
				if (options->option[i].value.bool) {
					*active_flag = MD_OPTION_DEACTIVATE_ACTIVE_INDEX;
				}
				break;
			case MD_OPTION_REMOVE_ACTIVE_INDEX:
				if (options->option[i].value.bool) {
					*active_flag = MD_OPTION_REMOVE_ACTIVE_INDEX;
				}
				break;
        		case MD_OPTION_REMOVE_FAULTY_INDEX:
        			if (options->option[i].value.bool) {
        				*faulty_flag = MD_OPTION_REMOVE_FAULTY_INDEX;
        			}
        			break;

			default:
				break;

		}
	}
	RETURN(rc);
}
static int swap_disks(md_volume_t * volume, int index, int index2){
	storage_object_t * tmp_object;
	mdp_disk_t tmp_disk;
	LOG_ENTRY;

	// swap the disk entries in the master SB
	tmp_disk = volume->super_block->disks[index];
	volume->super_block->disks[index] = volume->super_block->disks[index2];
	volume->super_block->disks[index2] = tmp_disk;

	volume->super_block->disks[index].number = index;
	volume->super_block->disks[index].raid_disk = index;
	volume->super_block->disks[index2].number = index2;
	volume->super_block->disks[index2].raid_disk = index2;
	// swap the object pointers
	tmp_object = volume->child_object[index];
	volume->child_object[index] = volume->child_object[index2];
	volume->child_object[index2] = tmp_object;
 /*
	tmp_super = volume->super_array[index];
	volume->super_array[index] = volume->super_array[index2];
	volume->super_array[index2] = tmp_super;

	
	for (i = 0; i < volume->nr_disks; i++) {
		tmp_disk = volume->super_array[i]->disks[index];
		volume->super_array[i]->disks[index] = volume->super_array[i]->disks[index2];
		volume->super_array[i]->disks[index2] = tmp_disk;
	}*/
	RETURN(0);
}


// Make sure all active disks are before all spare disks.
static int order_disks(md_volume_t * volume, int index){
	int      rc= 0;
	LOG_ENTRY;

	if (volume->super_block->disks[index].state & (1<< MD_DISK_PENDING_ACTIVE)) {
		// disk is active, needs to move up in list
		while (index && !(volume->super_block->disks[index - 1].state & (1<< MD_DISK_ACTIVE))) {
			swap_disks(volume, index, index-1);
			index--;
		}
	} else {
		// disk is spare, needs to move down in list below the Actives
		while ((index < volume->nr_disks - 1)
			&& (volume->super_block->disks[index + 1].state & (1<< MD_DISK_ACTIVE | 1<<MD_DISK_PENDING_ACTIVE))) {
			swap_disks(volume, index, index +1);
			index++;
		}
		// Also, move the spare above any faulty or removed disks
		while (index && (volume->super_block->disks[index - 1].state & (1<<MD_DISK_FAULTY | 1<<MD_DISK_REMOVED))) {
			swap_disks(volume, index, index-1);
			index--;
		}
	}


	RETURN(rc);
}

static int  deactivate_disk(md_volume_t * volume, char *active_disk){

	int      i, rc= 0;
	u_int32_t new_time;
	LOG_ENTRY;

	new_time = time(NULL);
	for (i = 0; i < volume->nr_disks; i++ ) {
		if (!strcmp(active_disk, volume->child_object[i]->name)) {
			// ok, found the disk to deactivate
       			volume->super_block->disks[i].state = 0;
       			volume->super_block->spare_disks++;
       			volume->super_block->active_disks--;
       			volume->super_block->raid_disks--;
			break;
		}
	}
	order_disks(volume, i);

	md_rediscover_volumes_for_region(volume->region);

	RETURN(rc);
}
static int  activate_spare(md_volume_t * volume, char *spare_disk){

	int      i, rc= 0;
	LOG_ENTRY;

	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		if (!volume->child_object[i]) {
			continue;
		}
		if (!strcmp(spare_disk, volume->child_object[i]->name)) {
			md_add_modify_object_to_list(volume, EVMS_MD_ACTIVATE,
						volume->super_block->disks[i].major,
						volume->super_block->disks[i].minor);
			// ok, found the disk to activate
			// leave state as spare, kernel will bring online
       			volume->super_block->disks[i].state |= (1<<MD_DISK_PENDING_ACTIVE);
       			volume->super_block->raid_disks++;

			break;
		}
	}

	order_disks(volume, i);

	md_rediscover_volumes_for_region(volume->region);

	RETURN(rc);
}

static int  remove_disk(md_volume_t * volume, char *active_disk){

	int      i,k, rc= 0;
	LOG_ENTRY;

	for (i = 0; i < volume->nr_disks; i++ ) {
		if (!strcmp(active_disk, volume->child_object[i]->name)) {
			// ok, found the disk to remove

			// this disk is going away, add it to our list of disks
			// to ioctl to the kernel.
			md_add_modify_object_to_list(volume, EVMS_MD_REMOVE,
						volume->super_block->disks[i].major,
						volume->super_block->disks[i].minor);

			md_remove_region_from_object(volume->region, volume->child_object[i]);
			KILL_SECTORS(volume->child_object[i],
				MD_NEW_SIZE_SECTORS(volume->child_object[i]->size),
				MD_RESERVED_SECTORS);
			md_deallocate_memory(volume->super_array[i]);

		       // collapse super array and object array
			for (k = i; k < volume->nr_disks - 1; k++) {
				volume->super_array[k]= volume->super_array[k+1];
				volume->child_object[k] = volume->child_object[k+1];
				volume->super_block->disks[k]= volume->super_block->disks[k+1];
				volume->super_block->disks[k].number = k;
				volume->super_block->disks[k].raid_disk = k;
			}

       			memset(&volume->super_block->disks[k],0,sizeof(mdp_disk_t));  //NULL out now empty disk entry
       			volume->super_block->active_disks--;
       			volume->super_block->working_disks--;
       			volume->super_block->raid_disks--;
       			volume->super_block->nr_disks--;
			volume->nr_disks--;
			break;
		}
	}

	md_rediscover_volumes_for_region(volume->region);
	
	RETURN(rc);
}

static int  add_active(md_volume_t * volume, char *avail_disk){

	int       rc= 0;
	u_int32_t new_time;
	storage_object_t * object;
	mdp_disk_t disk;

	LOG_ENTRY;

	rc = md_clone_superblock(volume, volume->nr_disks);
	object = find_object(avail_disk);

	if (rc || !object) {
		MESSAGE("Error adding object %s to array, can't find, or no more memory\n",avail_disk);
		RETURN(rc);
	}

	new_time = time(NULL);

	volume->child_object[volume->nr_disks] = object;
	md_append_region_to_object(volume->region, object);

	get_legacy_dev(volume, object->name, &disk.major, &disk.minor);
	disk.number = volume->nr_disks;
	disk.raid_disk = disk.number;
	disk.state = (1<<MD_DISK_NEW | 1<<MD_DISK_PENDING_ACTIVE);
       	memcpy(&volume->super_block->disks[volume->nr_disks], &disk, sizeof(mdp_disk_t));  //copy new disk entry to each SB
	md_add_modify_object_to_list(volume, EVMS_MD_ADD,
				disk.major,
				disk.minor);
	md_add_modify_object_to_list(volume, EVMS_MD_ACTIVATE,
				disk.major,
				disk.minor);
//      	volume->super_block->active_disks++;
      	volume->super_block->spare_disks++;	//new
      	volume->super_block->working_disks++;
      	volume->super_block->raid_disks++;
      	volume->super_block->nr_disks++;
       	volume->super_block->state &= ~(1 << MD_SB_CLEAN); // mark all dirty
	volume->commit_flag = 0;
       	volume->nr_disks++;

	order_disks(volume, volume->nr_disks - 1);

	md_rediscover_volumes_for_region(volume->region);

	RETURN(rc);
}

static int  add_spare(md_volume_t * volume, char *avail_disk){

	int       rc= 0,i;
	storage_object_t * object;
	mdp_disk_t disk;

	LOG_ENTRY;

	object = find_object(avail_disk);
	if (rc || !object) {
		MESSAGE("Error adding object %s to array, can't find, or no more memory\n",avail_disk);
		RETURN(rc);
	}

	rc = md_clone_superblock(volume, volume->nr_disks);

	// look for 1st hole in array
/*	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		if (volume->child_object[i] == NULL) {
			break;
		}
	}*/
	i = volume->nr_disks;

	volume->child_object[i] = object;
	md_append_region_to_object(volume->region, object);

	get_legacy_dev(volume, object->name, &disk.major, &disk.minor);
	disk.number = i;
	disk.raid_disk = disk.number;
	disk.state = (1<<MD_DISK_NEW);
       	memcpy(&volume->super_block->disks[i], &disk, sizeof(mdp_disk_t));  //copy new disk entry to each SB
	md_add_modify_object_to_list(volume, EVMS_MD_ADD,
				disk.major,
				disk.minor);
       	volume->super_block->spare_disks++;
       	volume->super_block->working_disks++;
       	volume->super_block->nr_disks++;
       	volume->nr_disks++;

	order_disks(volume, volume->nr_disks - 1); //need to make sure this new spare
						   // get put above any faulty disks

	md_rediscover_volumes_for_region(volume->region);
	
	RETURN(rc);
}

static int  remove_spare(md_volume_t * volume, char *spare_disk){

	int      i,k, rc= 0;
	u_int32_t new_time;
	LOG_ENTRY;

	new_time = time(NULL);
	for (i = 0; i < volume->nr_disks; i++ ) {
		if (!strcmp(spare_disk, volume->child_object[i]->name)) {
			// ok, found the disk to remove

			// this disk is going away, add it to our list of disks
			// to ioctl to the kernel.
			md_add_modify_object_to_list(volume, EVMS_MD_REMOVE,
						volume->super_block->disks[i].major,
						volume->super_block->disks[i].minor);

			md_remove_region_from_object(volume->region, volume->child_object[i]);

			KILL_SECTORS(volume->child_object[i],
				MD_NEW_SIZE_SECTORS(volume->child_object[i]->size),
				MD_RESERVED_SECTORS);
			md_deallocate_memory(volume->super_array[i]);

		       // collapse super array and object array
			for (k = i; k < volume->nr_disks -1; k++) {
				volume->super_array[k]= volume->super_array[k+1];
				volume->child_object[k] = volume->child_object[k+1];
				volume->super_block->disks[k]= volume->super_block->disks[k+1];
				volume->super_block->disks[k].number = k;
				volume->super_block->disks[k].raid_disk = k;
			}

       			memset(&volume->super_block->disks[k],0,sizeof(mdp_disk_t));  //NULL out now empty disk entry
       			volume->super_block->working_disks--;
       			volume->super_block->spare_disks--;
       			volume->super_block->nr_disks--;
			volume->nr_disks--;
			break;
		}
	}

	md_rediscover_volumes_for_region(volume->region);

	RETURN(rc);
}

static int  remove_faulty(md_volume_t * volume, char *faulty_disk){

	int      i,k, rc= 0;
	u_int32_t new_time;
	LOG_ENTRY;

	new_time = time(NULL);
	for (i = 0; i < volume->nr_disks; i++ ) {
		if (!strcmp(faulty_disk, volume->child_object[i]->name)) {
			// ok, found the disk to remove

			// this disk is going away, add it to our list of disks
			// to ioctl to the kernel.
			md_add_modify_object_to_list(volume, EVMS_MD_REMOVE,
						volume->super_block->disks[i].major,
						volume->super_block->disks[i].minor);

			md_remove_region_from_object(volume->region, volume->child_object[i]);
			KILL_SECTORS(volume->child_object[i],
				MD_NEW_SIZE_SECTORS(volume->child_object[i]->size),
				MD_RESERVED_SECTORS);
			md_deallocate_memory(volume->super_array[i]);

		       // collapse super array and object array
			for (k = i; k < volume->nr_disks -1; k++) {
				volume->super_array[k]= volume->super_array[k+1];
				volume->child_object[k] = volume->child_object[k+1];
				volume->super_block->disks[k]= volume->super_block->disks[k+1];
				volume->super_block->disks[k].number = k;
				volume->super_block->disks[k].raid_disk = k;
			}

       			memset(&volume->super_block->disks[k],0,sizeof(mdp_disk_t));  //NULL out now empty disk entry
       			volume->super_block->failed_disks--;
       			volume->super_block->nr_disks--;
//     			volume->super_block->raid_disks--;	// raid disks include faulty
			volume->nr_disks--;
			break;
		}
	}

	md_rediscover_volumes_for_region(volume->region);

	RETURN(rc);
}


static int raid1_set_info(storage_object_t * region,
			  option_array_t   * options){
	int           rc= -1;
	int active_flag = -1, spare_flag = -1, avail_flag = -1, faulty_flag = -1;
	char * active_disk = NULL;
	char * spare_disk  = NULL;
	char * avail_disk  = NULL;
	char * faulty_disk  = NULL;
	md_volume_t * volume = (md_volume_t *) region->private_data;

	LOG_ENTRY;
	// Make sure this is an MD region
	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s is not owned by MD RAID1\n", region->name);
		RETURN(EINVAL);
	}

	if (volume->flags & MD_CORRUPT || volume->flags & MD_DEGRADED){
		int    answer = 0;
		char * choice_text[3] = { "Don't Fix", "Fix", NULL };
		EngFncs->user_message(my_plugin, &answer,choice_text,
		       	"Errors have been detected on MD region %s. Check the message "
		       	"display for messages detailing the errors requiring fixing.  "
		       	"Selecting \"Fix\" will permanently change the array.  If you "
		       	"elect to not fix the region at this time, you may not modify "
		       	"the region. \n\n",
			volume->name);
		if (answer) {
			raid1_verify_and_fix_array(volume, 1);
		} else{
			RETURN(EINVAL);
		}
	}

	get_set_info_options(options, &active_disk, &active_flag, &spare_disk,
			&spare_flag, &avail_disk, &avail_flag, &faulty_disk, &faulty_flag);

	// if active disk selected andf proper flag set
	if (active_disk && (active_flag == MD_OPTION_DEACTIVATE_ACTIVE_INDEX)) {
		rc = deactivate_disk(volume, active_disk);
	} else if (active_disk && (active_flag == MD_OPTION_REMOVE_ACTIVE_INDEX)) {
		rc = remove_disk(volume, active_disk);
	}

	if (spare_disk && (spare_flag == MD_OPTION_ACTIVATE_SPARE_INDEX)) {
		rc = activate_spare(volume, spare_disk);
	} else if (spare_disk && (spare_flag == MD_OPTION_REMOVE_SPARE_INDEX)) {
		rc = remove_spare(volume, spare_disk);
	}

	if (avail_disk && (avail_flag == MD_OPTION_ADD_ACTIVE_INDEX)) {
		rc = add_active(volume, avail_disk);
	} else if (avail_disk && (avail_flag == MD_OPTION_ADD_SPARE_INDEX)) {
		rc = add_spare(volume, avail_disk);
	}

	if (faulty_disk) {
		rc = remove_faulty(volume, faulty_disk);
	}

	if (!rc) {
		EngFncs->set_changes_pending();
		region->flags |= SOFLAG_DIRTY;
	} else	{ // rc was set to -1 to trace changes
		if (rc == -1) {
			rc =0;
		}
	}

	RETURN(rc);
}

/* Function: raid1_get_info
 *
 *  Return MD-specific information about the specified region. If the
 *  name field is set, only return the "extra" information pertaining
 *  to that name.
 */
static int raid1_get_info( storage_object_t       * region,
			   char                   * name,
			   extended_info_array_t ** info_array )
{

	md_volume_t * volume = NULL;
	int           rc= 0;

	LOG_ENTRY;

	// Parameter check
	if (!region || !info_array) {
		RETURN(EFAULT);
	}

	// Make sure this is an MD region
	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s is not owned by MD RAID1\n", region->name);
		RETURN(EINVAL);
	}

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

	RETURN(rc);
}


/* Function: raid1_get_plugin_info
 *
 *  Return information about the MD plugin. There is no "extra"
 *  information about MD, so "name" should always be NULL.
 */
static int raid1_get_plugin_info( char                     * name,
				  extended_info_array_t   ** info_array )
{

	extended_info_array_t   * info = NULL;
	char buffer[50] = {0};
	int i = 0;
	int rc = 0;

	LOG_ENTRY;

	// Parameter check
	if (! info_array) {
		RETURN(EFAULT);
	}

	if (! name) {
		// Get memory for the info array
		if (! (info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*4))) {
			LOG_ERROR("Error allocating memory for info array\n");
			RETURN(ENOMEM);
		}

		// Short Name
		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid1_plugin->short_name);
		i++;

		// Long Name
		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid1_plugin->long_name);
		i++;

		// Plugin Type
		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Region Manager");
		i++;

		// Plugin Version
		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required API Version
		SET_STRING(info->info[i].name, "Required_Version");
		SET_STRING(info->info[i].title, "Required Plugin API Version");
		SET_STRING(info->info[i].desc, "This is the version of the engine that the plugin requires.It will not run on older versions of the Engine.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid1_plugin->required_api_version.major, raid1_plugin->required_api_version.minor, raid1_plugin->required_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;
	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		RETURN(EINVAL);
	}

	info->count = i;
	*info_array = info;
	RETURN(0);
}


/* Function: raid1_read
 *
 *  Perform a logical-to-physical remapping, and send the read down to
 *  the next plugin.
 */
static int raid1_read( storage_object_t * region,
		       lsn_t              lsn,
		       sector_count_t     count,
		       void             * buffer )
{

	md_volume_t *   volume = (md_volume_t *)region->private_data;
	int             rc = 0;
	int i;

	LOG_ENTRY;

	// Parameter check.
	if (!region || !buffer) {
		RETURN(EFAULT);
	}

	if (volume->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n ",volume->name);
		RETURN(0);
	}

	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to read past end of region %s sector=%d\n ",volume->name,lsn+count);
	}

	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		// check for null object, if missing, skip
		if (volume->child_object[i]) {
			// only read from Active disks, not the spares.
			if (volume->super_block->disks[i].state & ( 1 << MD_DISK_ACTIVE)) {
				rc = READ(volume->child_object[i], lsn, count, buffer);
				if (rc) {
					//BUGBUG disable mirror
					MESSAGE("Error reading from mirror %s of region %s sector=%d, Mirror disabled.\n ",
						volume->child_object[i]->name, volume->name,lsn+count);
				}else{
					break;
				}

			}
		}
	}
	RETURN(rc);
}


/* Function: raid1_write
 *
 *  Perform a logical-to-physical remapping, and send the write down to
 *  the next plugin.
 */
static int raid1_write( storage_object_t * region,
			lsn_t              lsn,
			sector_count_t     count,
			void             * buffer )
{

	md_volume_t *   volume = (md_volume_t *)region->private_data;
	int             rc = 0;
	int i, found=0;

	LOG_ENTRY;

	// Parameter check.
	if (!region || !buffer) {
		RETURN(EFAULT);
	}

	if (volume->flags & MD_CORRUPT) {
		MESSAGE("MD Object %s is corrupt, writing data is not allowed\n ",volume->name);
		RETURN(EIO);
	}
	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%d\n ",volume->name,lsn+count);
	}

	for (i = 0; (i < MAX_MD_DEVICES) && (found < volume->nr_disks); i++ ) {
		// check for null object, if missing, skip
		if (volume->child_object[i]) {
			found++;
			// only write on Active disks, not the spares.
			if (volume->super_block->disks[i].state & (1 << MD_DISK_ACTIVE)) {
				rc = WRITE(volume->child_object[i], lsn, count, buffer);
				if (rc) {
					//BUGBUG disable mirror
				}
			}
		}
	}
	RETURN(rc);
}


/* Function: raid1_direct_plugin_communication
 */
static int raid1_direct_plugin_communication( void    * thing,
					      boolean   target_kernel_plugin,
					      void    * arg )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}

static int free_region (ADDRESS object,
			TAG     object_tag,
			uint    object_size,
			ADDRESS object_handle,
			ADDRESS parameters) {

	int i;
	int nr_disks = 0;
	storage_object_t * region = (storage_object_t *) object;
	md_volume_t * volume = (md_volume_t *)region->private_data;

	for (i = 0; (i < MAX_MD_DEVICES) && (nr_disks < volume->nr_disks); i++) {
		if (volume->child_object[i]) {
			nr_disks++;
			md_deallocate_memory(volume->super_array[i]);
		}
	}
	md_remove_volume_from_list(volume);
	md_deallocate_memory(volume);

	RETURN(DLIST_SUCCESS);
}



void raid1_plugin_cleanup(void) {

	dlist_t regions_list;

	LOG_ENTRY;

	EngFncs->get_object_list(REGION, DATA_TYPE, raid1_plugin, 0, &regions_list);

	ForEachItem(regions_list, free_region, NULL, TRUE);

	DestroyList(&regions_list, FALSE);

	LOG_EXIT(0);
	return;
}


/* Function tables for the MD Region Manager */
static plugin_functions_t raid1_functions = {
	cleanup_evms_plugin     : raid1_plugin_cleanup,
	setup_evms_plugin       : raid1_setup_evms_plugin,
	can_delete              : raid1_can_delete,
	can_expand              : raid1_can_expand,
	can_expand_by           : raid1_can_expand_by,
	can_shrink              : raid1_can_shrink,
	can_shrink_by           : raid1_can_shrink_by,
	can_move                : raid1_can_move,
	can_set_volume          : raid1_can_set_volume,
	discover                : raid1_discover,
	create                  : raid1_create,
	delete                  : raid1_delete,
	expand                  : raid1_expand,
	shrink                  : raid1_shrink,
	move                    : raid1_move,
	set_volume              : raid1_set_volume,
	add_sectors_to_kill_list: raid1_add_sectors_to_kill_list,
	commit_changes          : raid1_commit_changes,
	get_option_count        : raid1_get_option_count,
	init_task               : raid1_init_task,
	set_option              : raid1_set_option,
	set_objects             : raid1_set_objects,
	get_info                : raid1_get_info,
	set_info		: raid1_set_info,
	get_plugin_info         : raid1_get_plugin_info,
	read                    : raid1_read,
	write                   : raid1_write,
	direct_plugin_communication : raid1_direct_plugin_communication
};



/* Function: PluginInit
 *
 *  Initializes the local plugin record
 */

plugin_record_t raid1_plugin_record = {
	id:                     SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 5),

	version:               {major:         MAJOR_VERSION,
				minor:         MINOR_VERSION,
				patchlevel:    PATCH_LEVEL},

	required_api_version:  {major:         3,
				minor:         0,
				patchlevel:    0},

	short_name:             "MDRaid1RegMgr",
	long_name:              "MD Raid 1 Region Manager",
	oem_name:               "IBM",

	functions:              {plugin:        &raid1_functions},

	container_functions:    NULL
};


