/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvm_options.c
 *
 * Description: This file contains all functions related to the user-available
 *              options in the LVM region manager.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"



/****** Volume Group Creation Option Handlers ******/


/* Function: lvm_create_container_parse_option_array
 *
 *	Parse the options array to find options pertaining to volume group
 *	creation. Each option can either be keyed by value or string, so we
 *	have to check for both. Acceptable options for group creation are:
 *	LVM_OPTION_VG_NAME
 *	LVM_OPTION_PE_SIZE
 *	Incorrect options are simply ignored.
 */
int lvm_create_container_parse_option_array(	option_array_t	* options,
						char		* vg_name,
						u_int32_t	* pe_size )
{
	int i, rc = 0;

	LOG_ENTRY;

	// Default values
	*pe_size = LVM_DEFAULT_PE_SIZE;

	for ( i = 0; i < options->count; i++ ) {

		// If only the name-based index is specified, get the number.
		if ( ! options->option[i].is_number_based ) {
			if ( ! strcmp(options->option[i].name, LVM_OPTION_VG_NAME_STR) ) {
				options->option[i].number = LVM_OPTION_VG_NAME_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_PE_SIZE_STR) ) {
				options->option[i].number = LVM_OPTION_PE_SIZE_INDEX;
			}
			else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		// Use the number-based index to record the option.
		switch ( options->option[i].number ) {
		case LVM_OPTION_VG_NAME_INDEX:
			strncpy(vg_name, options->option[i].value.s, NAME_LEN);
			break;
		case LVM_OPTION_PE_SIZE_INDEX:
			*pe_size = options->option[i].value.ui32;
			break;
		default:
			break;
		}
	}

	rc = lvm_create_container_verify_options(vg_name, pe_size);

	RETURN(rc);
}


/* Function: lvm_create_container_verify_options
 *
 *	Run through the options that have been collected, and verify that they
 *	are all valid for creating a new group. Correct any bad options if
 *	possible, or return an error.
 */
int lvm_create_container_verify_options(char		* vg_name,
					u_int32_t	* pe_size )
{
	int	rc = 0;
	LOG_ENTRY;

	if ( (rc = lvm_check_vg_name(vg_name)) ) {
		RETURN(rc);
	}

	lvm_check_pe_size(pe_size);

	RETURN(0);
}


/* Function: lvm_create_container_allocate_options
 *
 *	This is a helper function for lvm_init_options. It sets up the initial
 *	information in the option descriptor for a Create_Container task.
 */
int lvm_create_container_allocate_option_descriptor( option_desc_array_t * od )
{
	LOG_ENTRY;

	od->count = LVM_OPTION_PE_SIZE_INDEX + 1;

	// Option 0 is VG name
	SET_STRING(od->option[LVM_OPTION_VG_NAME_INDEX].name, LVM_OPTION_VG_NAME_STR);
	SET_STRING(od->option[LVM_OPTION_VG_NAME_INDEX].title, "Name for new LVM container");
	od->option[LVM_OPTION_VG_NAME_INDEX].type	= EVMS_Type_String;
	od->option[LVM_OPTION_VG_NAME_INDEX].size	= EVMS_VOLUME_NAME_SIZE;
	od->option[LVM_OPTION_VG_NAME_INDEX].flags	= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[LVM_OPTION_VG_NAME_INDEX].value.s	= lvm_engine->engine_alloc(EVMS_VOLUME_NAME_SIZE);

	// Option 1 is PE size
	SET_STRING(od->option[LVM_OPTION_PE_SIZE_INDEX].name, LVM_OPTION_PE_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_PE_SIZE_INDEX].title, "PE size for new container");
	SET_STRING(od->option[LVM_OPTION_PE_SIZE_INDEX].tip, "Acceptable range: 8kB to 16GB. Must be a power of 2.");
	od->option[LVM_OPTION_PE_SIZE_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_PE_SIZE_INDEX].unit	= EVMS_Unit_Sectors;
	od->option[LVM_OPTION_PE_SIZE_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_PE_SIZE_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[LVM_OPTION_PE_SIZE_INDEX].constraint_type = EVMS_Collection_List;
	SET_POWER2_LIST(od->option[LVM_OPTION_PE_SIZE_INDEX].constraint.list, LVM_MIN_PE_SIZE, LVM_MAX_PE_SIZE);
	od->option[LVM_OPTION_PE_SIZE_INDEX].value.ui32 = LVM_DEFAULT_PE_SIZE;

	RETURN(0);
}


/* Function: lvm_create_container_get_acceptable
 *
 *	This is a helper function for lvm_init_task. It determines which
 *	objects in EVMS are appropriate for a Create_Container task.
 */
int lvm_create_container_get_acceptable( dlist_t acceptable_segments )
{
	dlist_t	object_list;
	int	rc;

	LOG_ENTRY;

	// Find all top-level disks, segments, and regions in the system. This
	// will also retrieve LVM regions, so let the user beware. :)
	if ( (rc = lvm_engine->get_object_list(DISK|SEGMENT|REGION, DATA_TYPE, NULL, VALID_INPUT_OBJECT, &object_list)) ) {
		RETURN(rc);
	}
	CopyList(acceptable_segments, object_list, AppendToList);
	DestroyList(&object_list, FALSE);

	RETURN(rc);
}


/* Function: lvm_create_container_set_objects
 *
 *	Examine the selected_objects list in the task, and verify that each
 *	object is valid for inclusion in a new container. If any are not valid,
 *	remove them from the selected_objects and add them to the
 *	declined_objects. In create_container, no options will change based
 *	on which objects are selected.
 */
int lvm_create_container_set_objects(	task_context_t	* context,
					dlist_t		declined_objects,
					task_effect_t	* effect )
{
	storage_object_t	* segment;
	int			rc;

	LOG_ENTRY;

	// There are no side-effects from setting objects in a
	// create container task.
	*effect = 0;

	// Check that all of the segments are valid for use in a new group.
	for ( rc = GoToStartOfList(context->selected_objects);
	      !rc && (segment = lvm_get_list_item(context->selected_objects));
	      rc = NextItem(context->selected_objects) ) {

		rc = lvm_check_segment_for_group_inclusion(segment);
		if (rc) {
// Add this segment to declined objects.
			LOG_ERROR("One or more objects are invalid for container creation\n");
			RETURN(EINVAL);
		}

		// If any of the segments are too small for the default PE
		// size, try to reset the PE size.
		rc = lvm_check_segment_for_pe_size(segment, &(context->option_descriptors->option[LVM_OPTION_PE_SIZE_INDEX].value.ui32));
		if (rc) {
			LOG_DEBUG("Object %s is too small\n");
			LOG_DEBUG("Resetting initial PE size value to %d sectors\n",
				context->option_descriptors->option[LVM_OPTION_PE_SIZE_INDEX].value.ui32);
			*effect |= EVMS_Effect_Reload_Options;
		}
	}

	RETURN(0);
}


/* Function: lvm_create_container_set_option
 *
 *	This is a helper function for lvm_set_option. It sets one particular
 *	option in the option descriptor for a Create_Container task, and
 *	updates other option descriptor information accordingly.
 */
int lvm_create_container_set_option(	task_context_t	* context,
					u_int32_t	index,
					value_t		* value,
					task_effect_t	* effect )
{
	option_desc_array_t	* od = context->option_descriptors;
	storage_object_t	* segment;
	int			rc = 0;

	LOG_ENTRY;
	LOG_EXTRA("Setting option %d\n", index);

	switch (index) {

	case LVM_OPTION_VG_NAME_INDEX:
		if ( strlen(value->s) >= EVMS_VOLUME_NAME_SIZE ) {
			LOG_ERROR("Container name too long\n");
			rc = ENOSPC;
			break;
		}
		if ( (rc = lvm_check_vg_name(value->s)) ) {
			break;
		}
		strcpy(od->option[index].value.s, value->s);
		od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		break;

	case LVM_OPTION_PE_SIZE_INDEX:
		if ( lvm_check_pe_size(&(value->ui32)) ) {
			*effect |= EVMS_Effect_Inexact;
		}

		// Make sure every segment in the "selected" list is large
		// enough for the specified pe_size.
		for ( rc = GoToStartOfList(context->selected_objects);
		      !rc && (segment = lvm_get_list_item(context->selected_objects));
		      rc = NextItem(context->selected_objects) ) {

			rc = lvm_check_segment_for_pe_size(segment, &(value->ui32));
			if (rc) {
				LOG_ERROR("One or more objects too small for PE size.\n");
				rc = ENOSPC;
				break;
			}
		}

		if ( rc != ENOSPC ) {
			od->option[index].value.ui32 = value->ui32;
			od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
			rc = 0;
		}
		break;
	default:
		rc = EINVAL;
		break;
	}

	RETURN(rc);
}



/****** Logical Volume Creation Option Handlers ******/


/* Function: lvm_create_region_parse_option_array
 *
 *	Parse the option_array_t to find options pertaining to logical volume
 *	creation. Each option can either be keyed by value or string, so we
 *	have to check for both. Acceptable options for volume creation are:
 *	LVM_OPTION_LV_NAME
 *	LVM_OPTION_EXTENTS
 *	LVM_OPTION_LV_SIZE
 *	LVM_OPTION_STRIPES
 *	LVM_OPTION_STRIPE_SIZE
 *	LVM_OPTION_SNAPSHOT
 *	LVM_OPTION_CHUNK_SIZE
 *	LVM_OPTION_CONTIGUOUS
 *	LVM_OPTION_READ_ONLY
 *	LVM_OPTION_ZERO
 *	Incorrect options are simply ignored.
 */
int lvm_create_region_parse_option_array(option_array_t		* options,
					lvm_volume_group_t	* group,
					lvm_lv_create_options_t	* lv_opts )
{
	int i, j, rc;

	LOG_ENTRY;

	// Initialize the options
	memset(lv_opts, 0, sizeof(lvm_lv_create_options_t));
	lv_opts->stripes = 1;
	lv_opts->zero = TRUE;

	for ( i = 0; i < options->count; i++ ) {

		// If only the name-based index is specified, get the number.
		if ( ! options->option[i].is_number_based ) {
			if ( ! strcmp(options->option[i].name, LVM_OPTION_LV_NAME_STR) ) {
				options->option[i].number = LVM_OPTION_LV_NAME_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_EXTENTS_STR) ) {
				options->option[i].number = LVM_OPTION_EXTENTS_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_LV_SIZE_STR) ) {
				options->option[i].number = LVM_OPTION_LV_SIZE_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_STRIPES_STR) ) {
				options->option[i].number = LVM_OPTION_STRIPES_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_STRIPE_SIZE_STR) ) {
				options->option[i].number = LVM_OPTION_STRIPE_SIZE_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_SNAPSHOT_STR) ) {
				options->option[i].number = LVM_OPTION_SNAPSHOT_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_CHUNK_SIZE_STR) ) {
				options->option[i].number = LVM_OPTION_CHUNK_SIZE_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_SNAPSHOT_ORG_STR) ) {
				options->option[i].number = LVM_OPTION_SNAPSHOT_ORG_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_CONTIGUOUS_STR) ) {
				options->option[i].number = LVM_OPTION_CONTIGUOUS_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_READ_ONLY_STR) ) {
				options->option[i].number = LVM_OPTION_READ_ONLY_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_ZERO_STR) ) {
				options->option[i].number = LVM_OPTION_ZERO_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_PV_NAMES_STR) ) {
				options->option[i].number = LVM_OPTION_PV_NAMES_INDEX;
			}
			else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		// Use the number-based index to record the option.
		switch ( options->option[i].number ) {
		case LVM_OPTION_LV_NAME_INDEX:
			strncpy(lv_opts->lv_name, options->option[i].value.s, NAME_LEN);
			break;
		case LVM_OPTION_EXTENTS_INDEX:
			lv_opts->extents = options->option[i].value.ui32;
			break;
		case LVM_OPTION_LV_SIZE_INDEX:
			lv_opts->lv_size = options->option[i].value.ui32;
			break;
		case LVM_OPTION_STRIPES_INDEX:
			lv_opts->stripes = options->option[i].value.ui32;
			break;
		case LVM_OPTION_STRIPE_SIZE_INDEX:
			lv_opts->stripe_size = options->option[i].value.ui32;
			break;
		case LVM_OPTION_SNAPSHOT_INDEX:
			lv_opts->snapshot = options->option[i].value.bool;
			break;
		case LVM_OPTION_CHUNK_SIZE_INDEX:
			lv_opts->chunk_size = options->option[i].value.ui32;
			break;
		case LVM_OPTION_SNAPSHOT_ORG_INDEX:
			lvm_find_volume_by_name(options->option[i].value.s, group, &lv_opts->snapshot_org);
			break;
		case LVM_OPTION_CONTIGUOUS_INDEX:
			lv_opts->contiguous = (options->option[i].value.bool ? LV_CONTIGUOUS : 0);
			break;
		case LVM_OPTION_READ_ONLY_INDEX:
			lv_opts->read_only = options->option[i].value.bool;
			break;
		case LVM_OPTION_ZERO_INDEX:
			lv_opts->zero = options->option[i].value.bool;
			break;
		case LVM_OPTION_PV_NAMES_INDEX:
			for ( j = 0; j < options->option[i].value.list->count; j++ ) {
				if ( ! (lv_opts->pv_entries[j] = lvm_get_pv_for_name(options->option[i].value.list->value[j].s, group)) ) {
					LOG_ERROR("%s is not an object in container %s\n", options->option[i].value.list->value[j].s, group->container->name);
					RETURN(EINVAL);
				}
			}
			break;
		default:
			break;
		}
	}

	rc = lvm_create_region_verify_options(lv_opts, group);

	RETURN(rc);
}


/* Function: lvm_create_region_verify_options
 *
 *	Run through the options that have been collected, and verify that they
 *	are all valid for using in the specified group. Correct any bad options
 *	if possible, or return an error.
 */
int lvm_create_region_verify_options(	lvm_lv_create_options_t	* lv_opts,
					lvm_volume_group_t	* group )
{
	int rc = 0;

	LOG_ENTRY;

	// Check the name
	rc = lvm_check_lv_name(lv_opts->lv_name, group);
	if (rc) {
		LOG_ERROR("Error verifying region creation options\n");
		RETURN(rc);
	}

	// Make sure lv_size is a multiple of the PE size. If it is
	// currently zero, this won't change anything.
	lvm_check_lv_size(&lv_opts->lv_size, group->vg->pe_size);

	// Check/set extents and lv_size. This only checks that lv_size and
	// extents match. It does not yet check available space.
	rc = lvm_compare_lv_size_and_extents(&lv_opts->lv_size, &lv_opts->extents, group->vg->pe_size);
	if (rc) {
		LOG_ERROR("Error verifying region creation options\n");
		RETURN(rc);
	}

	// Check striping options
	if ( lv_opts->stripes > 1 ) {

		// Striped LVs cannot be contiguous.
		if ( lv_opts->contiguous ) {
			LOG_ERROR("Cannot perform contiguous allocation on a striped region\n");
			RETURN(EINVAL);
		}

		// Check for enough PVs for the specified number of stripes.
		if ( lv_opts->stripes > lvm_get_available_stripes(group) ) {
			LOG_ERROR("%d stripes more than %d available objects in container %s\n",
				lv_opts->stripes, group->pv_count, group->container->name);
			RETURN(EINVAL);
		}

		// Make sure extents is divisible by the number of stripes.
		// We want to use an equal number of PEs on each PV.
		rc = lv_opts->extents % lv_opts->stripes;
		if (rc) {
			lv_opts->extents += (lv_opts->stripes - rc);
			lv_opts->lv_size = lv_opts->extents * group->vg->pe_size;
			LOG_ERROR("Rounding size up to stripes boundary: %ld\n", lv_opts->lv_size);
		}

		// Check the stripe size
		rc = lvm_check_stripe_size(&lv_opts->stripe_size, group->vg->pe_size);
		if (rc) {
			LOG_ERROR("Error verifying region creation options\n");
			RETURN(rc);
		}
	}
	else {
		// No striping. Ignore stripe size.
		lv_opts->stripes = 1;
		lv_opts->stripe_size = 0;
	}

	// Check snapshot options
	if ( lv_opts->snapshot ) {

		// Snapshots cannot be striped
		if ( lv_opts->stripes > 1 ) {
			LOG_ERROR("Snapshot regions cannot be striped\n");
			RETURN(EINVAL);
		}

		// Make sure an original has been specified.
		if ( ! lv_opts->snapshot_org ) {
			LOG_ERROR("Must specify an original region for snapshotting\n");
			RETURN(EINVAL);
		}

		// Check the chunk size
		rc = lvm_check_chunk_size(&lv_opts->chunk_size, group->vg->pe_size);
		if (rc) {
			LOG_ERROR("Error verifying region creation options\n");
			RETURN(rc);
		}

		// Make sure that the snapshot original is not also a snapshot.
		if ( lv_opts->snapshot_org->lv->lv_access & LV_SNAPSHOT ) {
			LOG_ERROR("Snapshot original %s is already a snapshot\n", lv_opts->snapshot_org->region->name);
			RETURN(EINVAL);
		}

		// Make sure that the original is a compatibility volume.
		if ( ! lv_opts->snapshot_org->region->volume ||
		     lv_opts->snapshot_org->region->volume->object != lv_opts->snapshot_org->region ) {
			LOG_ERROR("Original %s is not a Compatibility Volume\n", lv_opts->snapshot_org->region->name);
			RETURN(EINVAL);
		}

		// Make sure read-only and zero are both on.
		lv_opts->read_only = TRUE;
		lv_opts->zero = TRUE;
	}
	else {
		// No snapshotting. Ignore chunk size.
		lv_opts->chunk_size = 0;
		lv_opts->snapshot_org = NULL;
	}

	// 16-bit extent numbers mean we can't have more than 65536 extents
	// in a volume.
	if ( lv_opts->extents > LVM_PE_T_MAX  ) {
		LOG_ERROR("Desired region size (%d extents) too large\n", lv_opts->extents);
		LOG_ERROR("Maximum of %d extents per region allowed\n", LVM_PE_T_MAX);
		RETURN(ENOSPC);
	}

	// Check for sufficient space in the group. This is only a rough check.
	// Space constraints on striping and contiguous will happen later.
	if ( lv_opts->extents > group->freespace->lv->lv_allocated_le ) {
		LOG_ERROR("Not enough freespace in container %s\n", group->container->name);
		LOG_ERROR("Specified size: %ld sectors\n", lv_opts->lv_size);
		LOG_ERROR("Available space: %ld sectors\n", group->freespace->lv->lv_size);
		RETURN(ENOSPC);
	}

	RETURN(0);
}


/* Function: lvm_create_region_allocate_option_descriptor
 *
 *	This is a helper function for lvm_init_options. It sets up the initial
 *	information in the option descriptor for a Create task.
 */
int lvm_create_region_allocate_option_descriptor( option_desc_array_t * od )
{
	LOG_ENTRY;

	od->count = LVM_OPTION_PV_NAMES_INDEX + 1;

	// Option 0 is LV Name
	SET_STRING(od->option[LVM_OPTION_LV_NAME_INDEX].name, LVM_OPTION_LV_NAME_STR);
	SET_STRING(od->option[LVM_OPTION_LV_NAME_INDEX].title, "Name for new LVM Region (LV)");
	od->option[LVM_OPTION_LV_NAME_INDEX].type	= EVMS_Type_String;
	od->option[LVM_OPTION_LV_NAME_INDEX].size	= EVMS_VOLUME_NAME_SIZE;
	od->option[LVM_OPTION_LV_NAME_INDEX].flags	= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[LVM_OPTION_LV_NAME_INDEX].value.s	= lvm_engine->engine_alloc(EVMS_VOLUME_NAME_SIZE);

	// Option 1 is Extents
	SET_STRING(od->option[LVM_OPTION_EXTENTS_INDEX].name, LVM_OPTION_EXTENTS_STR);
	SET_STRING(od->option[LVM_OPTION_EXTENTS_INDEX].title, "Number of logical extents");
	SET_STRING(od->option[LVM_OPTION_EXTENTS_INDEX].tip, "Extents are the unit of allocation of space for LVM regions. Specify number of Extents, and Size will be adjusted accordingly.");
	od->option[LVM_OPTION_EXTENTS_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_EXTENTS_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_EXTENTS_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
							  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
							  EVMS_OPTION_FLAGS_AUTOMATIC;

	// Option 2 is LV Size
	SET_STRING(od->option[LVM_OPTION_LV_SIZE_INDEX].name, LVM_OPTION_LV_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_LV_SIZE_INDEX].title, "Size of new region");
	SET_STRING(od->option[LVM_OPTION_LV_SIZE_INDEX].tip, "Specify Size, and the number of Extents will be adjusted accordingly.");
	od->option[LVM_OPTION_LV_SIZE_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_LV_SIZE_INDEX].unit	= EVMS_Unit_Sectors;
	od->option[LVM_OPTION_LV_SIZE_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_LV_SIZE_INDEX].flags	= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	// Option 3 is Stripes
	SET_STRING(od->option[LVM_OPTION_STRIPES_INDEX].name, LVM_OPTION_STRIPES_STR);
	SET_STRING(od->option[LVM_OPTION_STRIPES_INDEX].title, "Number of stripes");
	SET_STRING(od->option[LVM_OPTION_STRIPES_INDEX].tip, "Number of objects (PVs) to use in a striped volume. Leave at 1 for a linear volume.");
	od->option[LVM_OPTION_STRIPES_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_STRIPES_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_STRIPES_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
							  EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[LVM_OPTION_STRIPES_INDEX].value.ui32	= 1;

	// Option 4 is Stripe Size
	SET_STRING(od->option[LVM_OPTION_STRIPE_SIZE_INDEX].name, LVM_OPTION_STRIPE_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_STRIPE_SIZE_INDEX].title, "Stripe Size");
	SET_STRING(od->option[LVM_OPTION_STRIPE_SIZE_INDEX].tip, "Granularity at which data is striped across the underlying objects.");
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].unit	= EVMS_Unit_Sectors;
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
							  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
							  EVMS_OPTION_FLAGS_AUTOMATIC |
							  EVMS_OPTION_FLAGS_INACTIVE;

	// Option 5 is Snapshotting
	SET_STRING(od->option[LVM_OPTION_SNAPSHOT_INDEX].name, LVM_OPTION_SNAPSHOT_STR);
	SET_STRING(od->option[LVM_OPTION_SNAPSHOT_INDEX].title, "Snapshot Region");
	SET_STRING(od->option[LVM_OPTION_SNAPSHOT_INDEX].tip, "Must also specify an original region to be snapshotted. Snapshot regions cannot be striped (Number of Stripes must equal 1).");
	od->option[LVM_OPTION_SNAPSHOT_INDEX].type	= EVMS_Type_Boolean;
	od->option[LVM_OPTION_SNAPSHOT_INDEX].size	= sizeof(int);
	od->option[LVM_OPTION_SNAPSHOT_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
							  EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[LVM_OPTION_SNAPSHOT_INDEX].value.bool= FALSE;

	// Option 6 is Chunk Size
	SET_STRING(od->option[LVM_OPTION_CHUNK_SIZE_INDEX].name, LVM_OPTION_CHUNK_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_CHUNK_SIZE_INDEX].title, "Snapshot Chunk Size");
	SET_STRING(od->option[LVM_OPTION_CHUNK_SIZE_INDEX].tip, "Granularity at which data is copied from the original to the snapshot.");
	od->option[LVM_OPTION_CHUNK_SIZE_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_CHUNK_SIZE_INDEX].unit	= EVMS_Unit_Sectors;
	od->option[LVM_OPTION_CHUNK_SIZE_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_CHUNK_SIZE_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
							  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
							  EVMS_OPTION_FLAGS_AUTOMATIC |
							  EVMS_OPTION_FLAGS_INACTIVE;

	// Option 7 is Snapshot Original
	SET_STRING(od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].name, LVM_OPTION_SNAPSHOT_ORG_STR);
	SET_STRING(od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].title, "Original region");
	SET_STRING(od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].tip, "The region to be snapshotted.");
	od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].type		= EVMS_Type_String;
	od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].size		= EVMS_VOLUME_NAME_SIZE;
	od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].flags		= EVMS_OPTION_FLAGS_NOT_REQUIRED |
								  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
								  EVMS_OPTION_FLAGS_INACTIVE;
	od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].value.s	= lvm_engine->engine_alloc(EVMS_VOLUME_NAME_SIZE);

	// Option 8 is Contiguous
	SET_STRING(od->option[LVM_OPTION_CONTIGUOUS_INDEX].name, LVM_OPTION_CONTIGUOUS_STR);
	SET_STRING(od->option[LVM_OPTION_CONTIGUOUS_INDEX].title, "Contiguous Allocation");
	SET_STRING(od->option[LVM_OPTION_CONTIGUOUS_INDEX].tip, "Create this region contiguously on the underlying objects");
	od->option[LVM_OPTION_CONTIGUOUS_INDEX].type		= EVMS_Type_Boolean;
	od->option[LVM_OPTION_CONTIGUOUS_INDEX].size		= sizeof(int);
	od->option[LVM_OPTION_CONTIGUOUS_INDEX].flags		= EVMS_OPTION_FLAGS_NOT_REQUIRED |
								  EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[LVM_OPTION_CONTIGUOUS_INDEX].value.bool	= FALSE;

	// Option 9 is Read-Only
	SET_STRING(od->option[LVM_OPTION_READ_ONLY_INDEX].name, LVM_OPTION_READ_ONLY_STR);
	SET_STRING(od->option[LVM_OPTION_READ_ONLY_INDEX].title, "Read-only region");
	od->option[LVM_OPTION_READ_ONLY_INDEX].type		= EVMS_Type_Boolean;
	od->option[LVM_OPTION_READ_ONLY_INDEX].size		= sizeof(int);
	od->option[LVM_OPTION_READ_ONLY_INDEX].flags		= EVMS_OPTION_FLAGS_NOT_REQUIRED |
								  EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[LVM_OPTION_READ_ONLY_INDEX].value.bool	= FALSE;

	// Option 10 is Zero
	SET_STRING(od->option[LVM_OPTION_ZERO_INDEX].name, LVM_OPTION_ZERO_STR);
	SET_STRING(od->option[LVM_OPTION_ZERO_INDEX].title, "Zero first 1k of region");
	SET_STRING(od->option[LVM_OPTION_ZERO_INDEX].tip, "Clear the first 1 kB of data at the start of this region.");
	od->option[LVM_OPTION_ZERO_INDEX].type		= EVMS_Type_Boolean;
	od->option[LVM_OPTION_ZERO_INDEX].size		= sizeof(int);
	od->option[LVM_OPTION_ZERO_INDEX].flags		= EVMS_OPTION_FLAGS_NOT_REQUIRED |
							  EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[LVM_OPTION_ZERO_INDEX].value.bool	= TRUE;

	// Option 11 is PV Names
	SET_STRING(od->option[LVM_OPTION_PV_NAMES_INDEX].name, LVM_OPTION_PV_NAMES_STR);
	SET_STRING(od->option[LVM_OPTION_PV_NAMES_INDEX].title, "Objects (PVs) to place the region on");
	SET_STRING(od->option[LVM_OPTION_PV_NAMES_INDEX].tip, "Region will be allocated on only these objects. Leave blank for automatic allocation.");
	od->option[LVM_OPTION_PV_NAMES_INDEX].type		= EVMS_Type_String;
	od->option[LVM_OPTION_PV_NAMES_INDEX].size		= EVMS_VOLUME_NAME_SIZE;
	od->option[LVM_OPTION_PV_NAMES_INDEX].flags		= EVMS_OPTION_FLAGS_NOT_REQUIRED |
								  EVMS_OPTION_FLAGS_AUTOMATIC |
								  EVMS_OPTION_FLAGS_VALUE_IS_LIST;
	od->option[LVM_OPTION_PV_NAMES_INDEX].value.list	= lvm_engine->engine_alloc(sizeof(value_list_t) + MAX_PV*sizeof(value_t));
	od->option[LVM_OPTION_PV_NAMES_INDEX].value.list->count = 0;
	// Don't allocate the memory for the list strings yet.
	// Wait until set_option to do that.

	RETURN(0);
}


/* Function: lvm_create_region_get_acceptable
 *
 *	This is a helper function for lvm_get_acceptable_objects. It determines
 *	which objects in EVMS are appropriate for a Create task.
 */
int lvm_create_region_get_acceptable( dlist_t acceptable_regions )
{
	lvm_volume_group_t	* group;
	int			rc;

	LOG_ENTRY;

	// Add all freespace regions to the acceptable regions list.
	for ( rc = GoToStartOfList(lvm_group_list);
	      !rc && (group = lvm_get_list_item(lvm_group_list));
	      rc = NextItem(lvm_group_list) ) {

		if ( group->freespace->region->size != 0 ) {
			lvm_add_object_to_list(group->freespace->region, acceptable_regions);
		}
	}

	RETURN(0);
}


/* Function: lvm_create_region_set_objects
 *
 *	Examine the selected_objects list in the task, and verify that each
 *	object is valid for using to create a new region. If any are not valid,
 *	remove them from the selected_objects and add them to the
 *	declined_objects. 
 */
int lvm_create_region_set_objects(	task_context_t	* context,
					dlist_t		declined_objects,
					task_effect_t	* effect )
{
	lvm_logical_volume_t	* freespace;
	lvm_volume_group_t	* group;
	option_desc_array_t	* od = context->option_descriptors;
	u_int32_t		pe_size;
	u_int32_t		extents;
	u_int32_t		stripes;
	int			rc = 0;
	int			i, j;

	LOG_ENTRY;

	// Only one freespace region can be selected, so just grab the
	// first item on the list.
	if ( (rc = lvm_get_freespace_volume(context->selected_objects, &freespace))) {
		RETURN(rc);
	}

	group = freespace->group;
	pe_size = group->vg->pe_size;
	extents = (freespace->lv->lv_allocated_le < LVM_PE_T_MAX) ? freespace->lv->lv_allocated_le : LVM_PE_T_MAX;
	stripes = lvm_get_available_stripes(group);

	LOG_EXTRA("Setting object %s\n", freespace->region->name);

	// Now that we have a selected freespace, we can update the option
	// descriptor appropriately.

	// Update range of extents and set initial value to maximum.
	od->option[LVM_OPTION_EXTENTS_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_EXTENTS_INDEX].constraint.range, 1, extents, 1);
	od->option[LVM_OPTION_EXTENTS_INDEX].value.ui32 = extents;
	od->option[LVM_OPTION_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	// Update range of LV size and set initial value to maximum.
	od->option[LVM_OPTION_LV_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_LV_SIZE_INDEX].constraint.range, pe_size, pe_size * extents, pe_size);
	od->option[LVM_OPTION_LV_SIZE_INDEX].value.ui32 = extents * pe_size;
	od->option[LVM_OPTION_LV_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	// Update number of available stripes. Initial value is already set.
	od->option[LVM_OPTION_STRIPES_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_STRIPES_INDEX].constraint.range, 1, stripes, 1);

	// Update range of stripe sizes and initial value.
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].constraint_type = EVMS_Collection_List;
	SET_POWER2_LIST(od->option[LVM_OPTION_STRIPE_SIZE_INDEX].constraint.list, LVM_MIN_STRIPE_SIZE, ((LVM_MAX_STRIPE_SIZE < pe_size) ? LVM_MAX_STRIPE_SIZE : pe_size));
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].value.ui32 = (LVM_DEFAULT_STRIPE_SIZE < pe_size) ? LVM_DEFAULT_STRIPE_SIZE : pe_size;
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	// Update range of snapshot chunk sizes and initial value.
	od->option[LVM_OPTION_CHUNK_SIZE_INDEX].constraint_type = EVMS_Collection_List;
	SET_POWER2_LIST(od->option[LVM_OPTION_CHUNK_SIZE_INDEX].constraint.list, LVM_SNAPSHOT_MIN_CHUNK, ((LVM_SNAPSHOT_MAX_CHUNK < pe_size) ? LVM_SNAPSHOT_MAX_CHUNK : pe_size));
	od->option[LVM_OPTION_CHUNK_SIZE_INDEX].value.ui32 = (LVM_SNAPSHOT_DEF_CHUNK < pe_size) ? LVM_SNAPSHOT_DEF_CHUNK : pe_size;
	od->option[LVM_OPTION_CHUNK_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	// Setup list of possible snapshot originals.
	od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].constraint_type = EVMS_Collection_List;
	od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].constraint.list = lvm_engine->engine_alloc(sizeof(value_list_t) + group->volume_count* sizeof(value_t));
	for ( i = 1, j = 0; i < MAX_LV; i++ ) {
		if ( group->volume_list[i] &&
		     group->volume_list[i]->region->volume &&
		     group->volume_list[i]->region->volume->object == group->volume_list[i]->region &&
		     ! (group->volume_list[i]->lv->lv_access & LV_SNAPSHOT) ) {
			SET_STRING(od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].constraint.list->value[j].s, group->volume_list[i]->region->name);
			j++;
		}
	}
	od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].constraint.list->count = j;

	// Setup list of possible PVs.
	od->option[LVM_OPTION_PV_NAMES_INDEX].constraint_type = EVMS_Collection_List;
	od->option[LVM_OPTION_PV_NAMES_INDEX].constraint.list = lvm_engine->engine_alloc(sizeof(value_list_t) + group->pv_count*sizeof(value_t));
	for ( i = 1, j = 0; i < MAX_PV; i++ ) {
		if ( group->pv_list[i] &&
		     group->pv_list[i]->pv->pe_total != group->pv_list[i]->pv->pe_allocated ) {
			SET_STRING(od->option[LVM_OPTION_PV_NAMES_INDEX].constraint.list->value[j].s, group->pv_list[i]->segment->name);
			j++;
		}
	}
	od->option[LVM_OPTION_PV_NAMES_INDEX].constraint.list->count = j;

	*effect = EVMS_Effect_Reload_Options;

	RETURN(0);
}


/* Function: lvm_create_region_set_option
 *
 *	This is a helper function for lvm_set_option. It sets one particular
 *	option in the option descriptor for a Create task, and updates other
 *	option descriptor information accordingly.
 */
int lvm_create_region_set_option(task_context_t	* context,
				u_int32_t	index,
				value_t		* value,
				task_effect_t	* effect )
{
	option_desc_array_t	* od = context->option_descriptors;
	lvm_logical_volume_t	* freespace;
	lvm_logical_volume_t	* snap_org;
	lvm_volume_group_t	* group;
	u_int32_t		extents = 0;
	u_int32_t		lv_size = 0;
	u_int32_t		stripes = 1;
	int			rc = 0;
	int			i;

	LOG_ENTRY;
	*effect = 0;

	// Only one freespace region should be selected, so just grab the
	// first item on the list. Anything else on the list will be ignored.
	if ( (rc = lvm_get_freespace_volume(context->selected_objects, &freespace))) {
		RETURN(rc);
	}
	group = freespace->group;

	LOG_EXTRA("Setting option %d\n", index);

	switch (index) {

	case LVM_OPTION_LV_NAME_INDEX:
		// Make sure this name isn't in use already.
		if ( (rc = lvm_check_lv_name(value->s, group)) ) {
			LOG_ERROR("Invalid name: %s\n", value->s);
		}
		else {
			strncpy(od->option[index].value.s, value->s, NAME_LEN);
			od->option[index].flags	&= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		}
		break;

	case LVM_OPTION_EXTENTS_INDEX:
		// Make sure there are enough extents.
		extents = (group->freespace->lv->lv_allocated_le < LVM_PE_T_MAX)
				? group->freespace->lv->lv_allocated_le : LVM_PE_T_MAX;
		if ( value->ui32 > extents ) {
			LOG_ERROR("%ld extents chosen. Only %ld available.\n", value->ui32, extents);
			value->ui32 = extents;
			*effect |= EVMS_Effect_Inexact;
		}
		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		// Recalculate size based on extents.
		lv_size = value->ui32 * group->vg->pe_size;
		od->option[LVM_OPTION_LV_SIZE_INDEX].value.ui32 = lv_size;
		od->option[LVM_OPTION_LV_SIZE_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_LV_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_LV_SIZE_INDEX:
		// Make sure lv_size is a multiple of the PE size
		if ( lvm_check_lv_size(&value->ui32, group->vg->pe_size) ) {
			*effect |= EVMS_Effect_Inexact;
		}

		// Make sure there is enough space.
		extents = (group->freespace->lv->lv_allocated_le < LVM_PE_T_MAX)
				? group->freespace->lv->lv_allocated_le : LVM_PE_T_MAX;
		lv_size = extents * group->vg->pe_size;
		if ( value->ui32 > lv_size ) {
			LOG_ERROR("%ld sectors chosen for size. Only %ld available.\n", value->ui32, lv_size);
			value->ui32 = lv_size;
			*effect |= EVMS_Effect_Inexact;
		}
		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		// Recalculate extents based on lv_size
		extents = value->ui32 / group->vg->pe_size;
		od->option[LVM_OPTION_EXTENTS_INDEX].value.ui32 = extents;
		od->option[LVM_OPTION_EXTENTS_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_STRIPES_INDEX:
		// Make sure there are enough PVs available.
		stripes = lvm_get_available_stripes(group);
		if ( value->ui32 > stripes ) {
			LOG_ERROR("%ld stripes chosen. Only %ld available.\n", value->ui32, stripes);
			value->ui32 = stripes;
			*effect |= EVMS_Effect_Inexact;
		}
		od->option[index].value.ui32 = value->ui32;

		// Turn on/off related options
		if ( value->ui32 > 1 ) {
			od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_SNAPSHOT_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_SNAPSHOT_INDEX].value.bool = FALSE;
			od->option[LVM_OPTION_CONTIGUOUS_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_CONTIGUOUS_INDEX].value.bool = FALSE;
		}
		else {
			od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_SNAPSHOT_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_CONTIGUOUS_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
		}

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_STRIPE_SIZE_INDEX:
		// Make sure stripe_size is within range.
		lvm_check_stripe_size(&(value->ui32), group->vg->pe_size);
		od->option[index].value.ui32 = value->ui32;
		break;

	case LVM_OPTION_SNAPSHOT_INDEX:
		od->option[index].value.bool = value->bool;
		// Activate/adjust associated options
		if ( value->bool ) {
			od->option[LVM_OPTION_CHUNK_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].flags &= (~EVMS_OPTION_FLAGS_INACTIVE &
									    ~EVMS_OPTION_FLAGS_NOT_REQUIRED);

			// Snapshot volumes can't be striped, and must be read-only.
			od->option[LVM_OPTION_STRIPES_INDEX].value.ui32 = 1;
			od->option[LVM_OPTION_STRIPES_INDEX].flags	|= EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags	|= EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_READ_ONLY_INDEX].value.bool = TRUE;
		}
		else {
			od->option[LVM_OPTION_CHUNK_SIZE_INDEX].flags	|= EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_SNAPSHOT_ORG_INDEX].flags	|= (EVMS_OPTION_FLAGS_INACTIVE |
									    EVMS_OPTION_FLAGS_NOT_REQUIRED);
			od->option[LVM_OPTION_STRIPES_INDEX].flags	&= ~EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_READ_ONLY_INDEX].value.bool = FALSE;
		}

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_CHUNK_SIZE_INDEX:
		// Make sure chunk_size is within range.
		lvm_check_chunk_size(&(value->ui32), group->vg->pe_size);
		od->option[index].value.ui32 = value->ui32;
		break;

	case LVM_OPTION_SNAPSHOT_ORG_INDEX:
		if ( (rc = lvm_find_volume_by_name(value->s, group, &snap_org)) ) {
			LOG_ERROR("%s is not a region in container %s\n", value->s, group->container->name);
		}
		else if ( ! snap_org->region->volume ||
		          snap_org->region->volume->object != snap_org->region ) {
			LOG_ERROR("%s is not a compatibility volume. Cannot be snapshotted.\n", value->s);
			rc = EINVAL;
		}
		else if ( snap_org->lv->lv_access & LV_SNAPSHOT ) {
			LOG_ERROR("%s is a snapshot region. Cannot be snapshotted.\n", value->s);
			rc = EINVAL;
		}
		else {
			strncpy(od->option[index].value.s, value->s, EVMS_VOLUME_NAME_SIZE);
		}
		break;

	case LVM_OPTION_CONTIGUOUS_INDEX:
		od->option[index].value.bool = value->bool;
		if ( value->bool ) {
			// Can't stripe contiguous volumes.
			od->option[LVM_OPTION_STRIPES_INDEX].value.ui32 = 1;
			od->option[LVM_OPTION_STRIPES_INDEX].flags	|= EVMS_OPTION_FLAGS_INACTIVE;
			od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags	|= EVMS_OPTION_FLAGS_INACTIVE;
		}
		else {
			od->option[LVM_OPTION_STRIPES_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
		}

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_READ_ONLY_INDEX:
		od->option[index].value.bool = value->bool;
		break;

	case LVM_OPTION_ZERO_INDEX:
		od->option[index].value.bool = value->bool;
		break;

	case LVM_OPTION_PV_NAMES_INDEX:
		for ( i = 0; i < value->list->count; i++ ) {
			if ( od->option[index].value.list->value[i].s ) {
				lvm_engine->engine_free(od->option[index].value.list->value[i].s);
				od->option[index].value.list->value[i].s = NULL;
			}
			SET_STRING(od->option[index].value.list->value[i].s, value->list->value[i].s);
		}
		for ( ; i < od->option[index].value.list->count; i++ ) {
			if ( od->option[index].value.list->value[i].s ) {
				lvm_engine->engine_free(od->option[index].value.list->value[i].s);
				od->option[index].value.list->value[i].s = NULL;
			}
		}
		od->option[index].value.list->count = value->list->count;
		break;

	default:
		rc = EINVAL;
		break;
	}

	RETURN(rc);
}



/****** Logical Volume Expand Option Handlers ******/


/* Function: lvm_expand_region_parse_option_array
 *
 *	Parse the option_array_t to find options pertaining to logical volume
 *	expansion. Each option can either be keyed by value or string, so we
 *	have to check for both. Acceptable options for volume creation are:
 *	LVM_OPTION_EXPAND_SIZE
 *	LVM_OPTION_EXPAND_EXTENTS
 *	LVM_OPTION_EXPAND_PV_NAMES
 *	Incorrect options are simply ignored.
 */
int lvm_expand_region_parse_option_array(option_array_t		* options,
					lvm_volume_group_t	* group,
					lvm_logical_volume_t	* volume,
					lvm_lv_expand_options_t	* lv_opts )
{
	int i, j, rc = 0;

	LOG_ENTRY;

	// Initialize the options
	memset(lv_opts, 0, sizeof(lvm_lv_expand_options_t));

	for ( i = 0; !rc && i < options->count; i++ ) {

		// If only the name-based index is specified, get the number.
		if ( ! options->option[i].is_number_based ) {
			if ( ! strcmp(options->option[i].name, LVM_OPTION_EXPAND_EXTENTS_STR) ) {
				options->option[i].number = LVM_OPTION_EXPAND_EXTENTS_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_EXPAND_SIZE_STR) ) {
				options->option[i].number = LVM_OPTION_EXPAND_SIZE_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_EXPAND_PV_NAMES_STR) ) {
				options->option[i].number = LVM_OPTION_EXPAND_PV_NAMES_INDEX;
			}
			else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		// Use the number-based index to record the option.
		switch ( options->option[i].number ) {
		case LVM_OPTION_EXPAND_EXTENTS_INDEX:
			lv_opts->add_extents = options->option[i].value.ui32;
			break;
		case LVM_OPTION_EXPAND_SIZE_INDEX:
			lv_opts->add_size = options->option[i].value.ui32;
			break;
		case LVM_OPTION_EXPAND_PV_NAMES_INDEX:
			for ( j = 0; j < options->option[i].value.list->count; j++ ) {
				if ( ! (lv_opts->pv_entries[j] = lvm_get_pv_for_name(options->option[i].value.list->value[j].s, group)) ) {
					LOG_ERROR("%s is not an object in container %s\n", options->option[i].value.list->value[j].s, group->container->name);
					rc = EINVAL;
				}
			}
			break;
		default:
			break;
		}
	}

	if (!rc) {
		rc = lvm_expand_region_verify_options(lv_opts, group, volume);
	}

	RETURN(rc);
}


/* Function: lvm_expand_region_verify_options
 *
 *	Run through the options that have been collected, and verify that they
 *	are all valid for using in the specified group. Correct any bad options
 *	if possible, or return an error.
 */
int lvm_expand_region_verify_options(	lvm_lv_expand_options_t	* lv_opts,
					lvm_volume_group_t	* group,
					lvm_logical_volume_t	* volume )
{
	int rc = 0;

	LOG_ENTRY;

	// Make sure add_size is a multiple of the PE size. If it is
	// currently zero, this won't change anything.
	lvm_check_lv_size(&lv_opts->add_size, group->vg->pe_size);

	// Check/set extents and size. This only checks that size and
	// extents match. It does not yet check available space.
	rc = lvm_compare_lv_size_and_extents(&lv_opts->add_size, &lv_opts->add_extents, group->vg->pe_size);
	if (rc) {
		LOG_ERROR("Error verifying region expansion options\n");
		RETURN(rc);
	}

	// Make sure extents is divisible by the number of stripes. We need to
	// use an equal number of PEs on each PV. On a non-striped volume this
	// test will always fail.
	rc = lv_opts->add_extents % volume->lv->lv_stripes;
	if (rc) {
		lv_opts->add_extents += (volume->lv->lv_stripes - rc);
		lv_opts->add_size = lv_opts->add_extents * group->vg->pe_size;
		LOG_WARNING("Rounding size up to stripes boundary: %ld\n", lv_opts->add_size);
	}

	// 16-bit extent numbers mean we can't have more than
	// 65536 extents in a volume.
	if ( lv_opts->add_extents + volume->lv->lv_allocated_le > LVM_PE_T_MAX  ) {
		LOG_ERROR("Desired final region size (%d extents) too large\n", lv_opts->add_extents + volume->lv->lv_allocated_le);
		LOG_ERROR("Maximum of %d extents per region allowed\n", LVM_PE_T_MAX);
		RETURN(ENOSPC);
	}

	// Check for sufficient space in the group. This is only a
	// rough check. Space constraints on striping and contiguous
	// will happen later.
	if ( lv_opts->add_extents > group->freespace->lv->lv_allocated_le ) {
		LOG_ERROR("Not enough freespace in container %s\n", group->container->name);
		LOG_ERROR("Specified additional size of: %ld sectors\n", lv_opts->add_size);
		LOG_ERROR("Available space: %ld sectors\n", group->freespace->lv->lv_size);
		RETURN(ENOSPC);
	}

	RETURN(0);
}


/* Function: lvm_expand_region_allocate_options
 *
 *	This is a helper function for lvm_init_options. It sets up the initial
 *	information in the option descriptor for an Expand task.
 */
int lvm_expand_region_allocate_option_descriptor( option_desc_array_t * od )
{
	LOG_ENTRY;

	od->count = LVM_OPTION_EXPAND_PV_NAMES_INDEX + 1;

	// Option 0 is expansion extents
	SET_STRING(od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].name, LVM_OPTION_EXPAND_EXTENTS_STR);
	SET_STRING(od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].title, "Additional Extents");
	SET_STRING(od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].tip, "Number of extents to add to the selected LVM region. Only specify extents or size!");
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
								  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
								  EVMS_OPTION_FLAGS_AUTOMATIC;

	// Option 1 is expansion size
	SET_STRING(od->option[LVM_OPTION_EXPAND_SIZE_INDEX].name, LVM_OPTION_EXPAND_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_EXPAND_SIZE_INDEX].title, "Additional Size");
	SET_STRING(od->option[LVM_OPTION_EXPAND_SIZE_INDEX].tip, "Amount of space to add to the selected LVM region");
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].unit	= EVMS_Unit_Sectors;
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].flags	= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	// Option 2 is PV Names
	SET_STRING(od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].name, LVM_OPTION_EXPAND_PV_NAMES_STR);
	SET_STRING(od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].title, "Objects (PVs) to expand the region onto");
	SET_STRING(od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].tip, "Region will be expanded only onto these objects. Leave blank for automatic allocation.");
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].type		= EVMS_Type_String;
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].size		= EVMS_VOLUME_NAME_SIZE;
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].flags		= EVMS_OPTION_FLAGS_NOT_REQUIRED |
									  EVMS_OPTION_FLAGS_AUTOMATIC |
									  EVMS_OPTION_FLAGS_VALUE_IS_LIST;
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].value.list		= lvm_engine->engine_alloc(sizeof(value_list_t) + MAX_PV*sizeof(value_t));
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].value.list->count	= 0;
	// Don't allocate the memory for the list strings yet.
	// Wait until set_option to do that.

	RETURN(0);
}


/* Function: lvm_expand_region_get_acceptable
 *
 *	This is a helper function for lvm_get_acceptable_objects. It determines
 *	which objects in EVMS are appropriate for an Expand task.
 */
int lvm_expand_region_get_acceptable(	storage_object_t	* region,
					dlist_t			freespace_regions )
{
	lvm_logical_volume_t	* volume = region->private_data;
	lvm_logical_volume_t	* freespace = volume->group->freespace;
	int			rc = 0;

	LOG_ENTRY;

	// Find the freespace for this region's group. If it has any
	// available space, add it to the list.
	if ( freespace->region->size > 0 ) {
		rc = lvm_add_object_to_list(freespace->region, freespace_regions);
	}

	RETURN(rc);
}


/* Function: lvm_expand_region_set_objects
 *
 *	Examine the selected_objects list in the task, and verify that each
 *	object is valid for using to expand a region. If any are not valid,
 *	remove them from the selected_objects and add them to the
 *	declined_objects. 
 */
int lvm_expand_region_set_objects(	task_context_t	* context,
					dlist_t		declined_objects,
					task_effect_t	* effect )
{
	lvm_logical_volume_t	* volume = context->object->private_data;
	lvm_logical_volume_t	* freespace;
	lvm_volume_group_t	* group = volume->group;
	option_desc_array_t	* od = context->option_descriptors;
	u_int32_t		pe_size = group->vg->pe_size;
	u_int32_t		add_extents;
	sector_count_t		add_sectors;
	int			rc = 0;
	int			i, j;

	LOG_ENTRY;

	// Only one freespace region can be selected, so just grab the
	// first item on the list.
	if ( (rc = lvm_get_freespace_volume(context->selected_objects, &freespace))) {
		RETURN(rc);
	}

	if ( freespace->group != group ) {
		LOG_ERROR("Target object and selected object are not in same group.\n");
		LOG_ERROR("Target of expand is %s.\n", volume->region->name);
		LOG_ERROR("Selected object is %s.\n", freespace->region->name);
		RETURN(EINVAL);
	}

	// Calculate maximum number of extents that can be added.
	add_extents = freespace->lv->lv_allocated_le;
	if ( add_extents + volume->lv->lv_allocated_le > LVM_PE_T_MAX ) {
		add_extents = LVM_PE_T_MAX - volume->lv->lv_allocated_le;
	}

	// Check the additional space with the engine.
	add_sectors = add_extents * pe_size;
	rc = lvm_engine->can_expand_by(context->object, &add_sectors);
	if ( rc == EAGAIN ) {
		if ( add_sectors < pe_size ) {
			LOG_ERROR("Unable to expand region %s.\n", volume->region->name);
			LOG_ERROR("The Engine will only allow expanding by %lld sectors,\n", add_sectors);
			LOG_ERROR("but LVM must expand the region by at least %d sectors.\n", pe_size);
			RETURN(ENOSPC);
		}
		else if ( add_sectors < add_extents * pe_size ) {
			add_extents = add_sectors / pe_size;
		}
	}
	else if (rc) {
		LOG_ERROR("A parent object or fsim has disallowed the expand of region %s\n", context->object->name);
		RETURN(rc);
	}

	LOG_EXTRA("Setting selected object %s\n", freespace->region->name);

	// Now that we have a selected freespace, we can update the option
	// descriptor appropriately.

	// Update range of extents and set initial value.
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].constraint.range, 1, add_extents, 1);
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].value.ui32 = add_extents;
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	// Update range of LV size and set initial value.
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_EXPAND_SIZE_INDEX].constraint.range, pe_size, pe_size * add_extents, pe_size);
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].value.ui32 = add_extents * pe_size;
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	// Setup list of possible PVs.
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].constraint_type = EVMS_Collection_List;
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].constraint.list = lvm_engine->engine_alloc(sizeof(value_list_t) + group->pv_count*sizeof(value_t));
	for ( i = 1, j = 0; i < MAX_PV; i++ ) {
		if ( group->pv_list[i] &&
		     group->pv_list[i]->pv->pe_total != group->pv_list[i]->pv->pe_allocated ) {
			SET_STRING(od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].constraint.list->value[j].s, group->pv_list[i]->segment->name);
			j++;
		}
	}
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].constraint.list->count = j;

	*effect = EVMS_Effect_Reload_Options;

	RETURN(0);
}


/* Function: lvm_expand_region_set_option
 *
 *	This is a helper function for lvm_set_option. It sets one particular
 *	option in the option descriptor for an Expand task, and updates other
 *	option descriptor information accordingly.
 */
int lvm_expand_region_set_option(task_context_t	* context,
				u_int32_t	index,
				value_t		* value,
				task_effect_t	* effect )
{
	option_desc_array_t	* od = context->option_descriptors;
	lvm_logical_volume_t	* volume = context->object->private_data;
	lvm_logical_volume_t	* freespace;
	lvm_volume_group_t	* group = volume->group;
	u_int32_t		pe_size = group->vg->pe_size;
	u_int32_t		add_extents = 0;
	u_int32_t		add_size = 0;
	sector_count_t		add_sectors;
	int			rc = 0;
	int			i;

	LOG_ENTRY;
	*effect = 0;

	// Only one freespace region should be selected, so just grab the
	// first item on the list.
	if ( (rc = lvm_get_freespace_volume(context->selected_objects, &freespace))) {
		RETURN(rc);
	}

	// Make sure the selected freespace belongs
	// to the selected object's group.
	if ( freespace->group != group ) {
		LOG_ERROR("Target region and selected freespace are not in same container.\n");
		LOG_ERROR("Target of expand is %s.\n", volume->region->name);
		LOG_ERROR("Selected freespace is %s.\n", freespace->region->name);
		RETURN(EINVAL);
	}

	// Calculate maximum number of extents that can be added.
	add_extents = freespace->lv->lv_allocated_le;
	if ( add_extents + volume->lv->lv_allocated_le > LVM_PE_T_MAX ) {
		add_extents = LVM_PE_T_MAX - volume->lv->lv_allocated_le;
	}

	LOG_EXTRA("Setting option %d\n", index);

	switch (index) {

	case LVM_OPTION_EXPAND_EXTENTS_INDEX:
		// Make sure there are enough extents.
		if ( value->ui32 > add_extents ) {
			LOG_ERROR("%ld extents chosen. Only %ld available.\n", value->ui32, add_extents);
			value->ui32 = add_extents;
			*effect |= EVMS_Effect_Inexact;
		}

		// Check the additional space with the engine.
		add_sectors = value->ui32 * pe_size;
		rc = lvm_engine->can_expand_by(context->object, &add_sectors);
		if ( rc == EAGAIN ) {
			if ( add_sectors < pe_size ) {
				LOG_ERROR("Unable to expand region %s.\n", context->object->name);
				LOG_ERROR("The Engine will only allow expanding by %lld sectors,\n", add_sectors);
				LOG_ERROR("but LVM must expand the region by at least %d sectors.\n", pe_size);
				RETURN(ENOSPC);
			}
			else if ( add_sectors < value->ui32 * pe_size ) {
				value->ui32 = add_sectors / pe_size;
				*effect |= EVMS_Effect_Inexact;
				LOG_ERROR("A parent object or fsim has restricted the expand size for region %s.\n", context->object->name);
				LOG_ERROR("Rounding down to %ld extents.\n", value->ui32);
			}
			rc = 0;
		}
		else if (rc) {
			LOG_ERROR("A parent object or fsim has disallowed the expand of region %s\n", context->object->name);
			RETURN(rc);
		}

		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		// Recalculate size based on extents.
		add_size = value->ui32 * pe_size;
		od->option[LVM_OPTION_EXPAND_SIZE_INDEX].value.ui32 = add_size;
		od->option[LVM_OPTION_EXPAND_SIZE_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_EXPAND_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_EXPAND_SIZE_INDEX:
		// Make sure lv_size is a multiple of the PE size
		if ( lvm_check_lv_size(&value->ui32, pe_size) ) {
			*effect |= EVMS_Effect_Inexact;
		}

		// Make sure there is enough space.
		add_size = add_extents * pe_size;
		if ( value->ui32 > add_size ) {
			LOG_ERROR("%ld sectors chosen for size. Only %ld available.\n", value->ui32, add_size);
			value->ui32 = add_size;
			*effect |= EVMS_Effect_Inexact;
		}

		// Check the additional space with the engine.
		add_sectors = value->ui32;
		rc = lvm_engine->can_expand_by(context->object, &add_sectors);
		if ( rc == EAGAIN ) {
			if ( add_sectors < pe_size ) {
				LOG_ERROR("Unable to expand region %s.\n", context->object->name);
				LOG_ERROR("The Engine will only allow expanding by %lld sectors,\n", add_sectors);
				LOG_ERROR("but LVM must expand the region by at least %d sectors.\n", pe_size);
				RETURN(ENOSPC);
			}
			else if ( add_sectors < value->ui32 ) {
				value->ui32 = add_sectors;
				lvm_check_lv_size(&value->ui32, pe_size);
				*effect |= EVMS_Effect_Inexact;
				LOG_ERROR("A parent object or fsim has restricted the expand size for region %s.\n", context->object->name);
				LOG_ERROR("Rounding down to %ld sectors.\n", value->ui32);
			}
			rc = 0;
		}
		else if (rc) {
			LOG_ERROR("A parent object or fsim has disallowed the expand of region %s\n", context->object->name);
			RETURN(rc);
		}

		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		// Recalculate extents based on lv_size
		add_extents = value->ui32 / pe_size;
		od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].value.ui32 = add_extents;
		od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_EXPAND_PV_NAMES_INDEX:
		for ( i = 0; i < value->list->count; i++ ) {
			SET_STRING(od->option[index].value.list->value[i].s, value->list->value[i].s);
		}
		for ( ; i < od->option[index].value.list->count; i++ ) {
			if ( od->option[index].value.list->value[i].s ) {
				lvm_engine->engine_free(od->option[index].value.list->value[i].s);
				od->option[index].value.list->value[i].s = NULL;
			}
		}
		od->option[index].value.list->count = value->list->count;
		break;

	default:
		rc = EINVAL;
		break;
	}

	RETURN(rc);
}



/****** Logical Volume Shrink Option Handlers ******/


/* Function: lvm_shrink_region_parse_option_array
 *
 *	Parse the option_array_t to find options pertaining to logical volume
 *	shrink. Each option can either be keyed by value or string, so we
 *	have to check for both. Acceptable options for volume shrink are:
 *	LVM_OPTION_SHRINK_SIZE
 *	LVM_OPTION_SHRINK_EXTENTS
 *	Incorrect options are simply ignored.
 */
int lvm_shrink_region_parse_option_array(option_array_t		* options,
					lvm_volume_group_t	* group,
					lvm_logical_volume_t	* volume,
					u_int32_t		* remove_extents )
{
	u_int32_t	extents = 0;
	u_int32_t	size = 0;
	int		i, rc;

	LOG_ENTRY;

	// Initialize the options
	*remove_extents = 0;

	for ( i = 0; i < options->count; i++ ) {

		// If only the name-based index is specified, get the number.
		if ( ! options->option[i].is_number_based ) {
			if ( ! strcmp(options->option[i].name, LVM_OPTION_SHRINK_EXTENTS_STR) ) {
				options->option[i].number = LVM_OPTION_SHRINK_EXTENTS_INDEX;
			}
			else if ( ! strcmp(options->option[i].name, LVM_OPTION_SHRINK_SIZE_STR) ) {
				options->option[i].number = LVM_OPTION_SHRINK_SIZE_INDEX;
			}
			else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		// Use the number-based index to record the option.
		switch ( options->option[i].number ) {
		case LVM_OPTION_SHRINK_EXTENTS_INDEX:
			extents = options->option[i].value.ui32;
			break;
		case LVM_OPTION_SHRINK_SIZE_INDEX:
			size = options->option[i].value.ui32;
			break;
		default:
			break;
		}
	}

	rc = lvm_shrink_region_verify_options(&extents, &size, group, volume);

	*remove_extents = extents;

	RETURN(rc);
}


/* Function: lvm_shrink_region_verify_options
 *
 *	Run through the options that have been collected, and verify that they
 *	are all valid for using in the specified group. Correct any bad options
 *	if possible, or return an error.
 */
int lvm_shrink_region_verify_options(	u_int32_t		* extents,
					u_int32_t		* size,
					lvm_volume_group_t	* group,
					lvm_logical_volume_t	* volume )
{
	int rc = 0;

	LOG_ENTRY;

	// Make sure size is a multiple of the PE size. If it is
	// currently zero, this won't change anything.
	lvm_check_lv_size(size, group->vg->pe_size);

	// Check/set extents and size. This only checks that size and
	// extents match.
	rc = lvm_compare_lv_size_and_extents(size, extents, group->vg->pe_size);
	if (rc) {
		LOG_ERROR("Error verifying region shrink options\n");
		RETURN(rc);
	}

	// Make sure extents is divisible by the number of stripes. We need to
	// use an equal number of PEs on each PV. On a non-striped volume this
	// test will always fail.
	rc = *extents % volume->lv->lv_stripes;
	if (rc) {
		*extents -= rc;
		*size = *extents * group->vg->pe_size;
		LOG_ERROR("Rounding size down to stripes boundary: %ld sectors\n", *size);
	}

	// Make sure we are shrinking by at least one extent.
	// Kind of silly otherwise.
	if ( *extents == 0 ) {
		LOG_ERROR("Specified zero extents to remove. Please shrink by a non-zero amount.\n");
		RETURN(EINVAL);
	}

	// Make sure there is at least one extent left in the volume. To
	// remove all extents, use the delete function.
	if ( *extents >= volume->lv->lv_allocated_le ) {
		LOG_ERROR("Cannot shrink region %s to zero size\n", volume->region->name);
		RETURN(EINVAL);
	}

	RETURN(0);
}


/* Function: lvm_shrink_region_allocate_options
 *
 *	This is a helper function for lvm_init_options. It sets up the initial
 *	information in the option descriptor for a Shrink task.
 */
int lvm_shrink_region_allocate_option_descriptor( option_desc_array_t * od )
{
	LOG_ENTRY;

	od->count = LVM_OPTION_SHRINK_SIZE_INDEX + 1;

	// Option 0 is shrink extents
	SET_STRING(od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].name, LVM_OPTION_SHRINK_EXTENTS_STR);
	SET_STRING(od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].title, "Shrink by Extents");
	SET_STRING(od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].tip, "Number of extents to remove from the selected LVM region. Only specify extents or size!");
	od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
								  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
								  EVMS_OPTION_FLAGS_AUTOMATIC;

	// Option 1 is shrink size
	SET_STRING(od->option[LVM_OPTION_SHRINK_SIZE_INDEX].name, LVM_OPTION_SHRINK_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_SHRINK_SIZE_INDEX].title, "Shrink by Size");
	SET_STRING(od->option[LVM_OPTION_SHRINK_SIZE_INDEX].tip, "Amount of space to remove from the selected LVM region");
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].unit	= EVMS_Unit_Sectors;
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].size	= sizeof(u_int32_t);
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].flags	= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	RETURN(0);
}


/* Function: lvm_shrink_region_get_acceptable
 *
 *	This is a helper function for lvm_get_acceptable_objects. Since no
 *	objects are selected for shrinks in LVM, this function should be a
 *	no-op. But, the GUI gets mad if we don't put something, so toss the
 *	corresponding freespace region on the list.
 */
int lvm_shrink_region_get_acceptable(	storage_object_t	* region,
					dlist_t			freespace_regions )
{
	lvm_logical_volume_t	* volume = region->private_data;
	lvm_logical_volume_t	* freespace = volume->group->freespace;
	int			rc = 0;

	LOG_ENTRY;

	rc = lvm_add_object_to_list(freespace->region, freespace_regions);

	RETURN(rc);
}


/* Function: lvm_shrink_region_set_objects
 *
 *	Shrinks don't take objects in LVM, so this function is basically
 *	a no-op.
 */
int lvm_shrink_region_set_objects(	task_context_t	* context,
					dlist_t		declined_objects,
					task_effect_t	* effect )
{
	lvm_logical_volume_t	* volume = context->object->private_data;
	option_desc_array_t	* od = context->option_descriptors;
	u_int32_t		pe_size = volume->group->vg->pe_size;
	u_int32_t		remove_extents = volume->lv->lv_allocated_le - 1;
	sector_count_t		remove_sectors;
	int			rc;

	LOG_ENTRY;

	// Check the removal space with the engine.
	remove_sectors = remove_extents * pe_size;
	rc = lvm_engine->can_shrink_by(context->object, &remove_sectors);
	if ( rc == EAGAIN ) {
		if ( remove_sectors < pe_size ) {
			LOG_ERROR("Unable to shrink region %s.\n", context->object->name);
			LOG_ERROR("The Engine will only allow shrinking by %lld sectors,\n", remove_sectors);
			LOG_ERROR("but LVM must shrink the region by at least %d sectors.\n", pe_size);
			RETURN(ENOSPC);
		}
		else if ( remove_sectors < remove_extents * pe_size ) {
			remove_extents = remove_sectors / pe_size;
		}
	}
	else if (rc) {
		LOG_ERROR("A parent object or fsim has disallowed the shrink of region %s\n", context->object->name);
		RETURN(rc);
	}

	// Update range of extents and set initial value to minimum.
	od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].constraint.range, 1, remove_extents, 1);
	od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].value.ui32 = 1;
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	// Update range of LV size and set initial value to minimu.
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_SHRINK_SIZE_INDEX].constraint.range, pe_size, pe_size * remove_extents, pe_size);
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].value.ui32 = pe_size;
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	RETURN(0);
}


/* Function: lvm_shrink_region_set_option
 *
 *	This is a helper function for lvm_set_option. It sets one particular
 *	option in the option descriptor for a Shrink task, and updates other
 *	option descriptor information accordingly.
 */
int lvm_shrink_region_set_option(task_context_t	* context,
				u_int32_t	index,
				value_t		* value,
				task_effect_t	* effect )
{
	option_desc_array_t	* od = context->option_descriptors;
	lvm_logical_volume_t	* volume = context->object->private_data;
	lvm_volume_group_t	* group = volume->group;
	u_int32_t		pe_size = group->vg->pe_size;
	u_int32_t		remove_extents = 0;
	u_int32_t		remove_size = 0;
	sector_count_t		remove_sectors = 0;
	int			rc = 0;

	LOG_ENTRY;
	*effect = 0;

	// Calculate maximum number of extents that can be removed.
	remove_extents = volume->lv->lv_allocated_le - 1;

	LOG_EXTRA("Setting option %d\n", index);

	switch (index) {

	case LVM_OPTION_SHRINK_EXTENTS_INDEX:
		// Make sure there are enough extents.
		if ( value->ui32 > remove_extents ) {
			LOG_ERROR("%ld extents chosen. Only %ld allowed.\n", value->ui32, remove_extents);
			value->ui32 = remove_extents;
			*effect |= EVMS_Effect_Inexact;
		}

		// Check the removal space with the engine.
		remove_sectors = value->ui32 * pe_size;
		rc = lvm_engine->can_shrink_by(context->object, &remove_sectors);
		if ( rc == EAGAIN ) {
			if ( remove_sectors < pe_size ) {
				LOG_ERROR("Unable to shrink region %s.\n", context->object->name);
				LOG_ERROR("The Engine will only allow shrinking by %lld sectors,\n", remove_sectors);
				LOG_ERROR("but LVM must shrink the region by at least %d sectors.\n", pe_size);
				RETURN(ENOSPC);
			}
			else if ( remove_sectors < value->ui32 * pe_size ) {
				value->ui32 = remove_sectors / pe_size;
				*effect |= EVMS_Effect_Inexact;
				LOG_ERROR("A parent object or fsim has restricted the shrink size for region %s.\n", context->object->name);
				LOG_ERROR("Rounding down to %ld extents.\n", value->ui32);
			}
			rc = 0;
		}
		else if (rc) {
			LOG_ERROR("A parent object or fsim has disallowed the shrink of region %s\n", context->object->name);
			break;
		}

		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		// Recalculate size based on extents.
		remove_size = value->ui32 * pe_size;
		od->option[LVM_OPTION_SHRINK_SIZE_INDEX].value.ui32 = remove_size;
		od->option[LVM_OPTION_SHRINK_SIZE_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_SHRINK_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_SHRINK_SIZE_INDEX:
		// Make sure the size is a multiple of the PE size
		if ( lvm_check_lv_size(&value->ui32, pe_size) ) {
			*effect |= EVMS_Effect_Inexact;
		}

		// Make sure there is enough space.
		remove_size = remove_extents * pe_size;
		if ( value->ui32 > remove_size ) {
			LOG_ERROR("%ld sectors chosen for size. Only %ld allowed.\n", value->ui32, remove_size);
			value->ui32 = remove_size;
			*effect |= EVMS_Effect_Inexact;
		}

		// Check the removal space with the engine.
		remove_sectors = value->ui32;
		rc = lvm_engine->can_shrink_by(context->object, &remove_sectors);
		if ( rc == EAGAIN ) {
			if ( remove_sectors < pe_size ) {
				LOG_ERROR("Unable to shrink region %s.\n", context->object->name);
				LOG_ERROR("The Engine will only allow shrinking by %lld sectors,\n", remove_sectors);
				LOG_ERROR("but LVM must shrink the region by at least %d sectors.\n", pe_size);
				RETURN(ENOSPC);
			}
			else if ( remove_sectors < value->ui32 ) {
				value->ui32 = remove_sectors;
				lvm_check_lv_size(&value->ui32, pe_size);
				*effect |= EVMS_Effect_Inexact;
				LOG_ERROR("A parent object or fsim has restricted the shrink size for region %s.\n", context->object->name);
				LOG_ERROR("Rounding down to %ld sectors.\n", value->ui32);
			}
			rc = 0;
		}
		else if (rc) {
			LOG_ERROR("A parent object or fsim has disallowed the shrink of region %s\n", context->object->name);
			break;
		}

		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		// Recalculate extents based on lv_size
		remove_extents = value->ui32 / pe_size;
		od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].value.ui32 = remove_extents;
		od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	default:
		rc = EINVAL;
		break;
	}

	RETURN(rc);
}



/****** Volume Group Expand Option Handlers ******/



/* Function: lvm_expand_container_allocate_option_descriptor
 *
 *	This is a helper function for lvm_init_options. It sets up the initial
 *	information in the option descriptor for a Expand_Container task.
 */
int lvm_expand_container_allocate_option_descriptor( option_desc_array_t * od )
{
	LOG_ENTRY;

	od->count = 0;

	RETURN(0);
}


/* Function: lvm_expand_container_get_acceptable
 *
 *	This is a helper function for lvm_get_acceptable_objects. It determines
 *	which objects in EVMS are appropriate for an Expand_Container task. This
 *	is very similar to get_acceptable for create_container, but this
 *	function will check each segment for the correct PE size ratio before
 *	declaring it as acceptable.
 */
int lvm_expand_container_get_acceptable(storage_container_t	* container,
					dlist_t			acceptable_segments )
{
	lvm_volume_group_t	* group = container->private_data;
	storage_object_t	* segment;
	dlist_t			object_list;
	u_int32_t		pe_size;
	int 			rc;

	LOG_ENTRY;

	// Find all top-level disks, segments, and regions in the system. This
	// will also retrieve LVM regions, so let the user beware. :)
	if ( (rc = lvm_engine->get_object_list(DISK|SEGMENT|REGION, DATA_TYPE, NULL, VALID_INPUT_OBJECT, &object_list)) ) {
		RETURN(rc);
	}

	for ( rc = GoToStartOfList(object_list);
	      !rc && (segment = lvm_get_list_item(object_list));
	      rc = NextItem(object_list) ) {
		pe_size = group->vg->pe_size;
		rc = lvm_check_segment_for_pe_size(segment, &pe_size);
		if (!rc) {
			lvm_add_object_to_list(segment, acceptable_segments);
		}
	}
	DestroyList(&object_list, FALSE);

	RETURN(0);
}


/* Function: lvm_expand_container_set_objects
 *
 *	Examine the selected_objects list in the task, and verify that each
 *	object is valid for inclusion in the selected container. If any are not
 *	valid, remove them from the selected_objects and add them to the
 *	declined_objects. 
 */
int lvm_expand_container_set_objects(	task_context_t	* context,
					dlist_t		declined_objects,
					task_effect_t	* effect )
{
	lvm_volume_group_t	* group = context->container->private_data;
	storage_object_t	* segment;
	u_int32_t		pe_size;
	int			rc;

	LOG_ENTRY;

	// There are no side-effects from setting objects in an
	// expand container task.
	*effect = 0;

	// Check that all of the segments are valid for use in the 
	// selected group.
	for ( rc = GoToStartOfList(context->selected_objects);
	      !rc && (segment = lvm_get_list_item(context->selected_objects));
	      rc = NextItem(context->selected_objects) ) {

		rc = lvm_check_segment_for_group_inclusion(segment);
		if (rc) {
// Add this segment to declined objects.
			LOG_ERROR("One or more objects are invalid for container expansion\n");
			RETURN(EINVAL);
		}

		// If any of the segments are too small for the default PE
		// size, try to reset the PE size.
		pe_size = group->vg->pe_size;
		rc = lvm_check_segment_for_pe_size(segment, &pe_size);
		if (rc) {
// Add this segment to declined objects.
			LOG_ERROR("One or more objects are invalid for container expansion\n");
			RETURN(EINVAL);
		}
	}

	RETURN(0);
}


/* Function: lvm_expand_container_set_option
 *
 *	This is a helper function for lvm_set_option. Since there are no options
 *	for container expansion, this function is mostly a no-op.
 */
int lvm_expand_container_set_option(	task_context_t	* context,
					u_int32_t	index,
					value_t		* value,
					task_effect_t	* effect )
{
	LOG_ENTRY;

	*effect = 0;

	RETURN(0);
}



/****** Set Region Info Option Handlers ******/


/* Function: lvm_set_region_info_allocate_option_descriptor
 */
int lvm_set_region_info_allocate_option_descriptor( option_desc_array_t * od )
{
	LOG_ENTRY;

	od->count = 0;

	RETURN(0);
}


/* Function: lvm_set_region_info_set_option
 */
int lvm_set_region_info_set_option(	task_context_t	* context,
					u_int32_t	index,
					value_t		* value,
					task_effect_t	* effect )
{
//	lvm_logical_volume_t	* volume = context->object->private_data;
	int			rc = 0;

	LOG_ENTRY;

	*effect = 0;

	LOG_EXTRA("Setting option %d\n", index);

	RETURN(rc);
}

