/***************************************************************************
 *   Copyright (C) 2006, 2007 by Niklas Knutsson                           *
 *   nq@altern.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "eqonomizevalueedit.h"

#include <qobject.h>
#include <qlayout.h>
#include <kglobal.h>
#include <klocale.h>
#include <knuminput.h>
#include <qspinbox.h>
#include <qlineedit.h>
#include <math.h>

class Eqonomize_QSpinBox : public QSpinBox {
	public:
		Eqonomize_QSpinBox(int lower, int upper, int value, QWidget *parent, const char *name = 0) : QSpinBox(lower, upper, value, parent, name) {}
		QLineEdit *lineEdit() const {
			return editor();
		}
};
class Eqonomize_KDoubleSpinBox : public KDoubleSpinBox {
	public:
		Eqonomize_KDoubleSpinBox(double lower, double upper, double step, double value, int precision, QWidget *parent, const char *name = 0) : KDoubleSpinBox(lower, upper, step, value, precision, parent, name) {}
		QLineEdit *lineEdit() const {
			return editor();
		}
};

EqonomizeValueEdit::EqonomizeValueEdit(bool allow_negative, QWidget *parent, const char *name) : QWidget(parent, name) {
	layout = new QVBoxLayout(this);
	init(allow_negative ? INT_MIN / pow(10, KGlobal::locale()->fracDigits()) + 1.0 : 0.0, INT_MAX / pow(10, KGlobal::locale()->fracDigits()) - 1.0, 1.0, 0.0, KGlobal::locale()->fracDigits(), true);
}
EqonomizeValueEdit::EqonomizeValueEdit(double value, bool allow_negative, bool show_currency, QWidget *parent, const char *name) : QWidget(parent, name) {
	layout = new QVBoxLayout(this);
	init(allow_negative ? INT_MIN / pow(10, KGlobal::locale()->fracDigits()) + 1.0 : 0.0, INT_MAX / pow(10, KGlobal::locale()->fracDigits()) - 1.0, 1.0, value, KGlobal::locale()->fracDigits(), show_currency);
}
EqonomizeValueEdit::EqonomizeValueEdit(double value, int precision, bool allow_negative, bool show_currency, QWidget *parent, const char *name) : QWidget(parent, name) {
	layout = new QVBoxLayout(this);
	init(allow_negative ? INT_MIN / pow(10, precision) + 1.0 : 0.0, INT_MAX / pow(10, precision) - 1.0, 1.0, value, precision, show_currency);
}
EqonomizeValueEdit::EqonomizeValueEdit(double lower, double upper, double step, double value, int precision, bool show_currency, QWidget *parent, const char *name) : QWidget(parent, name) {	
	layout = new QVBoxLayout(this);
	init(lower, upper, step, value, precision, show_currency);
}
EqonomizeValueEdit::~EqonomizeValueEdit() {}
void EqonomizeValueEdit::init(double lower, double upper, double step, double value, int precision, bool show_currency) {
	i_precision = precision;
	if(i_precision > 0) {
		valueEdit = new Eqonomize_KDoubleSpinBox(lower, upper, step, value, precision, this);
	} else {
		valueEdit = new Eqonomize_QSpinBox((int) round(lower), (int) round(upper), (int) round(step), this);
		lineEdit()->setAlignment(Qt::AlignRight);
		setValue(value);
	}
	layout->addWidget(valueEdit);
	valueEdit->show();
	setFocusProxy(valueEdit);
	if(show_currency) {
		if(KGlobal::locale()->positivePrefixCurrencySymbol() && KGlobal::locale()->negativePrefixCurrencySymbol()) {
			valueEdit->setPrefix(KGlobal::locale()->currencySymbol() + " ");
		} else {
			valueEdit->setSuffix(QString(" ") + KGlobal::locale()->currencySymbol());
		}
	}
	if(i_precision > 0) {
		connect((KDoubleSpinBox*) valueEdit, SIGNAL(valueChanged(double)), this, SIGNAL(valueChanged(double)));
	} else {
		connect(valueEdit, SIGNAL(valueChanged(int)), this, SLOT(onValueChanged(int)));
	}
	connect(lineEdit(), SIGNAL(returnPressed()), this, SIGNAL(returnPressed()));
}
QLineEdit *EqonomizeValueEdit::lineEdit() const {
	if(i_precision > 0) return ((Eqonomize_KDoubleSpinBox*) valueEdit)->lineEdit();
	return ((Eqonomize_QSpinBox*) valueEdit)->lineEdit();
}
void EqonomizeValueEdit::selectAll() {
	lineEdit()->selectAll();
}
double EqonomizeValueEdit::value() const {
	if(i_precision > 0) return ((KDoubleSpinBox*) valueEdit)->value();
	else return (double) valueEdit->value();
}
double EqonomizeValueEdit::maxValue() const {
	if(i_precision > 0) return ((KDoubleSpinBox*) valueEdit)->maxValue();
	else return (double) valueEdit->maxValue();
}
void EqonomizeValueEdit::onValueChanged(int) {
	emit valueChanged((double) valueEdit->value());
}
void EqonomizeValueEdit::setValue(double d_value) {
	if(i_precision > 0) return ((KDoubleSpinBox*) valueEdit)->setValue(d_value);
	else return valueEdit->setValue((int) round(d_value));
}
void EqonomizeValueEdit::setMaxValue(double d_value) {
	if(i_precision > 0) return ((KDoubleSpinBox*) valueEdit)->setMaxValue(d_value);
	else return valueEdit->setMaxValue((int) round(d_value));
}
void EqonomizeValueEdit::setRange(double lower, double upper, double step, int precision) {
	if(precision != i_precision && (precision == 0 || i_precision == 0)) {
		double d_value = value();
		bool show_currency = !valueEdit->prefix().isEmpty() || !valueEdit->suffix().isEmpty();
		delete valueEdit;
		init(lower, upper, step, d_value, precision, show_currency);
		return;
	}
	i_precision = precision;
	if(i_precision > 0) return ((KDoubleSpinBox*) valueEdit)->setRange(lower, upper, step, precision);
	else return valueEdit->setRange((int) round(lower), (int) round(upper));
}
void EqonomizeValueEdit::setPrecision(int precision) {
	if(precision == i_precision) return;
	if(precision == 0 || i_precision == 0) {
		double d_value = value();
		bool show_currency = !valueEdit->prefix().isEmpty() || !valueEdit->suffix().isEmpty();
		bool b_neg = false;
		double d_step;
		if(i_precision > 0) {
			b_neg = ((KDoubleSpinBox*) valueEdit)->minValue() < 0.0;
			d_step = ((KDoubleSpinBox*) valueEdit)->lineStep();
		} else {
			b_neg = valueEdit->minValue() < 0;
			d_step = valueEdit->lineStep();
		}		
		delete valueEdit;
		init(b_neg ? INT_MIN / pow(10, precision) + 1.0 : 0.0, INT_MAX / pow(10, precision) - 1.0, d_step, d_value, precision, show_currency);
	} else {
		bool b_neg = ((KDoubleSpinBox*) valueEdit)->minValue() < 0.0;
		double d_step = ((KDoubleSpinBox*) valueEdit)->lineStep();
		i_precision = precision;
		((KDoubleSpinBox*) valueEdit)->setRange(b_neg ? INT_MIN / pow(10, precision) + 1.0 : 0.0, INT_MAX / pow(10, precision) - 1.0, d_step, precision);
	}
}

#include "eqonomizevalueedit.moc"
