# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import pgm
from pgm.graph.text import Text
from pgm.graph.image import Image

from progressbar_osd import ProgressBarOsd
from dock import Dock

import time, os


# The Time-Layouts
TIME_LAYOUT_NORMAL = 0
TIME_LAYOUT_ELAPSED = 1
TIME_LAYOUT_MINUS = 2

class PlayerOsd(Dock):

    def __init__(self, canvas, layer,
                 playing_length,
                 width, height,
                 bar_bg_image, bar_fg_image,
                 background_image):

        Dock.__init__(self, canvas, layer, width, height, background_image)

        self._time_position = 0
        self._time_string = ""
        self._time_layout = TIME_LAYOUT_NORMAL


        inner_width = width * 0.9
        inner_x = (width - inner_width)/2.0


        # build the title
        self._title = Text()
        self._title.label = ""
#        self._title.font_family = "MgOpen Cosmetica"
        self._title.ellipsize = pgm.TEXT_ELLIPSIZE_MIDDLE
        self._title.fg_color = (220, 220, 220, 255)
        self._title.bg_color = (0, 0, 0, 0)
        self.add(self._title)
        self._title.position = (inner_x, 0.05, 0.0)
        self._title.size = (inner_width, height*0.33)

        # build the time bar
        self._timebar = ProgressBarOsd(canvas, layer, playing_length,
                                       bar_bg_image, bar_fg_image)
        self._timebar.bg_color = (0, 0, 0, 0)
        self.add(self._timebar)
        self._timebar.position = (inner_x, height*(0.33+0.15), 0.0)
        self._timebar.size = (inner_width, height*0.2)
        self._timebar.visible = False
        
        # build the total time text
        self._time = Text()
        self._time.label = ""
#        self._time.font_family = "MgOpen Cosmetica"
        self._time.fg_color = (255, 255, 255, 255)
        self._time.bg_color = (0, 0, 0, 0)
        self._time.alignment = pgm.TEXT_ALIGN_RIGHT
        self.add(self._time)
        self._time.position = (inner_x, height*(0.33+0.2+0.15), 0.001)
        self._time.size = (inner_width, height*0.2)
        self._time.visible = False

        self._title.visible = True

        self.playing_length = playing_length
        
    def playing_length__set(self, length):
        """
        Set the length of the playing media (in milliseconds)
        """
        self._timebar.length = length
        if length > 1:
            self._timebar.visible = True
            self._show_time()
        else:
            self._timebar.visible = False

    def playing_length__get(self):
        """
        Get the length of the playing media (in milliseconds)
        """
        return self._timebar.length


    def title__set(self, title):
        """
        Set the title of the dock
        @param title: the name of a movie or a audio file
        @type title: string
        """
        self._title.label = title

    def title__get(self):
        """
        Get the current title of the dock
        """
        return self._title.label

    def time__set(self, position):
        """
        Set the time position to position
        """
        self._time_position = position
        self._time.visible = bool(position > 1)
        self._timebar.jump_to_position(position)
        self._show_time()

    def time__get(self):
        """
        Get the current time position
        """
        return self._time_position

    def _conv_secs(self, sec, format):
        """
        This returns a converted string of the seconds in the given format
        (which must be a string that is usable for time.strftime())
        If the intelligent_mode was swichted on and there was the word "days" in format
        it is replaced with the number of weeks and days
        """
        if "days" in format:
            if sec >= 60480000:
                weeks = int(sec / 604800000)
                if weeks != 1:
                    replacer = "%s weeks" % weeks
                else:
                    replacer = "1 week"

                days = int(((sec - (weeks * 604800000)) / 86400000))
                if days != 1:
                    replacer += " %s days" % days
                else:
                    replacer += " 1 day"
                format = format.replace("days" , replacer)
            elif sec >= 86400000:
                days = int(sec / 86400000)
                format = format.replace("days" , ("%s day(s)", days))
            else:
                format = format.replace("days","")

        return time.strftime(format, time.gmtime(sec/1000))

    def _show_time(self):
        format = self.conversion_string
        label = ""
        
        if self._time_layout == TIME_LAYOUT_MINUS:
            label = "- %s" % (self._conv_secs(
                                            self.playing_length
                                            -self.time, format))
        elif self._time_layout == TIME_LAYOUT_ELAPSED:
            label = "%s" % (self._conv_secs(self.time, format))
        else:
            if self.time > 1:
                label = self._conv_secs(self.time, format)
                if self.playing_length > 1:
                    total = self._conv_secs(self.playing_length, format)
                    label += " / %s" % total

        if label != self._time.label:
            self._time.label = label

    def conversion_string__set(self, string):
        """
        Set the conversion string for the time to string
        @param string: a string that could be parsed by time.strftime
        if there is none given, the player dock is set in the intelligent
            mode, and trys to guess what format is usefull
        """
        self._time_string = string
        self._show_time()
        # FIXME: missed total time formatting

    def conversion_string__get(self):
        """
        Returns the string that is currently used to parse in time.strftime
        """
        if self._time_string == "":
            format = "%Ss"
            if self.playing_length > 1:
                time_value = self.playing_length
            else:
                time_value = self.time
            if time_value >= 86400000:
                format = "days %H:%M"
            elif time_value >= 3600000:
                format = "%H:%M:%S"
            elif time_value >= 60000:
                format = "%M:%S"

            return format
        else:
            return self._time_string

    def time_layout__set(self, layout):
        """
        Set how the time should be shown
        """
        self._time_layout = layout
        self._show_time()

    def time_layout__get(self):
        """
        Get the currently used time layout
        """
        return self._time_layout
