/* easytag.c - 2000/04/28 */
/*
 *  EasyTAG - Tag editor for MP3 and OGG files
 *  Copyright (C) 2000  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <id3.h>

#include "easytag.h"
#include "browser.h"
#include "misc.h"
#include "bar.h"
#include "prefs.h"
#include "setting.h"
#include "scan.h"
#include "mpeg_header.h"
#include "id3tag.h"
#include "msgbox.h"
#include "et_core.h"
#include "mylocale.h"

#include "../pixmaps/EasyTAG.xpm"


/****************
 * Declarations *
 ****************/
guint idle_handler_id;
guint progressbar_index;  /* An integer to set the value of progress bar into the recurse fonction */

/* A variable to force to quit recursive functions (reading dirs) */
gboolean Stop_Recursive_File_Search = 0;
GtkWidget *QuitRecursionWindow = NULL;

/* Used to force to hide the msgbox when saving tag */
gboolean SF_HideMsgbox_Write_Tag;
/* To remember which button was pressed when saving tag */
gint SF_ButtonPressed_Write_Tag;
/* Used to force to hide the msgbox when renaming file */
gboolean SF_HideMsgbox_Rename_File;
/* To remember which button was pressed when renaming file */
gint SF_ButtonPressed_Rename_File;


/**************
 * Prototypes *
 **************/
GtkWidget *Create_Browser_Area (void);
GtkWidget *Create_File_Area    (void);
GtkWidget *Create_Tag_Area     (void);
GtkWidget *Create_Command_Area (void);
void Mini_Button_Clicked (GtkObject *object);
void Insert_Only_Digit (GtkEditable *editable, const gchar *text, gint length,gint *position,gpointer data);
void Disable_Command_Buttons (void);
void Clear_Tag_Entry_Fields (void);
void Clear_Header_Fields (void);
void Spinner_Button_Activate (void);

void Write_This_Tag (void);
void Rename_File    (void);
gint Save_File      (gboolean multiple_files);

void Display_Usage (void);

void Init_Load_Default_Dir (void);
void EasyTAG_Exit (void);
void Quit_MainWindow_Ok_Button (void);

guint Read_Directory_Recursively (gchar * path, gint recurse);
guint Read_Directory_Recursively_Get_Number_Of_File (gchar * path, gint recurse);
void Open_Quit_Recursion_Function_Window    (void);
void Destroy_Quit_Recursion_Function_Window (void);
void Quit_Recursion_Function_Button_Pressed (void);
void Quit_Recursion_Window_Key_Press (GtkWidget *window, GdkEvent *event);



/********
 * Main *
 ********/
int main (int argc, char *argv[])
{
    GtkWidget *Table;
    GtkWidget *HBox, *VBox;
    GdkPixmap *pixmap;
    GdkBitmap *mask;
    struct stat statbuf;


#ifdef ENABLE_NLS
    bindtextdomain("easytag",LOCALE);
    textdomain("easytag");
    /* Initialize i18n support */
    gtk_set_locale();
#endif

    /* Initialize GTK */
    gtk_init( &argc, &argv );

    /* Get home variable */
    HOME_VARIABLE = g_getenv("HOME");
    INIT_DIRECTORY = NULL;

    /* Check given arguments */
    if (argc>1)
    {
        if ( (strcmp(argv[1],"--version")==0) || (strcmp(argv[1],"-v")==0) ) // Query version
        {
            printf(_("%s %s by %s (compiled %s, %s)\n"),APPNAME,VERSION,AUTHOR,__TIME__,__DATE__);
            printf(_("E-mail: %s"),EMAIL"\n");
            printf(_("Web Page: %s"),WEBPAGE"\n");
            exit (0);
        }else if ( (strcmp(argv[1],"--help")==0) || (strcmp(argv[1],"-h")==0) ) // Query help
        {
            Display_Usage();
        }else if ( (strcmp(argv[1],".")==0) || (strcmp(argv[1],"./")==0) ) // Passed the current directory
        {
            INIT_DIRECTORY = g_get_current_dir();
        }else if (stat(argv[1],&statbuf)==0 && S_ISDIR(statbuf.st_mode)) // Passed an absolute path
        {
            INIT_DIRECTORY = g_strdup(argv[1]);
        }else if (stat(argv[1],&statbuf)==0 && S_ISREG(statbuf.st_mode)) // Passed an absolute path with filename
        {
            // When passing a file, we load only the directory
            INIT_DIRECTORY = g_dirname(argv[1]);
        }else
        {
            printf("Unknown parameter '%s'\n",argv[1]);
            Display_Usage();
        }
    }

    /* Starting message */
    g_print(_("Starting EasyTAG %s ...\n"),VERSION);
    g_print(_("Currently using id3lib version %d.%d.%d ...\n"),ID3LIB_MAJOR_VERSION,
                                                               ID3LIB_MINOR_VERSION,
                                                               ID3LIB_PATCH_VERSION);

    /* Load Config */
    Init_Config_Variables();
    Read_Config();
    /* Display_Config(); // <- for debugging */

    /* Initialization */
    Init_Mouse_Cursor();
    Init_OptionsWindow();
    Init_ScannerWindow();

    /* The main window */
    MainWindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW (MainWindow),APPNAME" "VERSION);
    gtk_window_set_policy(GTK_WINDOW(MainWindow),FALSE,TRUE,FALSE);
    if (SET_MAIN_WINDOW_SIZE)
        gtk_window_set_default_size(GTK_WINDOW(MainWindow),MAIN_WINDOW_WIDTH,MAIN_WINDOW_HEIGHT);

    gtk_signal_connect(GTK_OBJECT(MainWindow),"delete_event",GTK_SIGNAL_FUNC(Quit_MainWindow),NULL);
    gtk_signal_connect(GTK_OBJECT(MainWindow),"destroy",GTK_SIGNAL_FUNC(Quit_MainWindow),NULL);

    /* Minimised window icon  */
    gtk_widget_realize(MainWindow);
    pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,EasyTAG_xpm);
    gdk_window_set_icon(MainWindow->window,(GdkWindow *)NULL,pixmap,mask);

    Table = gtk_table_new(2,3,FALSE);
    gtk_container_add (GTK_CONTAINER(MainWindow),Table);
    gtk_widget_show(Table);

    /* Menu Bar */
    MenuArea = Create_Menu_Bar();
    gtk_table_attach(GTK_TABLE(Table),MenuArea,0,2,0,1,GTK_FILL|GTK_EXPAND,GTK_FILL,0,0);

    /* Tool Bar */
    ToolArea = Create_Tool_Bar();
    gtk_table_attach(GTK_TABLE(Table),ToolArea,0,2,1,2,GTK_FILL|GTK_EXPAND,GTK_FILL,0,0);
    
    /* The two panes: BrowserArea on the left, FileArea+TagArea on the right */
    MainWindowHPaned = gtk_hpaned_new();
    gtk_table_attach(GTK_TABLE(Table),MainWindowHPaned,0,1,2,3,GTK_FILL|GTK_EXPAND,GTK_FILL|GTK_EXPAND,0,0);
    gtk_paned_set_handle_size(GTK_PANED(MainWindowHPaned),8);
    gtk_paned_set_gutter_size(GTK_PANED(MainWindowHPaned),6);                       
    if (SET_PANE_HANDLE_POSITION1)
        gtk_paned_set_position(GTK_PANED(MainWindowHPaned),PANE_HANDLE_POSITION1);
    gtk_widget_show(MainWindowHPaned);

    /* Browser */
    BrowseArea = Create_Browser_Area();
    gtk_paned_pack1(GTK_PANED(MainWindowHPaned),BrowseArea,TRUE,TRUE);

    /* Vertical box for FileArea + TagArea */
    VBox = gtk_vbox_new(FALSE,0);
    gtk_paned_pack2(GTK_PANED(MainWindowHPaned),VBox,TRUE,FALSE);
    gtk_widget_show(VBox);

    /* File */
    FileArea = Create_File_Area();
    gtk_box_pack_start(GTK_BOX(VBox),FileArea,TRUE,TRUE,0);

    /* Tag */
    TagArea = Create_Tag_Area();
    gtk_box_pack_start(GTK_BOX(VBox),TagArea,TRUE,TRUE,0);

    /* Commands */
    CommandArea = Create_Command_Area();
    gtk_table_attach(GTK_TABLE(Table),CommandArea,1,2,2,3,GTK_FILL,GTK_FILL,0,0);

    /* Horizontal box for Status bar + Progress bar */
    HBox = gtk_hbox_new(FALSE,0);
    gtk_table_attach(GTK_TABLE(Table),HBox,0,2,3,4,GTK_FILL|GTK_EXPAND,GTK_FILL,0,0);
    gtk_widget_show(HBox);

    /* Status bar */
    StatusArea = Create_Status_Bar();
    gtk_box_pack_start(GTK_BOX(HBox),StatusArea,TRUE,TRUE,0);

    /* Progress bar */
    ProgressArea = Create_Progress_Bar();
    gtk_box_pack_end(GTK_BOX(HBox),ProgressArea,FALSE,FALSE,0);

    gtk_widget_show(MainWindow);

    /* Load the default dir when the UI is created and displayed to the screen */
    idle_handler_id = gtk_idle_add((GtkFunction)Init_Load_Default_Dir,NULL);

    /* Enter the event loop */
    gtk_main ();         
    return 0;
}


GtkWidget *Create_Browser_Area (void)
{
    GtkWidget *Frame;
    GtkWidget *Tree;

    Frame = gtk_frame_new(_("Browser"));
    gtk_container_border_width(GTK_CONTAINER(Frame),2);

    Tree = Create_Browser_Items(MainWindow);
    gtk_container_add(GTK_CONTAINER(Frame),Tree);

    /* Don't load init dir here because Tag area hasn't been yet created!.
     * It will be load at the end of the main function */
    //Browser_Tree_Select_Dir(DEFAULT_PATH_TO_MP3);

    gtk_widget_show(Frame);
    return Frame;
}


#include "../pixmaps/read_only.xpm"
#include "../pixmaps/broken.xpm"
GtkWidget *Create_File_Area (void)
{
    GtkWidget *VBox, *HBox;
    GtkWidget *Separator;
    GtkTooltips *Tips;


    FileFrame = gtk_frame_new(_("File"));
    gtk_container_border_width(GTK_CONTAINER(FileFrame),2);

    VBox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(FileFrame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);

    /* Tips */
    Tips = gtk_tooltips_new_1();

    /* HBox for FileEntry and IconBox */
    HBox = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(VBox),HBox,TRUE,TRUE,0);

    /* File index (position in list + list length) */
    FileIndex = gtk_label_new("0/0:");
    gtk_box_pack_start(GTK_BOX(HBox),FileIndex,FALSE,FALSE,0);

    /* File name */
    FileEntry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(HBox),FileEntry,TRUE,TRUE,2);
    gtk_entry_set_text(GTK_ENTRY(FileEntry),_("Select a directory to browse!"));
    
    /* Access status icon */
    ReadOnlyStatusIconBox = Create_Pixmap_Icon_With_Event_Box(read_only_xpm);
    gtk_box_pack_start(GTK_BOX(HBox),ReadOnlyStatusIconBox,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,ReadOnlyStatusIconBox,_("Read Only File"),NULL);
    BrokenStatusIconBox = Create_Pixmap_Icon_With_Event_Box(broken_xpm);
    gtk_box_pack_start(GTK_BOX(HBox),BrokenStatusIconBox,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,BrokenStatusIconBox,_("File Link Broken"),NULL);

    /* Create Popup Menu */
    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(FileEntry));


    /*
     *  File Infos
     */
    HeaderInfosTable = gtk_table_new(3,5,FALSE);
    gtk_container_add(GTK_CONTAINER(VBox),HeaderInfosTable);
    gtk_container_border_width(GTK_CONTAINER(HeaderInfosTable),2);
    gtk_table_set_col_spacings(GTK_TABLE(HeaderInfosTable),4);

    VersionLabel = gtk_label_new(_("MPEG"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),VersionLabel,0,1,0,1);
    VersionValueLabel = gtk_label_new(_("?, Layer ?"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),VersionValueLabel,1,2,0,1);
    gtk_misc_set_alignment(GTK_MISC(VersionLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(VersionValueLabel),0,0.5);

    BitrateLabel = gtk_label_new(_("Bitrate:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),BitrateLabel,0,1,1,2);
    BitrateValueLabel = gtk_label_new(_("? kb/s"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),BitrateValueLabel,1,2,1,2);
    gtk_misc_set_alignment(GTK_MISC(BitrateLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(BitrateValueLabel),0,0.5);

    SampleRateLabel = gtk_label_new(_("Freq:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),SampleRateLabel,0,1,2,3);
    SampleRateValueLabel = gtk_label_new(_("? Hz"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),SampleRateValueLabel,1,2,2,3);
    gtk_misc_set_alignment(GTK_MISC(SampleRateLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(SampleRateValueLabel),0,0.5);

    Separator = gtk_vseparator_new();
    gtk_table_attach(GTK_TABLE(HeaderInfosTable),Separator,2,3,0,4,GTK_FILL,GTK_FILL,0,0);

    ModeLabel = gtk_label_new(_("Mode:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),ModeLabel,3,4,0,1);
    ModeValueLabel = gtk_label_new(_("?"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),ModeValueLabel,4,5,0,1);
    gtk_misc_set_alignment(GTK_MISC(ModeLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(ModeValueLabel),0,0.5);

    SizeLabel = gtk_label_new(_("Size:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),SizeLabel,3,4,1,2);
    SizeValueLabel = gtk_label_new(_("? kb"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),SizeValueLabel,4,5,1,2);
    gtk_misc_set_alignment(GTK_MISC(SizeLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(SizeValueLabel),0,0.5);

    DurationLabel = gtk_label_new(_("Time:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),DurationLabel,3,4,2,3);
    DurationValueLabel = gtk_label_new(_("?"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),DurationValueLabel,4,5,2,3);
    gtk_misc_set_alignment(GTK_MISC(DurationLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(DurationValueLabel),0,0.5);

    gtk_widget_show(FileFrame);
    gtk_widget_show(VBox);
    gtk_widget_show(HBox);
    gtk_widget_show(FileIndex);
    gtk_widget_show(FileEntry);
    if (SHOW_HEADER_INFO)
        gtk_widget_show_all(HeaderInfosTable);
    return FileFrame;
}


#include "../pixmaps/sequence_track.xpm"
GtkWidget *Create_Tag_Area (void)
{
    GtkWidget *Separator;
    GtkWidget *Table;
    GtkWidget *Label;
    GtkWidget *Icon;
    GtkTooltips *Tips;

    TagFrame = gtk_frame_new(_("Tag"));
    gtk_container_border_width(GTK_CONTAINER(TagFrame),2);

    Table = gtk_table_new(6,10,FALSE);
    gtk_container_add(GTK_CONTAINER(TagFrame),Table);
    gtk_container_border_width(GTK_CONTAINER(Table),5);
    gtk_table_set_row_spacings(GTK_TABLE(Table),4);
    gtk_table_set_col_spacings(GTK_TABLE(Table),5);

    /* Tips */
    Tips = gtk_tooltips_new_1();

    /* Title */
    Label = gtk_label_new(_("Title:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    TitleEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),TitleEntry,1,9,0,1);
    gtk_widget_set_usize(TitleEntry,200,-1);

    TitleMButton = gtk_button_new();
    gtk_widget_set_usize(TitleMButton,10,10);
    gtk_table_attach(GTK_TABLE(Table),TitleMButton,9,10,0,1,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(TitleMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,TitleMButton,_("Tag all files with this title"),NULL);

    /* Attach Popup Menu */
    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(TitleEntry));


    /* Artist */
    Label = gtk_label_new(_("Artist:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    ArtistEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),ArtistEntry,1,9,1,2);

    ArtistMButton = gtk_button_new();
    gtk_widget_set_usize(ArtistMButton,10,10);
    gtk_table_attach(GTK_TABLE(Table),ArtistMButton,9,10,1,2,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(ArtistMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,ArtistMButton,_("Tag all files with this artist"),NULL);

    /* Attach Popup Menu */
    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(ArtistEntry));


    /* Album */
    Label = gtk_label_new(_("Album:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,2,3,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    AlbumEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),AlbumEntry,1,9,2,3);

    AlbumMButton = gtk_button_new();
    gtk_widget_set_usize(AlbumMButton,10,10);
    gtk_table_attach(GTK_TABLE(Table),AlbumMButton,9,10,2,3,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(AlbumMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,AlbumMButton,_("Tag all files with this album name"),NULL);

    /* Attach Popup Menu */
    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(AlbumEntry));


    /* Year */
    Label = gtk_label_new(_("Year:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    YearEntry = gtk_entry_new_with_max_length(4);
    gtk_table_attach_defaults(GTK_TABLE(Table),YearEntry,1,2,3,4);
    gtk_widget_set_usize(YearEntry,40,-1);
    gtk_signal_connect(GTK_OBJECT(YearEntry),"insert_text",GTK_SIGNAL_FUNC(Insert_Only_Digit),NULL);
    gtk_signal_connect(GTK_OBJECT(YearEntry),"activate",GTK_SIGNAL_FUNC(Parse_Date),NULL);
    gtk_signal_connect(GTK_OBJECT(YearEntry),"focus-out-event",GTK_SIGNAL_FUNC(Parse_Date),NULL);

    YearMButton = gtk_button_new();
    gtk_widget_set_usize(YearMButton,10,10);
    gtk_table_attach(GTK_TABLE(Table),YearMButton,2,3,3,4,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(YearMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,YearMButton,_("Tag all files with this year"),NULL);


    /* Small vertical separator */
    Separator = gtk_vseparator_new();
    gtk_table_attach(GTK_TABLE(Table),Separator,3,4,3,4,GTK_FILL,GTK_FILL,4,2);


    /* Track */
    TrackMButtonSequence = gtk_button_new();
    gtk_widget_set_usize(TrackMButtonSequence,10,10);
    gtk_table_attach(GTK_TABLE(Table),TrackMButtonSequence,4,5,3,4,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(TrackMButtonSequence),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,TrackMButtonSequence,_("Number all tracks sequentially. "
                                                     "Starts at 01 in each subdirectory."), NULL);
    // Pixmap into TrackMButtonSequence button
    Icon = Create_Pixmap_Icon(sequence_track_xpm);
    gtk_container_add(GTK_CONTAINER(TrackMButtonSequence),Icon);
    gtk_misc_set_alignment(GTK_MISC(Icon),1,1); /* strange: values would be 0,0 !*/

    Label = gtk_label_new(_("Track #:"));
    gtk_table_attach(GTK_TABLE(Table),Label,5,6,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    TrackEntry = gtk_combo_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),TrackEntry,6,7,3,4);
    gtk_widget_set_usize(TrackEntry,45,-1);
    gtk_signal_connect(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry)),"insert_text",
                GTK_SIGNAL_FUNC(Insert_Only_Digit),NULL);

    Label = gtk_label_new("/");
    gtk_table_attach(GTK_TABLE(Table),Label,7,8,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    TrackTotalEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),TrackTotalEntry,8,9,3,4);
    gtk_widget_set_usize(TrackTotalEntry,25,-1);
    gtk_signal_connect(GTK_OBJECT(GTK_ENTRY(TrackTotalEntry)),"insert_text",
        GTK_SIGNAL_FUNC(Insert_Only_Digit),NULL);

    TrackMButton = gtk_button_new();
    gtk_widget_set_usize(TrackMButton,10,10);
    gtk_table_attach(GTK_TABLE(Table),TrackMButton,9,10,3,4,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(TrackMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,TrackMButton,_("Tag all files with this track number"),NULL);


    /* Genre */
    Label = gtk_label_new(_("Genre:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,4,5,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    GenreEntry = gtk_combo_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),GenreEntry,1,9,4,5);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),TRUE);
    gtk_combo_set_use_arrows_always(GTK_COMBO(GenreEntry),TRUE);
    gtk_combo_set_case_sensitive(GTK_COMBO(GenreEntry),FALSE);
    Load_Genres_List_To_UI();
    // Help the user to fill the genre. Important: '...after' to get the last character entered
    gtk_signal_connect_after(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry)),"key_press_event",
                (GtkSignalFunc)Parse_Genre,GTK_COMBO(GenreEntry));

    GenreMButton = gtk_button_new();
    gtk_widget_set_usize(GenreMButton,10,10);
    gtk_table_attach(GTK_TABLE(Table),GenreMButton,9,10,4,5,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(GenreMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,GenreMButton,_("Tag all files with this genre"),NULL);


    /* Comment */
    Label = gtk_label_new(_("Comment:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,5,6,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    CommentEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),CommentEntry,1,9,5,6);

    CommentMButton = gtk_button_new();
    gtk_widget_set_usize(CommentMButton,10,10);
    gtk_table_attach(GTK_TABLE(Table),CommentMButton,9,10,5,6,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(CommentMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,CommentMButton,_("Tag all files with this comment"),NULL);

    /* Attach Popup Menu */
    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(CommentEntry));


    gtk_widget_show_all(TagFrame);
    return TagFrame;
}


#include "../pixmaps/go.xpm"
#include "../pixmaps/next.xpm"
#include "../pixmaps/previous.xpm"
#include "../pixmaps/scan.xpm"
#include "../pixmaps/scan_all.xpm"
#include "../pixmaps/delete.xpm"
#include "../pixmaps/delete_all.xpm"
#include "../pixmaps/undo.xpm"
#include "../pixmaps/undo_all.xpm"
#include "../pixmaps/redo.xpm"
#include "../pixmaps/redo_all.xpm"
#include "../pixmaps/save.xpm"
#include "../pixmaps/save_all.xpm"
#include "../pixmaps/exit.xpm"
GtkWidget *Create_Command_Area (void)
{
    GtkWidget *Frame;
    GtkWidget *MainVBox, *VBox, *HBox;
    GtkTooltips *Tips;
    GtkWidget *Icon;


    Frame = gtk_frame_new(_("Commands"));
    gtk_container_border_width(GTK_CONTAINER(Frame),2);

    VBox = gtk_vbox_new(FALSE,0);    /* This VBox is used just to have buttons aligned to the bottom */
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    MainVBox = gtk_vbox_new(FALSE,4);
    gtk_box_pack_end(GTK_BOX(VBox),MainVBox,FALSE,FALSE,0);
    gtk_container_border_width(GTK_CONTAINER(MainVBox),4);

    /*
     * Spin button
     */
    HBox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(MainVBox),HBox,FALSE,FALSE,0);

    SpinButtonAdjust = (GtkAdjustment *) gtk_adjustment_new(0,0,0,1,10,10);
    SpinButton = gtk_spin_button_new(SpinButtonAdjust,1.0,0);
    gtk_box_pack_start(GTK_BOX(HBox),SpinButton,TRUE,TRUE,0);
    gtk_signal_connect(GTK_OBJECT(SpinButton),"activate",(GtkSignalFunc)Spinner_Button_Activate,NULL);
    // FIXME !!
    //gtk_signal_connect(GTK_OBJECT(SpinButton),"changed",(GtkSignalFunc)Action_Select_Nth_File,NULL);

    GoButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(go_xpm);
    gtk_container_add(GTK_CONTAINER(GoButton),Icon);
    gtk_misc_set_alignment(GTK_MISC(Icon),0.5,0.4);
    gtk_box_pack_start(GTK_BOX(HBox),GoButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(GoButton),GTK_RELIEF_NONE);
    gtk_signal_connect(GTK_OBJECT(GoButton),"clicked",(GtkSignalFunc)Spinner_Button_Activate,NULL);


    /*
     * Buttons: First, Prev, Next, Last.
     */
    HBox = gtk_hbox_new(FALSE,3);
    gtk_box_pack_start(GTK_BOX(MainVBox),HBox,FALSE,FALSE,0);
    FirstButton = gtk_button_new();
    PrevButton  = gtk_button_new();
    NextButton  = gtk_button_new();
    LastButton  = gtk_button_new();
    gtk_widget_set_usize(FirstButton,6,-1);
    gtk_widget_set_usize(LastButton, 6,-1);
    gtk_box_pack_start(GTK_BOX(HBox),FirstButton,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(HBox),PrevButton,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(HBox),NextButton,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(HBox),LastButton,FALSE,FALSE,0);


    /* Pixmaps into buttons */
    Icon = Create_Pixmap_Icon(previous_xpm);
    gtk_container_add(GTK_CONTAINER(PrevButton),Icon);
    gtk_misc_set_alignment(GTK_MISC(Icon),0.4,0.5);
    Icon = Create_Pixmap_Icon(next_xpm);
    gtk_misc_set_alignment(GTK_MISC(Icon),0.5,0.5);
    gtk_container_add(GTK_CONTAINER(NextButton),Icon);

    gtk_signal_connect(GTK_OBJECT(FirstButton),"clicked",GTK_SIGNAL_FUNC(Action_Select_First_File),NULL);
    gtk_signal_connect(GTK_OBJECT(PrevButton),"clicked",GTK_SIGNAL_FUNC(Action_Select_Prev_File),NULL);
    gtk_signal_connect(GTK_OBJECT(NextButton),"clicked",GTK_SIGNAL_FUNC(Action_Select_Next_File),NULL);
    gtk_signal_connect(GTK_OBJECT(LastButton),"clicked",GTK_SIGNAL_FUNC(Action_Select_Last_File),NULL);

    /*
     * Buttons: Scan.
     */
    HBox = gtk_hbox_new(FALSE,3);
    gtk_box_pack_start(GTK_BOX(MainVBox),HBox,FALSE,FALSE,0);

    ScanButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(scan_xpm);
    gtk_container_add(GTK_CONTAINER(ScanButton),Icon);

    ScanAllButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(scan_all_xpm);
    gtk_container_add(GTK_CONTAINER(ScanAllButton),Icon);

    gtk_box_pack_start(GTK_BOX(HBox),ScanButton,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(HBox),ScanAllButton,TRUE,TRUE,0);

    gtk_signal_connect(GTK_OBJECT(ScanButton),"clicked",GTK_SIGNAL_FUNC(Action_Scan_File),NULL);
    gtk_signal_connect(GTK_OBJECT(ScanButton),"enter",GTK_SIGNAL_FUNC(Scan_Display_Info),NULL);
    gtk_signal_connect(GTK_OBJECT(ScanAllButton),"clicked",GTK_SIGNAL_FUNC(Action_Scan_All_Files),NULL);

    /*
     * Buttons: Remove.
     */
    HBox = gtk_hbox_new(FALSE,3);
    gtk_box_pack_start(GTK_BOX(MainVBox),HBox,FALSE,FALSE,0);

    RemoveButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(delete_xpm);
    gtk_container_add(GTK_CONTAINER(RemoveButton),Icon);

    RemoveAllButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(delete_all_xpm);
    gtk_container_add(GTK_CONTAINER(RemoveAllButton),Icon);

    gtk_box_pack_start(GTK_BOX(HBox),RemoveButton,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(HBox),RemoveAllButton,TRUE,TRUE,0);

    gtk_signal_connect(GTK_OBJECT(RemoveButton),"clicked",GTK_SIGNAL_FUNC(Action_Remove_Tag),NULL);
    gtk_signal_connect(GTK_OBJECT(RemoveAllButton),"clicked",GTK_SIGNAL_FUNC(Action_Remove_All_Tags),NULL);

    /*
     * Buttons: Undo.
     */
    HBox = gtk_hbox_new(FALSE,3);
    gtk_box_pack_start(GTK_BOX(MainVBox),HBox,FALSE,FALSE,0);

    UndoButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(undo_xpm);
    gtk_container_add(GTK_CONTAINER(UndoButton),Icon);

    UndoAllButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(undo_all_xpm);
    gtk_container_add(GTK_CONTAINER(UndoAllButton),Icon);

    gtk_box_pack_start(GTK_BOX(HBox),UndoButton,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(HBox),UndoAllButton,TRUE,TRUE,0);

    gtk_signal_connect(GTK_OBJECT(UndoButton),"clicked",GTK_SIGNAL_FUNC(Action_Undo_File),NULL);
    //gtk_signal_connect(GTK_OBJECT(UndoButton),"enter",GTK_SIGNAL_FUNC(Undo_Display_Last_Info),NULL);
    //gtk_signal_connect_after(GTK_OBJECT(UndoButton),"clicked",GTK_SIGNAL_FUNC(Undo_Display_Last_Info),NULL);
    gtk_signal_connect(GTK_OBJECT(UndoAllButton),"clicked",GTK_SIGNAL_FUNC(Action_Undo_All_Files),NULL);
    //gtk_signal_connect(GTK_OBJECT(UndoAllButton),"enter",GTK_SIGNAL_FUNC(Undo_Display_Last_Info),NULL);
    //gtk_signal_connect_after(GTK_OBJECT(UndoAllButton),"clicked",GTK_SIGNAL_FUNC(Undo_Display_Last_Info),NULL);

    /*
     * Buttons: Redo.
     */
    HBox = gtk_hbox_new(FALSE,3);
    gtk_box_pack_start(GTK_BOX(MainVBox),HBox,FALSE,FALSE,0);

    RedoButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(redo_xpm);
    gtk_container_add(GTK_CONTAINER(RedoButton),Icon);

    RedoAllButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(redo_all_xpm);
    gtk_container_add(GTK_CONTAINER(RedoAllButton),Icon);

    gtk_box_pack_start(GTK_BOX(HBox),RedoButton,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(HBox),RedoAllButton,TRUE,TRUE,0);

    gtk_signal_connect(GTK_OBJECT(RedoButton),"clicked",GTK_SIGNAL_FUNC(Action_Redo_File),NULL);
    gtk_signal_connect(GTK_OBJECT(RedoAllButton),"clicked",GTK_SIGNAL_FUNC(Action_Redo_All_Files),NULL);

    /*
     * Buttons: Apply/Save.
     */
    HBox = gtk_hbox_new(FALSE,3);
    gtk_box_pack_start(GTK_BOX(MainVBox),HBox,FALSE,FALSE,0);

    ApplyButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(save_xpm);
    gtk_container_add(GTK_CONTAINER(ApplyButton),Icon);

    ApplyAllButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(save_all_xpm);
    gtk_container_add(GTK_CONTAINER(ApplyAllButton),Icon);

    gtk_box_pack_start(GTK_BOX(HBox),ApplyButton,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(HBox),ApplyAllButton,TRUE,TRUE,0);

    gtk_signal_connect(GTK_OBJECT(ApplyButton),"clicked",GTK_SIGNAL_FUNC(Action_Save_File),NULL);
    gtk_signal_connect(GTK_OBJECT(ApplyAllButton),"clicked",GTK_SIGNAL_FUNC(Action_Save_All_Files),NULL);

    /*
     * Button: Quit.
     */
    QuitButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(exit_xpm);
    gtk_container_add(GTK_CONTAINER(QuitButton),Icon);
    gtk_box_pack_start(GTK_BOX(MainVBox),QuitButton,FALSE,FALSE,0);
    gtk_signal_connect(GTK_OBJECT(QuitButton),"clicked",GTK_SIGNAL_FUNC(Quit_MainWindow),NULL);


    /* Tips */
    Tips = gtk_tooltips_new_1();
    gtk_tooltips_set_delay(Tips,1000);
    gtk_tooltips_set_tip(Tips,SpinButton,_("Select position of file in list, and "
        "press Enter key or GO button"),NULL);
    gtk_tooltips_set_tip(Tips,GoButton,       _("Go to this file"),               NULL);
    gtk_tooltips_set_tip(Tips,FirstButton,    _("Select first file"),             NULL);
    gtk_tooltips_set_tip(Tips,PrevButton,     _("Select previous file"),          NULL);
    gtk_tooltips_set_tip(Tips,NextButton,     _("Select next file"),              NULL);
    gtk_tooltips_set_tip(Tips,LastButton,     _("Select last file"),              NULL);
    gtk_tooltips_set_tip(Tips,ScanButton,     _("Scan this file"),                NULL);
    gtk_tooltips_set_tip(Tips,ScanAllButton,  _("Scan all files"),                NULL);
    gtk_tooltips_set_tip(Tips,RemoveButton,   _("Remove tag of this file"),       NULL);
    gtk_tooltips_set_tip(Tips,RemoveAllButton,_("Remove all tags"),               NULL);
    gtk_tooltips_set_tip(Tips,UndoButton,     _("Undo last changes of this file"),NULL);
    gtk_tooltips_set_tip(Tips,UndoAllButton,  _("Undo last changes"),             NULL);
    gtk_tooltips_set_tip(Tips,RedoButton,     _("Redo last changes of this file"),NULL);
    gtk_tooltips_set_tip(Tips,RedoAllButton,  _("Redo last changes"),             NULL);
    gtk_tooltips_set_tip(Tips,ApplyButton,    _("Save this file"),                NULL);
    gtk_tooltips_set_tip(Tips,ApplyAllButton, _("Save all files"),                NULL);
    gtk_tooltips_set_tip(Tips,QuitButton,     _("Exit EasyTAG!"),                 NULL);

    if (SHOW_COMMAND_BUTTONS)    gtk_widget_show_all(Frame);
    else                         gtk_widget_hide_all(Frame);
    /* Select buttons to display or to hide */
    Update_Button_Visibility_Into_Command_Bar();

    return Frame;
}


/*
 * Actions when mini buttons are pressed: apply the field to all others files
 */
void Mini_Button_Clicked (GtkObject *object)
{
    GList *save_position = NULL;
    gchar *string_to_set = NULL;
    gchar *string_to_set1 = NULL;
    gchar *msg = NULL;


    if (!ETFileList) return;

    /* Save the current position in the list */
    save_position = ETFileList;
    /* Save the current displayed data */
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    if (object==GTK_OBJECT(TitleMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(TitleEntry),0,-1); // The string to apply to all other files
        ETFileList = g_list_first(ETFileList); // "Rewind" list to the first item
        while (ETFileList)
        {    
            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
            gtk_entry_set_text(GTK_ENTRY(TitleEntry),string_to_set);
            ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
            ETFileList = g_list_next(ETFileList);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("All files tagged with title '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed title from all files."));
    }
    else if (object==GTK_OBJECT(ArtistMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(ArtistEntry),0,-1);
        ETFileList = g_list_first(ETFileList);
        while (ETFileList)
        {    
            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
            gtk_entry_set_text(GTK_ENTRY(ArtistEntry),string_to_set);
            ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
            ETFileList = g_list_next(ETFileList);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("All files tagged with artist '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed artist from all files."));
    }
    else if (object==GTK_OBJECT(AlbumMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(AlbumEntry),0,-1);
        ETFileList = g_list_first(ETFileList);
        while (ETFileList)
        {    
            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
            gtk_entry_set_text(GTK_ENTRY(AlbumEntry),string_to_set);
            ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
            ETFileList = g_list_next(ETFileList);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("All files tagged with album '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed album name from all files."));
    }
    else if (object==GTK_OBJECT(YearMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(YearEntry),0,-1);
        ETFileList = g_list_first(ETFileList);
        while (ETFileList)
        {    
            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
            gtk_entry_set_text(GTK_ENTRY(YearEntry),string_to_set);
            ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
            ETFileList = g_list_next(ETFileList);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("All files tagged with year '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed year from all files."));
    }
    else if (object==GTK_OBJECT(TrackMButton))
    {
        /* Used of Track and Total Track values */
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(TrackEntry)->entry),0,-1);
        string_to_set1 = gtk_editable_get_chars(GTK_EDITABLE(TrackTotalEntry),0,-1);
        ETFileList = g_list_first(ETFileList);
        while (ETFileList)
        {    
            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
            gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),string_to_set);
            gtk_entry_set_text(GTK_ENTRY(TrackTotalEntry),string_to_set1);
            ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
            ETFileList = g_list_next(ETFileList);
        }

        if ( string_to_set != NULL && strlen(string_to_set)>0 ) //&& atoi(string_to_set)>0 )
        {
            if ( string_to_set1 != NULL && strlen(string_to_set1)>0 ) //&& atoi(string_to_set1)>0 )
            {
                msg = g_strdup_printf(_("All files tagged with track '%s/%s'."),string_to_set,string_to_set1);
            }else
            {
                msg = g_strdup_printf(_("All files tagged with track '%s'."),string_to_set);
            }
        }else
        {
            msg = g_strdup(_("Removed track number from all files."));
        }
    }
    /* This part doesn't set the same track number to all files, but sequence the tracks */
    else if (object==GTK_OBJECT(TrackMButtonSequence))
    {
        gchar *path = NULL;
        gchar *path1 = NULL;
        gint i = 0;

        /* FIX ME!: see to fill also the Total Track (it's a good idea?) */
        ETFileList = g_list_first(ETFileList);
        while (ETFileList)
        {       
            File_Name *FileNameCur = (File_Name *)((ET_File *)ETFileList->data)->FileNameCur->data;
            // Restart counter when new directory
            if (path1) g_free(path1);
            path1 = g_dirname(FileNameCur->value);
            if ( path && path1 && strcmp(path,path1)!=0 )
                i = 0;
            if (NUMBER_TRACK_FORMATED)
                string_to_set = g_strdup_printf("%.2d",++i);
            else
                string_to_set = g_strdup_printf("%d",++i);

            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
            gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),string_to_set);
            ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
            
            if (string_to_set) {g_free(string_to_set); string_to_set = NULL;}
            if (path) g_free(path);
            path = g_strdup(path1);

            ETFileList = g_list_next(ETFileList);
        }
        if (path)  g_free(path);
        if (path1) g_free(path1);
        msg = g_strdup_printf(_("All %d tracks numbered sequentially."), ETFileList_Length);
    }
    else if (object==GTK_OBJECT(GenreMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(GenreEntry)->entry),0,-1);
        ETFileList = g_list_first(ETFileList);
        while (ETFileList)
        {    
            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
            gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),string_to_set);
            ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
            ETFileList = g_list_next(ETFileList);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("All files tagged with genre '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed genre from all files."));
    }
    else if (object==GTK_OBJECT(CommentMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(CommentEntry),0,-1);
        ETFileList = g_list_first(ETFileList);
        while (ETFileList)
        {    
            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
            gtk_entry_set_text(GTK_ENTRY(CommentEntry),string_to_set);
            ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
            ETFileList = g_list_next(ETFileList);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("All files tagged with comment '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed comment from all files."));
    }

    /* Return to the saved position in the list */
    ETFileList = save_position;
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data); // Needed when sequencing tracks
    if (msg)
    {
        g_print("%s\n",msg);
        Statusbar_Message(msg,TRUE);
        g_free(msg);
    }
    if (string_to_set)  g_free(string_to_set);
    if (string_to_set1) g_free(string_to_set1);

    /* To update state of Undo button */
    Update_Command_Buttons_Sensivity();
}


/*
 * Action when First button is selected
 */
void Action_Select_First_File (void)
{
    if (!ETFileList || !ETFileList->prev)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the first item of the list */
    ET_File_List_First();

    Browser_List_Select_File(ETFileList_Index);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(SpinButton),(gfloat)ETFileList_Index);

    if (GTK_WIDGET_HAS_FOCUS(FirstButton)) 
        gtk_window_set_focus(GTK_WINDOW(MainWindow),NextButton);

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
}


/*
 * Action when Prev button is selected
 */
void Action_Select_Prev_File (void)
{
    if (!ETFileList || !ETFileList->prev)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the prev item of the list */
    ET_File_List_Previous();

    Browser_List_Select_File(ETFileList_Index);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(SpinButton),(gfloat)ETFileList_Index);

    /* Set focus to the next button only if we reach the first when clicking 
     * the prev button, and not when using keyboard shortcut */
    if (!ETFileList->prev && GTK_WIDGET_HAS_FOCUS(PrevButton)) 
        gtk_window_set_focus(GTK_WINDOW(MainWindow),NextButton);

//    if (!ETFileList->prev)
//        gdk_beep(); // Warm the user

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
}


/*
 * Action when Next button is selected
 */
void Action_Select_Next_File (void)
{
    if (!ETFileList || !ETFileList->next)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the next item of the list */
    ET_File_List_Next();
    
    Browser_List_Select_File(ETFileList_Index);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(SpinButton),(gfloat)ETFileList_Index);

    if (!ETFileList->next && GTK_WIDGET_HAS_FOCUS(NextButton)) 
        gtk_window_set_focus(GTK_WINDOW(MainWindow),PrevButton);
    
//    if (!ETFileList->next)
//        gdk_beep(); // Warm the user

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
}


/*
 * Action when Last button is selected
 */
void Action_Select_Last_File (void)
{
    if (!ETFileList || !ETFileList->next)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the last item of the list */
    ET_File_List_Last();

    Browser_List_Select_File(ETFileList_Index);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(SpinButton),(gfloat)ETFileList_Index);

    if (GTK_WIDGET_HAS_FOCUS(LastButton)) 
        gtk_window_set_focus(GTK_WINDOW(MainWindow),PrevButton);

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
}


/*
 * Select a file in the "main list" using the primary key of each item.
 */
void Action_Select_Nth_File_By_Key (gulong key)
{
    if (!ETFileList)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Search item with the same key in the list */
    ET_File_List_Nth_By_Key(key);
    Browser_List_Select_File(ETFileList_Index);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
}

/*
 * Select the nth file in list.
 * Note: element 'n' of the list, has position 'n-1' in the list.
 */
void Action_Select_Nth_File_By_Position (gulong num_item)
{
    if (!ETFileList || !num_item)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the nth item of the list */
    ETFileList = g_list_first(ETFileList);
    ETFileList = g_list_nth(ETFileList,num_item-1);
    ETFileList_Index = num_item;
    Browser_List_Select_File(ETFileList_Index);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
}


/*
 * Action when Scan button is selected
 */
void Action_Scan_File (void)
{
    if (!ETFileList) return;

    /* If Scan window is closed, then open it. If it's opened, scan tag */
    if (ScannerWindow)
    {
        Scan_Select_Mode_And_Run_Scanner();
    }else
    {
        Open_ScannerWindow();
        Statusbar_Message(_("Press the Scan button again to run the scanner."),TRUE);
    }
}


void Action_Scan_All_Files (void)
{
    gint progress_bar_index;

    if (!ETFileList) return;

    /* Select the first file */
    Action_Select_First_File();

    /* Initialize status bar */
    gtk_progress_configure(GTK_PROGRESS(ProgressBar),0,0,ETFileList_Length);
    progress_bar_index = 0;

    /* Check if scanner window is opened */
    if(!ScannerWindow)
    {
        Open_ScannerWindow();
        Statusbar_Message(_("Select Mode and Mask, and redo the same action"),TRUE);
        return;
    }

    while (TRUE)
    {
        Action_Scan_File();

        gtk_progress_set_value(GTK_PROGRESS(ProgressBar),++progress_bar_index);
        /* Needed to refresh status bar */
        while (gtk_events_pending())
            gtk_main_iteration();

        if (!ETFileList->next) break;
        Action_Select_Next_File();
    }

    /* Return to the first file */
    Action_Select_First_File();

    gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
    Statusbar_Message(_("All tags have been scanned"),TRUE);
}



/*
 * Action when Remove button is selected
 */
void Action_Remove_Tag (void)
{
    if (!ETFileList) return;

    Clear_Tag_Entry_Fields();
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Needed to run 'Id3tag_Display_Tag' to refresh data for undo */

    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
    Statusbar_Message(_("Tag removed"),TRUE);
}


void Action_Remove_All_Tags (void)
{
    gint progress_bar_index;

    if (!ETFileList) return;

    /* Select the first file */
    Action_Select_First_File();

    /* Initialize status bar */
    gtk_progress_configure(GTK_PROGRESS(ProgressBar),0,0,ETFileList_Length);
    progress_bar_index = 0;

    while (TRUE)
    {
        ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
        Action_Remove_Tag();

        gtk_progress_set_value(GTK_PROGRESS(ProgressBar),++progress_bar_index);
        /* Needed to refresh status bar */
        while (gtk_events_pending())
            gtk_main_iteration();

        if (!ETFileList->next) break;
        Action_Select_Next_File();
    }

    /* Return to the first file */
    Action_Select_First_File();

    gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
    Statusbar_Message(_("All tags have been scanned"),TRUE);
}



/*
 * Action when Undo button is selected.
 * Action_Undo_File: Undo the last "individual" changes for the file.
 * Action_Undo_All_File: Undo the last "commun" changes for all files of the list.
 */
gint Action_Undo_File (void)
{
    gint state;
    
    if (!ETFileList) return FALSE;

    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
    state = ET_Undo_File_Data((ET_File *)ETFileList->data);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
 
    Update_Command_Buttons_Sensivity();
    return state;
}


void Action_Undo_All_Files (void)
{
    if (!ETFileList) /*|| !ETHistoryFileList)*/ return;

    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
    ET_Undo_History_File_Data();
    Browser_List_Select_File(ETFileList_Index);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
 
    Update_Command_Buttons_Sensivity();
}



/*
 * Action when Redo button is selected.
 * Action_Redo_File: Redo the last "individual" changes for the file.
 * Action_Redo_All_File: Redo the last "commun" changes for all files of the list.
 */
gint Action_Redo_File (void)
{
    gint state;
    
    if (!ETFileList) return FALSE;

    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
    state = ET_Redo_File_Data((ET_File *)ETFileList->data);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
 
    Update_Command_Buttons_Sensivity();
    return state;
}


void Action_Redo_All_Files (void)
{
    if (!ETFileList) /*|| !ETHistoryFileList)*/ return;

    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
    ET_Redo_History_File_Data();
    Browser_List_Select_File(ETFileList_Index);
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
 
    Update_Command_Buttons_Sensivity();
}




/*
 * Save_All_Tags: Action when Apply button is selected
 * Write tags to files.
 */
void Write_This_Tag (void)
{
    gchar *cur_filename = ((File_Name *)((ET_File *)ETFileList->data)->FileNameCur->data)->value;

    Statusbar_Message(_("Writing tag(s)..."),TRUE);
    if ( ET_Save_File_Tag_To_HD((ET_File *)ETFileList->data)==FALSE )
    {
        gchar *msg;
        GtkWidget *msgbox;

        msg = g_strdup_printf(_("Can't write tag in file '%s'!\n(%s)"),
                    g_basename(cur_filename),g_strerror(errno)); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
    }else
    {
        Statusbar_Message(_("Tag(s) written"),TRUE);
    }
}
void Rename_File (void)
{
    FILE *file;
    gchar *tmpfilename = NULL;
    gchar *cur_filename = ((File_Name *)((ET_File *)ETFileList->data)->FileNameCur->data)->value;
    gchar *new_filename = ((File_Name *)((ET_File *)ETFileList->data)->FileNameNew->data)->value;

    Statusbar_Message(_("Renaming file(s)..."),TRUE);
    /* Check if a file already exists with the new name (detect if it's only a case change) */
    if ( (file=fopen(new_filename,"r+"))!=NULL && strcasecmp(cur_filename,new_filename)!=0 )
    {
        gchar *msg;
        GtkWidget *msgbox;

        fclose(file);
        msg = g_strdup_printf(_("Can't rename file \n'%s'\nbecause the following "
                    "file already exists:\n'%s'"),g_basename(cur_filename),g_basename(new_filename)); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        Statusbar_Message(_("File(s) not renamed..."),TRUE);
        return;
    }

    /* We use two stages to rename file, to avoid problem with some system
     * that doesn't allow to rename the file if only the case has changed. */
    tmpfilename = g_strconcat(cur_filename,".EasyTAG",NULL);
    if ( rename(cur_filename,tmpfilename)!=0 )
    {
        gchar *msg;
        GtkWidget *msgbox;

        /* Renaming file to the temporary filename has failed */
        msg = g_strdup_printf(_("Can't rename file '%s'\n to \n'%s'!\n(%s)"),
                                g_basename(cur_filename),g_basename(new_filename),g_strerror(errno)); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        Statusbar_Message(_("File(s) not renamed..."),TRUE);
        if (tmpfilename) g_free(tmpfilename);
        return;
    }

    /* Now, we rename the file to his final name */
    if ( rename(tmpfilename,new_filename)==0 )
    {
        /* Renaming file has succeeded */
        g_print(_("Renamed file '%s' to '%s'\n"),g_basename(cur_filename),g_basename(new_filename));

        ((ET_File *)ETFileList->data)->FileNameCur = ((ET_File *)ETFileList->data)->FileNameNew;
        /* Now the file was renamed, so mark his state */
        ET_Mark_File_Name_As_Saved((ET_File *)ETFileList->data);

        Statusbar_Message(_("File(s) renamed..."),TRUE);
    }else
    {
        gchar *msg;
        GtkWidget *msgbox;

        /* Renaming file has failed, but we try to set the initial name */
        rename(tmpfilename,cur_filename);

        msg = g_strdup_printf(_("Can't rename file '%s'\n to \n'%s'!\n(%s)"),
                    g_basename(cur_filename),g_basename(new_filename),g_strerror(errno)); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        Statusbar_Message(_("File(s) not renamed..."),TRUE);
    }
    if (tmpfilename) g_free(tmpfilename);
}

/*
 * Saving severals files, "multiple_files": TRUE->show cancel button + check button
 *                        "hide_msgbox"   : TRUE-> don't use the msgbox
 */
gint Save_File (gboolean multiple_files)
{
    gchar *msg = NULL;
    gint stop_loop = 0;
    gchar *cur_filename = ((File_Name *)((ET_File *)ETFileList->data)->FileNameCur->data)->value;
    File_Tag  *FileTag;
    File_Name *FileNameNew;


    if (!ETFileList || !ETFileList->data) return 0;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
    FileTag     = ((ET_File *)ETFileList->data)->FileTag->data;
    FileNameNew = ((ET_File *)ETFileList->data)->FileNameNew->data;

    /*
     * First part: write tag informations (artist, title,...)
     */
    if ( FileTag->saved == FALSE ) // This tag had been already saved ?
    {
        GtkWidget *msgbox = NULL;
        gint button;

        if (CONFIRM_WRITE_TAG && (!multiple_files || (multiple_files & !SF_HideMsgbox_Write_Tag)))
        {
            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
            
            msg = g_strdup_printf(_("Do you want to write the tag of file\n'%s' ?"),g_basename(cur_filename) );
            if (multiple_files)
            {
                msgbox = msg_box_new (_("Write Tag..."),msg,MSG_QUESTION,
                    BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
            }else
            {
                msgbox = msg_box_new (_("Write Tag..."),msg,MSG_QUESTION,BUTTON_YES,BUTTON_NO,0);
                msg_box_hide_check_button(MSG_BOX(msgbox));
            }
            g_free(msg);

            SF_ButtonPressed_Write_Tag = button = msg_box_run(MSG_BOX(msgbox));
            /* When check button in msgbox was activated */
            if (MSG_BOX(msgbox)->check_button_state)
                SF_HideMsgbox_Write_Tag = MSG_BOX(msgbox)->check_button_state;
            gtk_widget_destroy(msgbox);
        }else
        {
            if (multiple_files && SF_HideMsgbox_Write_Tag)
                button = SF_ButtonPressed_Write_Tag;
            else
                button = BUTTON_YES;
        }

        switch(button)
        {
            case BUTTON_YES:
                Write_This_Tag();
                break;
            case BUTTON_NO:
                break;
            case BUTTON_CANCEL:
            case -1:
                stop_loop = -1;
                return stop_loop;
                break;
        }
    }


    /*
     * Second part: rename the file
     */
    if ( FileNameNew->saved == FALSE ) // This filename had been already saved ?
    {
        GtkWidget *msgbox = NULL;
        gint button;

        if (CONFIRM_RENAME_FILE && (!multiple_files || (multiple_files & !SF_HideMsgbox_Rename_File)))
        {
            ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

            msg = g_strdup_printf(_("Do you want to rename the file \n'%s'\nto \n'%s' ?"),
                g_basename(((File_Name *)((ET_File *)ETFileList->data)->FileNameCur->data)->value),
                g_basename(((File_Name *)((ET_File *)ETFileList->data)->FileNameNew->data)->value) );
            if (multiple_files)
            {
                msgbox = msg_box_new (_("Rename File..."),msg,MSG_QUESTION,
                    BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
            }else
            {
                msgbox = msg_box_new (_("Rename File..."),msg,MSG_QUESTION,
                    BUTTON_YES,BUTTON_NO,0);
                msg_box_hide_check_button(MSG_BOX(msgbox));
            }
            g_free(msg);
            SF_ButtonPressed_Rename_File = button = msg_box_run(MSG_BOX(msgbox));
            if (MSG_BOX(msgbox)->check_button_state)
                SF_HideMsgbox_Rename_File = MSG_BOX(msgbox)->check_button_state;
            gtk_widget_destroy(msgbox);
        }else
        {
            if (multiple_files && SF_HideMsgbox_Rename_File)
                button = SF_ButtonPressed_Rename_File;
            else
                button = BUTTON_YES;
        }

        switch(button)
        {
            case BUTTON_YES:
                Rename_File();
                break;
            case BUTTON_NO:
                break;
            case BUTTON_CANCEL:
            case -1:
                stop_loop = -1;
                return stop_loop;
                break;
        }
    }

    /* Refresh file into browser list */
    Browser_List_Refresh_File_In_List((ET_File *)ETFileList->data);

    return 1;
}

/*
 * Action_Save_File_With_Answer: Function to save single/multiple file(s).
 *  - multiple_files = TRUE : when saving files, a msgbox appears with ability 
 *                            to do the same action for all files.
 *  - multiple_files = FALSE : appears only a msgbox to ak confirmation.
 */
gint Action_Save_File_With_Answer (gboolean multiple_files)
{
    gint returned_value;
    
    Disable_Command_Buttons();
    Browser_Area_Set_Sensitive(FALSE);
    Tag_Area_Set_Sensitive(FALSE);
    File_Area_Set_Sensitive(FALSE);

    returned_value = Save_File(multiple_files);

    Browser_Area_Set_Sensitive(TRUE);
    Tag_Area_Set_Sensitive(TRUE);
    File_Area_Set_Sensitive(TRUE);
    Update_Command_Buttons_Sensivity();
    return returned_value;
}
/*
 * Action_Save_File: Function to save a single file.
 */
void Action_Save_File (void)
{
    Action_Save_File_With_Answer(FALSE);
}



gint Action_Save_All_Files_With_Answer (void)
{
    gint progress_bar_index;
    gint saving_answer;
    GList *save_position = NULL;
    gint save_list_index;


    if (!ETFileList) return FALSE;

    /* Save the current position in the list */
    save_position = ETFileList;
    save_list_index = ETFileList_Index;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Rewind the list */
    ET_File_List_First();
    
    /* Initialize status bar */
    gtk_progress_configure(GTK_PROGRESS(ProgressBar),0,0,ETFileList_Length);
    progress_bar_index = 0;

    /* Set to unsensitive all command buttons (except Quit button) */
    Disable_Command_Buttons();
    Browser_Area_Set_Sensitive(FALSE);
    Tag_Area_Set_Sensitive(FALSE);
    File_Area_Set_Sensitive(FALSE);

    /* Show msgbox (if needed) to ask confirmation */
    SF_HideMsgbox_Write_Tag = 0;
    SF_HideMsgbox_Rename_File = 0;
    
    do
    {
        ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

        saving_answer = Save_File(TRUE);
        if (saving_answer == -1)
        {
            /* Stop saving files + reinit progress bar */
            gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
            Statusbar_Message(_("All tags have been saved..."),TRUE);
            /* To update state of command buttons */
            Update_Command_Buttons_Sensivity();
            Browser_Area_Set_Sensitive(TRUE);
            Tag_Area_Set_Sensitive(TRUE);
            File_Area_Set_Sensitive(TRUE);
            
            return -1; /* We stop all actions */
        }

        gtk_progress_set_value(GTK_PROGRESS(ProgressBar),++progress_bar_index);
        /* Needed to refresh status bar */
        while (gtk_events_pending())
            gtk_main_iteration();

    } while(ET_File_List_Next() != NULL);

    /* Return to the saved position in the list */
    ETFileList = save_position;
    ETFileList_Index = save_list_index;
    Browser_List_Select_File(ETFileList_Index);

    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    /* To update state of command buttons */
    Update_Command_Buttons_Sensivity();
    Browser_Area_Set_Sensitive(TRUE);
    Tag_Area_Set_Sensitive(TRUE);
    File_Area_Set_Sensitive(TRUE);

    gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
    Statusbar_Message(_("All tags have been saved..."),TRUE);

    return TRUE;
}

void Action_Save_All_Files (void)
{
    Action_Save_All_Files_With_Answer();
}


void Spinner_Button_Activate (void)
{
    Action_Select_Nth_File_By_Position(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(SpinButton)));
}


/*
 * Delete a file on the hard disk
 * TRUE if deleted, else FALSE
 */
gboolean Delete_File (gchar *filename)
{
    GtkWidget *msgbox = NULL;
    gchar *msg;
    gint button;

    if (!filename) return FALSE;
    
    msg = g_strdup_printf(_("Do you really want to delete definitively the file\n'%s' ?"),g_basename(filename) );
    msgbox = msg_box_new (_("Delete File..."),msg,MSG_QUESTION,BUTTON_YES,BUTTON_NO,0);
    msg_box_hide_check_button(MSG_BOX(msgbox));
    g_free(msg);

    button = msg_box_run(MSG_BOX(msgbox));
    if (button == BUTTON_YES)
    {
        if (remove(filename)==0)
        {
            msg = g_strdup_printf(_("File '%s' deleted"),g_basename(filename));
            Statusbar_Message(msg,FALSE);
            g_free(msg);
            return TRUE;
        }else
        {
            return FALSE;
        }
    }else
    {
        return FALSE;
    }
}


/*
 * Scans the specified directory: and load files into a list.
 */
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
void Read_Directory (gchar *path)
{
    DIR *dir;
    gchar *msg;
    guint nbrfile = 0;


    if (path == NULL) return;
    
    if ((dir=opendir(path)) == NULL)
    {
        g_print(_("Can't read directory '%s'! (%s)\n"),path,g_strerror(errno));
        return;
    }
    closedir(dir);

    ReadingDirectory = TRUE;    /* A flag to avoid to start an other reading */
    Set_Busy_Cursor();
    Statusbar_Message(_("Prepare reading of directory..."),FALSE);

    /* Initialize file list */
    ET_Free_File_List();
    ET_Free_History_File_List();
    Update_Command_Buttons_Sensivity();

    /* Initialize browser clist */
    Browser_List_Clear();
    
    /* Clear entry boxes  */
    Clear_Tag_Entry_Fields();
    Clear_Header_Fields();
    ETFileList_Index = ETFileList_Length = 0;
    gtk_label_set_text(GTK_LABEL(FileIndex),"0/0:");
    gtk_entry_set_text(GTK_ENTRY(FileEntry),_("Search in progress..."));

    ETFileList_TotalSize = 0;
    ETFileList_TotalDuration = 0;
    progressbar_index = 0;

    /* Open the window to quit recursion (only into recursion mode) */
    gtk_widget_set_sensitive(TBStopButton,BROWSE_SUBDIR);
    if (BROWSE_SUBDIR)
        Open_Quit_Recursion_Function_Window();

    /* Get the number of elements for progressbar */
    nbrfile = Read_Directory_Recursively_Get_Number_Of_File(path,BROWSE_SUBDIR);
    gtk_progress_configure(GTK_PROGRESS(ProgressBar),0,0,nbrfile);

    /* Read the directory recursively */
    msg = g_strdup_printf(_("Reading directory (%d items)..."),nbrfile);
    Statusbar_Message(msg,FALSE);
    g_free(msg);
    /* Search and load files */
    Read_Directory_Recursively(path,BROWSE_SUBDIR);

    /* Close window to quit recursion */
    Destroy_Quit_Recursion_Function_Window();
    Stop_Recursive_File_Search = 0;
    gtk_widget_set_sensitive(TBStopButton,FALSE);

    /* Update sensitivity of buttons and menus */
    Update_Command_Buttons_Sensivity();

    //ET_Debug_Print_File_List(__FILE__,__LINE__);

    if (ETFileList)
    {
        /* Index to display position in list */
        ETFileList_Index = 1;
        ETFileList_Length = ET_File_List_Get_Length();

        /* Spinner button configuration */
        SpinButtonAdjust = (GtkAdjustment *) gtk_adjustment_new(1,1,ETFileList_Length,1,10,10);
        gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(SpinButton),SpinButtonAdjust);
        gtk_spin_button_set_value(GTK_SPIN_BUTTON(SpinButton),1.0);

        /* Load the list attached to the TrackEntry */
        Load_Track_List_To_UI();
        
        /* Load the list of file into the browser clist widget */
        Browser_List_Load_Files();
        
        /* Display the first file */
        ET_File_List_First();
        ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
        Browser_List_Select_File(ETFileList_Index);

        if (BROWSE_SUBDIR)
            msg = g_strdup_printf(_("Found %d file(s) in this directory and subdirectories."),ETFileList_Length);
        else
            msg = g_strdup_printf(_("Found %d file(s) in this directory."),ETFileList_Length);
    }else
    {
        /* Clear entry boxes */
        Clear_Tag_Entry_Fields();
        Clear_Header_Fields();

        SpinButtonAdjust = (GtkAdjustment *) gtk_adjustment_new(0,0,0,0,0,0);
        gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(SpinButton),SpinButtonAdjust);
        gtk_spin_button_set_value(GTK_SPIN_BUTTON(SpinButton),0.0);

        if (BROWSE_SUBDIR)
            msg = g_strdup(_("No file found in this directory and subdirectories!"));
        else
            msg = g_strdup(_("No file found in this directory!"));
        if (FileEntry)
            gtk_entry_set_text(GTK_ENTRY(FileEntry),msg);
        if (FileIndex)
            gtk_label_set_text(GTK_LABEL(FileIndex),"0/0:");
    }
    gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
    Statusbar_Message(msg,TRUE);
    g_free (msg);
    Set_Unbusy_Cursor();
    ReadingDirectory = FALSE;
}



/*
 * Recurse the path to create a list of files. Return the number of files found
 */
guint Read_Directory_Recursively (gchar * path, gint recurse)
{
    DIR *dir;
    struct dirent *dirent;
    struct stat statbuf;
    gchar *filename;
    guint entries = 0;


    if ((dir = opendir(path)) != NULL)
    {
        while ((dirent = readdir(dir)) != NULL)
        {
            if (Stop_Recursive_File_Search == 1)
                return entries;

            if (dirent->d_name[0] != '.')
            {
                if (path[strlen(path)-1]!='/')
                    filename = g_strconcat(path,"/",dirent->d_name,NULL);
                else
                    filename = g_strconcat(path,dirent->d_name,NULL);

                lstat(filename, &statbuf);
                if (S_ISDIR(statbuf.st_mode))
                {
                    if (recurse)
                    {
                        guint i;
                        i = Read_Directory_Recursively(filename,recurse);
                        entries += i;
                    }
                }else if (S_ISREG(statbuf.st_mode) && ET_File_Is_Supported(filename))
                {
                    ETFileList = ET_Add_File_To_File_List(filename);
                    entries++;
                }
                g_free(filename);
                gtk_progress_set_value(GTK_PROGRESS(ProgressBar),++progressbar_index);
                while (gtk_events_pending())
                    gtk_main_iteration();
            }
        }
        closedir(dir);
    }
    return entries;
}


/*
 * Return the number of files into the directory, browsed recursively 
 * or not, just to initialize progressbar.
 */
guint Read_Directory_Recursively_Get_Number_Of_File (gchar * path, gint recurse)
{
    DIR *dir;
    struct dirent *dirent;
    struct stat statbuf;
    gchar *filename;
    guint entries = 0;


    if ((dir = opendir(path)) != NULL)
    {
        while ((dirent = readdir(dir)) != NULL)
        {
            if (Stop_Recursive_File_Search == 1)
                return entries;

            if (dirent->d_name[0] != '.')
            {
                if (path[strlen(path)-1]!='/')
                    filename = g_strconcat(path,"/",dirent->d_name,NULL);
                else
                    filename = g_strconcat(path,dirent->d_name,NULL);

                lstat(filename, &statbuf);
                if (S_ISDIR(statbuf.st_mode))
                {
                    if (recurse)
                    {
                        guint i;
                        i = Read_Directory_Recursively_Get_Number_Of_File(filename,recurse);
                        entries += i;
                    }
                }else
                {
                    entries++;
                }
                g_free(filename);
                /* Just to not block X events */
                while (gtk_events_pending())
                    gtk_main_iteration();
            }
        }
        closedir(dir);
    }
    return entries;
}
/*
 * Window with the 'STOP' button to stop recursion when reading directories
 */
#include "../pixmaps/stop.xpm"
void Open_Quit_Recursion_Function_Window (void)
{
    GtkWidget *button;
    GtkWidget *frame;
    gint mw_x, mw_y;
    gint ba_x, ba_y;


    if (QuitRecursionWindow != NULL) 
        return;
    QuitRecursionWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(QuitRecursionWindow),_("Searching..."));
    gtk_window_set_transient_for(GTK_WINDOW(QuitRecursionWindow),GTK_WINDOW(MainWindow));
    //gtk_window_set_policy(GTK_WINDOW(QuitRecursionWindow),FALSE,FALSE,TRUE);
    gtk_signal_connect(GTK_OBJECT(QuitRecursionWindow),"destroy",
        (GtkSignalFunc)Quit_Recursion_Function_Button_Pressed,NULL);
    gtk_signal_connect(GTK_OBJECT(QuitRecursionWindow),"delete_event",
        (GtkSignalFunc)Quit_Recursion_Function_Button_Pressed,NULL);
    gtk_signal_connect(GTK_OBJECT(QuitRecursionWindow),"key_press_event",
        (GtkSignalFunc)Quit_Recursion_Window_Key_Press,NULL);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame),GTK_SHADOW_IN);
    gtk_container_add(GTK_CONTAINER(QuitRecursionWindow),frame);
    gtk_container_set_border_width(GTK_CONTAINER(frame),2);

    /* Button to stop... */
    //button = gtk_button_new_with_label(_("  STOP the search...  "));
    button = Create_Button_With_Icon_And_Label(stop_xpm,_("  STOP the search...  "));
    gtk_container_set_border_width(GTK_CONTAINER(button),8);
    gtk_container_add(GTK_CONTAINER(frame),button);
    gtk_signal_connect(GTK_OBJECT(button),"clicked",
        (GtkSignalFunc)Quit_Recursion_Function_Button_Pressed,NULL);

    gtk_widget_show_all(QuitRecursionWindow);

    /* Note : the window must be shown before to move it */
    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gdk_window_get_position(BrowseArea->window,&ba_x,&ba_y);
    gtk_window_reposition(GTK_WINDOW(QuitRecursionWindow),
        mw_x + ba_x + (BrowseArea->allocation.width/2)  - (QuitRecursionWindow->allocation.width)/2,
        mw_y + ba_y + (BrowseArea->allocation.height/2) - (QuitRecursionWindow->allocation.height)/2);
}
void Destroy_Quit_Recursion_Function_Window (void)
{
    if (QuitRecursionWindow)
    {
        gtk_widget_destroy(QuitRecursionWindow);
        QuitRecursionWindow = (GtkWidget *)NULL;
        /*Statusbar_Message(_("Recursive file search interrupted."),FALSE);*/
    }
}
void Quit_Recursion_Function_Button_Pressed (void)
{
    Stop_Recursive_File_Search = 1;
    Destroy_Quit_Recursion_Function_Window();
}
void Quit_Recursion_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:    
                Destroy_Quit_Recursion_Function_Window();
                break;
        }
    }
}


/*
 * To stop the recursive search within directories
 */
void Action_Stop_Recursive_File_Search (void)
{
    Stop_Recursive_File_Search = 1;
    gtk_widget_set_sensitive(TBStopButton,FALSE);
}


/*
 * Update_Command_Buttons_Sensivity: Set to sensitive/unsensitive the state of each button into
 * the commands area and menu items in function of state of the "main list".
 */
void item_factory_set_sensitive (guint action, gboolean sensitive)
{
    GtkWidget *widget;

    widget = gtk_item_factory_get_widget_by_action(ItemFactory,action);
    if (widget) gtk_widget_set_sensitive(widget,sensitive);
}

void Update_Command_Buttons_Sensivity (void)
{
    if (!ETFileList)
    {
        /* No file found */
        
        /* Command buttons */
        gtk_widget_set_sensitive(SpinButton,FALSE);
        gtk_widget_set_sensitive(GoButton,FALSE);
        gtk_widget_set_sensitive(FirstButton,FALSE);
        gtk_widget_set_sensitive(PrevButton,FALSE);
        gtk_widget_set_sensitive(NextButton,FALSE);
        gtk_widget_set_sensitive(LastButton,FALSE);
        gtk_widget_set_sensitive(ScanButton,FALSE);
        gtk_widget_set_sensitive(RemoveButton,FALSE);
        gtk_widget_set_sensitive(UndoButton,FALSE);
        gtk_widget_set_sensitive(RedoButton,FALSE);
        gtk_widget_set_sensitive(ApplyButton,FALSE);
        gtk_widget_set_sensitive(ScanAllButton,FALSE);
        gtk_widget_set_sensitive(RemoveAllButton,FALSE);
        gtk_widget_set_sensitive(UndoAllButton,FALSE);
        gtk_widget_set_sensitive(RedoAllButton,FALSE);
        gtk_widget_set_sensitive(ApplyAllButton,FALSE);
        /* Tool bar buttons */
        gtk_widget_set_sensitive(TBFirstButton,FALSE);
        gtk_widget_set_sensitive(TBPrevButton,FALSE);
        gtk_widget_set_sensitive(TBNextButton,FALSE);
        gtk_widget_set_sensitive(TBLastButton,FALSE);
        gtk_widget_set_sensitive(TBScanButton,FALSE);
        gtk_widget_set_sensitive(TBRemoveButton,FALSE);
        gtk_widget_set_sensitive(TBUndoButton,FALSE);
        gtk_widget_set_sensitive(TBRedoButton,FALSE);
        gtk_widget_set_sensitive(TBApplyButton,FALSE);
        gtk_widget_set_sensitive(TBScanAllButton,FALSE);
        gtk_widget_set_sensitive(TBRemoveAllButton,FALSE);
        gtk_widget_set_sensitive(TBUndoAllButton,FALSE);
        gtk_widget_set_sensitive(TBApplyAllButton,FALSE);
        gtk_widget_set_sensitive(TBStopButton,FALSE);
        /* Commands into menu */
        item_factory_set_sensitive(SORT_ASCENDING_FILENAME,FALSE);
        item_factory_set_sensitive(SORT_DESCENDING_FILENAME,FALSE);
        item_factory_set_sensitive(SORT_ASCENDING_CREATION_DATE,FALSE);
        item_factory_set_sensitive(SORT_DESCENDING_CREATION_DATE,FALSE);
        item_factory_set_sensitive(SORT_ASCENDING_TRACK_NUMBER,FALSE);
        item_factory_set_sensitive(SORT_DESCENDING_TRACK_NUMBER,FALSE);
        item_factory_set_sensitive(PREV,FALSE);
        item_factory_set_sensitive(NEXT,FALSE);
        item_factory_set_sensitive(FIRST,FALSE);
        item_factory_set_sensitive(LAST,FALSE);
        item_factory_set_sensitive(SCAN,FALSE);
        item_factory_set_sensitive(REMOVE,FALSE);
        item_factory_set_sensitive(UNDO,FALSE);
        item_factory_set_sensitive(REDO,FALSE);
        item_factory_set_sensitive(SAVE,FALSE);
        item_factory_set_sensitive(SCAN_ALL,FALSE);
        item_factory_set_sensitive(REMOVE_ALL,FALSE);
        item_factory_set_sensitive(UNDO_ALL,FALSE);
        item_factory_set_sensitive(REDO_ALL,FALSE);
        item_factory_set_sensitive(SAVE_ALL,FALSE);
        item_factory_set_sensitive(WRITE_PLAYLIST,FALSE);
        item_factory_set_sensitive(RUN_MP3_PLAYER,FALSE);
        /* File and Tag frames */
        gtk_widget_set_sensitive(GTK_BIN(FileArea)->child,FALSE);
        gtk_widget_set_sensitive(GTK_BIN(TagArea)->child,FALSE);
        return;
    }else
    {
        /* File and Tag frames */
        gtk_widget_set_sensitive(GTK_BIN(FileArea)->child,TRUE);
        gtk_widget_set_sensitive(GTK_BIN(TagArea)->child,TRUE);
        /* Command buttons */
        gtk_widget_set_sensitive(SpinButton,TRUE);
        gtk_widget_set_sensitive(GoButton,TRUE);
        gtk_widget_set_sensitive(ScanButton,TRUE);
        gtk_widget_set_sensitive(RemoveButton,TRUE);
        gtk_widget_set_sensitive(ScanAllButton,TRUE);
        gtk_widget_set_sensitive(RemoveAllButton,TRUE);
        gtk_widget_set_sensitive(ApplyAllButton,TRUE);
        /* Tool bar buttons */
        gtk_widget_set_sensitive(TBScanButton,TRUE);
        gtk_widget_set_sensitive(TBRemoveButton,TRUE);
        gtk_widget_set_sensitive(TBScanAllButton,TRUE);
        gtk_widget_set_sensitive(TBRemoveAllButton,TRUE);
        gtk_widget_set_sensitive(TBApplyAllButton,TRUE);
        gtk_widget_set_sensitive(TBStopButton,FALSE);
        /* Commands into menu */
        item_factory_set_sensitive(SORT_ASCENDING_FILENAME,TRUE);
        item_factory_set_sensitive(SORT_DESCENDING_FILENAME,TRUE);
        item_factory_set_sensitive(SORT_ASCENDING_CREATION_DATE,TRUE);
        item_factory_set_sensitive(SORT_DESCENDING_CREATION_DATE,TRUE);
        item_factory_set_sensitive(SORT_ASCENDING_TRACK_NUMBER,TRUE);
        item_factory_set_sensitive(SORT_DESCENDING_TRACK_NUMBER,TRUE);
        item_factory_set_sensitive(SCAN,TRUE);
        item_factory_set_sensitive(REMOVE,TRUE);
        item_factory_set_sensitive(SCAN_ALL,TRUE);
        item_factory_set_sensitive(REMOVE_ALL,TRUE);
        item_factory_set_sensitive(WRITE_PLAYLIST,TRUE);
        item_factory_set_sensitive(RUN_MP3_PLAYER,TRUE);
        /* Enable undo commands if there are undo data for the current file */
        if (ET_File_Data_Has_Undo_Data((ET_File *)ETFileList->data))
        {
            gtk_widget_set_sensitive(UndoButton,TRUE);
            gtk_widget_set_sensitive(TBUndoButton,TRUE);
            item_factory_set_sensitive(UNDO,TRUE);
        }else
        {
            gtk_widget_set_sensitive(UndoButton,FALSE);
            gtk_widget_set_sensitive(TBUndoButton,FALSE);
            item_factory_set_sensitive(UNDO,FALSE);
        }
        /* Enable redo commands if there are redo data for the current file */
        if (ET_File_Data_Has_Redo_Data((ET_File *)ETFileList->data))
        {
            gtk_widget_set_sensitive(RedoButton,TRUE);
            gtk_widget_set_sensitive(TBRedoButton,TRUE);
            item_factory_set_sensitive(REDO,TRUE);
        }else
        {
            gtk_widget_set_sensitive(RedoButton,FALSE);
            gtk_widget_set_sensitive(TBRedoButton,FALSE);
            item_factory_set_sensitive(REDO,FALSE);
        }
        /* Enable undo command if there are data into main undo list */
        if (ET_History_File_List_Has_Undo_Data())
        {
            gtk_widget_set_sensitive(UndoAllButton,TRUE);
            gtk_widget_set_sensitive(TBUndoAllButton,TRUE);
            item_factory_set_sensitive(UNDO_ALL,TRUE);
        }else
        {
            gtk_widget_set_sensitive(UndoAllButton,FALSE);
            gtk_widget_set_sensitive(TBUndoAllButton,FALSE);
            item_factory_set_sensitive(UNDO_ALL,FALSE);
        }
        /* Enable redo commands if there are data into main redo list */
        if (ET_History_File_List_Has_Redo_Data())
        {
            gtk_widget_set_sensitive(RedoAllButton,TRUE);
            gtk_widget_set_sensitive(TBRedoAllButton,TRUE);
            item_factory_set_sensitive(REDO_ALL,TRUE);
        }else
        {
            gtk_widget_set_sensitive(RedoAllButton,FALSE);
            gtk_widget_set_sensitive(TBRedoAllButton,FALSE);
            item_factory_set_sensitive(REDO_ALL,FALSE);
        }
        /* Enable save file command if file has been changed */
        // Desactivated because problem with only one file in the list
        //if (ET_Check_If_File_Is_Saved((ET_File *)ETFileList->data))
        //{
        //    gtk_widget_set_sensitive(ApplyButton,FALSE);
        //    gtk_widget_set_sensitive(TBApplyButton,FALSE);
        //    item_factory_set_sensitive(SAVE,FALSE);
        //}else
        //{
            gtk_widget_set_sensitive(ApplyButton,TRUE);
            gtk_widget_set_sensitive(TBApplyButton,TRUE);
            item_factory_set_sensitive(SAVE,TRUE);
        //}
        /* Enable save all files command if there is at least one modified file in the list */
        //if (ET_Check_If_All_Files_Are_Saved())
        //{
        //    gtk_widget_set_sensitive(ApplyAllButton,FALSE);
        //    gtk_widget_set_sensitive(TBApplyAllButton,FALSE);
        //    item_factory_set_sensitive(SAVE_ALL,FALSE);
        //}else
        //{
            gtk_widget_set_sensitive(ApplyAllButton,TRUE);
            gtk_widget_set_sensitive(TBApplyAllButton,TRUE);
            item_factory_set_sensitive(SAVE_ALL,TRUE);
        //}
    }

    if (!ETFileList->prev)    /* Is it the 1rst item ? */
    {
        gtk_widget_set_sensitive(FirstButton,FALSE);
        gtk_widget_set_sensitive(PrevButton,FALSE);
        gtk_widget_set_sensitive(TBFirstButton,FALSE);
        gtk_widget_set_sensitive(TBPrevButton,FALSE);
        item_factory_set_sensitive(PREV,FALSE);
        item_factory_set_sensitive(FIRST,FALSE);
    }else
    {
        gtk_widget_set_sensitive(FirstButton,TRUE);
        gtk_widget_set_sensitive(PrevButton,TRUE);
        gtk_widget_set_sensitive(TBFirstButton,TRUE);
        gtk_widget_set_sensitive(TBPrevButton,TRUE);
        item_factory_set_sensitive(PREV,TRUE);
        item_factory_set_sensitive(FIRST,TRUE);
    }
    if (!ETFileList->next)    /* Is it the last item ? */
    {
        gtk_widget_set_sensitive(NextButton,FALSE);
        gtk_widget_set_sensitive(LastButton,FALSE);
        gtk_widget_set_sensitive(TBNextButton,FALSE);
        gtk_widget_set_sensitive(TBLastButton,FALSE);
        item_factory_set_sensitive(NEXT,FALSE);
        item_factory_set_sensitive(LAST,FALSE);
    }else
    {
        gtk_widget_set_sensitive(NextButton,TRUE);
        gtk_widget_set_sensitive(LastButton,TRUE);
        gtk_widget_set_sensitive(TBNextButton,TRUE);
        gtk_widget_set_sensitive(TBLastButton,TRUE);
        item_factory_set_sensitive(NEXT,TRUE);
        item_factory_set_sensitive(LAST,TRUE);
    }
}

/*
 * Just to disable buttons when we are saving files (do not disable Quit button)
 */
void Disable_Command_Buttons (void)
{
    gtk_widget_set_sensitive(SpinButton,FALSE);
    gtk_widget_set_sensitive(GoButton,FALSE);
    gtk_widget_set_sensitive(FirstButton,FALSE);
    gtk_widget_set_sensitive(PrevButton,FALSE);
    gtk_widget_set_sensitive(NextButton,FALSE);
    gtk_widget_set_sensitive(LastButton,FALSE);
    gtk_widget_set_sensitive(ScanButton,FALSE);
    gtk_widget_set_sensitive(RemoveButton,FALSE);
    gtk_widget_set_sensitive(UndoButton,FALSE);
    gtk_widget_set_sensitive(RedoButton,FALSE);
    gtk_widget_set_sensitive(ApplyButton,FALSE);
    gtk_widget_set_sensitive(ScanAllButton,FALSE);
    gtk_widget_set_sensitive(RemoveAllButton,FALSE);
    gtk_widget_set_sensitive(UndoAllButton,FALSE);
    gtk_widget_set_sensitive(RedoAllButton,FALSE);
    gtk_widget_set_sensitive(ApplyAllButton,FALSE);
    gtk_widget_set_sensitive(TBFirstButton,FALSE);
    gtk_widget_set_sensitive(TBPrevButton,FALSE);
    gtk_widget_set_sensitive(TBNextButton,FALSE);
    gtk_widget_set_sensitive(TBLastButton,FALSE);
    gtk_widget_set_sensitive(TBScanButton,FALSE);
    gtk_widget_set_sensitive(TBRemoveButton,FALSE);
    gtk_widget_set_sensitive(TBUndoButton,FALSE);
    gtk_widget_set_sensitive(TBApplyButton,FALSE);
    gtk_widget_set_sensitive(TBScanAllButton,FALSE);
    gtk_widget_set_sensitive(TBRemoveAllButton,FALSE);
    gtk_widget_set_sensitive(TBUndoAllButton,FALSE);
    gtk_widget_set_sensitive(TBApplyAllButton,FALSE);
    item_factory_set_sensitive(PREV,FALSE);
    item_factory_set_sensitive(NEXT,FALSE);
    item_factory_set_sensitive(FIRST,FALSE);
    item_factory_set_sensitive(LAST,FALSE);
    item_factory_set_sensitive(SCAN,FALSE);
    item_factory_set_sensitive(REMOVE,FALSE);
    item_factory_set_sensitive(UNDO,FALSE);
    item_factory_set_sensitive(REDO,FALSE);
    item_factory_set_sensitive(SAVE,FALSE);
    item_factory_set_sensitive(SCAN_ALL,FALSE);
    item_factory_set_sensitive(REMOVE_ALL,FALSE);
    item_factory_set_sensitive(UNDO_ALL,FALSE);
    item_factory_set_sensitive(REDO_ALL,FALSE);
    item_factory_set_sensitive(SAVE_ALL,FALSE);
}

/*
 * Disable (FALSE) / Enable (TRUE) all user widgets in the tag area
 */
void Tag_Area_Set_Sensitive (gboolean activate)
{
    /* TAG Area */
    gtk_widget_set_sensitive(GTK_WIDGET(TitleEntry),     activate);
    gtk_widget_set_sensitive(GTK_WIDGET(ArtistEntry),    activate);
    gtk_widget_set_sensitive(GTK_WIDGET(AlbumEntry),     activate);
    gtk_widget_set_sensitive(GTK_WIDGET(YearEntry),      activate);
    gtk_widget_set_sensitive(GTK_WIDGET(TrackEntry),     activate);
    gtk_widget_set_sensitive(GTK_WIDGET(TrackTotalEntry),activate);
    gtk_widget_set_sensitive(GTK_WIDGET(CommentEntry),   activate);
    gtk_widget_set_sensitive(GTK_WIDGET(GenreEntry),     activate);
    /* Mini buttons */
    gtk_widget_set_sensitive(GTK_WIDGET(TitleMButton),  activate);
    gtk_widget_set_sensitive(GTK_WIDGET(ArtistMButton), activate);
    gtk_widget_set_sensitive(GTK_WIDGET(AlbumMButton),  activate);
    gtk_widget_set_sensitive(GTK_WIDGET(YearMButton),   activate);
    gtk_widget_set_sensitive(GTK_WIDGET(TrackMButton),  activate);
    gtk_widget_set_sensitive(GTK_WIDGET(TrackMButtonSequence),activate);
    gtk_widget_set_sensitive(GTK_WIDGET(CommentMButton),activate);
    gtk_widget_set_sensitive(GTK_WIDGET(GenreMButton),  activate);
}

/*
 * Disable (FALSE) / Enable (TRUE) all user widgets in the file area
 */
void File_Area_Set_Sensitive (gboolean activate)
{
    gtk_widget_set_sensitive(GTK_WIDGET(FileEntry),activate);
}

/*
 * Clear the entries of tag
 */
void Clear_Tag_Entry_Fields (void)
{
    if (!TitleEntry) return;

    gtk_entry_set_text(GTK_ENTRY(TitleEntry),"");
    gtk_entry_set_text(GTK_ENTRY(ArtistEntry),"");
    gtk_entry_set_text(GTK_ENTRY(AlbumEntry),"");
    gtk_entry_set_text(GTK_ENTRY(YearEntry),"");
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),"");
    gtk_entry_set_text(GTK_ENTRY(TrackTotalEntry),"");
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),"");
    gtk_entry_set_text(GTK_ENTRY(CommentEntry),"");
}


/*
 * Clear the header informations
 */
void Clear_Header_Fields (void)
{
    if (!VersionValueLabel) return;

    /* Default values are MPs data */
    gtk_label_set_text(GTK_LABEL(VersionLabel),        _("MPEG"));
    gtk_label_set_text(GTK_LABEL(VersionValueLabel),   _("?, Layer ?"));
    gtk_label_set_text(GTK_LABEL(BitrateValueLabel),   _("? kb/s"));
    gtk_label_set_text(GTK_LABEL(SampleRateValueLabel),_("? Hz"));
    gtk_label_set_text(GTK_LABEL(ModeLabel),           _("Mode:"));
    gtk_label_set_text(GTK_LABEL(ModeValueLabel),      _("?"));
    gtk_label_set_text(GTK_LABEL(SizeValueLabel),      _("?"));
    gtk_label_set_text(GTK_LABEL(DurationValueLabel),  _("?"));
}




/*
 * Load the default directory when the user interface is completely displayed
 * to avoid bad visualization effect at startup.
 */
void Init_Load_Default_Dir (void)
{
    ETFileList = (GList *)NULL;
    if (INIT_DIRECTORY)
        Browser_Tree_Select_Dir(INIT_DIRECTORY);
    else
        Browser_Load_Default_Directory();
        //Browser_Tree_Select_Dir(DEFAULT_PATH_TO_MP3);
    gtk_idle_remove(idle_handler_id);
}




/*
 * Entry_Popup_Menu_Handler: show the popup menu when the third mouse button is pressed.
 */
gboolean Entry_Popup_Menu_Handler (GtkMenu *menu, GdkEventButton *event)
{
    if (event && (event->type==GDK_BUTTON_PRESS) && (event->button==3))
    {
        gtk_menu_popup(menu,NULL,NULL,NULL,NULL,event->button,event->time);
        return TRUE;
    }
    return FALSE;
}

void Convert_P20_And_Undescore_Into_Spaces (GtkWidget *entry)
{
    if (GTK_IS_ENTRY(entry))
    {
        gchar *text = gtk_entry_get_text_1(entry);
        Scan_Convert_Underscore_Into_Space(text);
        Scan_Convert_P20_Into_Space(text);
        gtk_entry_set_text(GTK_ENTRY(entry),text);
    }
}
void Convert_Space_Into_Undescore (GtkWidget *entry)
{
    if (GTK_IS_ENTRY(entry))
    {
        gchar *text = gtk_entry_get_text_1(entry);
        Scan_Convert_Space_Into_Undescore(text);
        gtk_entry_set_text(GTK_ENTRY(entry),text);
    }
}
void Convert_All_Uppercase (GtkWidget *entry)
{
    if (GTK_IS_ENTRY(entry))
    {
        gchar *text = gtk_entry_get_text_1(entry);
        //g_print("s: %d\n",GTK_EDITABLE(GTK_ENTRY(entry))->selection_start_pos);
        //g_print("e: %d\n",GTK_EDITABLE(GTK_ENTRY(entry))->selection_end_pos);
        Scan_Process_Fields_All_Uppercase(text);
        gtk_entry_set_text(GTK_ENTRY(entry),text);
    }
}
void Convert_All_Downcase (GtkWidget *entry)
{
    if (GTK_IS_ENTRY(entry))
    {
        gchar *text = gtk_entry_get_text_1(entry);
        Scan_Process_Fields_All_Downcase(text);
        gtk_entry_set_text(GTK_ENTRY(entry),text);
    }
}
void Convert_Letter_Uppercase (GtkWidget *entry)
{
    if (GTK_IS_ENTRY(entry))
    {
        gchar *text = gtk_entry_get_text_1(entry);
        Scan_Process_Fields_Letter_Uppercase(text);
        gtk_entry_set_text(GTK_ENTRY(entry),text);
    }
}
void Convert_First_Letters_Uppercase (GtkWidget *entry)
{
    if (GTK_IS_ENTRY(entry))
    {
        gchar *text = gtk_entry_get_text_1(entry);
        Scan_Process_Fields_First_Letters_Uppercase(text);
        gtk_entry_set_text(GTK_ENTRY(entry),text);
    }
}
void Convert_Remove_Space (GtkWidget *entry)
{
    if (GTK_IS_ENTRY(entry))
    {
        gchar *text = gtk_entry_get_text_1(entry);
        Scan_Process_Fields_Remove_Space(text);
        gtk_entry_set_text(GTK_ENTRY(entry),text);
    }
}
void Convert_Insert_Space (GtkWidget *entry)
{
    if (GTK_IS_ENTRY(entry))
    {
        /* Memory must be allocated cause size of string may increase */
        guint string_len = 2 * strlen(gtk_entry_get_text_1(entry));
        gchar *text = g_malloc(string_len+1);
        strncpy(text,gtk_entry_get_text_1(entry),string_len);
        Scan_Process_Fields_Insert_Space(text);
        gtk_entry_set_text(GTK_ENTRY(entry),text);
        g_free(text);
    }
}
void Convert_Only_One_Space (GtkWidget *entry)
{
    if (GTK_IS_ENTRY(entry))
    {
        gchar *text = gtk_entry_get_text_1(entry);
        Scan_Process_Fields_Keep_One_Space(text);
        gtk_entry_set_text(GTK_ENTRY(entry),text);
    }
}

/*
 * Popup menu attached to all entries of tag + filename + rename combobox.
 * Displayed when pressing the right mouse button and contains functions to process ths strings.
 */
void Attach_Popup_Menu_To_Tag_Entries (GtkEntry *entry)
{
    GtkWidget *PopupMenu;
    GtkWidget *MenuItem;


    PopupMenu = gtk_menu_new();
    gtk_signal_connect_object(GTK_OBJECT(entry),"button_press_event",
        (GtkSignalFunc)Entry_Popup_Menu_Handler,GTK_OBJECT(PopupMenu));

    /* Menu items */
    MenuItem = gtk_menu_item_new_with_label(_("Convert '_' and '%20' to spaces"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_P20_And_Undescore_Into_Spaces,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("Convert ' ' to '_'"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Space_Into_Undescore,GTK_OBJECT(entry));

    /* Separator */
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("All uppercase"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_All_Uppercase,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("All downcase"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_All_Downcase,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("First letter uppercase"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Letter_Uppercase,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("First letter uppercase of each word"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_First_Letters_Uppercase,GTK_OBJECT(entry));

    /* Separator */
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Remove spaces"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Remove_Space,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("Insert space before uppercase letter"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Insert_Space,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("Remove duplicate spaces or underscores"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Only_One_Space,GTK_OBJECT(entry));

    gtk_widget_show_all(PopupMenu);
}



/*
 * Display usage informations
 */
void Display_Usage (void)
{
    fprintf(stderr,
        _("\nUsage: easytag [option] "
          "\n   or: easytag [directory]\n"
          "\n"
          "Option:\n"
          "-------\n"
          "-h, --help        Display this text and exit.\n"
          "-v, --version     Print basic informations and exit.\n"
          "\n"
          "Directory:\n"
          "----------\n"
          "/path_to/files    Use an absolute path to load,\n"
          "., ./             Load the current directory.\n"
          "\n"));
    exit(0);
}



/*
 * Exit the program
 */
void EasyTAG_Exit (void)
{
    g_print(_("EasyTAG: Normal exit.\n"));
    gtk_main_quit();
    exit(0);
}

void Quit_MainWindow_Confirmed (void)
{
    /* Save config if wanted */
    if (SAVE_CONFIG_BEFORE_EXIT)
    {
        Apply_Changes(); /* Can be replaced by Save_Changes() */
        Save_Config_To_File();
    }
    EasyTAG_Exit();
}

void Quit_MainWindow_Save_And_Quit (void)
{
    /* Save modified tags */
    if (Action_Save_All_Files_With_Answer() == -1) return;
    Quit_MainWindow_Confirmed();
}

void Quit_MainWindow (void)
{
    GtkWidget *msgbox;
    gint button;
    GList *list;


    /* If you change the displayed data and quit immediately */
    if (ETFileList)
        ET_Save_File_Data_From_UI((ET_File *)ETFileList->data); // To detect change before exiting

    /* Save combobox history lists before exit */
    list = gtk_object_get_data(GTK_OBJECT(BrowserEntry),"History");
    Save_Path_Entry_List(list);

    /* Exit ? */
    if(CONFIRM_BEFORE_EXIT) 
    {
        if (ET_Check_If_All_Files_Are_Saved() != TRUE)
        {
            /* Some files haven't been saved */
            msgbox = msg_box_new (_("Confirm..."),_("Some files have been modified but not "
                "saved...\nDo you want to save them before exiting the program?"),
                MSG_QUESTION,BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            button = msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);
            switch(button)
            {
                case BUTTON_YES:
                    Quit_MainWindow_Save_And_Quit();
                    break;
                case BUTTON_NO:
                    Quit_MainWindow_Confirmed();
                    break;
                case BUTTON_CANCEL:
                case -1:
                    return;
            }
        } else
        {
            msgbox = msg_box_new (_("Confirm..."),_(" Do you really want to exit the program? "),
                MSG_QUESTION,BUTTON_YES,BUTTON_NO,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            button = msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);
            switch(button)
            {
                case BUTTON_YES:
                    Quit_MainWindow_Confirmed();
                    break;
                case BUTTON_NO:
                case -1:
                    return;
                    break;
            }
        }
    }else if (ET_Check_If_All_Files_Are_Saved() != TRUE)
    {
        /* Some files aren't saved */
        msgbox = msg_box_new (_("Confirm..."),_("Some files have been modified but not "
            "saved...\nDo you want to save them before exiting the program?"),
            MSG_QUESTION,BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        button = msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        switch(button)
        {
            case BUTTON_YES:
                Quit_MainWindow_Save_And_Quit();
                break;
            case BUTTON_NO:
                Quit_MainWindow_Confirmed();
                break;
            case BUTTON_CANCEL:
            case -1:
                return;
        }
    }else
    {
        Quit_MainWindow_Confirmed();
    }
}



