/* browser.c - 2000/04/28 */
/*
 *  EasyTAG - Tag editor for MP3 and OGG files
 *  Copyright (C) 2000  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/* Some parts of this browser are taken from:
 * XMMS - Cross-platform multimedia player
 * Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas,
 * Thomas Nilsson and 4Front Technologies
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdk.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <dirent.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#include "easytag.h"
#include "browser.h"
#include "et_core.h"
#include "msgbox.h"
#include "bar.h"
#include "misc.h"
#include "setting.h"
#include "mylocale.h"

/* Pixmaps */
#include "../pixmaps/opened_folder.xpm"
#include "../pixmaps/closed_folder.xpm"
#include "../pixmaps/closed_folder_locked.xpm"

/* Returns the GList item for the nth row (taken from gtkclist.c) */
#define    ROW_ELEMENT(clist, row) (((row) == (clist)->rows - 1) ? \
                                   (clist)->row_list_end : \
                                   g_list_nth ((clist)->row_list, (row)))


/****************
 * Declarations *
 ****************/
static GdkPixmap *opened_folder_pixmap, *closed_folder_pixmap, *closed_folder_locked_pixmap;
static GdkBitmap *opened_folder_mask,   *closed_folder_mask,   *closed_folder_locked_mask;

GtkWidget *BrowserTree;    // Tree of directories
GtkWidget *BrowserList;    // List of files
GtkWidget *BrowserEntry;
gchar     *BrowserCurrentPath = NULL; // Path selected in the browser area (BrowserEntry or BrowserTree)

GtkCTreeNode *RootNode, *node, *nextnode;
GtkWidget *RenameDirectoryWindow = NULL;
GtkWidget *RunProgramWindow = NULL;



/**************
 * Prototypes *
 **************/

static gint Browser_Tree_Key_Press        (GtkWidget *ctree, GdkEvent *event);
void        Browser_Tree_Set_Node_Visible (GtkCTree *ctree, GtkCTreeNode *node);
void        Browser_Tree_Disable          (void);
void        Browser_Tree_Enable           (void);
void        Browser_Tree_Initialize       (void);
void        Browser_Tree_Node_Selected    (GtkCTree *tree, GtkCTreeNode *node, gint column, gpointer data);

static gint Browser_List_Key_Press       (GtkWidget *clist, GdkEvent *event);
void        Browser_List_Set_Row_Visible (GtkCList *ctree, gint row);
void        Browser_List_Disable         (void);
void        Browser_List_Enable          (void);
void        Browser_List_Row_Selected    (GtkCList *clist, gint row, gint column,
                                          GdkEventButton *event, gpointer data);
void        Browser_List_Delete_File     (void);

void        Browser_Entry_Activated (void);
void        Browser_Entry_Disable   (void);
void        Browser_Entry_Enable    (void);

gchar      *Browser_Get_Current_Path       (void);
void        Browser_Update_Current_Path    (gchar *path);
void        Browser_Load_Home_Directory    (void);
void        Browser_Load_Default_Directory (void);
void        Browser_Reload_Directory       (void);

static gboolean check_for_subdir(gchar *path);
static void destroy_ctb (gpointer data);
static void destroy_clb (gpointer data);

gboolean    Check_For_Access_Permission (gchar *path);
static void expand_cb (GtkWidget *widget, GtkCTreeNode *parent_node);
static void collapse_cb(GtkWidget *ctree, GtkCTreeNode *parent_node);
static int  filetreeent_compare_func(const void *a, const void *b);

/* Pop up menus */
gboolean    Browser_Popup_Menu (GtkMenu *menu, GdkEventButton *event);
GtkWidget  *Create_Browser_Tree_Popup_Menu (GtkCTree *ctree);
GtkWidget  *Create_Browser_List_Popup_Menu (GtkCList *clist);

/* For window to rename a directory */
void        Browser_Open_Rename_Directory_Window (void);
void        Destroy_Rename_Directory_Window (void);
void        Rename_Directory (GtkObject *combobox);
void        Rename_Directory_Window_Key_Press (GtkWidget *window, GdkEvent *event);

/* For window to run a program with the directory */
void        Browser_Open_Run_Program_Window (void);
void        Destroy_Run_Program_Window (void);
void        Run_Program_Window_Key_Press (GtkWidget *window, GdkEvent *event);
void        Run_Program (GtkObject *combobox);



/*************
 * Functions *
 *************/
/*
 * Load home directory
 */
void Browser_Load_Home_Directory (void)
{
    Browser_Tree_Select_Dir(HOME_VARIABLE);
}


/*
 * Load default directory
 */
void Browser_Load_Default_Directory (void)
{
    if (DEFAULT_PATH_TO_MP3 && strlen(DEFAULT_PATH_TO_MP3)>0)
        Browser_Tree_Select_Dir(DEFAULT_PATH_TO_MP3);
    else
        Browser_Tree_Select_Dir(HOME_VARIABLE);
}


/*
 * Get the path from the selected node (row) in the browser
 * Warning: return NULL if no row selected int the tree.
 */
gchar *Browser_Tree_Get_Path_Of_Selected_Node (void)
{
    GList *SelectedNode = NULL;
    DirNode *dirnode = NULL;

    SelectedNode = ((GtkCList*)BrowserTree)->selection;
    if (!SelectedNode) 
        return NULL;
    dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(BrowserTree),SelectedNode->data);

    if (!dirnode) 
        return NULL;
    else
        return dirnode->path;
}


/*
 * Set the 'path' within the variable BrowserCurrentPath.
 */
void Browser_Update_Current_Path (gchar *path)
{
    gchar *tmp_path;

    if (path == NULL) return;

    /* We use a temporary variable to avoid problem when freeing 'BrowserCurrentPath'
     * if the parameter given to this function is also 'BrowserCurrentPath' */
    tmp_path = g_strdup(path);
    if (BrowserCurrentPath!=NULL)
        g_free(BrowserCurrentPath);
    BrowserCurrentPath = g_strdup(tmp_path);    
    g_free(tmp_path);
}


/*
 * Return the current path
 */
gchar *Browser_Get_Current_Path (void)
{
    return BrowserCurrentPath;
}


/*
 * Reload the current directory.
 */
void Browser_Reload_Directory (void)
{
    if (BrowserCurrentPath != NULL)
        Browser_Tree_Select_Dir(BrowserCurrentPath);
}


/*
 * Set the current path (selected node) in browser as default path (within config variable).
 */
void Set_Current_Path_As_Default (void)
{
    if (DEFAULT_PATH_TO_MP3 != NULL)
        g_free(DEFAULT_PATH_TO_MP3);
    DEFAULT_PATH_TO_MP3 = g_strdup(BrowserCurrentPath);
    Statusbar_Message(_("New default path for files selected"),TRUE);
}


/*
 * When you press the key 'enter' in the BrowserEntry to validate the text (browse the directory)
 */
void Browser_Entry_Activated (void)
{
    gchar *path;

    Add_To_Combo_Box_History(GTK_OBJECT(BrowserEntry));
    path = gtk_entry_get_text_1(BrowserEntry);
    Browser_Tree_Select_Dir(path);
}

/*
 * Set a text into the BrowserEntry (and don't activate it)
 */
void Browser_Entry_Set_Text (gchar *text)
{
    if (text && BrowserEntry)
    {
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(BrowserEntry)->entry),text);
        /* Justify to the rigth, text into BrowserEntry */
        gtk_editable_set_position(GTK_EDITABLE(GTK_COMBO(BrowserEntry)->entry),-1);
    }
}


/*
 * Key Press events into browser tree
 */
static gint Browser_Tree_Key_Press (GtkWidget *ctree, GdkEvent *event)
{
    GdkEventKey *kevent;
    GList *SelectedNode;

    if (event && event->type==GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        SelectedNode = ((GtkCList*)ctree)->selection;
        if (!SelectedNode) 
            return FALSE;
        switch(kevent->keyval)
        {
            case GDK_KP_Enter:    /* Enter key in Num Pad */
            case GDK_Return:      /* 'Normal' Enter key */
            case GDK_t:           /* Expand/Collapse node */
            case GDK_T:           /* Expand/Collapse node */
                gtk_ctree_toggle_expansion(GTK_CTREE(ctree),SelectedNode->data);
                return TRUE;
                break;

            case GDK_e:           /* Expand node */
            case GDK_E:           /* Expand node */
                gtk_ctree_expand(GTK_CTREE(ctree),SelectedNode->data);
                return TRUE;
                break;

            case GDK_c:           /* Collapse node */
            case GDK_C:           /* Collapse node */
                gtk_ctree_collapse(GTK_CTREE(ctree),SelectedNode->data);
                return TRUE;
                break;
            //default:
            //    gtk_widget_event(MainWindow,(GdkEvent *)event);
            //    break;
        }
    }
    return FALSE;
}


/*
 * Key Press events into browser list
 */
gboolean Browser_List_Stop_Timer (guint *flag)
{
    *flag = FALSE;
    return FALSE;
}
static gint Browser_List_Key_Press (GtkWidget *clist, GdkEvent *event)
{
    gchar *string;
    static gchar *key_string = NULL;
    static guint BrowserListTimerId = 0;
    static gboolean timer_is_running = FALSE;
    gint row;
    FileRow *filerow = NULL;
    GdkEventKey *kevent;

    kevent = (GdkEventKey *)event;
    if (event && event->type==GDK_KEY_PRESS)
    {
        if ( ((GtkCList*)clist)->selection )
        {
            switch(kevent->keyval)
            {
                case GDK_Delete:
                    Browser_List_Delete_File();
                    return TRUE;
            }
        }
    }

    /*
     * Tries to select file corresponding to the character sequence entered
     */
    if ( strlen(kevent->string)>0 ) // Alphanumeric key?
    {
        // If the timer is running: concatenate the character of the pressed key, else starts a new string
        string = key_string;
        if (timer_is_running)
            key_string = g_strconcat(key_string,kevent->string,NULL);
        else
            key_string = g_strdup(kevent->string);
        if (string) g_free(string);

        // Remove the current timer
        if (BrowserListTimerId)
        {
            gtk_timeout_remove(BrowserListTimerId);
            BrowserListTimerId = 0;
        }
        // Start a new timer of 500ms
        BrowserListTimerId = gtk_timeout_add(500,(GtkFunction)Browser_List_Stop_Timer,&timer_is_running);
        timer_is_running = TRUE;

        // Browse the list keeping the current classification
        for (row=0; row<((GtkCList *)clist)->rows; row++)
        {
            filerow = gtk_clist_get_row_data(GTK_CLIST(clist),row);
            if (strncasecmp(g_basename(filerow->path),key_string,strlen(key_string))==0 )
            {
                gtk_clist_select_row(GTK_CLIST(clist),row,0);
                break;
            }
        }
    }
    return FALSE;
}


/*
 * Collapse (close) tree recursively up to the root node.
 */
void Browser_Tree_Collapse (void)
{
    gtk_ctree_collapse_recursive(GTK_CTREE(BrowserTree),RootNode);
    /* But keep the main directory opened */
    gtk_ctree_expand(GTK_CTREE(BrowserTree),RootNode);
}


/*
 * Set a row (or node) visible in the ctree (by scrolling the tree) if the row is hidden
 */
void Browser_Tree_Set_Node_Visible (GtkCTree *ctree, GtkCTreeNode *node)
{
    GtkVisibility visible;

    visible = gtk_ctree_node_is_visible(ctree,node);
    switch (visible)
    {
        case GTK_VISIBILITY_NONE:
        case GTK_VISIBILITY_PARTIAL:
            gtk_ctree_node_moveto(ctree,node,0,0.1,0.0);
            break;
        case GTK_VISIBILITY_FULL:
            break;
    }
}


/*
 * Set a row visible in the clist (by scrolling the list) if the row is hidden
 */
void Browser_List_Set_Row_Visible (GtkCList *clist, gint row)
{
    GtkVisibility visible;

    visible = gtk_clist_row_is_visible(clist,row);
    switch (visible)
    {
        case GTK_VISIBILITY_NONE:
        case GTK_VISIBILITY_PARTIAL:
            gtk_clist_moveto(clist,row,0,0.5,0.0);
            break;
        case GTK_VISIBILITY_FULL:
            break;
    }
}


/*void Browser_Tree_Node_Selected (GtkCTree *tree, GList *node, gint column, gpointer data)*/
void Browser_Tree_Node_Selected (GtkCTree *tree, GtkCTreeNode *node, gint column, gpointer data)
{
    DirNode *dirnode;
    static int counter = 0;


    /* Open the node */
    if (OPEN_SELECTED_BROWSER_NODE)
        gtk_ctree_expand(GTK_CTREE(tree),node);

    /* Don't start a new reading, if an other one is running...*/
    if (ReadingDirectory == TRUE) return;

    Browser_Tree_Set_Node_Visible(tree,node);
    dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(tree),node);
    if (dirnode->path)
    {
        /* Save the current displayed data */
        if (ETFileList && ETFileList->data)
        {
            ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
            Update_Command_Buttons_Sensivity(); // Not clean to put this here...
        }
        /* Check if all files have been saved before changing the directory */
        if (ET_Check_If_All_Files_Are_Saved() != TRUE)
        {
            GtkWidget *msgbox = NULL;
            gint button;

            msgbox = msg_box_new (_("Confirm..."),_("Some files have been modified but not "
                "saved...\nDo you want to save them before changing the directory?"),
                MSG_QUESTION,BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            button = msg_box_run(MSG_BOX(msgbox));
            switch(button)
            {
                case BUTTON_YES:
                    if (Action_Save_All_Files_With_Answer()==-1) return;
                    break;
                case BUTTON_NO:
                    break;
                case BUTTON_CANCEL:
                case -1:
                    return;
            }
        }
        
        /* Memorize the current path */
         Browser_Update_Current_Path(dirnode->path);
        
        /* Display the selected path into the BrowserEntry */
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(BrowserEntry)->entry),dirnode->path);

        /* Start to read the directory */
        /* The first time 'counter' is equal to zero and if we don't want to load 
         * directory on startup, we skip the 'reading', but we must read it */
        if (LOAD_ON_STARTUP || counter)
            Read_Directory(dirnode->path);
        else
            /* As we don't use the function 'Read_Directory' we must add this function here */
            Update_Command_Buttons_Sensivity();
        counter++;
    }
}



/*
 * Browser_Tree_Select_Dir: Select the directory corresponding to the 'path' in
 * the tree browser, but it doesn't read it!
 * Check if path is correct before selecting it. And returns 1 on success, else 0.
 */
gint Browser_Tree_Select_Dir (gchar *current_path)
{
    struct stat stbuf;
    gchar *currentdir,*pos,*tpath,*tpathnew;
    gboolean leaf;
    gboolean path_status; // Returned value : TRUE if path is correct

    /* If path is invalid: inform the user, but load the first directories 
     * of the full path while parent directories are valid */
    if ( stat(current_path,&stbuf)==-1)
    {
        GtkWidget *msgbox;
        gchar *msg;

        msg = g_strdup_printf(_("The entered path is invalid!:\n%s\n(%s)"),
            current_path,g_strerror(errno));
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        path_status = FALSE;
    }else
    {
        /* The path is correct */
        Browser_Update_Current_Path(current_path);
        path_status = TRUE;
    }

    /* Load current_path */
    if(current_path && *current_path)
    {
        currentdir=g_strdup(current_path);
        tpath=g_strdup("/");
        gtk_ctree_expand(GTK_CTREE(BrowserTree),RootNode);
        pos=(gchar*)strtok(currentdir,"/");
        node=gtk_ctree_find_by_row_data_custom(GTK_CTREE(BrowserTree),NULL,"/",filetreeent_compare_func);
        do
        {
            tpathnew=g_strconcat(tpath,pos,"/",NULL);
            g_free(tpath);
            tpath=tpathnew;
            nextnode=gtk_ctree_find_by_row_data_custom(GTK_CTREE(BrowserTree),node,tpath,
                                                       filetreeent_compare_func);
            if(!nextnode)
                break;
            node=nextnode;
            pos=(gchar*)strtok(NULL,"/");
            gtk_ctree_get_node_info(GTK_CTREE(BrowserTree),node,NULL,NULL,NULL,NULL,NULL,NULL,&leaf,NULL);
            Browser_Tree_Set_Node_Visible(GTK_CTREE(BrowserTree),node);
            if(!leaf && pos)
                gtk_ctree_expand(GTK_CTREE(BrowserTree),node);
            else
            {
                gtk_ctree_select(GTK_CTREE(BrowserTree),node);
                break;
            }
        }
        while(pos);
        g_free(tpath);
        g_free(currentdir);
    }else
    {
        gtk_ctree_expand(GTK_CTREE(BrowserTree),RootNode);
    }

    return path_status;
}


/*
 * Callback to select-row event
 */
void Browser_List_Row_Selected (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
    FileRow *filerow = NULL;
    
    filerow = gtk_clist_get_row_data(GTK_CLIST(clist),row);
    Action_Select_Nth_File_By_Key(filerow->key);
}



/*
 * Load the list of supported files into the browser clist.
 */
static void destroy_clb(gpointer data)
{
    FileRow *row = data;
    // Data in structure 'row' aren't allocated!
    g_free(row);
}
void Browser_List_Load_Files (void)
{
    GdkColor LIGHT_BLUE = {0, 0xddd1, 0xeeec, 0xffff}; // Light blue
    gboolean activate_bg_color = 0;
    GList *save_position;

    if (ETFileList && BrowserList)
    {
        /* Save position of current list */
        save_position = ETFileList;
        
        gtk_clist_freeze(GTK_CLIST(BrowserList));
        gtk_clist_clear(GTK_CLIST(BrowserList));
        ET_File_List_First();
        while (ETFileList)
        {
            FileRow *filerow;
            gchar *row_text[] = {NULL};
            gint row;
            gchar *current_filename = ((File_Name *)((ET_File *)ETFileList->data)->FileNameCur->data)->value;
            
            /* Clist displays the current filename (name on HD) */
            row_text[0] = g_strdup(g_basename(current_filename));
            row = gtk_clist_append(GTK_CLIST(BrowserList),row_text);
            g_free(row_text[0]);

            /* Data attached to each row */
            filerow = g_malloc0(sizeof(FileRow));
            filerow->key  = ((ET_File *)ETFileList->data)->ETFileKey;
            filerow->path = ((File_Name *)((ET_File *)ETFileList->data)->FileNameCur->data)->value;
            filerow->ETFile = (ET_File *)ETFileList->data;
            gtk_clist_set_row_data_full(GTK_CLIST(BrowserList),row,filerow,destroy_clb);

            /* Change background color when changing directory (the first row must not be changed) */
            if (row != 0)
            {
                gchar *dir1;
                gchar *dir2;
                gchar *previous_filename = ((File_Name *)((ET_File *)ETFileList->prev->data)->FileNameCur->data)->value;

                if (strcmp(dir1=g_dirname(previous_filename),
                           dir2=g_dirname(current_filename))!=0 )
                {
                    activate_bg_color = !activate_bg_color;
                }
                if (dir1) g_free(dir1);
                if (dir2) g_free(dir2);
            }
            if (activate_bg_color == TRUE)
                gtk_clist_set_background(GTK_CLIST(BrowserList),row,&LIGHT_BLUE);

            // Set foregroung color of the row
            Browser_List_Set_Foregroung_Row_Color(row);

            if (ET_File_List_Next() == NULL) break;
        }
        /* Return saved position of list */
        ETFileList = save_position;
        gtk_clist_thaw(GTK_CLIST(BrowserList));
    }
}


/*
 * Update state of files in the list after changes (without clearing the clist!)
 *  - Refresh filename is file saved,
 *  - Change color is something change on the file
 */
void Browser_List_Refresh_Whole_List (void)
{
    FileRow *filerow;
    guint row;
    gchar *current_filename;

    if (!ETFileList || !BrowserList || GTK_CLIST(BrowserList)->rows==0)
        return;

    gtk_clist_freeze(GTK_CLIST(BrowserList));
    
    // Browse the full list for changes
    for (row=0;row<GTK_CLIST(BrowserList)->rows;row++)
    {
        // Refresh filename
        filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),row);
        current_filename = g_basename( ((File_Name *)filerow->ETFile->FileNameCur->data)->value );
        gtk_clist_set_text(GTK_CLIST(BrowserList),row,0,current_filename);
        
        // Set foregroung color of the row
        Browser_List_Set_Foregroung_Row_Color(row);
    }

    gtk_clist_thaw(GTK_CLIST(BrowserList));
}


/*
 * Update state of one file in the list after changes (without clearing the clist!)
 *  - Refresh filename is file saved,
 *  - Change color is something change on the file
 */
void Browser_List_Refresh_File_In_List (ET_File *ETFile)
{
    GList   *SelectedRow = NULL;
    FileRow *filerow = NULL;
    gboolean row_found = FALSE;
    guint    row = 0;
    gchar   *current_filename;
    
    if (!ETFileList || !BrowserList || GTK_CLIST(BrowserList)->rows==0)
        return;

    // Found the row of the modified file (when found: row_found=TRUE)
    // 1/3. Try with the selected file in clist
    SelectedRow = ((GtkCList*)BrowserList)->selection;
    if (SelectedRow && SelectedRow->data!=NULL)
    {
        row = (gint)SelectedRow->data;
        filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),row);
        if (filerow && (filerow->ETFile==ETFile))
            row_found = TRUE;
    }

    // 2/3. Fails, often it's just the row before (if we aren't on the first row)
    if (row_found==FALSE && row>0)
    {
        filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),--row);
        if (filerow && (filerow->ETFile==ETFile))
            row_found = TRUE;
    }

    // 3/3. Fails, now we browse the full list to find it
    if (row_found==FALSE)
    {
        for (row=0;row<GTK_CLIST(BrowserList)->rows;row++)
        {
            filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),row);
            if (filerow && (filerow->ETFile==ETFile))
            {
                row_found = TRUE;
                break;
            }
        }
    }

    if (row_found==FALSE) //Error somewhere...
        return;
    
    gtk_clist_freeze(GTK_CLIST(BrowserList));
    current_filename = g_basename( ((File_Name *)filerow->ETFile->FileNameCur->data)->value );
    gtk_clist_set_text(GTK_CLIST(BrowserList),row,0,current_filename);

    // Set foregroung color of the row
    Browser_List_Set_Foregroung_Row_Color(row);

    gtk_clist_thaw(GTK_CLIST(BrowserList));
}


/*
 * Set the foregroung color of the row:
 *  - red for the row, if file or tag have changed.
 *  - default color, if no changes.
 */
void Browser_List_Set_Foregroung_Row_Color (guint row)
{
    GdkColor RED   = {0, 0xffff, 0x0000, 0x0000};
    FileRow *filerow = NULL;

    filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),row);
    if (filerow==NULL || filerow->ETFile==NULL)
        return;

    // Set foreground color to red if filename or tag changed
    if ( ET_Check_If_File_Is_Saved(filerow->ETFile) == FALSE )
    {
        gtk_clist_set_foreground(GTK_CLIST(BrowserList),row,&RED);
    }else
    {
        gtk_clist_set_foreground(GTK_CLIST(BrowserList),row,NULL);
    }
}


void Browser_List_Delete_File (void)
{
    GList   *SelectedRow = NULL;
    FileRow *filerow = NULL;
    guint    row = 0;
    
    if (!ETFileList || !BrowserList || GTK_CLIST(BrowserList)->rows==0)
        return;

    SelectedRow = ((GtkCList*)BrowserList)->selection;
    if (SelectedRow && SelectedRow->data!=NULL)
    {
        row = (gint)SelectedRow->data;
        filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),row);
        if (filerow)
        {
            if (ET_Delete_File(filerow->ETFile)==TRUE)
            {
                // File deleted, we remove the corresponding line in the list
                gtk_clist_remove(GTK_CLIST(BrowserList),row);
                // Note: info of the new current file must be displayed
                ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);
                gtk_clist_select_row(GTK_CLIST(BrowserList),row,0);
            }
        }
    }
    
}


/*
 * Select the specified line into clist, whitout sending the signal.
 * Note: the row 1 for the user correspond to row 0 for clist
 */
void Browser_List_Select_File (gint row)
{
    if (row > 0)
    {
        gtk_signal_handler_block_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Selected),NULL);
        gtk_clist_select_row(GTK_CLIST(BrowserList),row-1,0);
        Browser_List_Set_Row_Visible(GTK_CLIST(BrowserList),row-1);
        gtk_signal_handler_unblock_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Selected),NULL);
    }
}


void Browser_List_Scroll_Vertical(GtkCList *clist, GtkScrollType scroll_type,
                                  gfloat position, gpointer user_data)
{
    if (!clist) return;
    
    /* As this function was called by signal gtk_signal_..._after..., "clist->focus_row" contains
     * the number of the row which receive the signal */
    gtk_clist_select_row(GTK_CLIST(clist),(gint)clist->focus_row,0);
}


void Browser_List_Clear (void)
{
    gtk_clist_clear(GTK_CLIST(BrowserList));
}


/*
 * Disable (FALSE) / Enable (TRUE) all user widgets in the browser area (Tree + List + Entry)
 */
void Browser_Area_Set_Sensitive (gboolean activate)
{
    gtk_widget_set_sensitive(GTK_WIDGET(BrowserEntry),activate);
    gtk_widget_set_sensitive(GTK_WIDGET(BrowserTree), activate);
    gtk_widget_set_sensitive(GTK_WIDGET(BrowserList), activate);
}


/*
 * Browser_Popup_Menu_Handler : displays the corresponding menu
 * Create_Browser_Tree_Popup_Menu: Create a popup menu for the tree browser
 * Create_Browser_List_Popup_Menu: Create a popup menu for the list of files of browser
 */
gboolean Browser_Popup_Menu_Handler (GtkMenu *menu, GdkEventButton *event)
{
    if (event && (event->type==GDK_BUTTON_PRESS) && (event->button==3))
    {
        gtk_menu_popup(menu,NULL,NULL,NULL,NULL,event->button,event->time);
        return TRUE;
    }
    return FALSE;
}
GtkWidget *Create_Browser_Tree_Popup_Menu (GtkCTree *ctree)
{
    GtkWidget *BrowserPopupMenu;
    GtkWidget *MenuItem;


    BrowserPopupMenu = gtk_menu_new();
    gtk_signal_connect_object(GTK_OBJECT(BrowserTree),"button_press_event",
                  (GtkSignalFunc)Browser_Popup_Menu_Handler,
                  GTK_OBJECT(BrowserPopupMenu));

    MenuItem = gtk_menu_item_new_with_label(_("Go to Home Directory"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Load_Home_Directory,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Go to Default Directory"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Load_Default_Directory,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Set Current Path as Default"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Set_Current_Path_As_Default,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Rename Directory"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Open_Rename_Directory_Window,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Reload Directory"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Reload_Directory,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Browse Directory with ..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Open_Run_Program_Window,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    CheckMenuItemBrowseSubdirPopupMenu = gtk_check_menu_item_new_with_label(_("Browse Sub-directories"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(CheckMenuItemBrowseSubdirPopupMenu),BROWSE_SUBDIR);
    gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(CheckMenuItemBrowseSubdirPopupMenu),TRUE);
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),CheckMenuItemBrowseSubdirPopupMenu);
    gtk_signal_connect(GTK_OBJECT(CheckMenuItemBrowseSubdirPopupMenu),"toggled",
        (GtkSignalFunc)Check_Menu_Item_Toggled_Browse_Subdir,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Collapse Tree"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Tree_Collapse,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Refresh Tree"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Tree_Rebuild,NULL); /* '..._connect_object' to send NULL parameter to Browser_Tree_Rebuild */

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Run XMMS"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Run_Xmms,NULL);

    gtk_widget_show_all(BrowserPopupMenu);
    return BrowserPopupMenu;
}
GtkWidget  *Create_Browser_List_Popup_Menu (GtkCList *clist)
{
    GtkWidget *BrowserPopupMenu;
    GtkWidget *MenuItem;

    // Note: there is an interesting example with GtkItemFactoryEntry and gtk_item_factory_set_translate_func in XMMS sources xmms/playlistwin.c
    BrowserPopupMenu = gtk_menu_new();
    gtk_signal_connect_object(GTK_OBJECT(BrowserList),"button_press_event",
                  (GtkSignalFunc)Browser_Popup_Menu_Handler,
                  GTK_OBJECT(BrowserPopupMenu));

    MenuItem = gtk_menu_item_new_with_label(_("Delete this file"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_List_Delete_File,NULL);
    
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by filename..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Sort_File_list_By_Ascending_Filename,NULL);
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by filename..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Sort_File_list_By_Descending_Filename,NULL);
    
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by creation date..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Sort_File_List_By_Ascending_Creation_Date,NULL);
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by creation date..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Sort_File_List_By_Descending_Creation_Date,NULL);
    
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by track number..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Sort_File_List_By_Ascending_Track_Number,NULL);
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by track number..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Sort_File_List_By_Descending_Track_Number,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Reload the directory"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Reload_Directory,NULL);

    gtk_widget_show_all(BrowserPopupMenu);
    return BrowserPopupMenu;
}


gint Find_Node (gconstpointer a, gconstpointer b)
{
    gchar *data1;
    gchar *data2;

    if (!a || !b || !((DirNode*)a)->path) return -1;
    data1 = g_strdup((gchar *)((DirNode*)a)->path);
    data2 = g_strdup((gchar *)b);

    if ( strncmp(data1,data2,strlen(data2))==0 && strlen(data1)>strlen(data2) )
    {
        g_free(data1);
        g_free(data2);
        return 0;
    }else
    {
        g_free(data1);
        g_free(data2);
        return -1;
    }
}

/*
 * Destroy the whole tree up to the root node
 */
void Browser_Tree_Initialize (void)
{
    GtkCTreeNode *node;
    DirNode *dirnode;
    gchar *node_text = "dummy";

    gtk_clist_freeze(GTK_CLIST(BrowserTree));
    dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(BrowserTree),RootNode);
    while ( (node=gtk_ctree_find_by_row_data_custom(GTK_CTREE(BrowserTree),NULL,dirnode->path,Find_Node)) )
    {
        gtk_ctree_remove_node(GTK_CTREE(BrowserTree),node);
    }
    /* Reinit node data */
    dirnode->scanned = 0;
    /* Insert dummy node to display: '+' */
    node = gtk_ctree_insert_node(GTK_CTREE(BrowserTree),RootNode,NULL,&node_text,4,
        NULL,NULL,NULL,NULL,TRUE,TRUE);
    gtk_clist_thaw(GTK_CLIST(BrowserTree));
}

/*
 * Browser_Tree_Rebuild: Refresh the tree browser by destroying it and rebuilding it.
 * Opens tree nodes corresponding to 'path_to_load' if this parameter isn't NULL.
 * If NULL, selects the current path.
 */
void Browser_Tree_Rebuild (gchar *path_to_load)
{
    if (path_to_load != NULL)
    {
        Browser_Tree_Initialize();
        Browser_Tree_Select_Dir(path_to_load);
    }else
    {
        gchar *current_path = NULL;

        /* Memorize the current path to load it again at the end */
        if (Browser_Tree_Get_Path_Of_Selected_Node() != NULL)
            current_path = g_strdup(Browser_Tree_Get_Path_Of_Selected_Node());
        /* If no node selected, get path from BrowserEntry or default path */
        else if (BrowserCurrentPath != NULL)
            current_path = g_strdup(BrowserCurrentPath);
        else if (strlen(gtk_entry_get_text_1(BrowserEntry)) > 0)
            current_path = g_strdup(gtk_entry_get_text_1(BrowserEntry));
        else
            current_path = g_strdup(DEFAULT_PATH_TO_MP3);
        
        Browser_Tree_Initialize();
        /* Reload the memorized path */
//        gtk_signal_handler_block_by_func(GTK_OBJECT(BrowserTree),GTK_SIGNAL_FUNC(Browser_Tree_Node_Selected),NULL);
        Browser_Tree_Select_Dir(current_path);
//        gtk_signal_handler_unblock_by_func(GTK_OBJECT(BrowserTree),GTK_SIGNAL_FUNC(Browser_Tree_Node_Selected),NULL);
        if (current_path) g_free(current_path);
    }
}



static gboolean check_for_subdir(gchar *path)
{
    DIR *dir;
    struct dirent *dirent;
    struct stat statbuf;
    gchar *npath;
    
    if( (dir=opendir(path)) )
    {
        while( (dirent=readdir(dir)) )
        {
            if(dirent->d_name[0]!='.')
            {
                npath = g_strconcat(path,dirent->d_name,"/",NULL);
                stat(npath,&statbuf);
                g_free(npath);
                if(S_ISDIR(statbuf.st_mode))
                {
                    closedir(dir);
                    return TRUE;
                }
            }
        }
        closedir(dir);
    }
    return FALSE;
}

/*
 * Check if you have access permissions for directory path. Returns 1 if ok, else 0.
 */
gboolean Check_For_Access_Permission (gchar *path)
{
    DIR *dir;

    if( (dir=opendir(path)) == NULL )
    {
        if (errno == EACCES)
            return FALSE;
    }else
    {
        closedir(dir);
    }
    return TRUE;
}


/*
 * Destroy data attached to each node of the tree
 */
static void destroy_ctb(gpointer data)
{
    DirNode *node = data;
    g_free(node->path);
    g_free(node);
}

static void expand_cb(GtkWidget *ctree, GtkCTreeNode *parent_node)
{
    DIR *dir;
    struct dirent *dirent;
    gchar *path;
    gchar *text;
    gchar *dummy = "dummy";
    struct stat statbuf;
    GtkCTreeNode *node, *sub_node;
    DirNode *parent_dirnode, *dirnode;
    gboolean has_subdir = FALSE;
    
    
    parent_dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(ctree), parent_node);
    if(!parent_dirnode->scanned)
    {
        gtk_clist_freeze(GTK_CLIST(ctree));
        node = gtk_ctree_find_by_row_data(GTK_CTREE(ctree), parent_node, NULL);
        gtk_ctree_remove_node(GTK_CTREE(ctree),node);
        if( (dir=opendir(parent_dirnode->path)) )
        {
            while( (dirent=readdir(dir)) )
            {
                path = g_strconcat(parent_dirnode->path,dirent->d_name,NULL);
                stat(path,&statbuf);
                if(S_ISDIR(statbuf.st_mode)&&dirent->d_name[0]!='.')
                {
                    dirnode = g_malloc0(sizeof(DirNode));
                    dirnode->path = g_strconcat(path,"/",NULL);
                    text = dirent->d_name;

                    if(check_for_subdir(dirnode->path))
                        has_subdir=TRUE;
                    else
                        has_subdir=FALSE;

                    /* Select pixmap for accessible/unaccessible directory */
                    if (Check_For_Access_Permission(path))
                    {
                        node = gtk_ctree_insert_node(GTK_CTREE(ctree),
                            parent_node,NULL,&text,4, 
                            closed_folder_pixmap,closed_folder_mask,
                            opened_folder_pixmap,opened_folder_mask, 
                            !has_subdir, FALSE);
                    }
                    else
                    {
                        node = gtk_ctree_insert_node(GTK_CTREE(ctree),
                            parent_node,NULL,&text,4, 
                            closed_folder_locked_pixmap,closed_folder_locked_mask,
                            opened_folder_pixmap,opened_folder_mask, 
                            !has_subdir, FALSE);
                    }

                    gtk_ctree_node_set_row_data_full(GTK_CTREE(ctree),node,dirnode,destroy_ctb);

                    if(has_subdir)
                        sub_node=gtk_ctree_insert_node(GTK_CTREE(ctree),
                            node,NULL,&dummy,4,NULL,NULL,NULL,NULL,FALSE,FALSE);
                }
                g_free(path);
            }
            closedir(dir);
            gtk_ctree_sort_node(GTK_CTREE(ctree),parent_node);
        }
        gtk_clist_thaw(GTK_CLIST(ctree));    
        parent_dirnode->scanned=TRUE;
    }
}

static void collapse_cb(GtkWidget *ctree, GtkCTreeNode *parent_node)
{
    DirNode *dirnode;
    GtkCTreeNode *dummy_node, *child_node;
    GtkCTreeRow *row;
    gchar *dummy = "dummy";

    if (KEEP_TREE_BROWSER_IN_MEMORY) return;
    
    dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(ctree),parent_node);
    if (dirnode && dirnode->scanned)
    {
        gtk_signal_handler_block_by_func(GTK_OBJECT(ctree),GTK_SIGNAL_FUNC(collapse_cb),NULL);
        gtk_clist_freeze(GTK_CLIST(ctree));

        row = GTK_CTREE_ROW(parent_node);
        // Remove all childrens
        while ( (child_node=row->children) )
            gtk_ctree_remove_node(GTK_CTREE(ctree),child_node);
        // Insert a dummy node to show the hot spot '+' (Note: by definition, this node has sub nodes...)
        dummy_node=gtk_ctree_insert_node(GTK_CTREE(ctree),parent_node,NULL,&dummy,4,NULL,NULL,NULL,NULL,FALSE,FALSE);
        // Reinitialize attached data to rescan the dir when expanding the node
        dirnode->scanned = 0;

        gtk_clist_thaw(GTK_CLIST(ctree));    
        gtk_signal_handler_unblock_by_func(GTK_OBJECT(ctree),GTK_SIGNAL_FUNC(collapse_cb),NULL);
    }
}

static int filetreeent_compare_func(const void *a, const void *b)
{
    if(!a || !b || !((DirNode*)a)->path)
        return -1;
    return strcmp(((DirNode*)a)->path,(gchar*)b);
}


/*
 * Create item of the browser (Entry + Tree + List).
 */
GtkWidget *Create_Browser_Items (GtkWidget *parent)
{
    GtkWidget *VerticalBox;
    GtkWidget *ScrollWindowCTree;
    GtkWidget *ScrollWindowCList;
    gchar *RootText = "/";
    gchar *node_text = "dummy";
    DirNode *dirnode;
    GtkTooltips *Tips;
    GtkWidget *PopupMenu;
    GList *History_List;


    Tips = gtk_tooltips_new_1();

    VerticalBox = gtk_vbox_new(FALSE,2);
    gtk_container_set_border_width(GTK_CONTAINER(VerticalBox),2);


    /*
     * The entry box for displaying path
     */
    BrowserEntry = gtk_combo_new();
    gtk_combo_disable_activate(GTK_COMBO(BrowserEntry));
    /* History list */
    History_List = Load_Path_Entry_List();
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(BrowserEntry),History_List);
    gtk_object_set_data(GTK_OBJECT(BrowserEntry),"History",History_List);

    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(BrowserEntry)->entry),"activate",GTK_SIGNAL_FUNC(Browser_Entry_Activated),NULL);
    gtk_box_pack_start(GTK_BOX(VerticalBox),BrowserEntry,FALSE,TRUE,0);
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(BrowserEntry)->entry),_("No path selected!"));
    gtk_tooltips_set_tip(Tips,GTK_COMBO(BrowserEntry)->entry,_("Enter a directory to browse."),NULL);


    /*
     * The ScrollWindow and the CTree
     */
    ScrollWindowCTree = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindowCTree),
                                   GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    //gtk_box_pack_start(GTK_BOX(VerticalBox),ScrollWindowCTree,TRUE,TRUE,0);
    //gtk_widget_set_usize (GTK_WIDGET(ScrollWindowCTree),100,-1);

    /* The tree */
    BrowserTree = gtk_ctree_new(1,0);
    gtk_container_add(GTK_CONTAINER(ScrollWindowCTree),BrowserTree);
    gtk_ctree_set_indent(GTK_CTREE(BrowserTree),13);
    gtk_ctree_set_line_style(GTK_CTREE(BrowserTree),GTK_CTREE_LINES_DOTTED);
    gtk_ctree_set_expander_style(GTK_CTREE(BrowserTree),GTK_CTREE_EXPANDER_SQUARE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(BrowserTree),0,TRUE);
    //gtk_clist_set_selection_mode (GTK_CLIST(BrowserTree),GTK_SELECTION_EXTENDED);
    //gtk_clist_set_selection_mode (GTK_CLIST(BrowserTree),GTK_SELECTION_BROWSE);

    /* Create pixmaps */
    if(!opened_folder_pixmap)
    {
        opened_folder_pixmap = gdk_pixmap_create_from_xpm_d(parent->window,
                    &opened_folder_mask,NULL,opened_folder_xpm);
        closed_folder_pixmap = gdk_pixmap_create_from_xpm_d(parent->window,
                    &closed_folder_mask,NULL,closed_folder_xpm);
        closed_folder_locked_pixmap = gdk_pixmap_create_from_xpm_d(parent->window,
                    &closed_folder_locked_mask,NULL,closed_folder_locked_xpm);
    }
    
    /* Signaux */
    gtk_signal_connect(GTK_OBJECT(BrowserTree),"tree_expand",    (GtkSignalFunc)expand_cb,NULL);
    gtk_signal_connect(GTK_OBJECT(BrowserTree),"tree_collapse",  (GtkSignalFunc)collapse_cb,NULL);
    gtk_signal_connect(GTK_OBJECT(BrowserTree),"tree_select_row",(GtkSignalFunc)Browser_Tree_Node_Selected,NULL);
    gtk_signal_connect(GTK_OBJECT(BrowserTree),"key_press_event",(GtkSignalFunc)Browser_Tree_Key_Press,NULL);
    gtk_signal_connect_after(GTK_OBJECT(BrowserTree),"scroll_vertical",(GtkSignalFunc)Browser_List_Scroll_Vertical,NULL);

    /* Create the root node */
    RootNode = gtk_ctree_insert_node(GTK_CTREE(BrowserTree),NULL,NULL,&RootText,4, 
                    closed_folder_pixmap,closed_folder_mask,
                    opened_folder_pixmap,opened_folder_mask, 
                    FALSE, FALSE);

    dirnode = g_malloc0(sizeof(DirNode));
    dirnode->path = g_strdup("/");
    gtk_ctree_node_set_row_data_full(GTK_CTREE(BrowserTree),RootNode,dirnode,destroy_ctb);

    /* Insert dummy node to display: '+' */
    node = gtk_ctree_insert_node(GTK_CTREE(BrowserTree),RootNode,NULL,&node_text,4, 
                    NULL,NULL,NULL,NULL,TRUE,TRUE);

    /* 
     * Create Popup Menu on browser ctree
     */
    PopupMenu = Create_Browser_Tree_Popup_Menu(GTK_CTREE(BrowserTree));

    /* Open root node */
    /*gtk_ctree_expand(GTK_CTREE(BrowserTree),RootNode);*/

    /* Load current path */
    /* Don't load it now! */


    /*
     * The ScrollWindow and the CList
     */
    ScrollWindowCList = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindowCList),
                                   GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);

    /* The clist */
    BrowserList = gtk_clist_new(1);
    gtk_container_add(GTK_CONTAINER(ScrollWindowCList),BrowserList);
    gtk_clist_set_column_auto_resize(GTK_CLIST(BrowserList),0,TRUE);
    gtk_clist_set_reorderable(GTK_CLIST(BrowserList),FALSE);
    gtk_clist_set_selection_mode(GTK_CLIST(BrowserList),GTK_SELECTION_SINGLE);

    gtk_signal_connect(GTK_OBJECT(BrowserList),"key_press_event",(GtkSignalFunc)Browser_List_Key_Press,NULL);
    gtk_signal_connect(GTK_OBJECT(BrowserList),"select_row",(GtkSignalFunc)Browser_List_Row_Selected,NULL);
    /* Note: "after" for this signal to obtain the new focus_row of clist and not the previous one */
    gtk_signal_connect_after(GTK_OBJECT(BrowserList),"scroll_vertical",
        (GtkSignalFunc)Browser_List_Scroll_Vertical,NULL);

    /* 
     * Create Popup Menu on browser ctree
     */
    PopupMenu = Create_Browser_List_Popup_Menu(GTK_CLIST(BrowserList));


    /*
     * The pane for the CTree and the CList
     */
    BrowserHPaned = gtk_hpaned_new();
    gtk_box_pack_start(GTK_BOX(VerticalBox),BrowserHPaned,TRUE,TRUE,0);
    gtk_paned_set_handle_size(GTK_PANED(BrowserHPaned),8);
    gtk_paned_set_gutter_size(GTK_PANED(BrowserHPaned),6);                       
    gtk_paned_pack1(GTK_PANED(BrowserHPaned),ScrollWindowCTree,TRUE,TRUE); // Left side
    gtk_paned_pack2(GTK_PANED(BrowserHPaned),ScrollWindowCList,TRUE,TRUE); // Right side
    if (SET_PANE_HANDLE_POSITION2)
        gtk_paned_set_position(GTK_PANED(BrowserHPaned),PANE_HANDLE_POSITION2);

    gtk_widget_show_all(VerticalBox);
    
    /* Set home variable as current path */
    Browser_Update_Current_Path(HOME_VARIABLE);
    
    return VerticalBox;
}





/*
 * The window to Rename a directory into the browser.
 */
void Browser_Open_Rename_Directory_Window (void)
{
    GtkWidget *Frame;
    GtkWidget *VBox;
    GtkWidget *Label;
    GtkWidget *Combo;
    GtkWidget *ButtonBox;
    GtkWidget *Button;
    GtkWidget *Separator;
    GList *combo_list = NULL;
    gchar *directory_parent = NULL;
    gchar *directory_name = NULL;
    gchar *address = NULL;
    gchar *string;
    gint mw_x, mw_y;
    gint ba_x, ba_y;


    if (RenameDirectoryWindow != NULL) 
    {
        gdk_window_raise(RenameDirectoryWindow->window);
        return;
    }

    /* We get the full path but we musn't display the parent directories */
    directory_parent = g_strdup(BrowserCurrentPath);
    if (!directory_parent || strlen(directory_parent)==0)
        return;
    if (strlen(directory_parent)>1 && directory_parent[strlen(directory_parent)-1]=='/')
        directory_parent[strlen(directory_parent)-1]=0;
    address = strrchr(directory_parent,'/');
    if (!address) return;
    directory_name = g_strdup(address+1);
    *(address+1) = 0;
    if (!directory_name || strlen(directory_name)==0)
        return;
    
    
    RenameDirectoryWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(RenameDirectoryWindow),_("Rename the directory"));
    gtk_window_set_transient_for(GTK_WINDOW(RenameDirectoryWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(RenameDirectoryWindow),FALSE,TRUE,TRUE);
    gtk_signal_connect(GTK_OBJECT(RenameDirectoryWindow),"destroy",
        (GtkSignalFunc)Destroy_Rename_Directory_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RenameDirectoryWindow),"delete_event",
        (GtkSignalFunc)Destroy_Rename_Directory_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RenameDirectoryWindow),"key_press_event",
        (GtkSignalFunc)Rename_Directory_Window_Key_Press,NULL);

    Frame = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(RenameDirectoryWindow),Frame);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),2);

    VBox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);

    string = g_strdup_printf(_("Rename the directory '%s' to : "),directory_name);
    Label = gtk_label_new(_(string));
    g_free(string);
    gtk_box_pack_start(GTK_BOX(VBox),Label,TRUE,TRUE,0);
    gtk_label_set_line_wrap(GTK_LABEL(Label),TRUE);

    /* The combobox to rename the directory */
    Combo = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(VBox),Combo,TRUE,TRUE,0);
    gtk_combo_set_value_in_list(GTK_COMBO(Combo),FALSE,FALSE);
    gtk_widget_set_usize(GTK_WIDGET(Combo),300,-1);
    /* Set the directory into the combobox */
    combo_list = g_list_append(combo_list,"");
    combo_list = g_list_append(combo_list,directory_name);
//    gtk_combo_set_popdown_strings(GTK_COMBO(Combo),combo_list);
//*** FIX ME! When activated, can't change letter case
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(Combo)->entry),directory_name);
    /* We attach usefull data to the combobox */
    gtk_object_set_data(GTK_OBJECT(Combo),"Parent_Directory",directory_parent);
    gtk_object_set_data(GTK_OBJECT(Combo),"Current_Directory",directory_name);
    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(GTK_COMBO(Combo)->entry));

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    ButtonBox = gtk_hbutton_box_new ();
    gtk_box_pack_start(GTK_BOX(VBox),ButtonBox,FALSE,FALSE,0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(ButtonBox),GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(ButtonBox),10);

    /* Button to save: rename directory */
    Button = Create_Button_With_Pixmap(BUTTON_APPLY);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Rename_Directory,GTK_OBJECT(Combo));
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(Combo)->entry),"changed",(GtkSignalFunc)Entry_Changed_Disable_Object,GTK_OBJECT(Button));
    
    /* Button to cancel */
    Button = Create_Button_With_Pixmap(BUTTON_CANCEL);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_widget_grab_default(Button);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Destroy_Rename_Directory_Window,NULL);

    gtk_widget_show_all(RenameDirectoryWindow);

    /* Note : the window must be shown before moving it.
     * The window is centered over the Browser Area */
    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gdk_window_get_position(BrowseArea->window,&ba_x,&ba_y);
    gtk_window_reposition(GTK_WINDOW(RenameDirectoryWindow),
        mw_x + ba_x + (BrowseArea->allocation.width/2)  - (RenameDirectoryWindow->allocation.width)/2,
        mw_y + ba_y + (BrowseArea->allocation.height/2) - (RenameDirectoryWindow->allocation.height)/2);
}
void Destroy_Rename_Directory_Window (void)
{
    if (RenameDirectoryWindow)
    {
        gtk_widget_destroy(RenameDirectoryWindow);
        RenameDirectoryWindow = (GtkWidget *)NULL;
    }
}
void Rename_Directory (GtkObject *combobox)
{
    gchar *directory_parent;
    gchar *directory_last_name;
    gchar *directory_new_name;
    gchar *last_path;
    gchar *tmp_path;
    gchar *new_path;
    struct stat statbuf;


    if (!GTK_IS_COMBO(combobox)) return;
    
    directory_parent    = gtk_object_get_data(GTK_OBJECT(combobox),"Parent_Directory");
    directory_last_name = gtk_object_get_data(GTK_OBJECT(combobox),"Current_Directory");
    directory_new_name  = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combobox)->entry),0,-1);

    /* Check if a name for the directory have been supplied */
    if (!directory_new_name || strlen(directory_new_name)<1)
    {
        GtkWidget *msgbox;

        msgbox = msg_box_new (_("Error..."),_("You must type a directory name!"),MSG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        return;
    }

    /* If the directory name haven't been changed, we do nothing! */
    if (directory_last_name && directory_new_name && strcmp(directory_last_name,directory_new_name)==0)
    {
        Destroy_Rename_Directory_Window();
        g_free(directory_new_name);
        return;
    }

    /* Build the current and new absolute paths */
    last_path = g_strconcat(directory_parent,directory_last_name,NULL);
    new_path  = g_strconcat(directory_parent,directory_new_name,NULL);
    /* Temporary path (useful when changing only string case) */
    tmp_path = g_strconcat(new_path,".EastTAG",NULL);


    /* Check if the new directory name doesn't already exists (case change is accepted) */
    stat(new_path,&statbuf);
    if(S_ISDIR(statbuf.st_mode) && strcmp(last_path,new_path)==0)
    {
        gchar *msg;
        GtkWidget *msgbox;

        msg = g_strdup_printf(_("Can't rename because this directory name "
                                "already exists!\n(%s)"),new_path); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        return;
    }

    /* Rename the directory from 'last name' to 'tmp name' */
    if ( rename(last_path,tmp_path)!=0 )
    {
        gchar *msg;
        GtkWidget *msgbox;

        msg = g_strdup_printf(_("Can't rename directory \n'%s'\n to \n'%s'!\n(%s)"),
                    last_path,tmp_path,g_strerror(errno)); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        return;
    }
    /* Rename the directory from 'tmp name' to 'new name' (final name) */
    if ( rename(tmp_path,new_path)==0 )
    {
        ET_Update_Directory_Name_Into_File_List(last_path,new_path);
    }else
    {
        gchar *msg;
        GtkWidget *msgbox;

        msg = g_strdup_printf(_("Can't rename directory \n'%s'\n to \n'%s'!\n(%s)"),
                    tmp_path,new_path,g_strerror(errno)); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        return;
    }

    g_free(tmp_path);
    g_free(directory_new_name);

    Destroy_Rename_Directory_Window();
    Browser_Tree_Rebuild(new_path);
    Statusbar_Message(_("Directory renamed"),TRUE);
}
void Rename_Directory_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Destroy_Rename_Directory_Window();
                break;
        }
    }
}






void Browser_Open_Run_Program_Window (void)
{
    GtkWidget *Frame;
    GtkWidget *VBox;
    GtkWidget *Label;
    GtkWidget *Combo;
    GtkWidget *ButtonBox;
    GtkWidget *Button;
    GtkWidget *Separator;
    GtkTooltips *Tips;
    GList *History_List;
    gchar *current_directory = NULL;
    gint mw_x, mw_y;
    gint ba_x, ba_y;


    if (RunProgramWindow != NULL) 
    {
        gdk_window_raise(RunProgramWindow->window);
        return;
    }

    // Current directory
    current_directory = g_strdup(BrowserCurrentPath);
    if (!current_directory || strlen(current_directory)==0)
        return;
    
    RunProgramWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(RunProgramWindow),_("Browse Directory with ..."));
    gtk_window_set_transient_for(GTK_WINDOW(RunProgramWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(RunProgramWindow),FALSE,TRUE,TRUE);
    gtk_signal_connect(GTK_OBJECT(RunProgramWindow),"destroy",(GtkSignalFunc)Destroy_Run_Program_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RunProgramWindow),"delete_event",(GtkSignalFunc)Destroy_Run_Program_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RunProgramWindow),"key_press_event",(GtkSignalFunc)Run_Program_Window_Key_Press,NULL);

    Tips = gtk_tooltips_new_1();

    Frame = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(RunProgramWindow),Frame);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),2);

    VBox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);

    Label = gtk_label_new(_("Program to run :"));
    gtk_box_pack_start(GTK_BOX(VBox),Label,TRUE,TRUE,0);
    gtk_label_set_line_wrap(GTK_LABEL(Label),TRUE);

    /* The combobox to enter the program to run */
    Combo = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(VBox),Combo,TRUE,TRUE,0);
    gtk_combo_set_value_in_list(GTK_COMBO(Combo),FALSE,FALSE);
    gtk_widget_set_usize(GTK_WIDGET(Combo),150,-1);
    gtk_tooltips_set_tip(Tips,GTK_COMBO(Combo)->entry,_("Enter the program to run. It will "
        "receive the current directory as parameter."),NULL);

    /* History list */
    History_List = Load_Run_Program_List();
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(Combo),History_List);
    gtk_object_set_data(GTK_OBJECT(Combo),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(Combo)->entry)),"activate",
        Add_To_Combo_Box_History,GTK_OBJECT(Combo));
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(Combo)->entry)),"focus_out_event",
        Add_To_Combo_Box_History,GTK_OBJECT(Combo));

    /* We attach usefull data to the combobox (into Run_Program) */
    gtk_object_set_data(GTK_OBJECT(Combo),"Current_Directory",current_directory);

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    ButtonBox = gtk_hbutton_box_new ();
    gtk_box_pack_start(GTK_BOX(VBox),ButtonBox,FALSE,FALSE,0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(ButtonBox),GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(ButtonBox),10);

    /* Button to save: rename directory */
    Button = Create_Button_With_Pixmap(BUTTON_EXECUTE);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Run_Program,GTK_OBJECT(Combo));
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(Combo)->entry),"changed",(GtkSignalFunc)Entry_Changed_Disable_Object,GTK_OBJECT(Button));
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(Combo)->entry),"changed",NULL);

    /* Button to cancel */
    Button = Create_Button_With_Pixmap(BUTTON_CANCEL);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_widget_grab_default(Button);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Destroy_Run_Program_Window,NULL);

    gtk_widget_show_all(RunProgramWindow);

    /* Note : the window must be shown before moving it.
     * The window is centered over the Browser Area */
    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gdk_window_get_position(BrowseArea->window,&ba_x,&ba_y);
    gtk_window_reposition(GTK_WINDOW(RunProgramWindow),
        mw_x + ba_x + (BrowseArea->allocation.width/2)  - (RunProgramWindow->allocation.width)/2,
        mw_y + ba_y + (BrowseArea->allocation.height/2) - (RunProgramWindow->allocation.height)/2);
    
}
void Destroy_Run_Program_Window (void)
{
    if (RunProgramWindow)
    {
        gtk_widget_destroy(RunProgramWindow);
        RunProgramWindow = (GtkWidget *)NULL;
    }
}
void Run_Program_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Destroy_Run_Program_Window();
                break;
        }
    }
}
void Run_Program (GtkObject *combobox)
{
    gchar *current_directory;
    gchar *program_name;
    gchar *argv[3];
    gchar *msg;
    pid_t pid;


    if (!GTK_IS_COMBO(combobox)) return;
    
    current_directory = gtk_object_get_data(GTK_OBJECT(combobox),"Current_Directory");
    program_name      = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combobox)->entry),0,-1);

    /* Check if a name for the program have been supplied */
    if (!program_name || strlen(program_name)<1)
    {
        GtkWidget *msgbox;

        msgbox = msg_box_new (_("Error..."),_("You must type a program name!"),MSG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        return;
    }

    pid = fork();
    switch(pid)
    {
        case -1:
            g_warning(_("Can't fork another process!\n"));
            break;
        case 0:
            argv[0] = program_name;
            argv[1] = current_directory;
            argv[2] = NULL;
            execvp(argv[0],argv);

            msg = g_strdup_printf(_("Executed command : '%s %s'"),program_name,current_directory);
            Statusbar_Message(msg,TRUE);
            g_free(msg);
            break;
        default:
            break;
    }

    // Save list attached to the combobox
    Save_Run_Program_List(gtk_object_get_data(GTK_OBJECT(combobox),"History"));

    Destroy_Run_Program_Window();
}



