/* szap -- simple zapping tool for the Linux DVB API
 *
 * szap operates on VDR (http://www.cadsoft.de/people/kls/vdr/index.htm)
 * satellite channel lists (e.g. from http://www.dxandy.de/cgi-bin/dvbchan.pl).
 * szap assumes you have a "Universal LNB" (i.e. with LOFs 9750/10600 MHz).
 *
 * Compilation: `gcc -Wall -I../../ost/include -O2 szap.c -o szap`
 *  or, if your DVB driver is in the kernel source tree:
 *              `gcc -Wall -DDVB_IN_KERNEL -O2 szap.c -o szap`
 *
 * Copyright (C) 2001 Johannes Stezenbach (js@convergence.de)
 * for convergence integrated media
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/poll.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>

#include <stdint.h>
#include <sys/time.h>

#ifdef DVB_IN_KERNEL
#  include <linux/ost/sec.h>
#  include <linux/ost/frontend.h>
#  include <linux/ost/dmx.h>
#else
#  include <ost/sec.h>
#  include <ost/frontend.h>
#  include <ost/dmx.h>
#endif


#ifndef TRUE
#define TRUE (1==1)
#endif
#ifndef FALSE
#define FALSE (1==0)
#endif

/* location of channel list file */
#define CHANNEL_FILE "channels.conf"

/* one line of the VDR channel file has the following format:
 * ^name:frequency_MHz:polarization:sat_no:symbolrate:vpid:apid:?:service_id$
 */


#define SECDEVICE "/dev/ost/sec"
#define FRONTENDDEVICE "/dev/ost/frontend"
#define DEMUXDEVICE "/dev/ost/demux"


/* LNB hi/lo band switch frequency in kHz */
#define SWITCHFREQ 11700000
/* LNB hi/lo band local oscillator frequencies in kHz */
#define LOF_HI 10600000
#define LOF_LO 9750000


static char *usage_str =
  "usage: szap -l\n"
  "         list known channels\n"
  "       szap -n channel-number\n"
  "       szap 'channel name'\n"
  "         zap to channel via number or full name (case insensitive)\n";


static void usage (void)
{
  fprintf (stderr, usage_str);
  exit (1);
}


static int set_demux (dmxfd, pid, audio)
{
  struct dmxPesFilterParams pesfilter;

  pesfilter.pid = pid;
  pesfilter.input = DMX_IN_FRONTEND;
  pesfilter.output = DMX_OUT_DECODER;
  pesfilter.pesType = (audio ? DMX_PES_AUDIO : DMX_PES_VIDEO);
  pesfilter.flags = DMX_IMMEDIATE_START;
  if (ioctl (dmxfd, DMX_SET_PES_FILTER, &pesfilter) == -1
      && pid && pid != 0x1fff)
    {
      fprintf (stderr, "DMX_SET_PES_FILTER failed "
	       "(PID = 0x%04x): %d %m\n", pid, errno);
      return FALSE;
    }

  return TRUE;
}


/* digital satellite equipment control,
 * specification is available from http://www.eutelsat.com/ 
 */
static int diseqc (int secfd, int sat_no, int pol, int hi_lo)
{
  struct secCmdSequence secseq;
  struct secCommand scmd;

  scmd.type = SEC_CMDTYPE_DISEQC;
  scmd.u.diseqc.addr = 0x10;		/* any switch */
  scmd.u.diseqc.cmd = 0x38;		/* set port group 0 */
  scmd.u.diseqc.numParams = 1;
  /* param: high nibble: reset bits, low nibble set bits,
   * bits are: option, position, polarizaion, band
   */
  scmd.u.diseqc.params[0] = 0xF0 | (((sat_no * 4) & 0x0F)
                                    | (hi_lo?1:0) | (pol?0:2));

  /* tone/volt for backwards compatiblity with non-diseqc LNBs */
  secseq.voltage = pol ? SEC_VOLTAGE_13 : SEC_VOLTAGE_18;
  secseq.continuousTone = hi_lo ? SEC_TONE_ON : SEC_TONE_OFF;
  secseq.miniCommand = SEC_MINI_NONE;
  secseq.numCommands = 1;
  secseq.commands = &scmd;

  if (ioctl (secfd, SEC_SEND_SEQUENCE, &secseq) == -1)
    {
      perror ("send SEC sequence failed");
      return FALSE;
    }
  return TRUE;
}

static int do_tune(int fefd, uint ifreq, uint sr)
{
  FrontendParameters tuneto;
  FrontendEvent ev;
  struct pollfd pfd;
  int prc;

  /* discard stale QPSK events */
  while (1)
    {
      if (ioctl (fefd, FE_GET_EVENT, &ev) == -1)
	break;
    }

  tuneto.Frequency = ifreq;
  tuneto.u.qpsk.SymbolRate = sr;
  tuneto.u.qpsk.FEC_inner = FEC_AUTO;
  if (ioctl (fefd, FE_SET_FRONTEND, &tuneto) == -1)
    {
      perror ("FE_SET_FRONTEND failed");
      return FALSE;
    }

  /* wait for tunig to complete, with timout */
  pfd.fd = fefd;
  pfd.events = POLLIN | POLLPRI;
  prc = poll (&pfd, 1, 5000);
  if (prc == -1)
    {
      perror ("FE_GET_EVENT failed");
      return FALSE;
    }
  else if (prc == 0)
    {
      perror ("FE_GET_EVENT timed out");
      return FALSE;
    }

  if (ioctl (fefd, FE_GET_EVENT, &ev) == -1)
    {
      perror ("FE_GET_EVENT failed");
      return FALSE;
    }

  if (ev.type != FE_COMPLETION_EV)
    {
      fprintf (stderr, "tuning failed\n");
      return FALSE;
    }

  return TRUE;
}


static int zap_to (sat_no, freq, pol, sr, vpid, apid)
{
  int secfd, fefd, videofd, audiofd;
  uint ifreq;
  int hiband, result;
  FrontendInfo fe_info;

  secfd = open (SECDEVICE, O_RDWR);;
  if (secfd == -1)
    {
      perror ("opening SEC failed");
      return FALSE;
    }
  fefd = open (FRONTENDDEVICE, O_RDWR | O_NONBLOCK);
  if (fefd == -1)
    {
      perror ("opening frontend failed");
      close (secfd);
      return FALSE;
    }
  result = ioctl (fefd, FE_GET_INFO, &fe_info);
  if (result == -1 || fe_info.type != FE_QPSK)
    {
      perror ("ioctl on fefd failed");
      close (fefd);
      close (secfd);
      return FALSE;
    }

  videofd = open (DEMUXDEVICE, O_RDWR);
  if (videofd == -1)
    {
      perror ("opening video demux failed");
      close (fefd);
      close (secfd);
      return FALSE;
    }
  audiofd = open (DEMUXDEVICE, O_RDWR);
  if (audiofd == -1)
    {
      perror ("opening audio demux failed");
      close (videofd);
      close (fefd);
      close (secfd);
      return FALSE;
    }

  hiband = (freq >= SWITCHFREQ);
  if (hiband)
    ifreq = freq - LOF_HI;
  else
    ifreq = freq - LOF_LO;

  result = FALSE;
  if (diseqc (secfd, sat_no, pol, hiband))
    if (do_tune (fefd, ifreq, sr))
      if (set_demux (videofd, vpid, 0))
	if (set_demux (audiofd, apid, 1))
	  result = TRUE;

  close (audiofd);
  close (videofd);
  close (fefd);
  close (secfd);

  return result;
}


static int read_channels (const char *filename, int list_channels,
			  uint chan_no, const char *chan_name)
{
  FILE *cfp;
  char buf[4096];
  char *field, *tmp;
  uint line = 0;
  uint freq, pol, sat_no, sr, vpid, apid;

  cfp = fopen (filename, "r");
  if (! cfp)
    {
      fprintf (stderr, "error opening channel list '%s': %d %m\n",
	       filename, errno);
      return FALSE;
    }

  while (! feof (cfp))
    {
      if (fgets (buf, sizeof (buf), cfp))
	{
	  line++;
	  if (chan_no && chan_no != line)
	    continue;

	  tmp = buf;
	  field = strsep (&tmp, ":");
	  if (!field)
	    goto syntax_err;
	  if (list_channels)
	    {
	      printf ("%03u %s\n", line, field);
	      continue;
	    }
	  if (chan_name && strcasecmp (chan_name, field) != 0)
	    continue;

	  printf ("zapping to '%s':\n", field);

	  field = strsep (&tmp, ":");
	  if (!field)
	    goto syntax_err;
	  freq = strtoul (field, NULL, 0);
	  field = strsep (&tmp, ":");
	  if (!field)
	    goto syntax_err;
	  pol = (field[0] == 'h' ? 0: 1);
	  field = strsep (&tmp, ":");
	  if (!field)
	    goto syntax_err;
	  sat_no = strtoul (field, NULL, 0);
	  field = strsep (&tmp, ":");
	  if (!field)
	    goto syntax_err;
	  sr = strtoul (field, NULL, 0) * 1000;
	  field = strsep (&tmp, ":");
	  if (!field)
	    goto syntax_err;
	  vpid = strtoul (field, NULL, 0);
	  field = strsep (&tmp, ":");
	  if (!field)
	    goto syntax_err;
	  apid = strtoul (field, NULL, 0);

	  printf ("sat %u, frequency = %u MHz %c, symbolrate %u, "
		  "vpid = 0x%04x, apid = 0x%04x\n",
		  sat_no, freq, pol ? 'V' : 'H', sr, vpid, apid);

	  fclose (cfp);
	  if (zap_to (sat_no, freq * 1000, pol, sr, vpid, apid))
	    return TRUE;
	  return FALSE;

	syntax_err:
	  fprintf (stderr, "syntax error in line %u: '%s'\n", line, buf);
	}
      else if (ferror (cfp))
	{
	  fprintf (stderr, "error reading channel list '%s': %d %m\n",
		   filename, errno);
	  fclose (cfp);
	  return FALSE;
	}
      else
	break;
    }
  fclose (cfp);
  if (!list_channels)
    {
      fprintf (stderr, "channel not found\n");
      return FALSE;
    }
  return TRUE;
}


int main (int argc, char *argv[])
{
  const char *home;
  char chanfile[2 * PATH_MAX];
  int list_channels = 0;
  uint chan_no = 0;
  const char *chan_name = NULL;

  if (argc == 2 && strcmp (argv[1], "-l") == 0)
    list_channels = 1;
  else if (argc == 3 && strcmp (argv[1], "-n") == 0)
    chan_no = strtoul (argv[2], NULL, 0);
  else if (argc == 2)
    chan_name = argv[1];
  else
    usage ();

  home = getenv ("HOME");
  if (! home)
    {
      fprintf (stderr, "error: $HOME not set\n");
      return TRUE;
    }
  strncpy (chanfile, home, sizeof (chanfile));
  strcat (chanfile, "/.szap/" CHANNEL_FILE);

  if (! read_channels (chanfile, list_channels, chan_no, chan_name))
    return TRUE;

  return FALSE;
}
