/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002-2004 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <config.h>

#include <gnome.h>
#include <gdk-pixbuf/gdk-pixdata.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gconf/gconf.h>
#include <time.h>
#include <string.h>
#include <curl/curl.h>

#include "drivel.h"
#include "msg_queue.h"
#include "login.h"
#include "network.h"
#include "journal.h"

GMutex *net_mutex;

void
display_error_dialog (DrivelClient *dc, const gchar *header, const gchar *mesg)
{
	GtkWidget *dialog;
	gchar *error_header, *error_mesg, *error_string;
	
	if (header)
		error_header = g_strdup (header);
	else
		error_header = g_strdup (_("Error"));
	
	if (mesg)
		error_mesg = g_strdup (mesg);
	else
		error_mesg = g_strdup (
				_("Oh bother, there's a server error.  Please try again later."));
	
	error_string = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">"
			"%s</span>\n\n%s", error_header, error_mesg);
	
	dialog = gtk_message_dialog_new_with_markup (
			GTK_WINDOW (dc->current_window),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_ERROR,
			GTK_BUTTONS_CLOSE,
			error_string);
		
	gtk_dialog_run (GTK_DIALOG (dialog));
		
	gtk_widget_destroy (dialog);
	g_free (error_header);
	g_free (error_mesg);
	g_free (error_string);
	
	return;
}

void fill_time(DrivelClient *dc)
{
    time_t exact_time;
    struct tm *local_time;
    
    /* we need the year, month, day, hour, and minute in order to post */
    exact_time = time(NULL);
    local_time = localtime(&exact_time);
    
    dc->time.year = (local_time->tm_year + 1900);
    dc->time.month = (local_time->tm_mon + 1);
    dc->time.day = local_time->tm_mday;
    dc->time.hour = local_time->tm_hour;
    dc->time.minute = local_time->tm_min;
    
    return;
}

void
drivel_push_current_window (DrivelClient *dc, GtkWidget *window)
{
	dc->window_list = g_slist_prepend (dc->window_list, window);
	dc->current_window = window;
	
	return;
}

GtkWidget *
drivel_pop_current_window (DrivelClient *dc)
{
	GtkWidget *window;
	
	window = dc->window_list->data;
	dc->window_list = g_slist_remove (dc->window_list, window);
	dc->current_window = dc->window_list->data;
	
	return window;
}

GtkWidget *
drivel_get_current_window (GSList *list)
{
	GtkWidget *window;
	
	window = list->data;
	
	return window;
}

gboolean
validate_username (const gchar *username)
{
	gchar *temp, *p;
	gunichar c;
	gboolean retval = TRUE;

	if (!g_utf8_validate (username, -1, NULL))
		return FALSE;
	
	temp = g_strdup (username);
	p = temp;
	
	while (*p)
	{
		c = g_utf8_get_char_validated (p, -1);

		if (!g_unichar_isalnum (c) && c != '_')
		{
			retval = FALSE;
			break;
		}
		
		p = g_utf8_next_char (p);
	}
	
	g_free (temp);
	
	return retval;
}

void
drivel_button_list_clear (DrivelButtonVAList *bval)
{
	GSList *item, *entries;
	gulong signal;
	GtkWidget *entry;
	
	if (!bval)
		return;
	
	entries = bval->entries;
	for (item = bval->signals; item; item = item->next)
	{
		signal = GPOINTER_TO_INT (item->data);
		entry = entries->data;
		
		g_signal_handler_disconnect (entry, signal);
		entries = entries->next;
	}
	
	g_free (bval);
	bval = NULL;
	
	return;
}

gchar *
get_default_text (GConfClient *client, const gchar *key, const gchar *standard_text)
{
	gchar *string;
	
	string = gconf_client_get_string (client, key, NULL);
	
	if (!string)
		string = g_strdup (standard_text);
	
	return string;
}

void
list_free_item (gpointer data, gpointer user_data)
{
	g_free (data);
	data = NULL;
	
	return;
}

void
menu_list_free_item (gpointer data, gpointer user_data)
{
	LJMenuItem *menu_item = (LJMenuItem *) data;
	
	g_free (menu_item->label);
	g_free (menu_item->url);
	
	return;
}

void
friends_list_free_item (gpointer data, gpointer user_data)
{
	LJFriend *friend = (LJFriend *) data;
	
	g_free (friend->username);
	g_free (friend->name);
	g_free (friend->bg);
	g_free (friend->fg);
	
	return;
}

gint
compare_usernames (gconstpointer a, gconstpointer b)
{
	const LJFriend *friend = a;
	const gchar *username = b;
	
	return (strncmp (friend->username, username, strlen (friend->username)));
}

/* 'picture_store' must already be a valid GtkListStore */
void
fill_picture_menu (DrivelClient *dc, GtkListStore *picture_store)
{
	gint i;
	gchar *pickw;
	const gchar *text;
	GdkPixbuf *pixbuf, *pixbuf_small;
	GtkTreeIter iter;

	gtk_list_store_clear (picture_store);
	
	for (i = 0; i < dc->pictures + 1; i++)
	{
		gchar *pickf;

		pickw = g_strdup_printf ("pickw_%d", i);
		text = g_hash_table_lookup (dc->picture_keywords, pickw);
		if (i > 0)
		{
			const gchar *file;
			file = g_hash_table_lookup (dc->picture_filenames, pickw);
			pickf = g_build_filename (dc->config_directory, "pictures", file, NULL);
		} else
			pickf = dc->default_picture_file;
		if (pickf) {
			GError *error = NULL;
			pixbuf = gdk_pixbuf_new_from_file (pickf, &error);
			if (error)
				g_warning ("Couldn't load %s: %s\n", pickf, error->message);
			if (pixbuf && !error)
			{
				pixbuf_small = gdk_pixbuf_scale_simple (pixbuf, 20, 20, GDK_INTERP_BILINEAR);
				g_object_unref (G_OBJECT (pixbuf));
			}
			else
				pixbuf_small = NULL;
			if (error)
				g_error_free (error);
		} else
			pixbuf_small = NULL;
		gtk_list_store_append (picture_store, &iter);
		gtk_list_store_set (picture_store, &iter, 0, pixbuf_small, 1, text, -1);
		
		g_free (pickw);
	}
	
	return;
}

static void
load_stored_moods (GHashTable *mood_icons, gint moods, GConfClient *client)
{
	gchar *gconf_key, *key, *value;
	gint i, offset;

	offset = 0;
	
	for (i = 1; i < moods + 1; i++)
	{		
		gconf_key = g_strdup_printf ("/apps/drivel/moods/mood_%d", i + offset);
		
		value = g_strdup_printf ("%d", i + offset);
		key = gconf_client_get_string (client, gconf_key, NULL);
		
		/* skip non-existant moods */
		if (!key)
		{
			offset++;
			
			g_free (gconf_key);
			g_free (value);
			
			continue;
		}

		g_hash_table_insert (mood_icons, key, value);
		
		g_free (gconf_key);
	}
	
	return;
}

static void
gconf_data_free (DrivelGConfData *data, GConfClient *client, DrivelIDs *id)
{
	if (id)
		remove_gconf_notifies (client, id);

	g_free (data->user);
	g_free (data->password);
	g_free (data->save_password);
	g_free (data->autologin);
	
	g_free (data->default_server_list);
	g_free (data->default_server);
	g_free (data->default_mood);
	g_free (data->default_music);
	g_free (data->default_picture);
	g_free (data->default_security);
	g_free (data->default_comment);
	g_free (data->default_autoformat);
	
	g_free (data->proxy);
	g_free (data->proxy_url);
	g_free (data->proxy_port);
	g_free (data->proxy_user);
	g_free (data->proxy_pass);
	
	g_free (data->tray);
	
	g_free (data->entry_x);
	g_free (data->entry_y);
	g_free (data->entry_width);
	g_free (data->entry_height);
	g_free (data->entry_max);
	
	g_free (data->min_start);
	g_free (data->min_post);
	g_free (data->highlight_syntax);
	g_free (data->spellcheck);
	
	return;
}

void
drivel_gconf_data_fill (DrivelGConfData *data, const gchar *username, GConfClient *client, DrivelIDs *id, DrivelClient *dc)
{
	gchar *base;
	
	g_return_if_fail (username);
	
	if (data->user)
		gconf_data_free (data, client, id);
	
	base = g_strdup_printf ("/apps/drivel/users/%s", username);
	data->user = base;
	data->password = g_strdup_printf ("%s/password", base);
	data->save_password = g_strdup_printf ("%s/save_password", base);
	data->autologin = g_strdup_printf ("%s/autologin", base);
	
	data->default_server_list = g_strdup_printf ("%s/default_server_list", base);
	data->default_server = g_strdup_printf ("%s/default_server", base);
	data->default_mood = g_strdup_printf ("%s/default_mood", base);
	data->default_music = g_strdup_printf ("%s/default_music", base);
	data->default_picture = g_strdup_printf ("%s/default_picture", base);
	data->default_security = g_strdup_printf ("%s/default_security", base);
	data->default_comment = g_strdup_printf ("%s/default_comment", base);
	data->default_autoformat = g_strdup_printf ("%s/default_autoformat", base);
	data->expander_open = g_strdup_printf ("%s/expander_open", base);

	data->proxy = g_strdup ("/system/http_proxy/use_http_proxy");
	data->proxy_url = g_strdup ("/system/http_proxy/host");
	data->proxy_port = g_strdup ("/system/http_proxy/port");
	data->proxy_user = g_strdup ("/system/http_proxy/authentication_user");
	data->proxy_pass = g_strdup ("/system/http_proxy/authentication_password");
	
	data->tray = g_strdup_printf ("%s/tray", base);
	
	data->entry_x = g_strdup_printf ("%s/entry_x", base);
	data->entry_y = g_strdup_printf ("%s/entry_y", base);
	data->entry_height = g_strdup_printf ("%s/entry_height", base);
	data->entry_width = g_strdup_printf ("%s/entry_width", base);
	data->entry_max = g_strdup_printf ("%s/entry_max", base);
	
	data->min_start = g_strdup_printf ("%s/min_start", base);
	data->min_post = g_strdup_printf ("%s/min_post", base);
	data->highlight_syntax = g_strdup_printf ("%s/highlight_syntax", base);
	data->spellcheck = g_strdup_printf ("%s/spellcheck", base);
	
	gconf_client_add_dir (client, base, GCONF_CLIENT_PRELOAD_NONE, NULL);

	add_gconf_notifies (dc);
	
	return;
}

static void
drivel_register_icons (void)
{
	GtkIconFactory *factory;
	GtkIconSet *icon;
	GdkPixbuf *pixbuf;
	GtkStockItem items[] =
	{
		{
			"drivel-post",
			_("_Post"),
			0,
			0,
			NULL
		},
		{
			"drivel-update",
			_("U_pdate"),
			0,
			0,
			NULL
		},
		{
			"drivel-add-friend",
			_("_Add..."),
			0,
			0,
			NULL
		},
		{
			"drivel-add-question",
			_("Add _Question"),
			0,
			0,
			NULL
		},
		{
			"drivel-add-answer",
			_("Add _Answer"),
			0,
			0,
			NULL
		},
		{
			"drivel-insert-button",
			_("_Insert"),
			0,
			0,
			NULL
		},
		{
			"drivel-edit",
			_("_Edit..."),
			0,
			0,
			NULL
		},
		{
			"drivel-login",
			_("_Log In"),
			0,
			0,
			NULL
		},
		{
			"drivel-save-draft",
			_("_Save Draft"),
			0,
			0,
			NULL
		},
		{
			"drivel-dont-save",
			_("_Don't Save"),
			0,
			0,
			NULL
		},
		{
			"drivel-sign-up",
			_("Sign _Up..."),
			0,
			0,
			NULL
		},
		{
			"drivel-insert-link",
			_("Insert Link..."),
			0,
			0,
			NULL
		},
		{
			"drivel-insert-image",
			_("Insert Image..."),
			0,
			0,
			NULL
		},
		{
			"drivel-what-is-lj",
			_("What's a LiveJournal?"),
			0,
			0,
			NULL
		}
	};
	
	factory = gtk_icon_factory_new ();
	
	/* create our own icons */
	pixbuf = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "livejournal.png",
		NULL);
	icon = gtk_icon_set_new_from_pixbuf (pixbuf);
	gtk_icon_factory_add (factory, "drivel-post", icon);
	
	pixbuf = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "drivel" 
		G_DIR_SEPARATOR_S "insert_image.png", NULL);
	icon = gtk_icon_set_new_from_pixbuf (pixbuf);
	gtk_icon_factory_add (factory, "drivel-insert-image", icon);
	
	pixbuf = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "drivel" 
		G_DIR_SEPARATOR_S "insert_object.png", NULL);
	icon = gtk_icon_set_new_from_pixbuf (pixbuf);
	gtk_icon_factory_add (factory, "drivel-insert-link", icon);
	
	/* adjust stock icons */
	icon = gtk_icon_factory_lookup_default (GTK_STOCK_ADD);
	gtk_icon_factory_add (factory, "drivel-add-friend", icon);

	icon = gtk_icon_factory_lookup_default (GTK_STOCK_ADD);
	gtk_icon_factory_add (factory, "drivel-add-question", icon);

	icon = gtk_icon_factory_lookup_default (GTK_STOCK_ADD);
	gtk_icon_factory_add (factory, "drivel-add-answer", icon);

	icon = gtk_icon_factory_lookup_default (GTK_STOCK_GO_FORWARD);
	gtk_icon_factory_add (factory, "drivel-insert-button", icon);
	
	icon = gtk_icon_factory_lookup_default (GTK_STOCK_PROPERTIES);
	gtk_icon_factory_add (factory, "drivel-edit", icon);
	
	icon = gtk_icon_factory_lookup_default (GTK_STOCK_GO_FORWARD);
	gtk_icon_factory_add (factory, "drivel-login", icon);
	
	icon = gtk_icon_factory_lookup_default (GTK_STOCK_SAVE);
	gtk_icon_factory_add (factory, "drivel-save-draft", icon);
	
	icon = gtk_icon_factory_lookup_default (GTK_STOCK_NO);
	gtk_icon_factory_add (factory, "drivel-dont-save", icon);
	
	icon = gtk_icon_factory_lookup_default (GTK_STOCK_SAVE);
	gtk_icon_factory_add (factory, "drivel-update", icon);
	
	icon = gtk_icon_factory_lookup_default (GTK_STOCK_HELP);
	gtk_icon_factory_add (factory, "drivel-what-is-lj", icon);
	
	gtk_icon_factory_add_default (factory);
	gtk_stock_add (items, G_N_ELEMENTS (items));
	
	return;
}

void
drivel_fill_journal_null (DrivelClient *dc)
{
	dc->journal_mood = NULL;
	dc->journal_music = NULL;
	dc->journal_picture = NULL;
	dc->journal_security = NULL;
	dc->journal_comment = NULL;
	dc->journal_autoformat = NULL;
}

static gchar *
init_config_directory (void)
{
	const gchar *home;
	gchar *directory, *pictures;
	GnomeVFSFileInfo info;

	home = g_get_home_dir ();
	if (!home)
		return NULL;
	
	/* try and create the directory, if necessary. */
	directory = g_build_filename (home, ".gnome2", "drivel.d", NULL);
	if (gnome_vfs_get_file_info (directory, &info, 
			GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE) != GNOME_VFS_OK)
	{
		if (gnome_vfs_make_directory (directory, 0700) != GNOME_VFS_OK)
		{
			g_free(directory);
			return NULL;
		}
	}
	pictures = g_build_filename (directory, "pictures", NULL);
	if (gnome_vfs_get_file_info (pictures, &info, 
			GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE) != GNOME_VFS_OK)
	{
		if (gnome_vfs_make_directory (pictures, 0700) != GNOME_VFS_OK)
		{
			g_free(directory);
			g_free(pictures);
			return NULL;
		}
	}
	g_free(pictures);
	
	return directory;
}

static void
drivel_init (DrivelClient *dc)
{
	dc->client = gconf_client_get_default ();
	dc->gconf = g_new0 (DrivelGConfData, 1);
	dc->mood_icons = g_hash_table_new_full (g_str_hash, g_str_equal,
			hash_table_item_free, hash_table_item_free);
	dc->mood_list = gconf_client_get_list (dc->client, "/apps/drivel/moods/mood_list",
			GCONF_VALUE_STRING, NULL);
	dc->moods = gconf_client_get_int (dc->client, "/apps/drivel/moods/moods",
			NULL);
	dc->net = g_new0 (DrivelNet, 1);
	/* FIXME: this needs to be done a per-account basis */
	dc->net->api = BLOG_API_LJ;
	dc->net->server = NULL;
	dc->net->curl_share = curl_share_init ();
	if (!dc->net->curl_share)
		g_warning ("Could not access the curl_share interface.");
	if (curl_share_setopt (dc->net->curl_share, CURLSHOPT_SHARE, 
			CURL_LOCK_DATA_DNS | CURL_LOCK_DATA_COOKIE))
		g_warning ("Could not enable DNS sharing for libcurl.");
	dc->net->fast_servers = 0;
	dc->net->msg_queue = msg_queue_setup ();
	dc->net->cancel = gnome_vfs_cancellation_new ();
	dc->gconf->user = NULL;
	dc->username = NULL;
	dc->password = NULL;
	dc->picture_keywords = NULL;
	dc->picture_filenames = NULL;
	dc->open_dialog = NULL;
	dc->save_dialog = NULL;
	dc->network_progress = NULL;
	dc->response = 0;
	dc->message_dialog = NULL;
	dc->use_fast_servers = TRUE;
	dc->proxy_user = NULL;
	dc->proxy_pass = NULL;
	dc->proxy_url = NULL;
	dc->proxy_proper_format = NULL;
	dc->journals = 0;
	dc->journal_list = NULL;
	dc->active_journal = NULL;
	dc->menu_list = NULL;
	dc->friends_list = NULL;
	dc->login_send_data = NULL;
	dc->invalid_input = FALSE;
	dc->edit_entry = FALSE;
	dc->window_list = NULL;
	dc->time_since_checkfriends = 0;
	dc->checking_friends = FALSE;
	dc->friends_update = FALSE;
	dc->lastupdate = g_strdup ("");
	dc->picture_store = NULL;
	dc->draft_filename = NULL;
	
	drivel_fill_journal_null (dc);
	
	if (!dc->moods)
		dc->mood_list = g_slist_prepend (dc->mood_list, "");
	
	load_stored_moods (dc->mood_icons, dc->moods, dc->client);
	
	gconf_client_add_dir (dc->client, "/apps/drivel/global_settings", GCONF_CLIENT_PRELOAD_NONE, NULL);

	dc->config_directory = init_config_directory();
	
	return;
}

gint
main (gint argc, gchar **argv)
{
	DrivelClient *dc;
	GnomeProgram *program;
	GError *error;
	
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
	
	error = NULL;
	if (gconf_init (argc, argv, &error) == FALSE)
	{
		g_assert (error != NULL);
		g_message (_("GConf init failed: %s"), error->message);
		g_error_free (error);
		exit (EXIT_FAILURE);
	}
	
	if (gnome_vfs_init () == FALSE)
		g_error (_("Could not initialize GnomeVFS!\n"));
	
	program = gnome_program_init (PACKAGE, VERSION, 
			LIBGNOMEUI_MODULE, argc, argv,
			GNOME_PARAM_HUMAN_READABLE_NAME, _("Drivel Journal Editor"),
			GNOME_PARAM_APP_DATADIR, DATADIR,
			NULL);
	
	curl_global_init (CURL_GLOBAL_NOTHING);

	net_mutex = g_mutex_new ();
	
	dc = g_new0 (DrivelClient, 1);

	drivel_init (dc);

	drivel_register_icons ();

	net_start_thread (dc);
	
	login_window_build (dc);
	
	gdk_threads_enter ();
	
	gtk_main ();
	
	gdk_threads_leave ();
	
	return 0;
}
