/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002-2004 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <gnome.h>
#include <glade/glade.h>
#include <gdk-pixbuf/gdk-pixdata.h>
#include <string.h>
#include <curl/curl.h>

#include "drivel_request.h"
#include "blog_lj.h"
#include "network.h"
#include "about.h"
#include "journal.h"
#include "md5.h"
#include "login.h"

extern GMutex *net_mutex;

void
about_cb(GtkWidget *widget, gpointer data)
{
	about_show (data);
	
	return;
}

static gboolean
get_sensitivity (GtkWidget *entry)
{
	gchar *text;
	gint len;
	gboolean retval;
	
	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	len = strlen (text);
	
	if (len > 0)
		retval = TRUE;
	else
		retval = FALSE;
	
	g_free (text);
	
	return retval;
}

void
set_user_login_prefs (DrivelClient *dc, const gchar *username)
{
	gchar *user_password, *proxy_url, *server;
	gboolean user_save_password, user_autologin;

	user_password = gconf_client_get_string (dc->client, dc->gconf->password, NULL);
	user_save_password = gconf_client_get_bool (dc->client, dc->gconf->save_password, NULL);
	user_autologin = gconf_client_get_bool (dc->client, dc->gconf->autologin, NULL);
	
	if (!user_password)
		user_password = g_strdup ("");
	
	gtk_entry_set_text (GTK_ENTRY (GTK_BIN (dc->login_name)->child), username);
	gtk_entry_set_text (GTK_ENTRY (dc->login_password), user_password);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->save_password), user_save_password);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->autologin), user_autologin);

	server = gconf_client_get_string (dc->client, dc->gconf->default_server, NULL);
	if (!server)
		server = g_strdup ("www.livejournal.com");
	g_mutex_lock (net_mutex);
	g_free (dc->net->server);
	dc->net->server = server;
	g_mutex_unlock (net_mutex);
	
	g_free (dc->proxy_url);
	dc->proxy = gconf_client_get_bool (dc->client, dc->gconf->proxy, NULL);
	proxy_url = gconf_client_get_string (dc->client, dc->gconf->proxy_url, NULL);
	if (proxy_url)
		dc->proxy_url = g_strdup (proxy_url);
	else
		dc->proxy_url = NULL;
	dc->proxy_port = gconf_client_get_int (dc->client, dc->gconf->proxy_port, NULL);
	
	g_free (user_password);
	
	return;
}

static void
user_list_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean valid;
	GSList *list, *current, *user_list = NULL;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	
	for (list = gconf_value_get_list (value); list; list = list->next)
		user_list = g_slist_append (user_list, g_strdup (gconf_value_get_string (list->data)));

	if (user_list)
	{
		/* clear out the old list... */
		model = gtk_combo_box_get_model (GTK_COMBO_BOX (dc->login_name));
		valid = gtk_tree_model_get_iter_first (model, &iter);
		while (valid)
			valid = gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
		
		/* and then create the new */
		user_list = g_slist_sort (user_list, string_compare);
		for (current = user_list; current; current = current->next)
			gtk_combo_box_append_text (GTK_COMBO_BOX (dc->login_name), current->data);
	}
	
	return;
}

static gboolean
valid_username (const gchar *username, GSList *user_list)
{
	gboolean retval = FALSE;
	
	if (g_slist_find_custom (user_list, username, string_compare))
		retval = TRUE;

	return retval;
}

static gboolean
login_focus_out_cb (GtkWidget *entry, GdkEventFocus *event, gpointer data)
{
	gchar *username;
	DrivelClient *dc = (DrivelClient *) data;
	
	username = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

	if (!validate_username (username))
	{		
		g_free (username);
			
		return FALSE;
	}

	if (username && strcmp (username, ""))
		drivel_gconf_data_fill (dc->gconf, username, dc->client, &dc->id, dc);
	
	g_mutex_lock (net_mutex);
	g_free (dc->username);
	dc->username = username;
	g_mutex_unlock (net_mutex);
	
	return FALSE;
}

static void
login_changed_cb (GtkWidget *entry, gpointer data)
{
	gboolean state;
	gchar *username;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = get_sensitivity (entry);

	if (state)
	{
		gtk_widget_set_sensitive (dc->login_password, TRUE);
		gtk_widget_set_sensitive (dc->sign_up_button, FALSE);
		gtk_widget_set_sensitive (dc->sign_up_menu, FALSE);
		
		username = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
		
		if (!validate_username (username))
		{
			display_error_dialog (dc, _("Bad username"), 
					_("Username contains invalid characters."));
		
			g_free (username);
			
			return;
		}
		
		if (username && valid_username (username,
				gconf_client_get_list (dc->client, "/apps/drivel/global_settings/user_list", GCONF_VALUE_STRING, NULL)))
		{
			drivel_gconf_data_fill (dc->gconf, username, dc->client, &dc->id, dc);
			
			gconf_client_set_string (dc->client, "/apps/drivel/global_settings/current_user", username, NULL);
			
			set_user_login_prefs (dc, username);
		}
		
		g_mutex_lock (net_mutex);
		g_free (dc->username);
		dc->username = username;
		g_mutex_unlock (net_mutex);
	}
	else
	{
		g_mutex_lock (net_mutex);
		g_free (dc->username);
		dc->username = NULL;
		g_mutex_unlock (net_mutex);
		
		gtk_editable_delete_text (GTK_EDITABLE (dc->login_password), 0, -1);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->save_password), FALSE);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->autologin), FALSE);
		gtk_widget_set_sensitive (dc->login_password, FALSE);
		gtk_widget_set_sensitive (dc->save_password, FALSE);
		gtk_widget_set_sensitive (dc->autologin, FALSE);
		gtk_widget_set_sensitive (dc->login_button, FALSE);
		gtk_widget_set_sensitive (dc->login_menu, FALSE);
		gtk_widget_set_sensitive (dc->sign_up_button, TRUE);
		gtk_widget_set_sensitive (dc->sign_up_menu, TRUE);
	}

	return;
}

static void
password_changed_cb (GtkWidget *entry, gpointer data)
{
	gboolean state, username;
	gchar *password;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = get_sensitivity (entry);
	
	if (state)
	{
		gtk_widget_set_sensitive (dc->save_password, TRUE);
		gtk_widget_set_sensitive (dc->login_button, TRUE);
		gtk_widget_set_sensitive (dc->login_menu, TRUE);
		
		g_mutex_lock (net_mutex);
		if (dc->username)
			username = TRUE;
		else
			username = FALSE;
		g_mutex_unlock (net_mutex);
		
		if (username && 
				gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->save_password)))
		{
			password = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

			gconf_client_set_string (dc->client, dc->gconf->password, password, NULL);
			g_free (password);
		}
	}
	else
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->save_password), FALSE);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->autologin), FALSE);
		gtk_widget_set_sensitive (dc->save_password, FALSE);
		gtk_widget_set_sensitive (dc->autologin, FALSE);
		gtk_widget_set_sensitive (dc->login_button, FALSE);
		gtk_widget_set_sensitive (dc->login_menu, FALSE);
	}
		
	return;
}

static void
save_password_cb (GtkWidget *button, gpointer data)
{
	gboolean state, username;
	gchar *password;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));

	g_mutex_lock (net_mutex);
	if (dc->username)
		username = TRUE;
	else
		username = FALSE;
	g_mutex_unlock (net_mutex);
	
	if (username && dc->gconf->user)
		gconf_client_set_bool (dc->client, dc->gconf->save_password, state, NULL);
		
	if (state)
	{
		gtk_widget_set_sensitive (dc->autologin, TRUE);
		
		if (username && dc->gconf->user)
		{
			password = gtk_editable_get_chars (GTK_EDITABLE (dc->login_password), 0, -1);

			gconf_client_set_string (dc->client, dc->gconf->password, password, NULL);
			g_free (password);
		}
	}
	else
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->autologin), FALSE);
		gtk_widget_set_sensitive (dc->autologin, FALSE);
		
		if (username && dc->gconf->user)
			gconf_client_set_string (dc->client, dc->gconf->password, "", NULL);
	}
	
	return;
}

static void
autologin_cb (GtkWidget *button, gpointer data)
{
	gboolean state, username;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	
	g_mutex_lock (net_mutex);
	if (dc->username)
		username = TRUE;
	else
		username = FALSE;
	g_mutex_unlock (net_mutex);
	
	if (username && dc->gconf->user)
		gconf_client_set_bool (dc->client, dc->gconf->autologin, state, NULL);
	
	return;
}

static void
login (DrivelClient *dc)
{
	gchar *login, *password, *version, *server;
	guchar md5_password[16 * 2 + 1];
	guint i;
	md5_state_t state;
	md5_byte_t digest[16];
	DrivelRequest *dr;
	
	login = gtk_editable_get_chars (GTK_EDITABLE (GTK_BIN (dc->login_name)->child), 0, -1);
	password = gtk_editable_get_chars (GTK_EDITABLE (dc->login_password), 0, -1);
	version = g_strdup_printf ("GNOME-Drivel/%s", VERSION);
	
	server = gconf_client_get_string (dc->client, dc->gconf->default_server, NULL);
	if (!server)
		server = g_strdup ("www.livejournal.com");
	g_mutex_lock (net_mutex);
	g_free (dc->net->server);
	dc->net->server = server;
	g_mutex_unlock (net_mutex);
	
	/* encrypt the password */
	md5_init (&state);
	md5_append (&state, password, strlen (password));
	md5_finish (&state, digest);

	for (i = 0; i < 16; i++)
		g_snprintf (md5_password + i * 2, 3, "%02x", digest[i]);
	md5_password [16 * 2] = '\0';
	
	g_mutex_lock (net_mutex);
	g_free (dc->username);
	g_free (dc->password);
	dc->username = g_strdup (login);
	dc->password = g_strdup (md5_password);
	g_mutex_unlock (net_mutex);
	
	/* build & enqueue the login request */
	
	dr = blog_lj_build_login_request (login, version, dc->moods);
	net_enqueue_request (dc, dr);
	
	g_free (login);
	g_free (password);
	g_free (version);
	
	return;
}

static void
login_button_cb (GtkWidget *widget, gpointer data)
{
	DrivelClient *dc = (DrivelClient *)data;
	
	login (dc);
	
	return;
}

static void
sign_up_cb (GtkWidget *menu, gpointer data)
{
	gnome_url_show ("http://www.livejournal.com/create.bml", NULL);
	
	return;
}

static void
help_cb (GtkWidget *menu, gpointer data)
{
	gnome_help_display ("drivel.xml", NULL, NULL);
	
	return;
}

static void
what_is_lj_cb (GtkWidget *menu, gpointer data)
{
	gnome_url_show ("http://www.livejournal.com/", NULL);
	
	return;
}

void
login_window_build (DrivelClient *dc)
{	
	GladeXML *xml;
	GtkWidget *menubar;
	GtkActionGroup *action_group;
	GtkUIManager *ui_manager;
	GtkAccelGroup *accel_group;
	GSList *user_list, *current;
	gchar *user_login;

	static GtkActionEntry entries[] =
	{
		{ "FileMenu", NULL, N_("_Journal") },
		{ "EditMenu", NULL, N_("_Edit") },
		{ "HelpMenu", NULL, N_("_Help") },
		{ "SignUp", "drivel-sign-up", N_("Sign _Up..."), NULL, "tooltip", G_CALLBACK (sign_up_cb) },
		{ "Login", "drivel-login", NULL, NULL, NULL, G_CALLBACK (login_button_cb) },
		{ "Quit", GTK_STOCK_QUIT, NULL, NULL, NULL, G_CALLBACK (exit_cb) },
		{ "Preferences", GTK_STOCK_PREFERENCES, N_("_Preferences"), NULL, "tooltip", G_CALLBACK (edit_preferences_cb) },
		{ "Contents", GTK_STOCK_HELP, N_("_Contents"), "F1", NULL, G_CALLBACK (help_cb) },
		{ "WhatIsLJ", "drivel-what-is-lj", NULL, NULL, NULL, G_CALLBACK (what_is_lj_cb) },
		{ "About", GNOME_STOCK_ABOUT, N_("_About"), NULL, "tooltip", G_CALLBACK (about_cb) }
	};
	
	static const gchar *ui_description =
	{
		"<ui>"
		"  <menubar name='MainMenu'>"
		"    <menu action='FileMenu'>"
		"      <menuitem action='SignUp' />"
		"      <menuitem action='Login' />"
		"      <separator name='sep1'/>"
		"      <menuitem action='Quit' />"
		"    </menu>"
		"    <menu action='EditMenu'>"
		"      <menuitem action='Preferences' />"
		"    </menu>"
		"    <menu action='HelpMenu'>"
		"      <menuitem action='Contents' />"
		"      <menuitem action='WhatIsLJ' />"
		"      <menuitem action='About' />"
		"    </menu>"
		"  </menubar>"
		"</ui>"
	};

	xml = load_glade_xml ();
	if (!xml)
		return;
	
	dc->login_window = glade_xml_get_widget (xml, "login_window");
	gtk_window_set_icon_from_file (GTK_WINDOW (dc->login_window),
			DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S
			"livejournal.png", NULL);

	gtk_image_set_from_file (GTK_IMAGE (glade_xml_get_widget (xml, "login_window_image")),
			DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "drivel"
			G_DIR_SEPARATOR_S "drivel_splash.png");

	dc->login_name = glade_xml_get_widget (xml, "login_window_username");
	gtk_entry_set_max_length (GTK_ENTRY (GTK_BIN (dc->login_name)->child), 15);
	gtk_entry_set_width_chars (GTK_ENTRY (GTK_BIN (dc->login_name)->child), 15);
	gtk_entry_set_activates_default (GTK_ENTRY (GTK_BIN (dc->login_name)->child), TRUE);
		
	dc->login_password = glade_xml_get_widget (xml, "login_window_password");
	dc->save_password = glade_xml_get_widget (xml, "login_window_save_password");
	dc->autologin = glade_xml_get_widget (xml, "login_window_auto_login");
	
	dc->sign_up_button = glade_xml_get_widget (xml, "login_window_sign_up");
	gtk_button_set_use_stock (GTK_BUTTON (dc->sign_up_button), TRUE);
	dc->login_button = glade_xml_get_widget (xml, "login_window_login");
	gtk_button_set_use_stock (GTK_BUTTON (dc->login_button), TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET (dc->login_button), FALSE);
	
	drivel_push_current_window (dc, dc->login_window);
	
	/* setup the menus */
	
	action_group = gtk_action_group_new ("MenuActions");
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (action_group, entries, G_N_ELEMENTS (entries), dc);
	
	ui_manager = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (ui_manager, action_group, 0);
	
	accel_group = gtk_ui_manager_get_accel_group (ui_manager);
	gtk_window_add_accel_group (GTK_WINDOW (dc->login_window), accel_group);
	
	if (!gtk_ui_manager_add_ui_from_string (ui_manager, ui_description, -1, NULL))
		g_error ("Error while building login window menu");
	
	menubar = gtk_ui_manager_get_widget (ui_manager, "/MainMenu");
	gnome_app_set_menus (GNOME_APP (dc->login_window), GTK_MENU_BAR (menubar));
	
	gtk_widget_show_all (dc->login_window);

	dc->sign_up_menu = gtk_ui_manager_get_widget (ui_manager, "/MainMenu/FileMenu/SignUp");
	dc->login_menu = gtk_ui_manager_get_widget (ui_manager, "/MainMenu/FileMenu/Login");
	gtk_widget_set_sensitive (dc->login_menu, FALSE);
	
	/* do this before connecting the changed signal */
	user_list = gconf_client_get_list (dc->client, "/apps/drivel/global_settings/user_list",
			GCONF_VALUE_STRING, NULL);
	gconf_client_notify_add (dc->client, "/apps/drivel/global_settings/user_list", user_list_changed_cb,
			dc, NULL, NULL);
	
	if (user_list)
	{
		user_list = g_slist_sort (user_list, string_compare);
		for (current = user_list; current; current = current->next)
			gtk_combo_box_append_text (GTK_COMBO_BOX (dc->login_name), current->data);
	}
	gtk_entry_set_text (GTK_ENTRY (GTK_BIN (dc->login_name)->child), "");
	
	/* connect the signals */
	
	g_signal_connect (G_OBJECT (GTK_BIN (dc->login_name)->child), "focus_out_event",
			G_CALLBACK (login_focus_out_cb), dc);
	g_signal_connect (G_OBJECT (GTK_BIN (dc->login_name)->child), "changed", 
			G_CALLBACK (login_changed_cb), dc);
	g_signal_connect (G_OBJECT (dc->login_password), "changed",
			G_CALLBACK (password_changed_cb), dc);
	g_signal_connect (G_OBJECT (dc->save_password), "toggled",
			G_CALLBACK (save_password_cb), dc);
	g_signal_connect (G_OBJECT (dc->autologin), "toggled",
			G_CALLBACK (autologin_cb), dc);
	g_signal_connect (G_OBJECT (dc->sign_up_button), "clicked",
			G_CALLBACK (sign_up_cb), NULL);
	g_signal_connect (G_OBJECT (dc->login_button), "clicked",
			G_CALLBACK (login_button_cb), dc);
	g_signal_connect (G_OBJECT (dc->login_window), "delete_event",
			G_CALLBACK (delete_event_cb), dc);

	/* setup the user's preferences */
	user_login = gconf_client_get_string (dc->client, "/apps/drivel/global_settings/current_user", NULL);	
	
	if (user_login && user_login [0] != '\0')
	{
		drivel_gconf_data_fill (dc->gconf, user_login, dc->client, &dc->id, dc);
		
		set_user_login_prefs (dc, user_login);
	}
	
	g_free (user_login);
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->autologin)))
		login_button_cb (dc->login_button, dc);
	
	gtk_editable_select_region (GTK_EDITABLE (GTK_BIN (dc->login_name)->child), 0, -1);
	gtk_widget_grab_default (dc->login_button);
	
	return;
}
