require 'runit/testcase'
require 'runit/cui/testrunner'
require 'drb/drb'
require 'drb/extservm'
require 'timeout'

class TestService
  @@scripts = %w(ut_drb.rb ut_array.rb ut_port.rb ut_large.rb)

  def initialize(uri=nil)
    @manager = DRb::ExtServManager.new
    @@scripts.each do |nm|
      DRb::ExtServManager.command[nm] = "ruby #{nm}"
    end
    @server = DRb::DRbServer.new(uri, @manager)
  end
  attr_reader :manager, :server
end

class Onecky
  include DRbUndumped
  def initialize(n)
    @num = n
  end

  def to_i
    @num.to_i
  end

  def sleep(n)
    Kernel.sleep(n)
    to_i
  end
end

class FailOnecky < Onecky
  class OneckyError < RuntimeError; end
  def to_i
    raise(OneckyError, @num.to_s)
  end
end


class DRbCoreTest < RUNIT::TestCase
  def setup
    @ext = $manager.service('ut_drb.rb')
    @there = @ext.front
  end

  def teardown
    @ext.stop_service
  end

  def test_01
    assert_equal(@there.hello, "hello")
    onecky = Onecky.new('3')
    assert_equal(@there.sample(onecky, 1, 2), 6)
    ary = @there.to_a
    assert_equal(ary.size, 1)
    assert_kind_of(DRb::DRbObject, ary[0])
  end

  def test_01_02_loop
    onecky = Onecky.new('3')
    50.times do 
      assert_equal(@there.sample(onecky, 1, 2), 6)
      ary = @there.to_a
      assert_equal(ary.size, 1)
      assert_kind_of(DRb::DRbObject, ary[0])
    end
  end

  def test_02_unknown
    obj = @there.unknown_obj
    assert_kind_of(DRb::DRbUnknown, obj)
    assert_equal(obj.name, 'DRbEx')

    assert_exception(DRb::DRbUnknownError) do
      @there.unknown_error
    end

    onecky = FailOnecky.new('3')

    assert_exception(FailOnecky::OneckyError) do
      @there.sample(onecky, 1, 2)
    end
  end

  def test_03
    assert_equal(@there.sum(1, 1, 1, 1, 1, 1, 1, 1), 8)
    assert_exception(ArgumentError) do
      @there.sum(1, 1, 1, 1, 1, 1, 1, 1, 1)
    end
    assert_exception(TypeError) do
      @there.sum('1' * 1024)
    end
  end

  def test_04
    assert_respond_to('sum', @there)
    assert(!(@there.respond_to? "foobar"))
  end

  def test_05_eq
    a = @there.to_a[0]
    b = @there.to_a[0]
    assert(a.id != b.id)
    assert(a != b)
    assert(a.hash != b.hash)
    assert(! a.eql?(b))
    require 'drb/eq'
    assert(a == b)
    assert_equal(a, b)
    assert(a == @there)
    assert_equal(a.hash, b.hash)
    assert_equal(a.hash, @there.hash)
    assert(a.eql?(b))
    assert(a.eql?(@there))
  end

  def test_06_timeout
    ten = Onecky.new(10)
    assert_exception(TimeoutError) do
      @there.do_timeout(ten)
    end
    assert_exception(TimeoutError) do
      @there.do_timeout(ten)
    end
  end

end

class DRbYieldTest < RUNIT::TestCase
  def setup
    @ext = $manager.service('ut_drb.rb')
    @there = @ext.front
  end

  def teardown
    @ext.stop_service
  end

  def test_01_one
    one = nil
    @there.echo_yield_1([]) {|one|}
    assert_equal(one, [])
    
    one = nil
    @there.echo_yield_1(1) {|one|}
    assert_equal(one, 1)
    
    one = nil
    @there.echo_yield_1(nil) {|one|}
    assert_equal(one, nil)
  end

  def test_02_two
    one = two = nil
    @there.echo_yield_2([], []) {|one, two|}
    assert_equal(one, [])
    assert_equal(two, [])

    one = two = nil
    @there.echo_yield_2(1, 2) {|one, two|}
    assert_equal(one, 1)
    assert_equal(two, 2)

    one = two = nil
    @there.echo_yield_2(3, nil) {|one, two|}
    assert_equal(one, 3)
    assert_equal(two, nil)
  end

  def test_03_many
    s = nil
    @there.echo_yield {|*s|}
    assert_equal(s, [])
    @there.echo_yield(nil) {|*s|}
    assert_equal(s, [nil])
    @there.echo_yield(1) {|*s|}
    assert_equal(s, [1])
    @there.echo_yield(1, 2) {|*s|}
    assert_equal(s, [1, 2])
    @there.echo_yield(1, 2, 3) {|*s|}
    assert_equal(s, [1, 2, 3])
    @there.echo_yield([], []) {|*s|}
    assert_equal(s, [[], []])
    @there.echo_yield([]) {|*s|}
    assert_equal(s, []) # !
  end
end

class RubyYieldTest < DRbYieldTest
  def echo_yield(*arg)
    yield(*arg)
  end

  def echo_yield_1(a)
    yield(a)
  end

  def echo_yield_2(a, b)
    yield(a, b)
  end

  def setup
    @there = self
  end
  
  def teardown
  end
end

class DRbAryTest < RUNIT::TestCase
  def setup
    @ext = $manager.service('ut_array.rb')
    @there = @ext.front
  end

  def teardown
    @ext.stop_service
  end

  def test_01
    assert_kind_of(DRb::DRbObject, @there)
  end

  def test_02_collect
    ary = @there.collect do |x| x + x end
    assert_kind_of(Array, ary)
    assert_equal(ary, [2, 4, 'IIIIII', 8, 'fivefive', 12])
  end

  def test_03_redo
    ary = []
    count = 0
    @there.each do |x|
      count += 1
      ary.push x
      redo if count == 3
    end
    assert_equal(ary, [1, 2, 'III', 'III', 4, 'five', 6])
  end

  def test_04_retry
    retried = false
    ary = []
    @there.each do |x|
      ary.push x
      if x == 4 && !retried
	retried = true
	retry
      end
    end
    assert_equal(ary, [1, 2, 'III', 4, 1, 2, 'III', 4, 'five', 6])
  end

  def test_05_break
    ary = []
    @there.each do |x|
      ary.push x
      break if x == 4
    end
    assert_equal(ary, [1, 2, 'III', 4])
  end

  def test_06_next
    ary = []
    @there.each do |x|
      next if String === x
      ary.push x
    end
    assert_equal(ary, [1, 2, 4, 6])
  end
end

class DRbMServerTest < RUNIT::TestCase
  def setup
    @ext = $manager.service('ut_drb.rb')
    @there = @ext.front
    @server = (1..3).collect do |n|
      DRb::DRbServer.new(nil, Onecky.new(n.to_s))
    end
  end

  def teardown
    @server.each do |s|
      s.stop_service
    end
    @ext.stop_service
  end

  def test_01
    assert_equal(@there.sample(@server[0].front, @server[1].front, @server[2].front), 6)
  end
end

class DRbReusePortTest < DRbAryTest
  def setup
    sleep 1
    @ext = $manager.service('ut_port.rb')
    @there = @ext.front
  end
end

class DRbLargeTest < RUNIT::TestCase
  def setup
    sleep 1
    @ext = $manager.service('ut_large.rb')
    @there = @ext.front
  end

  def test_01_large_ary
    ary = [2] * 102400
    assert_equal(102400, @there.size(ary))
    assert_equal(204800, @there.sum(ary))
  end

  def test_02_large_ary
    ary = ["Hello, World"] * 102400
    assert_equal(102400, @there.size(ary))
  end

  def test_03_large_ary
    ary = [Thread.current] * 102400
    assert_equal(102400, @there.size(ary))
  end
end

if __FILE__ == $0
  $testservice = TestService.new
  $manager = $testservice.manager

  RUNIT::CUI::TestRunner.run(DRbCoreTest.suite)
  RUNIT::CUI::TestRunner.run(RubyYieldTest.suite)
  RUNIT::CUI::TestRunner.run(DRbYieldTest.suite)
  RUNIT::CUI::TestRunner.run(DRbAryTest.suite)
  RUNIT::CUI::TestRunner.run(DRbMServerTest.suite)
  RUNIT::CUI::TestRunner.run(DRbReusePortTest.suite)
  RUNIT::CUI::TestRunner.run(DRbLargeTest.suite)
end
