/* doscan - Denial Of Service Capable Auditing of Networks
 * Copyright (C) 2003 Florian Weimer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef ENGINE_TCP_H
#define ENGINE_TCP_H

#include "ticks.h"

#include <pcre.h>

struct scan_host_t;             /* from scan.h */

/* These routines deal with transactions over half-duplex TCP
   connections.  You can post read and write requests and get notified
   if they complete. */

typedef enum {
  ENGINE_TCP_MAKE_COPY = 1, /* make copy on call and free afterwards */
  ENGINE_TCP_NO_COPY = 2, /* make no copy, and do not free (constant) */
  ENGINE_TCP_NO_COPY_AND_FREE = 3 /* make no copy, but free after send */
} engine_tcp_copy_t;
/* This flags control the behavior of the send routine below. */

typedef void (*engine_tcp_callback_t) (struct scan_host_t*);
/* Callback on successful write completion, or on connection
   termination.  (Both callbacks are different, but share the same
   interface.) */

typedef void (*engine_tcp_receive_callback_t) (struct scan_host_t*,
                                               const char *, unsigned);
/* Callback on successful reception of data. */

typedef struct {
  char *buffer;
  unsigned size, offset;
  engine_tcp_copy_t copy;
  pcre *regexp;
  ticks_t connect_time;
  engine_tcp_callback_t completion_callback, close_callback;
  engine_tcp_receive_callback_t receive_callback;
} engine_tcp_t;
/* You must not modify this structure.  It is internal.  Instead, use
   the routines below. */

void engine_tcp_open (struct scan_host_t *, engine_tcp_callback_t, engine_tcp_callback_t);
/* Initialize the host and engine structure.

   This routine assumes that the data member of the host structure
   already points to a memory chunk, with a engine_tcp_t object at the
   beginning.

   The first callback is called when the connection has been
   established succesfully.  The second one is called when a
   connection is terminated.  No callback is invoked if the connection
   attempt fails. */

void engine_tcp_send (struct scan_host_t *, const char *, unsigned,
                      engine_tcp_copy_t, engine_tcp_callback_t);
/* Post a send request.  Memory is handled according to the COPY
   parameter. */

void engine_tcp_receive (struct scan_host_t *, char *, unsigned,
                         engine_tcp_receive_callback_t);
/* Post a receive request.  If the specified buffer is null, a new one
   is allocated and automatically freed afterwards.*/

void engine_tcp_receive_until_match (struct scan_host_t *, pcre *,
                                     char *, unsigned,
                                     engine_tcp_receive_callback_t);
/* Post a receive request, complete if buffer matches regexp.  If the
   specified buffer is null, a new one is allocated and automatically
   freed afterwards. */

void engine_tcp_receive_packet (struct scan_host_t *, unsigned,
                                engine_tcp_receive_callback_t);
/* Post a receive request, and read as much as available (up to SIZE
   bytes).  */

ticks_t engine_tcp_connect_time (struct scan_host_t *);
/* Return the time the connection was established. */

void engine_tcp_close (struct scan_host_t *);
/* Close a TCP connection.  The close callback is invoked. */

#endif /* ENGINE_TCP_H */
