#!/usr/bin/env python
"Templates for generating dolfin::Form subclasses."

__author__ = "Martin Alnes (martinal@simula.no) and Anders Logg (logg@simula.no)"
__date__ = "2008-11-06 -- 2009-05-14"
__copyright__ = "Copyright (C) 2008-2009 Martin Alnes and Anders Logg"
__license__  = "GNU GPL version 3 or any later version"

from itertools import chain

# Template code for form class
form_class_template = """\
class %(classname)s: public dolfin::Form
{
public:

%(constructors)s

  // Destructor
  ~%(classname)s()
  {}

  /// Return the number of the coefficient with this name
  virtual dolfin::uint coefficient_number(const std::string& name) const
  {
%(coefficient_number)s
  }

  /// Return the name of the coefficient with this number
  virtual std::string coefficient_name(dolfin::uint i) const
  {
%(coefficient_name)s
  }

%(members)s
};"""

# Template code for Form constructor
form_constructor_template = """\
  // Constructor
  %(classname)s(%(argumentlist)s):
    dolfin::Form(%(rank)d, %(num_coefficients)d)%(initializerlist)s
  {
%(body)s
  }"""

# Unused comments
form_constructor_comments = """
  // Create form on given function space(s)
  // Create form on given function space(s) (shared data)
  // Create form on given function space(s) with given coefficient(s)
  // Create form on given function space(s) with given coefficient(s) (shared data)
"""

def generate_form_constructor(classname,
                              ufc_form_classname,
                              rank,
                              num_coefficients,
                              coefficient_names,
                              shared_function_space,
                              shared_coefficients,
                              shared_mesh,
                              include_coefficients):
    initializerlist = []
    argumentlist = []
    lines = []

    function_space_names = ["V%d" % i for i in range(rank)]

    # Check if we have a form of positive rank or a functional
    if rank:

        # Generate argument list
        for name in function_space_names:
            if shared_function_space:
                argument = "boost::shared_ptr<const dolfin::FunctionSpace> %s" % name
            else:
                argument = "const dolfin::FunctionSpace& %s" % name
            argumentlist.append(argument)

        # Initialize function spaces
        for i, name in enumerate(function_space_names):
            if shared_function_space:
                code = "    _function_spaces[%d] = %s;" % (i, name)
            else:
                code = "    _function_spaces[%d] = reference_to_no_delete_pointer(%s);" % (i, name)
            lines.append(code)

        lines.append("")

    else:

        # Handle initialization of mesh for functionals
        if shared_mesh:
            argument = "boost::shared_ptr<const dolfin::Mesh> mesh"
            code = "    _mesh = mesh;"
        else:
            argument = "const dolfin::Mesh& mesh"
            code = "    _mesh = reference_to_no_delete_pointer(mesh);"

        argumentlist.append(argument)
        lines.append(code)

    # Initialize Coefficient objects with a reference to this Form
    for (number, name) in enumerate(coefficient_names):
        initializerlist.append("%s(*this, %d)" % (name, number))

    if include_coefficients:

        # Generate argument list
        for name in coefficient_names:
            if shared_coefficients:
                a = "boost::shared_ptr<const dolfin::GenericFunction> %s" % name
            else:
                a = "const dolfin::GenericFunction& %s" % name
            argumentlist.append(a)

        # Initialize coefficients
        for name in coefficient_names:
            if shared_coefficients:
                code = "    this->%s = *%s;" % (name, name)
            else:
                code = "    this->%s = %s;" % (name, name)
            lines.append(code)

        lines.append("")

    # Construct UFC form
    code = "    _ufc_form = boost::shared_ptr<const ufc::form>(new %s());" % ufc_form_classname
    lines.append(code)

    # Chain statements together
    if initializerlist:
        initializerlist = ", " + ", ".join(initializerlist)
    else:
        initializerlist = ""
    argumentlist = ", ".join(argumentlist)
    body = "\n".join(lines)

    # Apply class template
    args = {"classname": classname,
            "rank": rank,
            "num_coefficients": num_coefficients,
            "argumentlist": argumentlist,
            "initializerlist": initializerlist,
            "body": body}
    code = form_constructor_template % args
    return code

def generate_form_class(classname,
                        ufc_form_classname,
                        functionspace_classnames,
                        basespace_classnames,
                        coefficientspace_classnames,
                        coefficient_classnames,
                        coefficient_names):

    """Generate code for a dolfin::Form subclass.

    functionspace_classnames:     [0, r+n) --> str
    basespace_classnames:         [0, r)   --> str
    coefficientspace_classnames:  [0, n)   --> str
    coefficient_classnames:       [0, n)   --> str
    coefficient_names:            [0, n)   --> str
    """

    rank = len(basespace_classnames)
    num_coefficients = len(coefficient_names)

    typedefs = []

    # Typedefs for test and trial spaces
    for i in range(rank):
        a, b = functionspace_classnames[i], basespace_classnames[i]
        if not a == b:
            code = "  typedef %s %s;" % (a, b)
            typedefs.append(code)

    # Typedefs for coefficient spaces
    for i in range(num_coefficients):
        a, b = functionspace_classnames[i+rank], coefficientspace_classnames[i]
        if not a == b:
            code = "  typedef %s %s;" % (a, b)
            typedefs.append(code)

    # Member variables for coefficients
    members = []
    for i in range(num_coefficients):
        code = "  dolfin::CoefficientAssigner %s;" % coefficient_names[i]
        members.append(code)

    # Build constructors
    constructors = []
    for shared_mesh in ((False, True) if rank == 0 else (False,)):
        for shared_function_space in ((False, True) if rank >0 else (False,)):

            # Generate constructor with no coefficients
            code = generate_form_constructor(classname,
                                             ufc_form_classname,
                                             rank,
                                             num_coefficients,
                                             coefficient_names,
                                             shared_function_space,
                                             False,
                                             shared_mesh,
                                             False)
            constructors.append(code)

            # Generate constructors with coefficients
            if coefficient_names:
                for shared_coefficients in (False, True):
                    code = generate_form_constructor(classname,
                                                     ufc_form_classname,
                                                     rank,
                                                     num_coefficients,
                                                     coefficient_names,
                                                     shared_function_space,
                                                     shared_coefficients,
                                                     shared_mesh,
                                                     True)
                    constructors.append(code)

    # Generate coefficient name<->number mapping functions
    if num_coefficients:
        ifstr = "if "
        coefficient_number = ''
        coefficient_name   = '    switch (i)\n    {\n'
        for i, name in enumerate(coefficient_names):
            coefficient_number += '    %s(name == "%s")\n      return %d;\n' % (ifstr, name, i)
            coefficient_name   += '    case %d:\n      return "%s";\n' % (i, name)
            ifstr = "else if "
        coefficient_name   += '    }\n'
        coefficient_number += '\n    dolfin::error("Invalid coefficient.");\n'
        coefficient_number += '    return 0;'
        coefficient_name   += '\n    dolfin::error("Invalid coefficient.");\n'
        coefficient_name   += '    return "unnamed";'
        update_coefficients = "\n".join("    %s = coefficients.%s.function_pointer();" % (name, name) for name in coefficient_names)
    else:
        coefficient_number  = '\n    dolfin::error("No coefficients.");\n'
        coefficient_number += '    return 0;'
        coefficient_name    = '\n    dolfin::error("No coefficients.");\n'
        coefficient_name   += '    return "unnamed";'
        update_coefficients = ''

    # Chain statements together
    constructors = "\n\n".join(constructors)
    members = "\n".join(["  // Typedefs"] + typedefs + ["", "  // Coefficients"] + members)

    # Apply class template
    args = {"classname": classname,
            "constructors": constructors,
            "coefficient_number": coefficient_number,
            "coefficient_name": coefficient_name,
            "update_coefficients": update_coefficients,
            "members": members }
    code = form_class_template % args
    return code

def _test_form_constructors():
    classname = "MyForm"
    ufc_form_classname = "my_ufc_form"
    coefficients = []

    rank = 0
    shared_function_space, shared_coefficients = False, False
    print generate_form_constructor(classname, ufc_form_classname, rank, num_coefficients, coefficients, shared_function_space, shared_coefficients, True, None)

    rank = 2
    print generate_form_constructor(classname, ufc_form_classname, rank, num_coefficients, coefficients, shared_function_space, shared_coefficients, True, None)

    shared_function_space, shared_coefficients = True, True
    print generate_form_constructor(classname, ufc_form_classname, rank, num_coefficients, coefficients, shared_function_space, shared_coefficients, True, None)

    coefficients = ["C0", "C1", "C2"]
    for shared_function_space in (True, False):
        for shared_coefficients in (True, False):
           print
           print "shared_function_space, shared_coefficients =", shared_function_space, shared_coefficients
           print generate_form_constructor(classname, ufc_form_classname, rank, num_coefficients, coefficients, shared_function_space, shared_coefficients, True, None)

def _test_form():
    # An arbitrary form
    from ufl import FiniteElement, TestFunction, TrialFunction, Coefficient, dx
    from ufl.algorithms import FormData
    element = FiniteElement("CG", "triangle", 1)
    v = TestFunction(element)
    u = TrialFunction(element)
    f = Coefficient(element)
    g = Coefficient(element)
    a = f*g*u*v*dx

    # ---

    # Input from form compiler
    formdata = FormData(a)
    prefix = "Prefix"
    classname = "MyForm"
    ufc_form_classname = "my_ufc_form"
    coefficient_names = ["f", "g"]

    # ---

    # Define lists of names for each coefficient
    import re
    def strip_special(s):
        r = re.compile("([(),. -\"']+)")
        s = r.subn("_", s)
        s = s[0].strip("_")
        return s
    element_strings = [strip_special(repr(e)) for e in formdata.elements]

    # Unique function space class for each unique element in sfc, may be different in ffc:
    functionspace_classnames = ["%sFunctionSpace%s" % (prefix, e) for e in element_strings]

    # ---

    # Derived input
    rank = formdata.rank
    num_coefficients = formdata.num_coefficients

    # Fall back on default names
    if not coefficient_names:
        coefficient_names = [None]*num_coefficients
    coefficient_names = [cn if cn else "w%d" % i for (i,cn) in enumerate(coefficient_names)]

    # FIXME: Move typedefs outside of Form?
    # Easy names to use in dolfin C++ (typedefs inside of form class, need no prefix)
    typedef_prefix = ""
    basespace_classnames = ["%sFunctionSpace%d" % (typedef_prefix, i) for i in range(rank)]
    if rank == 1:
        basespace_classnames[0] = "%sTestSpace" % typedef_prefix
    if rank == 2:
        basespace_classnames[0] = "%sTestSpace" % typedef_prefix
        basespace_classnames[1] = "%sTrialSpace" % typedef_prefix
    coefficientspace_classnames = ["%sCoefficientSpace_%s" % (typedef_prefix, cn) for cn in coefficient_names]
    coefficient_classnames      = ["%sCoefficient_%d" % (typedef_prefix, i) for i in range(num_coefficients)]

    # Generate form code
    print generate_form_class(classname, ufc_form_classname, functionspace_classnames, basespace_classnames,
                              coefficientspace_classnames, coefficient_classnames, coefficient_names)

if __name__ == "__main__":
    #_test_form_constructors()
    _test_form()
