/*
 * Copyright (C) Michael Stickel <michael@cubic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <linux/version.h>
#include <linux/config.h>

#if !defined(CONFIG_I2C) && !(defined(CONFIG_I2C_MODULE) && CONFIG_I2C_MODULE==1)
#error DMX4Linux I2C devices need I2C support in your kernel
#endif

#include <linux/module.h>
#include <linux/slab.h>

#include <linux/init.h>

#include <linux/i2c.h>
#include <linux/i2c-dev.h>

#include <dmx/dmxdev.h>


#define ONDEBUG(arg...)  arg

typedef struct
{
  DMXUniverse        *universe;

  struct i2c_client  *clients[64];  /* 512 / 8 = 64 TDA8444 per universe */
  int                 num_clients;

  wait_queue_head_t   tx_waitqueue;     /* wait for data to transmit */
  int                 tx_thread_pid;     /* -1 if not running */
  int                 tx_thread_state;  /* 0 not running, 1 running, 2 whould-exiting */

  unsigned char       buffer[512];
} I2C_Universe;



MODULE_AUTHOR("Michael Stickel <michael@cubic.org>");
MODULE_DESCRIPTION("DMX-interface for the I2C bus supporting device: TDA8444");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,17)
MODULE_LICENSE("GPL");
#endif


static int  i2c_transmitter_thread (void *user_data)
{
  I2C_Universe *i2c_u = (I2C_Universe *)user_data;
  struct i2c_client *client = NULL;
  unsigned char i2c_data[512]; /* 6 bit values for comparation */
  unsigned char i2c_buffer[9];
  int   i;
  int   j;

  if (!i2c_u)
    return -1;

  /*
   * This thread doesn't need any user-level access,
   * so get rid of all our resources
   */
  exit_files(current);  /* daemonize doesn't do exit_files */

/*  daemonize(); */

  strcpy(current->comm, "dmx_i2cd");

  i2c_u->tx_thread_state = 1;

  do
    {
      if(interruptible_sleep_on_timeout (&i2c_u->tx_waitqueue, 250))
        {
          /* immer zu 8 byte bloecken clustern */
          for (i=0; i<512/8 && i<i2c_u->num_clients; i++)
            {
	      /*
               * check if a channel in this block has changed
               * and update the buffer. If so flush the whole buffer.
               */
              char changed = 0;

              i2c_buffer[0] = 0;
              client = i2c_u->clients[i];

              for (j=0; j<8; j++)
                {
                  int nr = i*8+j;
                  unsigned char newval = i2c_u->buffer[nr]>>2;
                  if (i2c_data[nr] != newval)
                    changed=1;
                  i2c_buffer[j+1] = i2c_data[nr] = newval;
                }
              if (changed)
		i2c_master_send(client, i2c_buffer, 9);
            }
        }
      /* else timeout */

    } while (!signal_pending(current));
  /* while (i2c_u->tx_thread_state!=2) */

  i2c_u->tx_thread_state = 0;

  return 0;
}



static int start_i2cdmx_thread (I2C_Universe *u)
{
  if (u && u->tx_thread_state==0)
    {
      int pid = kernel_thread (i2c_transmitter_thread, (void *)u, CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
      if (pid >= 0)
        {
          u->tx_thread_pid = pid;
          printk (KERN_INFO "pid for dmx_i2c thread is %d\n", pid);
          return 0;
        }
    }
  return -1;
}


void stop_i2cdmx_thread (I2C_Universe *u)
{
  if (u && u->tx_thread_pid > 2) /* never kill init! */
    {
      int ret;

      ONDEBUG(printk (KERN_INFO "attempting to kill process with id %d\n", u->tx_thread_pid));

      ret = kill_proc(u->tx_thread_pid, SIGTERM, 0);
      if (!ret)
        {
          /* Wait 10 seconds */
          int count = 10 * 100;
          
          while (u->tx_thread_state && --count)
            {
              current->state = TASK_INTERRUPTIBLE;
              schedule_timeout(1);
            }
          if (!count)
            printk (KERN_INFO "giving up on killing i2c-thread");
        }
    }
}




static int  i2c_write_universe (DMXUniverse *u, off_t offs, DMXSlotType *dmx_buffer, size_t size)
{
  if (u && dmx_buffer && size > 0 && offs+size < 512)
    {
      I2C_Universe *i2c_u = (I2C_Universe *)u->user_data;
      if (i2c_u)
        {
          memcpy (&i2c_u->buffer[offs], dmx_buffer, size);
          wake_up_interruptible (&i2c_u->tx_waitqueue);

          return size;
        }
    }
  return -EINVAL;
}


static int i2c_delete_universe (DMXUniverse *u)
{
  if (u)
    {
      I2C_Universe *i2c_u = (I2C_Universe *)u->user_data;
      if (i2c_u)
        {
/*
 * free all i2c clients that this universe has used
 */
          stop_i2cdmx_thread (i2c_u);

          DMX_FREE(i2c_u);
        }
    }
  return 0;
}

static int i2c_create_universe (DMXUniverse *u, DMXPropList *pl)
{
  long           adapter_id=0;
  I2C_Universe  *i2c_u;

  if (u->interface && u->interface->universes)
    return -1;  /* only one can be created at the moment */

  i2c_u = DMX_ALLOC(I2C_Universe);
  if (i2c_u)
    {
      struct i2c_client  *prev = NULL;
      int                tda8444_id = 0;
      int                i;

      DMXProperty *p = pl->find(pl, "adapter");
      if (p)
        p->get_long(p, &adapter_id);


      for (i=0; i<64; i++)
        i2c_u->clients[i] = NULL;
      i2c_u->num_clients = 0;
      i2c_u->universe = u;

      do
        {
          struct i2c_client  *next = i2c_get_client (tda8444_id, (int)adapter_id, prev);

          if (next && next!=prev)
            i2c_u->clients[i2c_u->num_clients++] = next;
          else
            next=NULL;
          prev=next;
        }
      while (i2c_u->num_clients<64 && prev);



      printk (KERN_INFO "dmx_i2c: found %d TDA8444 at:", i2c_u->num_clients);
      for (i=0; i<i2c_u->num_clients; i++)
        {
          struct i2c_client *client = i2c_u->clients[i];
          if (client)
             printk (" 0x%02X", client->addr);
        }
      printk("\n");

      if((p = pl->find(pl, "slots")))
        p->set_long(p, i2c_u->num_clients*8);
      
      u->user_data = (void *)i2c_u;

      strcpy(u->connector, "no");  /* may put the i2c-adapter id here */
      u->conn_id = 0;  /* as well as here */

      i2c_u->tx_thread_state=0;
      i2c_u->tx_thread_pid = -1;
      init_waitqueue_head (&i2c_u->tx_waitqueue);

      if (start_i2cdmx_thread (i2c_u) == 0)
        {
          printk (KERN_INFO "dmx_i2c thread started\n");
          u->write_slots = i2c_write_universe;
          u->user_delete = i2c_delete_universe;
          return 0;
        }
      else
          printk (KERN_INFO "unable to start dmx_i2c thread\n");

      DMX_FREE(i2c_u);
    }
  return -1;
}


int adapter=0;
MODULE_PARM(adapter,"i");
MODULE_PARM_DESC(adapter, "The adapter id to use for seek of TDA8444 DAC's. Default is 0");

DMXFamily *i2c_family = NULL;


int __init dmx_i2c_init(void)
{
  i2c_family = dmx_create_family ("I2C");
  if (i2c_family)
    {
      DMXDriver *drv_tda8444 = i2c_family->create_driver (i2c_family, "tda8444", i2c_create_universe, NULL);

      /*
       * use the given buses with a list of
       * given devices.
       */
      if (drv_tda8444)
        {
          DMXInterface *dmxif = drv_tda8444->create_interface (drv_tda8444, NULL);
          if (dmxif)
            dmxif->create_universe (dmxif, 0, dmxproplist_vacreate ("adapter=%l", (long)adapter));

          return 0;
        }

      i2c_family->delete(i2c_family,0);
    }

  return 0;
}

/*
 * Called by the kernel-module-loader before the module is unloaded.
 */
void __exit dmx_i2c_exit(void)
{
  if (i2c_family)
    i2c_family->delete(i2c_family, 0);
}

#ifdef MODULE
module_init(dmx_i2c_init);
module_exit(dmx_i2c_exit);
#endif
