/*
  Copyright (c) 1998 - 2011
  ILK   - Tilburg University
  CLiPS - University of Antwerp
 
  This file is part of dimbl

  dimbl is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  dimbl is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, see <http://www.gnu.org/licenses/>.

  For questions and suggestions, see:
      http://ilk.uvt.nl/software.html
  or send mail to:
      timbl@uvt.nl
*/

#include <ostream>
#include <iomanip>
#include <fstream>
#include <sys/time.h>

#include "timbl/TimblAPI.h"

using namespace std;
using namespace Timbl;

#include "dimbl/DimProcs.h"

worker::~worker(){
  if ( exp )
    delete exp;
}

void split( const string& file, int num ){
  int cnt=0;
  string line;
  ifstream is( file.c_str() );
  while ( getline( is, line ) )
    ++cnt;
  if ( cnt ){
    is.clear();
    is.seekg(0);
    int to_do = cnt/num + 1;
    for ( int i=0; i < num; ++i ){
      string oname = file + "-" + toString( i+1 );
      ofstream os( oname.c_str() );
      if ( os ){
	int done = 0;
	while ( done < to_do &&
		getline( is, line ) ){
	  os << line << endl;
	  ++done;
	}
	if ( i == num-1 ){
	  while ( getline( is, line ) ){
	    os << line << endl;
	  }
	}
      }
    }
  }
}

settings::settings( TimblOpts& Opts ){
  status = true;
  wFileSpecified=false;
  distanceMetric = 0;
  nn = 2;
  numThreads = 1;
  do_distance = false;
  do_distrib = false;
  do_neighb = false;
  do_confusion = false;
  do_class_stats = false;
  do_advanced_stats = false;
  beam = 0;
  progress = 10000;
  estimate = 0;
  IF = UnknownInputFormat;
  bool mood;
  string value;
  if ( Opts.Find( 'a', value, mood ) ){
    Algorithm tst;
    if ( !string_to(value,tst) || tst != IB1 ){
      cerr << "unsupported algorithm. only IB1 is possible!" << endl;
      status = false;
    }
  }
  if ( Opts.Find( 'S', value, mood ) ){
    numThreads = stringTo<int>( value );
    Opts.Delete( 'S' );
  }
  if ( Opts.Find( 'k', value, mood ) ){
    nn = stringTo<int>(value) + 1;
    Opts.Delete( 'k' );
  }
  Opts.Add( 'k', toString<int>( nn ), true );
  if ( Opts.Find( "Beam", value, mood ) ){
    beam = stringTo<int>(value);
    Opts.Delete( "Beam" );
  }
  if ( Opts.Find( 'e', value, mood ) ){
    estimate = stringTo<int>(value);
    Opts.Delete( 'e' );
  }
  if ( Opts.Find( 'F', value, mood ) ){
    if ( !stringTo<InputFormatType>( value, IF ) ){
      cerr << "illegal value for -F option: " << value << endl;
      status = false;
    }
  }	
  if ( Opts.Find( 'p', value, mood ) ){
    progress = stringTo<int>(value);
    Opts.Delete( 'p' );
  }
  if ( Opts.Find( 'f', value, mood ) ){
    trainFileName = value;
    Opts.Delete( 'f' );
  }
  if ( Opts.Find( 't', value, mood ) ){
    testFileName = value;
    Opts.Delete( 't' );
  }
  if ( Opts.Find( 'i', value, mood ) ){
    treeInFileName = value;
    Opts.Delete( 'i' );
  }
  if ( Opts.Find( 'I', value, mood ) ){
    treeOutFileName = value;
    Opts.Delete( 'I' );
  }
  if ( Opts.Find( 'o', value, mood ) ){
    outFileName = value;
    Opts.Delete( 'o' );
  }
  else
    outFileName = testFileName + ".out";
  if ( trainFileName.empty() && treeInFileName.empty() ){
    cerr << "missing trainfile or -i option" << endl;
    status = false;
  }
  else if ( testFileName.empty() && treeOutFileName.empty() ){
    cerr << "missing testfile" << endl;
    status = false;
  }
  else {
    Weighting W = GR;
    if ( Opts.Find( 'w', value, mood ) ){
      // user specified weighting
      if ( !string_to( value, W ) ){
	// No valid weighting, so assume it also has a filename
	vector<string> parts;
	size_t num = split_at( value, parts, ":" );
	if ( num == 2 ){
	  if ( !string_to( parts[1], W ) ){
	    cerr << "invalid weighting option: " << value << endl;
	    status = false;
	  }
	  if ( status ){
	    weightsFileName = parts[0];
	    wFileSpecified = true;
	    Opts.Delete( 'w' );
	  }
	}
	else if ( num == 1 ){
	  weightsFileName = value;
	  wFileSpecified = true;
	  W = GR;
	  Opts.Delete( 'w' );
	} 
	else {
	  cerr << "invalid weighting option: " << value << endl;
	  status = false;
	}
      }
      else {
	// valid Weight, but maybe a number, so replace anyway
	Opts.Delete( 'w' );
      }
      if ( trainFileName.empty() && !weightsFileName.empty() ){
	cerr << "a weights filename is specied. "
	     << "That is incompatible with the -i option" << endl;
	status = false;
      }
      Opts.Add( 'w', to_string(W), false );
    }
  }
  if ( status ) {
    if ( Opts.Find( 'd', value, mood ) ){
      // user specified distance metric
      DecayType decay = UnknownDecay;
      double decay_alfa = 1.0;
      double decay_beta = 1.0;
      string::size_type pos1 = value.find( ":" );
      if ( pos1 == string::npos ){
	pos1 = value.find_first_of( "0123456789" );
	if ( pos1 != string::npos ){
	  if ( ! ( stringTo<DecayType>( string( value, 0, pos1 ), decay ) &&
		   stringTo<double>( string( value, pos1 ), decay_alfa ) ) ){
	    cerr << "illegal value for -d option: " << value << endl;
	    status = false;
	  }
	}
	else if ( !stringTo<DecayType>( value, decay ) ){
	  cerr << "illegal value for -d option: " << value << endl;
	  status = false;
	}
      }
      else {
	string::size_type pos2 = value.find( ':', pos1+1 );
	if ( pos2 == string::npos ){
	  pos2 = value.find_first_of( "0123456789", pos1+1 );
	  if ( pos2 != string::npos ){
	    if ( ! ( stringTo<DecayType>( string( value, 0, pos1 ),
					  decay ) &&
		     stringTo<double>( string( value, pos2 ), 
				       decay_alfa ) ) ){
	      cerr << "illegal value for -d option: " << value << endl;
	      status = false;
	    }
	  }
	  else {
	    cerr << "illegal value for -d option: " << value << endl;
	    status = false;
	  }
	}
	else {
	  if ( ! ( stringTo<DecayType>( string( value, 0, pos1 ), decay ) &&
		   stringTo<double>( string( value, pos1+1, pos2-pos1-1 ), 
				     decay_alfa ) &&
		   stringTo<double>( string( value, pos2+1 ), 
				     decay_beta ) ) ){
	    cerr << "illegal value for -d option: " << value << endl;
	    status = false;
	  }
	}
      }
      if ( status ){
	switch ( decay  ){
	case Zero:
	  distanceMetric = new zeroDecay;
	  break;
	case InvDist:
	  distanceMetric = new invDistDecay;
	  break;
	case InvLinear:
	  distanceMetric = new invLinDecay;
	  break;
	case ExpDecay:
	  distanceMetric = new expDecay( decay_alfa, decay_beta );
	  break;
	default:
	  cerr << "ignoring unknown decay" << toString( decay ) << endl;
	}
	Opts.Delete( 'd' );
      }
    }
  }
  if ( status ){
    if ( Opts.Find( 'v', value, mood ) ){
      if ( value.find( "di" ) != string::npos )
	do_distance = true;
      if ( value.find( "db" ) != string::npos )
	do_distrib = true;
      if ( value.find( "k" ) != string::npos )
	do_neighb = true;
      if ( value.find( "cm" ) != string::npos ){
	do_confusion = true;
	do_advanced_stats = true;
      }
      if ( value.find( "cs" ) != string::npos ){
	do_class_stats = true;
	do_advanced_stats = true;
      }
      if ( value.find( "as" ) != string::npos )
	do_advanced_stats = true;
      Opts.Delete( 'v' );
    }
    Opts.Add( 'v', "S", true );
    if ( treeOutFileName.empty() ){
      inp.open( testFileName.c_str() );
      if ( !inp ){
	cerr << "unable to open " << testFileName << endl;
	status = false;
      }
    }
    out.open( outFileName.c_str() );
    if ( !out ){
      cerr << "Unable to open outputfle " << outFileName << endl;
      status = false;
    }
  }
}

inline string curTime(){
  time_t lTime;
  struct tm *curtime;
  char *time_string;
  time(&lTime);
  curtime = localtime(&lTime);
  time_string = asctime(curtime);
  time_string[24] = '\0'; // defeat the newline!
  return time_string;
}  

void time_stamp( ostream& os, const string& line, int number ) {
  os << line;
  if ( number > -1 ){
    os.width(6);
    os.setf(ios::right, ios::adjustfield);
    os << number << " @ ";
  }
  else
    os << "        ";
  os << curTime() << endl;
}
  
void show_speed_summary( ostream& os, int lines, 
			 const timeval& Start ) {
  timeval Time;
  gettimeofday( &Time, 0 );
  long int uSecsUsed = (Time.tv_sec - Start.tv_sec) * 1000000 +
    (Time.tv_usec - Start.tv_usec);
  double secsUsed = (double)uSecsUsed / 1000000 + DBL_EPSILON;
  int oldPrec = os.precision(4);
  os << setprecision(4);
  os.setf( ios::fixed, ios::floatfield );
  os << "Seconds taken: " << secsUsed << " (";
  os << setprecision(2);
  os << lines / secsUsed << " p/s)" << endl;
  os << setprecision(oldPrec);
}

