/*****************************************************************

  ddj.c

  Copyright (c) 1999 by Mike Oliphant - oliphant@gtk.org

    http://www.nostatic.org/ddj

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

*****************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/times.h>
#include <pthread.h>
#include <X11/Xlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <math.h>
#include "mysql.h"
#include "ddj.h"
#include "xpm.h"
#include "parsecfg.h"
#include "dialog/dialog.h"

void ShutDown(GtkWidget *widget, GdkEvent *event, gpointer data);
void Debug(char *fmt,...);
int AddToString(char *str,int maxlen,char *fmt,...);
char *ChopWhite(char *buf);
gboolean ReadCDRomLabel(char *device,char *label);
void StartClock(void);
void StopClock(void);
void ResetClock(void);
int GetClockSecs(void);
void DisplayMsg(char *msg);
void BoolDialog(char *question,char *yes,GtkSignalFunc yesfunc,
		char *no,GtkSignalFunc nofunc);
void UpdateGTK(void);
void CheckWindowPosition(void);
gint TimeOut(gpointer data);
void DoSaveConfig(void);
void KillMP3(void);
void RedirectIO(gboolean redirect_output);
char *FindRoot(char *str);
void MakeArgs(char *str,char **args,int maxargs);
pid_t LaunchProg(char *prog);
void MinMax(void);
void ToggleControlButtons(void);
GtkWidget *ImageButton(char **xpm);
GtkWidget *Loadxpm(char **xpm);
GtkTooltips *MakeToolTip(void);
void ParseMP3Cmd(char *instr,char *outstr,char *filename);
void ParseFileFmt(char *instr,char *outstr);
void PlayCB(void);
char *MakePath(char *str);
#ifdef BUILTIN_MP3
gboolean ThreadPlay(void *arg);
#endif
void Play(void);
void StopPlay(void);
gboolean NextSong(void);
gboolean PrevSong(void);
void SelectRow(GtkWidget *widget,gint row,gint column,
	       GdkEventButton *event,gpointer data);
int DBNum(char *str);
void DBString(char *dest,char *src,int len);
gboolean LookUpSong(int id,Song *song,Artist *artist,Artist *disc_artist,
		    Disc *disc);
void UpdateScrollDisplay(gboolean init);
void SelectSong(int song);
void DeleteUser(void);
void CreateUser(GtkWidget *widget,gpointer data);
void DoInitSQL(void);
void FillUserCList(void);
void LogInAsRoot(void);
void SelectUser(GtkWidget *widget,gint row,gint column,
		GdkEventButton *event,gpointer data);
void MakeConfigPage(void);
void Homepage(void);
void MakeAboutPage(void);
GtkWidget *MakeNewPage(char *name);
void MakePlayPage(void);
void FillArtistCList(void);
void ArtistChanged(void);
void FillDiscCList(void);
void QueryGenres(void);
void ScrambleInts(int *list,int size);
void GeneratePlaylist(void);
void AddToPlayList(int discid);
void SetSongOrder(GtkWidget *widget,gpointer data);
void FillArtistCombo(void);
void FillDiscCombo(void);
void FillGenreCombo(void);
void UpdateSongTitle(void);
void UpdateSongArtist(void);
void Beat(void);
void BeatReset(void);
void MakeEditPage(void);
void DBScan(void);
void MakeQueryPage(void);
void MakeStyles(void);
GdkColor *MakeColor(int red,int green,int blue);
GtkStyle *MakeStyle(GdkColor *fg,GdkColor *bg,gboolean do_grade);

GdkColor gdkblack;
GdkColor gdkwhite;
GdkColor *color_LCD;
GdkColor *color_dark_grey;

GtkStyle *style_wb;
GtkStyle *style_LCD;
GtkStyle *style_dark_grey;

MYSQL mysql;
char sql_host[61]="localhost";
char sql_user[17]="ddj";
char sql_pswd[17]="";
char sql_dbname[33]="ddj_mp3";

GtkWidget *window;
GtkWidget *winbox;
GtkWidget *control_box;
GtkWidget *control_button_box;
GtkWidget *play_clist;
gchar *play_titles[]={"Title","Artist","Disc","Length"};
GtkWidget *artist_clist;
gchar *artist_titles[]={"Artist"};
GtkWidget *disc_clist;
gchar *disc_titles[]={"Disc","Artist"};
GtkWidget *genre_clist;
gchar *genre_titles[]={"Genre"};
GtkWidget *user_clist;
gchar *user_titles[]={"User","Host"};
GtkWidget *main_notebook;
GtkWidget *play_page;
GtkWidget *query_page;
GtkWidget *edit_page;
GtkWidget *config_page;
GtkWidget *about_page;
GtkWidget *current_song_label;
GtkWidget *play_time_label;
gboolean have_db=FALSE;
gboolean minimized=FALSE;
gboolean control_buttons_visible=TRUE;
int current_song=0;
gboolean playing=FALSE;
gboolean paused=FALSE;
char mp3exename[256]="/usr/bin/mpg123 %f";
char mp3fileformat[256]="~/mp3/%a/%d/%n.mp3";
char scroll_display_fmt[256]="  %n -- %a -- %d  ";
pid_t mp3_pid;
gboolean do_debug=FALSE;
int num_songs=0;
int song_order=SONG_ORDER_RANDOM;
GtkWidget *artist_combo;
GtkWidget *disc_combo;
GtkWidget *genre_combo;
GtkWidget *song_title_entry;
GtkWidget *beat_spin;
GtkWidget *year_spin_button;
int num_beats=0;
clock_t start_time=0,init_time,elapsed_time=0;
gboolean clock_ticking=FALSE;
GtkWidget *low_bpm_spin, *high_bpm_spin;
int display_win_size=20;
gboolean have_songs=FALSE;
GtkWidget *root_pswd_entry;
gboolean have_root_pswd=FALSE;
GtkWidget *tmp_user_name_entry;
GtkWidget *tmp_user_host_entry;
GtkWidget *tmp_user_pswd_entry;

Song song_data;
Artist artist_data,disc_artist_data;
Disc disc_data;

gchar *id3_genres[] = {
  "Blues",
  "Classic Rock",
  "Country",
  "Dance",
  "Disco",
  "Funk",
  "Grunge",
  "Hip-Hop",
  "Jazz",
  "Metal",
  "New Age",
  "Oldies",
  "Other",
  "Pop",
  "R&B",
  "Rap",
  "Reggae",
  "Rock",
  "Techno",
  "Industrial",
  "Alternative",
  "Ska",
  "Death Metal",
  "Pranks",
  "Soundtrack",
  "Euro-Techno",
  "Ambient",
  "Trip-Hop",
  "Vocal",
  "Jazz+Funk",
  "Fusion",
  "Trance",
  "Classical",
  "Instrumental",
  "Acid",
  "House",
  "Game",
  "Sound Clip",
  "Gospel",
  "Noise",
  "AlternRock",
  "Bass",
  "Soul",
  "Punk",
  "Space",
  "Meditative",
  "Instrumental Pop",
  "Instrumental Rock",
  "Ethnic",
  "Gothic",
  "Darkwave",
  "Techno-Industrial",
  "Electronic",
  "Pop-Folk",
  "Eurodance",
  "Dream",
  "Southern Rock",
  "Comedy",
  "Cult",
  "Gangsta",
  "Top 40",
  "Christian Rap",
  "Pop/Funk",
  "Jungle",
  "Native American",
  "Cabaret",
  "New Wave",
  "Psychadelic",
  "Rave",
  "Showtunes",
  "Trailer",
  "Lo-Fi",
  "Tribal",
  "Acid Punk",
  "Acid Jazz",
  "Polka",
  "Retro",
  "Musical",
  "Rock & Roll",
  "Hard Rock",
  "Folk",
  "Folk/Rock",
  "National Folk",
  "Swing",
  "Bebob",
  "Latin",
  "Revival",
  "Celtic",
  "Bluegrass",
  "Avantgarde",
  "Gothic Rock",
  "Progressive Rock",
  "Psychedelic Rock",
  "Symphonic Rock",
  "Slow Rock",
  "Big Band",
  "Chorus",
  "Easy Listening",
  "Acoustic",
  "Humour",
  "Speech",
  "Chanson",
  "Opera",
  "Chamber Music",
  "Sonata",
  "Symphony",
  "Booty Bass",
  "Primus",
  "Porn Groove",
  "Satire",
  "Slow Jam",
  "Club",
  "Tango",
  "Samba",
  "Folklore",
  "Ballad",
  "Power Ballad",
  "Rhythmic Soul",
  "Freestyle",
  "Duet",
  "Punk Rock",
  "Drum Solo",
  "Acapella",
  "Euro-house",
  NULL
};

CFGEntry cfg_entries[]={
  {"sql_host",CFG_ENTRY_STRING,61,sql_host},
  {"sql_dbname",CFG_ENTRY_STRING,33,sql_dbname},
  {"sql_user",CFG_ENTRY_STRING,17,sql_user},
  {"sql_pswd",CFG_ENTRY_STRING,17,sql_pswd},
  {"mp3fileformat",CFG_ENTRY_STRING,256,mp3fileformat},
  {"mp3exename",CFG_ENTRY_STRING,256,mp3exename},
  {"",CFG_ENTRY_LAST,0,NULL}
};

void ShutDown(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  KillMP3();

  gtk_main_quit();
}

void Debug(char *fmt,...)
{
  va_list args;

  if(do_debug) {
    args=va_start(args,fmt);

    vprintf(fmt,args);
  }

  va_end(args);
}

void UpdateGTK(void)
{
  while(gtk_events_pending())
    gtk_main_iteration();
}

/* See if the window is going off of the screen, and if so, move it */

void CheckWindowPosition(void)
{
  int x,y,width,height,newx,newy;
  int x_pos,y_pos;
  int scr_width,scr_height;
  gboolean doit=FALSE;

  gdk_window_get_origin(window->window,&x,&y);
  gdk_window_get_size(window->window,&width,&height);
  scr_width=gdk_screen_width();
  scr_height=gdk_screen_height();

  newx=x; newy=y;

  if((x+width)>scr_width) {
    newx=scr_width-width;
    doit=TRUE;
  }

  if((y+height)>scr_height) {
    newy=scr_height-height;
    doit=TRUE;
  }

  if(doit) {
    gdk_window_get_root_origin(window->window,&x_pos,&y_pos);

    gdk_window_move(window->window,newx-(x-x_pos),newy-(y-y_pos));
  }
}

/* Add to a string without overflowing (must be null-terminated already */

int AddToString(char *str,int maxlen,char *fmt,...)
{
  va_list args;
  int left,curr_len;
  char *offset;
  int written;

  curr_len=strlen(str);

  if(curr_len>(maxlen-2)) return 0;

  left=maxlen-curr_len;

  offset=str+curr_len;

  args=va_start(args,fmt);

  written=vsnprintf(offset,left,fmt,args);

  va_end(args);

  return (written>left)?left:written;
}

/* Get rid of whitespace at the beginning or end of a string */

char *ChopWhite(char *buf)
{
  int pos;

  for(pos=strlen(buf)-1;(pos>=0)&&isspace(buf[pos]);pos--);

  buf[pos+1]='\0';

  for(;isspace(*buf);buf++);

  return buf;
}

gboolean ReadCDRomLabel(char *device,char *label)
{
  int fd;

  fd=open(device,O_RDONLY);

  if(fd<0) return FALSE;
 

  if(lseek(fd,(long)0x8028,SEEK_SET) != (long)0x8028) {
    close(fd);

    return FALSE;
  }

  if(read(fd,label,11)!=11) {
    close(fd);

    return FALSE;
  }

  close(fd);

  label[11]='\0';

  return TRUE;
}

void StartClock(void)
{
  struct tms t;

  if(!clock_ticking) {
    init_time=times(&t);
    clock_ticking=TRUE;
  }
}

void StopClock(void)
{
  clock_t end_time;
  struct tms t;

  if(clock_ticking) {
    end_time=times(&t);
  
    elapsed_time+=(end_time-init_time);
    clock_ticking=FALSE;
  }
}

void ResetClock(void)
{
  StopClock();
  elapsed_time=0;
}

int GetClockSecs(void)
{
  clock_t curr_time;
  struct tms t;

  if(!clock_ticking) return elapsed_time/CLK_TCK;
  else {
    curr_time=times(&t);

    return (elapsed_time+(curr_time-init_time))/CLK_TCK;
  }
}

gint TimeOut(gpointer data)
{
  int secs;
  char buf[6];
  static int counter=0;

  if(playing) {
    if(!paused||(counter++%4)) {
      secs=GetClockSecs();

      snprintf(buf,6,"%02d:%02d",secs/60,secs%60);

      gtk_label_set_text(GTK_LABEL(play_time_label),buf);
    }
    else {
      gtk_label_set_text(GTK_LABEL(play_time_label),"     ");
    }

    if(waitpid(mp3_pid,NULL,WNOHANG)) {
      waitpid(mp3_pid,NULL,0);

      playing=FALSE;
      if(NextSong()) Play();
    }
  }

  UpdateScrollDisplay(FALSE);

  return TRUE;
}

void DoSaveConfig(void)
{
  char filename[256];

  snprintf(filename,256,"%s/.ddj",getenv("HOME"));

  if(!SaveConfig(filename,"DDJ",1,cfg_entries))
    printf("Error: Unable to save config file to %s\n",filename);
}

void KillMP3(void)
{
  if(playing) {
    if(paused) {
      kill(mp3_pid,SIGCONT);
      paused=FALSE;
    }

    kill(mp3_pid,SIGINT);
    waitpid(mp3_pid,NULL,0);
    ResetClock();

    gtk_label_set_text(GTK_LABEL(play_time_label),"00:00");
  }
}

void RedirectIO(gboolean redirect_output)
{
  int fd;

  fd=open("/dev/null",O_RDWR);
  dup2(fd,0);

  if(redirect_output) {
    dup2(fd,1);
    dup2(fd,2);
  }

  close(fd); 
}

char *FindRoot(char *str)
{
  char *c;

  for(c=str+strlen(str);c>str;c--) {
    if(*c=='/') return c+1;
  }

  return c;
}

void MakeArgs(char *str,char **args,int maxargs)
{
  int arg;
  gboolean inquotes=FALSE;

  for(arg=0,args[0]=str;*str&&arg<(maxargs-1);str++) {
    if(*str=='"') {
      if(inquotes) *str='\0';
      else args[arg]=str+1;

      inquotes=!inquotes;
    }
    else if(*str==' '&&!inquotes) {
      *str='\0';
      args[++arg]=str+1;
    }
  }

  args[++arg]=NULL;
}

pid_t LaunchProg(char *prog)
{
  pid_t pid;
  char *args[50];
  char *exename,*argstr;

  exename=strtok(prog," ");

  if(!exename) {
    printf("Error: Unable to launch [%s]\n",prog);

    return -1;
  }

  args[0]=FindRoot(exename);

  argstr=strtok(NULL,"");

  if(argstr)
    MakeArgs(argstr,args+1,49);
  else args[1]=NULL;

  pid=fork();

  if(pid==-1) {
    printf("Error: Unable to launch [%s]\n",prog);

    return -1;
  }

  if(pid==0) {
    close(ConnectionNumber(GDK_DISPLAY()));
    RedirectIO(TRUE);
    execv(exename,args);
    _exit(0);
  }

  return pid;
}

void MinMax(void)
{
  if(minimized) {
    gtk_container_border_width(GTK_CONTAINER(winbox),3);
    gtk_widget_show(main_notebook);
  }
  else {
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,TRUE);
    gtk_container_border_width(GTK_CONTAINER(winbox),0);
    gtk_widget_hide(main_notebook);

    gtk_label_set(GTK_LABEL(current_song_label),"");

    UpdateGTK();
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,FALSE);

    UpdateScrollDisplay(FALSE);
  }

  minimized=!minimized;
}

void ToggleControlButtons(void)
{
  if(control_buttons_visible) {
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,TRUE);
    gtk_widget_hide(control_button_box);

    UpdateGTK();
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,FALSE);
  }
  else {
    gtk_widget_show(control_button_box);
  }

  control_buttons_visible=!control_buttons_visible;
}


GtkWidget *ImageButton(char **xpm)
{
  GtkWidget *button;
  GtkWidget *image;

  button=gtk_button_new();

  image=Loadxpm(xpm);
  gtk_container_add(GTK_CONTAINER(button),image);
  gtk_widget_show(image);

  return button;
}

GtkWidget *Loadxpm(char **xpm)
{
  GdkBitmap *mask;
  GtkStyle *style;
  GtkWidget *pixmapwid;
  GdkPixmap *pixmap;

  style=gtk_widget_get_style(window);

  pixmap=gdk_pixmap_create_from_xpm_d(window->window,&mask,
				      &style->bg[GTK_STATE_NORMAL],
				      (gchar **)xpm);

  pixmapwid=gtk_pixmap_new(pixmap,mask);

  return pixmapwid;
}

GtkTooltips *MakeToolTip(void)
{
  GtkTooltips *tip;

  tip=gtk_tooltips_new();

  gtk_tooltips_set_delay(tip,1250);

  return tip;
}

void ParseMP3Cmd(char *instr,char *outstr,char *filename)
{
  int left=1024;

  for(;*instr&&(left>0);instr++) {
    if(*instr=='%') {
      switch(*(++instr)) {
      case '%':
	snprintf(outstr,left,"%%");
	break;
      case 'f':
	snprintf(outstr,left,"\"%s\"",filename);
	break;
      }
      
      left-=strlen(outstr);
      outstr+=strlen(outstr);
    }
    else {
      if(left>0) {
	*outstr++=*instr;
	left--;
      }
    }
  }
  
  *outstr='\0';
}

void ParseFileFmt(char *instr,char *outstr)
{
  int left=1024;
  char *tok;
  struct passwd *pwd;

  *outstr='\0';

  /* Expand ~ in dir -- modeled loosely after code from gtkfind by
                        Matthew Grossman */

  if(*instr=='~') {
    instr++;

    if((*instr=='\0') || (*instr=='/')) {   /* This user's dir */
      left-=snprintf(outstr,left,"%s",getenv("HOME"));
    }
    else {  /* Another user's dir */
      tok=strchr(instr,'/');

      if(tok) {   /* Ugly, but it works */
	*tok='\0';
	pwd=getpwnam(instr);
	instr+=strlen(instr);
	*tok='/';
      }
      else {
	pwd=getpwnam(instr);
	instr+=strlen(instr);
      }

      if(!pwd)
	printf("Error: unable to translate filename. No such user as %s\n",
	       tok);
      else {
	left-=snprintf(outstr,left,"%s",pwd->pw_dir);
      }
    }
    
    outstr+=strlen(outstr);
  }

  for(;*instr&&left>0;instr++) {
    if(*instr=='%') {
      switch(*(++instr)) {
      case '%':
	snprintf(outstr,left,"%%");
	break;
      case 't':
	snprintf(outstr,left,"%02d",song_data.track_num+1);
	break;
      case 'n':
	snprintf(outstr,left,"%s",song_data.title);
	break;
      case 'a':
	snprintf(outstr,left,"%s",artist_data.name);
	break;
      case 'A':
	if(disc_artist_data.id==artist_data.id)
	  snprintf(outstr,left,"%s",artist_data.name);
	else snprintf(outstr,left,"%s",disc_artist_data.name);
	break;
      case 'd':
	snprintf(outstr,left,"%s",disc_data.title);
	break;
      }
	  
      left-=strlen(outstr);
      outstr+=strlen(outstr);
    }
    else {
      if(left>0) {
	*outstr++=*instr;
	left--;
      }
    }
  }
  
  *outstr='\0';
}

void PlayCB(void)
{
  if(!have_songs) return;

  if(playing) {
    if(paused) {
      kill(mp3_pid,SIGCONT);
      paused=FALSE;

      StartClock();
    }
    else {
      kill(mp3_pid,SIGSTOP);
      paused=TRUE;

      StopClock();
    }
  }
  else Play();
}

char *MakePath(char *str)
{
  int len;

  len=strlen(str)-1;

  if(str[len]!='/') {
    str[len+1]='/';
    str[len+2]='\0';
  }

  return str;
}

#ifdef BUILTIN_MP3
pthread_t mp3_thread;
static char mp3_thread_filename[1024];

gboolean ThreadPlay(void *arg)
{
  strncpy(mp3_thread_filename,(char *)arg,1024);

  PlayMP3(mp3_thread_filename);

  pthread_exit(0);
}

#endif

void Play(void)
{
  char buf[1024];
  char exe_string[1024];

  if(!have_db) return;

#ifdef BUILTIN_MP3
  //  pthread_create(&mp3_thread,NULL,(void *)ThreadPlay,(void *)buf);
  ThreadPlay((void *)buf);
#else
  ParseMP3Cmd(mp3exename,exe_string,song_data.filename);

  Debug("MP3 exe is [%s]\n",exe_string);

  mp3_pid=LaunchProg(exe_string);
#endif

  if(mp3_pid!=-1) {
    playing=TRUE;
    
    ConnectSQL();
    
    snprintf(buf,1024,"UPDATE song SET num_plays=num_plays+1 WHERE id=%d",
	     song_data.id);
    
    mysql_query(&mysql,buf);
    
    snprintf(buf,1024,"UPDATE song SET last_play=NOW() WHERE id=%d",
	     song_data.id);
    
    mysql_query(&mysql,buf);
    
    snprintf(buf,1024,"UPDATE artist SET num_plays=num_plays+1 WHERE id=%d",
	     song_data.artistid);
    
    mysql_query(&mysql,buf);
    
    snprintf(buf,1024,"UPDATE artist SET last_play=NOW() WHERE id=%d",
	     song_data.artistid);
    
    mysql_query(&mysql,buf);
    
    snprintf(buf,1024,"UPDATE disc SET num_plays=num_plays+1 WHERE id=%d",
	     song_data.discid);
    
    mysql_query(&mysql,buf);
    
    snprintf(buf,1024,"UPDATE disc SET last_play=NOW() WHERE id=%d",
	     song_data.discid);
    
    mysql_query(&mysql,buf);
    
    DisconnectSQL();
  }

  ResetClock();
  StartClock();
}

void StopPlay(void)
{
  if(playing) {
    KillMP3();
    playing=FALSE;
    paused=FALSE;
  }
}

gboolean NextSong(void)
{
  if(current_song<(num_songs-1)) {
    gtk_clist_select_row(GTK_CLIST(play_clist),
			 current_song+1,0);

    return TRUE;
  }
  else {
    StopPlay();

    return FALSE;
  }
}

gboolean PrevSong(void)
{
  if(!playing||(GetClockSecs()<2)) {
      if(current_song)
	gtk_clist_select_row(GTK_CLIST(play_clist),
			     current_song-1,0);
      else {
	StopPlay();
	return FALSE;
      }
  }
    
  if(playing) {
    KillMP3();
    Play();
  }

  return TRUE;
}

void SelectRow(GtkWidget *widget,gint row,gint column,
	       GdkEventButton *event,gpointer data)
{
  SelectSong(row);

  if(gtk_clist_row_is_visible(GTK_CLIST(play_clist),row)!=GTK_VISIBILITY_FULL)
    gtk_clist_moveto(GTK_CLIST(play_clist),row,0,0,0);

  if(playing) {
    KillMP3();
    Play();
  }
}

int DBNum(char *str)
{
  if(!str) return -1;
  else return atoi(str);
}

void DBString(char *dest,char *src,int len)
{
  if(!src||!*src) *dest='\0';
  else strncpy(dest,src,len);
}

gboolean LookUpSong(int id,Song *song,Artist *artist,Artist *disc_artist,
		    Disc *disc)
{
  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char query[256];
  char *tok;

  if(!have_db) return FALSE;

  ConnectSQL();

  snprintf(query,256,"SELECT filename,title,artistid,discid,genre,track_num,
start_frame,num_frames,play_time,bpm,num_plays,last_play,year FROM song
where id=%d",id);

  if(mysql_query(&mysql,query)||
     !(res = mysql_store_result(&mysql))||
     !(row=mysql_fetch_row(res))) {
    printf("Query error\n");
    
    DisconnectSQL();
    return FALSE;
  }

  song->id=id;
  DBString(song->filename,row[0],256);
  DBString(song->title,row[1],80);
  song->artistid=DBNum(row[2]);
  song->discid=DBNum(row[3]);
  DBString(song->genre,row[4],20);
  song->track_num=DBNum(row[5]);
  song->start_frame=DBNum(row[6]);
  song->num_frames=DBNum(row[7]);
  song->mins=song->secs=-1;
  if(row[8]) {
    tok=strtok(row[8],":");

    if(tok) {
      song->mins=atoi(tok);

      tok=strtok(NULL,"");
      
      if(tok) song->secs=atoi(tok);
    }
  }
  song->bpm=DBNum(row[9]);
  song->year=DBNum(row[12]);
  if(song->year==-1) song->year=0;

  mysql_free_result(res);

  snprintf(query,80,"SELECT * FROM artist where id=%d",song->artistid);

  if(mysql_query(&mysql,query)||
     !(res = mysql_store_result(&mysql))||
     !(row=mysql_fetch_row(res))) {
    printf("Query error\n");
    
    DisconnectSQL();
    return TRUE;
  }

  artist->id=song->artistid;
  DBString(artist->name,row[1],80);

  mysql_free_result(res);

  snprintf(query,80,"SELECT * FROM disc where id=%d",song->discid);

  if(mysql_query(&mysql,query)||
     !(res = mysql_store_result(&mysql))||
     !(row=mysql_fetch_row(res))) {
    printf("Query error\n");
    
    DisconnectSQL();
    return TRUE;
  }

  disc->id=song->discid;
  DBString(disc->title,row[1],80); 
  disc->artistid=DBNum(row[2]);

  mysql_free_result(res);

  snprintf(query,80,"SELECT * FROM artist where id=%d",disc->artistid);

  if(mysql_query(&mysql,query)||
     !(res = mysql_store_result(&mysql))||
     !(row=mysql_fetch_row(res))) {
    printf("Query error\n");
    
    DisconnectSQL();
    return TRUE;
  }

  disc_artist->id=disc->artistid;
  DBString(disc_artist->name,row[1],80);

  mysql_free_result(res);

  DisconnectSQL();

  return TRUE;
}

void UpdateScrollDisplay(gboolean init)
{
  static char display_window[1024];
  static char display_buf[1024];
  static int display_buf_len=0;
  static int display_pos=0;
  static int curr_display_size=0;
  int left;
  int points_per_char;

  if(!have_songs) return;

  points_per_char=gdk_string_width(current_song_label->style->font,"W");
  display_win_size=current_song_label->allocation.width/points_per_char;

  if(display_win_size<15) display_win_size=15;

  if(init||(curr_display_size!=display_win_size)) {
    ParseFileFmt(scroll_display_fmt,display_buf);

    curr_display_size=display_win_size;
    display_pos=0;
    display_buf_len=strlen(display_buf);
  }
  else {
    if(display_buf_len<display_win_size)
      return;
    
    display_pos++;
    
    if(display_pos==display_buf_len) display_pos=0;
  }

  left=display_buf_len-display_pos;

  if((left>display_win_size)||(display_buf_len<=display_win_size))
    snprintf(display_window,display_win_size,"%s",display_buf+display_pos);
  else {
    sprintf(display_window,"%s",display_buf+display_pos);
    snprintf(display_window+left,display_win_size-left,"%s",display_buf);
  }

  gtk_label_set(GTK_LABEL(current_song_label),display_window);
}

void SelectSong(int song)
{
  int id;

  if(!have_songs) return;

  id=(int)gtk_clist_get_row_data(GTK_CLIST(play_clist),song);

  LookUpSong(id,&song_data,&artist_data,&disc_artist_data,&disc_data);

  UpdateScrollDisplay(TRUE);

  gtk_entry_set_text(GTK_ENTRY(song_title_entry),song_data.title);

  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(artist_combo)->entry),
		     artist_data.name);

  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(disc_combo)->entry),
		     disc_data.title);

  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(genre_combo)->entry),
		     song_data.genre);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(beat_spin),song_data.bpm);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(year_spin_button),song_data.year);

  gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,TRUE);
  UpdateGTK();
  gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,FALSE);

  current_song=song;
  num_beats=0;
}

void MakePlayPage(void)
{
#if (GTK_MINOR_VERSION != 0)
  GtkWidget *scroll;
#endif

  play_page=MakeNewPage("Playlist");

  play_clist=gtk_clist_new_with_titles(4,play_titles);
  gtk_widget_set_usize(play_clist,560,250);

  gtk_signal_connect(GTK_OBJECT(play_clist),"select_row",
		     GTK_SIGNAL_FUNC(SelectRow),
		     NULL);

  gtk_clist_set_selection_mode(GTK_CLIST(play_clist),GTK_SELECTION_SINGLE);
  gtk_clist_set_reorderable(GTK_CLIST(play_clist),TRUE);

  gtk_clist_column_titles_passive(GTK_CLIST(play_clist));

  gtk_clist_set_column_width(GTK_CLIST(play_clist),0,200);
  gtk_clist_set_column_width(GTK_CLIST(play_clist),1,100);
  gtk_clist_set_column_width(GTK_CLIST(play_clist),2,150);
  gtk_clist_set_column_width(GTK_CLIST(play_clist),3,50);

#if (GTK_MINOR_VERSION == 0)
  gtk_clist_set_policy(GTK_CLIST(play_clist),GTK_POLICY_AUTOMATIC,
		       GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(play_page),play_clist);
  gtk_widget_show(play_clist);
#else
  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),play_clist);
  gtk_widget_show(play_clist);

  gtk_container_add(GTK_CONTAINER(play_page),scroll);
  gtk_widget_show(scroll);
#endif
}

void FillArtistCList(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;

  if(!have_db) return;

  gtk_clist_clear(GTK_CLIST(artist_clist));

  ConnectSQL();

  if(mysql_query(&mysql,"SELECT artist FROM artist order by artist")) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  if(!(res = mysql_store_result(&mysql))) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  while((row=mysql_fetch_row(res))) {
    gtk_clist_append(GTK_CLIST(artist_clist),row);
  }

  mysql_free_result(res);
  DisconnectSQL();
}

void ArtistChanged(void)
{
  FillDiscCList();
}

void FillDiscCList(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  char query[1024];
  GList *alist;
  char *artist_name;
  gboolean first=TRUE;
  char tartist[256];
  int pos;

  if(!have_db) return;

  gtk_clist_clear(GTK_CLIST(disc_clist));

  *query='\0';

  AddToString(query,1024,"SELECT title,artist,disc.id FROM disc,artist WHERE
disc.artistid=artist.id");

  alist=GTK_CLIST(artist_clist)->selection;

  while(alist) {
    gtk_clist_get_text(GTK_CLIST(artist_clist),
		       GPOINTER_TO_INT(alist->data),0,
		       &artist_name);
    
    EscapeQuote(artist_name,tartist,256);

    if(first) {
      first=FALSE;
      AddToString(query,1024," AND (artist.artist='%s'",
		  tartist);
    }
    else
      AddToString(query,1024," OR artist.artist='%s'",
		  tartist);

    alist=alist->next;
  }

  if(!first) AddToString(query,1024,")");

  AddToString(query,1024," order by title");

  if(strlen(query)==1023) {
    printf("Error: Query is too long\n");

    return;
  }

  ConnectSQL();

  if(mysql_query(&mysql,query)) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  if(!(res = mysql_store_result(&mysql))) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  pos=0;

  while((row=mysql_fetch_row(res))) {
    gtk_clist_append(GTK_CLIST(disc_clist),row);

    gtk_clist_set_row_data(GTK_CLIST(disc_clist),pos++,(gpointer)atoi(row[2]));
  }

  mysql_free_result(res);
  DisconnectSQL();
}

void QueryGenres(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;

  if(!have_db) return;

  gtk_clist_clear(GTK_CLIST(genre_clist));

  ConnectSQL();

  if(mysql_query(&mysql,"SELECT distinct genre FROM song")) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  if(!(res = mysql_store_result(&mysql))) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  while((row=mysql_fetch_row(res))) {
    gtk_clist_append(GTK_CLIST(genre_clist),row);
  }

  mysql_free_result(res);
  DisconnectSQL();
}

void ScrambleInts(int *list,int size)
{
  int p1,p2,tmp;

  for(p1=0;p1<size;p1++) {
    p2=RRand(size);

    tmp=list[p1];
    list[p1]=list[p2];
    list[p2]=tmp;
  }
}

void GeneratePlaylist(void)
{
  GList *disc_list;
  int num_discs,disc;
  int *num_list,pos;

  if(!have_db) return;

  have_songs=FALSE;

  StopPlay();

  gtk_clist_freeze(GTK_CLIST(play_clist));
  gtk_clist_clear(GTK_CLIST(play_clist));

  if(song_order==SONG_ORDER_DISC) {
    disc_list=GTK_CLIST(disc_clist)->selection;
  
    if(disc_list) {
      num_discs=g_list_length(disc_list);

      num_list=(int *)malloc(num_discs*sizeof(int));
      
      disc=0;

      while(disc_list) {
	num_list[disc++]=(int)gtk_clist_get_row_data
	  (GTK_CLIST(disc_clist),GPOINTER_TO_INT(disc_list->data));

	disc_list=disc_list->next;
      }
    }
    else {
      num_discs=g_list_length(GTK_CLIST(disc_clist)->row_list);

      num_list=(int *)malloc(num_discs*sizeof(int));

      for(disc=0;disc<num_discs;disc++)
	num_list[disc]=(int)gtk_clist_get_row_data(GTK_CLIST(disc_clist),
						   disc);
    }

    ScrambleInts(num_list,num_discs);

    for(pos=0;pos<num_discs;pos++)
      AddToPlayList(num_list[pos]);
  }
  else AddToPlayList(-1);

  gtk_clist_thaw(GTK_CLIST(play_clist));
  gtk_widget_queue_resize(play_clist);

  gtk_clist_select_row(GTK_CLIST(play_clist),0,0);    
}

void AddToPlayList(int discid)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  char query[1024];
  GList *artist_list,*disc_list,*genre_list;
  gboolean first=TRUE;
  char *artist_name,*disc_name,*genre_name;
  int insert_pos;
  int low_bpm,high_bpm;
  char tartist[256],tdisc[256],tgenre[256];
  
  num_songs=g_list_length(GTK_CLIST(play_clist)->row_list);

  *query='\0';

  AddToString(query,1024,"SELECT song.title,artist,disc.title,play_time,
song.id FROM artist,disc,song WHERE artist.id=song.artistid AND
disc.id=song.discid");

  artist_list=GTK_CLIST(artist_clist)->selection;

  while(artist_list) {
    gtk_clist_get_text(GTK_CLIST(artist_clist),
		       GPOINTER_TO_INT(artist_list->data),0,
		       &artist_name);

    EscapeQuote(artist_name,tartist,256);

    if(first) {
      first=FALSE;
      AddToString(query,1024," AND (artist.artist='%s'",
		  tartist);
    }
    else
      AddToString(query,1024," OR artist.artist='%s'",
		  tartist);

    artist_list=artist_list->next;
  }

  if(!first) AddToString(query,1024,")");

  if(discid==-1) {
    first=TRUE;

    disc_list=GTK_CLIST(disc_clist)->selection;

    while(disc_list) {
      gtk_clist_get_text(GTK_CLIST(disc_clist),
			 GPOINTER_TO_INT(disc_list->data),0,
			 &disc_name);
      
      EscapeQuote(disc_name,tdisc,256);
      
      if(first) {
	first=FALSE;
	AddToString(query,1024," AND (disc.title='%s'",
		    tdisc);
      }
      else
	AddToString(query,1024," OR disc.title='%s'",tdisc);
      
      disc_list=disc_list->next;
    }

    if(!first) AddToString(query,1024,")");
  }
  else
    AddToString(query,1024," AND (disc.id=%d)",discid);

  first=TRUE;

  genre_list=GTK_CLIST(genre_clist)->selection;

  while(genre_list) {
    gtk_clist_get_text(GTK_CLIST(genre_clist),
		       GPOINTER_TO_INT(genre_list->data),0,
		       &genre_name);

    EscapeQuote(genre_name,tgenre,256);

    if(first) {
      first=FALSE;
      AddToString(query,1024," AND (song.genre='%s'",
		  tgenre);
    }
    else
      AddToString(query,1024," OR song.genre='%s'",
		  tgenre);

    genre_list=genre_list->next;
  }

  if(!first) AddToString(query,1024,")");

  low_bpm=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(low_bpm_spin));
  high_bpm=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(high_bpm_spin));

  if((low_bpm>LOW_BPM)||(high_bpm<HIGH_BPM)) {
    AddToString(query,1024," AND song.bpm BETWEEN %d and %d",
		low_bpm,high_bpm);
  }

  if(song_order==SONG_ORDER_DISC)
    AddToString(query,1024," order by discid,track_num");
  else if(song_order==SONG_ORDER_OLDEST)
    AddToString(query,1024," order by song.last_play");
 else if(song_order==SONG_ORDER_SLOWEST)
    AddToString(query,1024,
		" AND song.bpm != 'NULL' order by bpm");

  if(strlen(query)==1023) {
    printf("Error: Query is too long\n");

    return;
  }

  Debug("Query is [%s]\n",query);

  ConnectSQL();

  if(mysql_query(&mysql,query)) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  if(!(res = mysql_store_result(&mysql))) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  while((row=mysql_fetch_row(res))) {
    if(song_order==SONG_ORDER_RANDOM) {
      insert_pos=RRand(num_songs+1);
      gtk_clist_insert(GTK_CLIST(play_clist),insert_pos,row);
    }
    else {
      insert_pos=num_songs;
      gtk_clist_append(GTK_CLIST(play_clist),row);
    }

    gtk_clist_set_row_data(GTK_CLIST(play_clist),insert_pos,
			   (gpointer)atoi(row[4]));

    num_songs++;
  }

  mysql_free_result(res);
  DisconnectSQL();

  if(num_songs) have_songs=TRUE;
}

void SetSongOrder(GtkWidget *widget,gpointer data)
{
  song_order=(int)data;
}

void FillArtistCombo(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  GtkWidget *item;

  if(!have_db) return;

  ConnectSQL();

  if(mysql_query(&mysql,"SELECT artist FROM artist order by artist")) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  if(!(res = mysql_store_result(&mysql))) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  while((row=mysql_fetch_row(res))) {
    item=gtk_list_item_new_with_label(row[0]);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(artist_combo)->list),item);
    gtk_widget_show(item);
  }

  mysql_free_result(res);
  DisconnectSQL();
}

void FillDiscCombo(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  GtkWidget *item;

  if(!have_db) return;

  ConnectSQL();

  if(mysql_query(&mysql,"SELECT title FROM disc order by title")) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  if(!(res = mysql_store_result(&mysql))) {
    printf("Query error\n");

    DisconnectSQL();
    return;
  }

  while((row=mysql_fetch_row(res))) {
    item=gtk_list_item_new_with_label(row[0]);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(disc_combo)->list),item);
    gtk_widget_show(item);
  }

  mysql_free_result(res);
  DisconnectSQL();
}

void FillGenreCombo(void)
{
  GtkWidget *item;
  int genre;

  for(genre=0;id3_genres[genre];genre++) {
    item=gtk_list_item_new_with_label(id3_genres[genre]);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(genre_combo)->list),item);
    gtk_widget_show(item);
  }
}

void UpdateSongTitle(void)
{
  char query[1024];
  char ttitle[256];

  strncpy(song_data.title,gtk_entry_get_text(GTK_ENTRY(song_title_entry)),80);
  EscapeQuote(song_data.title,ttitle,256);

  snprintf(query,1024,"UPDATE song set title='%s' where id=%d",ttitle,
	   song_data.id);

  SQLQuery(query);
}

void UpdateSongArtist(void)
{
  char query[1024];
  char ttitle[256];

  strncpy(song_data.title,gtk_entry_get_text(GTK_ENTRY(song_title_entry)),80);
  EscapeQuote(song_data.title,ttitle,256);

  snprintf(query,1024,"UPDATE song set title='%s' where id=%d",ttitle,
	   song_data.id);

  SQLQuery(query);
}

void UpdateSongDisc(void)
{
}

void UpdateSongGenre(void)
{
  char query[1024];
  char tgenre[256];

  strncpy(song_data.genre,
	  gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(genre_combo)->entry)),20);

  EscapeQuote(song_data.genre,tgenre,256);

  snprintf(query,1024,"UPDATE song set genre='%s' where id=%d",tgenre,
	   song_data.id);

  SQLQuery(query);
}

void UpdateSongBPM(void)
{
  char query[1024];

  song_data.bpm=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(beat_spin));
  if(song_data.bpm==0) song_data.bpm=-1;

  if(song_data.bpm==-1)
    snprintf(query,1024,"UPDATE song set bpm=NULL where id=%d",song_data.id);
  else
    snprintf(query,1024,"UPDATE song set bpm=%d where id=%d",song_data.bpm,
	     song_data.id);

  SQLQuery(query);
}

void UpdateSongYear(void)
{
  char query[1024];

  song_data.year=
    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(year_spin_button));

  if(song_data.year)
    snprintf(query,1024,"UPDATE song set year=%d where id=%d",song_data.year,
	     song_data.id);
  else
    snprintf(query,1024,"UPDATE song set bpm=NULL where id=%d",
	     song_data.id);

  SQLQuery(query);
}

void Beat(void)
{
  clock_t end_time;
  struct tms t;
  gfloat bpm;

  num_beats++;

  if(num_beats==1)
    start_time=times(&t);
  else {
    end_time=times(&t);

    bpm=rint((gfloat)(num_beats-1)/
      (((gfloat)(end_time-start_time))/(gfloat)(CLK_TCK*60)));

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(beat_spin),bpm);
  }
}

void BeatReset(void)
{
  num_beats=0;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(beat_spin),0.0);
}

void MakeEditPage(void)
{
  GtkWidget *vbox,*vbox2,*hbox,*hbox2;
  GtkWidget *label;
  GtkWidget *button;
  GtkObject *adj;
  GtkWidget *ebox;
  GtkWidget *bpm_image;
  GtkWidget *frame;
  
  edit_page=MakeNewPage("Edit");

  vbox=gtk_vbox_new(FALSE,3);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Title:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  song_title_entry=gtk_entry_new_with_max_length(80);
  gtk_signal_connect(GTK_OBJECT(song_title_entry),"activate",
		     GTK_SIGNAL_FUNC(UpdateSongTitle),NULL);
  gtk_signal_connect(GTK_OBJECT(song_title_entry),"focus_out_event",
		     GTK_SIGNAL_FUNC(UpdateSongTitle),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),song_title_entry,TRUE,TRUE,0);
  gtk_widget_show(song_title_entry);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Artist:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  artist_combo=gtk_combo_new();
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(artist_combo)->entry),FALSE);

  FillArtistCombo();

  gtk_box_pack_start(GTK_BOX(hbox),artist_combo,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(artist_combo)->entry),"activate",
		     GTK_SIGNAL_FUNC(UpdateSongArtist),NULL);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(artist_combo)->entry),
		     "focus_out_event",
		     GTK_SIGNAL_FUNC(UpdateSongArtist),NULL);
  gtk_widget_show(artist_combo);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Disc Name:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  disc_combo=gtk_combo_new();
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(disc_combo)->entry),FALSE);

  FillDiscCombo();

  gtk_box_pack_start(GTK_BOX(hbox),disc_combo,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(disc_combo)->entry),"activate",
		     GTK_SIGNAL_FUNC(UpdateSongDisc),NULL);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(disc_combo)->entry),
		     "focus_out_event",
		     GTK_SIGNAL_FUNC(UpdateSongDisc),NULL);
  gtk_widget_show(disc_combo);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Genre:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  genre_combo=gtk_combo_new();

  FillGenreCombo();

  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(genre_combo)->entry),"activate",
		     GTK_SIGNAL_FUNC(UpdateSongGenre),NULL);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(genre_combo)->entry),
		     "focus_out_event",
		     GTK_SIGNAL_FUNC(UpdateSongGenre),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),genre_combo,TRUE,TRUE,0);
  gtk_widget_show(genre_combo);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Year:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  adj=gtk_adjustment_new(0,0,9999,1.0,5.0,0);

  year_spin_button=gtk_spin_button_new(GTK_ADJUSTMENT(adj),0.5,0);
  gtk_signal_connect(GTK_OBJECT(year_spin_button),"changed",
		     GTK_SIGNAL_FUNC(UpdateSongYear),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),year_spin_button,TRUE,TRUE,0);
  gtk_widget_show(year_spin_button);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  frame=gtk_frame_new(NULL);

  ebox=gtk_event_box_new();
  gtk_widget_set_style(ebox,style_wb);

  hbox=gtk_hbox_new(FALSE,3);

  bpm_image=Loadxpm(bpm_xpm);
  gtk_box_pack_start(GTK_BOX(hbox),bpm_image,TRUE,TRUE,0);
  gtk_widget_show(bpm_image);

  vbox2=gtk_vbox_new(FALSE,3);

  button=gtk_button_new_with_label("Beat");
  gtk_widget_set_style(button,style_dark_grey);
  gtk_widget_set_style(GTK_BIN(button)->child,style_wb);
  gtk_signal_connect(GTK_OBJECT(button),"pressed",
		     GTK_SIGNAL_FUNC(Beat),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Click here on the beat",NULL);
  gtk_box_pack_start(GTK_BOX(vbox2),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_button_new_with_label("Update");
  gtk_widget_set_style(button,style_dark_grey);
  gtk_widget_set_style(GTK_BIN(button)->child,style_wb);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(UpdateSongBPM),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Update the database with current BPM",NULL);
  gtk_box_pack_start(GTK_BOX(vbox2),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(hbox),vbox2,TRUE,TRUE,0);
  gtk_widget_show(vbox2);

  vbox2=gtk_vbox_new(FALSE,3);

  button=gtk_button_new_with_label("Reset");
  gtk_widget_set_style(button,style_dark_grey);
  gtk_widget_set_style(GTK_BIN(button)->child,style_wb);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(BeatReset),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Reset the BPM calclulation",NULL);
  gtk_box_pack_start(GTK_BOX(vbox2),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  hbox2=gtk_hbox_new(FALSE,3);
  
  label=gtk_label_new("BPM:");
  gtk_widget_set_style(label,style_wb);
  gtk_box_pack_start(GTK_BOX(hbox2),label,TRUE,FALSE,0);
  gtk_widget_show(label);

  adj=gtk_adjustment_new(120,LOW_BPM,HIGH_BPM,1.0,5.0,0);

  beat_spin=gtk_spin_button_new(GTK_ADJUSTMENT(adj),0.5,0);
  gtk_widget_set_name(beat_spin,"darkgrey");
  gtk_box_pack_start(GTK_BOX(hbox2),beat_spin,TRUE,FALSE,0);
  gtk_widget_show(beat_spin);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox2,TRUE,FALSE,0);
  gtk_widget_show(hbox2);

  gtk_box_pack_start(GTK_BOX(hbox),vbox2,TRUE,TRUE,0);
  gtk_widget_show(vbox2);

  gtk_container_add(GTK_CONTAINER(ebox),hbox);
  gtk_widget_show(hbox);

  gtk_container_add(GTK_CONTAINER(frame),ebox);
  gtk_widget_show(frame);

  gtk_box_pack_start(GTK_BOX(vbox),frame,FALSE,FALSE,0);
  gtk_widget_show(ebox);

  gtk_container_add(GTK_CONTAINER(edit_page),vbox);
  gtk_widget_show(vbox);
}

void DBScan(void)
{
  FillArtistCList();
  QueryGenres();
  FillDiscCList();
}

void MakeQueryPage(void)
{
  GtkWidget *vpaned,*hpaned;
  GtkWidget *vbox,*hbox;
  GtkWidget *button;
  GtkWidget *label;
  GtkObject *adj;
  GSList *group;
#if (GTK_MINOR_VERSION != 0)
  GtkWidget *scroll;
#endif

  query_page=MakeNewPage("Query");

  vbox=gtk_vbox_new(FALSE,3);

  hpaned=gtk_hpaned_new ();
  gtk_paned_set_position (GTK_PANED (hpaned), 350);
  gtk_paned_set_gutter_size (GTK_PANED (hpaned), 11);

  vpaned=gtk_vpaned_new ();
  gtk_paned_set_gutter_size (GTK_PANED (vpaned), 11);
  
  /* The Artist CList */

  artist_clist=gtk_clist_new_with_titles(1,artist_titles);

  gtk_signal_connect_after(GTK_OBJECT(artist_clist),"button_release_event",
			   GTK_SIGNAL_FUNC(ArtistChanged),NULL);
  gtk_signal_connect_after(GTK_OBJECT(artist_clist),"key_release_event",
			   GTK_SIGNAL_FUNC(ArtistChanged),NULL);

  gtk_clist_set_selection_mode(GTK_CLIST(artist_clist),GTK_SELECTION_EXTENDED);

  gtk_clist_column_titles_passive(GTK_CLIST(artist_clist));

  gtk_clist_set_column_width(GTK_CLIST(artist_clist),0,150);

#if (GTK_MINOR_VERSION == 0)
  gtk_clist_set_policy(GTK_CLIST(artist_clist),GTK_POLICY_AUTOMATIC,
		       GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(hpaned), artist_clist);
  gtk_widget_show(artist_clist);
#else
  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),artist_clist);
  gtk_widget_show(artist_clist);

  gtk_container_add(GTK_CONTAINER(hpaned), scroll);
  gtk_widget_show(scroll);
#endif

  /* The Genre CList */

  genre_clist=gtk_clist_new_with_titles(1,genre_titles);

  gtk_clist_set_selection_mode(GTK_CLIST(genre_clist),GTK_SELECTION_EXTENDED);

  gtk_clist_column_titles_passive(GTK_CLIST(genre_clist));

  gtk_clist_set_column_width(GTK_CLIST(genre_clist),0,150);

#if (GTK_MINOR_VERSION == 0)
  gtk_clist_set_policy(GTK_CLIST(genre_clist),GTK_POLICY_AUTOMATIC,
		       GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(hpaned), genre_clist);
  gtk_widget_show(genre_clist);
#else
  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),genre_clist);
  gtk_widget_show(genre_clist);

  gtk_container_add(GTK_CONTAINER(hpaned), scroll);
  gtk_widget_show(scroll);
#endif

  gtk_container_add (GTK_CONTAINER (vpaned), hpaned);
  gtk_box_pack_start(GTK_BOX(vbox),vpaned,TRUE,TRUE,0);
  gtk_widget_show(vpaned);
  gtk_widget_show(hpaned);

  /* The Disc CList */

  disc_clist=gtk_clist_new_with_titles(2,disc_titles);

  gtk_clist_set_selection_mode(GTK_CLIST(disc_clist),GTK_SELECTION_EXTENDED);

  gtk_clist_column_titles_passive(GTK_CLIST(disc_clist));

  gtk_clist_set_column_width(GTK_CLIST(disc_clist),0,250);
  gtk_clist_set_column_width(GTK_CLIST(disc_clist),1,250);

#if (GTK_MINOR_VERSION == 0)
  gtk_clist_set_policy(GTK_CLIST(disc_clist),GTK_POLICY_AUTOMATIC,
		       GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (vpaned), disc_clist);
  gtk_widget_show(disc_clist);
#else
  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),disc_clist);
  gtk_widget_show(disc_clist);

  gtk_container_add(GTK_CONTAINER(vpaned),scroll);
  gtk_widget_show(scroll);
#endif

  /* BPM selection */

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("BPM: More than");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  adj=gtk_adjustment_new(LOW_BPM,LOW_BPM,HIGH_BPM,1.0,5.0,0);

  low_bpm_spin=gtk_spin_button_new(GTK_ADJUSTMENT(adj),0.5,0);
  gtk_box_pack_start(GTK_BOX(hbox),low_bpm_spin,TRUE,FALSE,0);
  gtk_widget_show(low_bpm_spin);

  label=gtk_label_new("Less than");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  adj=gtk_adjustment_new(HIGH_BPM,LOW_BPM,HIGH_BPM,1.0,5.0,0);

  high_bpm_spin=gtk_spin_button_new(GTK_ADJUSTMENT(adj),0.5,0);
  gtk_box_pack_start(GTK_BOX(hbox),high_bpm_spin,TRUE,FALSE,0);
  gtk_widget_show(high_bpm_spin);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  /* Play order */

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Play order:");
  gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,0);
  gtk_widget_show(label);

  button=gtk_radio_button_new_with_label(NULL,"Random");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetSongOrder),
		     (gpointer)SONG_ORDER_RANDOM);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button),TRUE);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  group=gtk_radio_button_group(GTK_RADIO_BUTTON(button));

  button=gtk_radio_button_new_with_label(group,"Disc/Track");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetSongOrder),
		     (gpointer)SONG_ORDER_DISC);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(button)),"Age");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetSongOrder),
		     (gpointer)SONG_ORDER_OLDEST);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(button)),"BPM");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetSongOrder),
		     (gpointer)SONG_ORDER_SLOWEST);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(TRUE,3);

  button=gtk_button_new_with_label("Generate Playlist");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(GeneratePlaylist),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_button_new_with_label("Re-Scan Database");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(DBScan),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  gtk_container_add(GTK_CONTAINER(query_page),vbox);
  gtk_widget_show(vbox);
}

void DeleteUser(void)
{
  char *user,*host;

  if(!have_root_pswd) {
    DisplayMsg("You must log in as root first");

    return;
  }

  user=gtk_entry_get_text(GTK_ENTRY(tmp_user_name_entry));
  if(!*user) {
    DisplayMsg("Please enter a username");

    return;
  }

  host=gtk_entry_get_text(GTK_ENTRY(tmp_user_host_entry));
  if(!*host) {
    DisplayMsg("Please enter a host.\n(Use 'localhost' for local machine)");

    return;
  }

  DeleteSQLUser(user,host,gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  ReloadSQL(gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  FillUserCList();
}

void CreateUser(GtkWidget *widget,gpointer data)
{
  char *user,*host,*pswd;

  if(!have_root_pswd) {
    DisplayMsg("You must log in as root first");

    return;
  }

  user=gtk_entry_get_text(GTK_ENTRY(tmp_user_name_entry));
  if(!*user) {
    DisplayMsg("Please enter a username");

    return;
  }

  host=gtk_entry_get_text(GTK_ENTRY(tmp_user_host_entry));
  if(!*host) {
    DisplayMsg("Please enter a host.\n(Use 'localhost' for local machine)");

    return;
  }

  pswd=gtk_entry_get_text(GTK_ENTRY(tmp_user_pswd_entry));
  if(!*pswd) {
    DisplayMsg("Please enter a password");

    return;
  }

  MakeSQLUser(user,host,pswd,gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  ReloadSQL(gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  FillUserCList();

  gtk_clist_select_row(GTK_CLIST(user_clist),
		       g_list_length(GTK_CLIST(user_clist)->row_list)-1,0);
}

void DoInitSQL(void)
{
  if(!have_root_pswd)
    DisplayMsg("You must log in as root first");
  else {
    InitSQL(gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));
    ReloadSQL(gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));
  }
}

void FillUserCList(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  char query[256];

  gtk_clist_clear(GTK_CLIST(user_clist));

  if(!(mysql_connect(&mysql,sql_host,"root",
		     gtk_entry_get_text(GTK_ENTRY(root_pswd_entry))))) {
    printf("SQL error: Unable to connect to server\n");

    return;
  }

  if(mysql_select_db(&mysql,"mysql")) {
    printf("SQL error: Unable to access mysql database\n");

    mysql_close(&mysql);

    return;
  }

  snprintf(query,256,"SELECT User,Host FROM db WHERE Db='%s'",sql_dbname);

  if(mysql_query(&mysql,query)) {
    printf("Query error\n");

    mysql_close(&mysql);
    return;
  }

  if(!(res = mysql_store_result(&mysql))) {
    printf("Query error\n");

    mysql_close(&mysql);
    return;
  }

  while((row=mysql_fetch_row(res))) {
    gtk_clist_append(GTK_CLIST(user_clist),row);
  }

  mysql_free_result(res);

  mysql_close(&mysql);
}


void LogInAsRoot(void)
{
  have_root_pswd=TRUE;

  FillUserCList();
}

void SelectUser(GtkWidget *widget,gint row,gint column,
	       GdkEventButton *event,gpointer data)
{
  char *buf;

  if(gtk_clist_row_is_visible(GTK_CLIST(user_clist),row)!=GTK_VISIBILITY_FULL)
    gtk_clist_moveto(GTK_CLIST(user_clist),row,0,0,0);

  gtk_clist_get_text(GTK_CLIST(user_clist),row,0,&buf);
  gtk_entry_set_text(GTK_ENTRY(tmp_user_name_entry),buf);

  gtk_clist_get_text(GTK_CLIST(user_clist),row,1,&buf);
  gtk_entry_set_text(GTK_ENTRY(tmp_user_host_entry),buf);
}

void MakeConfigPage(void)
{
  GtkWidget *cfgbox,*vbox,*vbox2,*hbox,*hbox2;
  GtkWidget *entry;
  GtkWidget *button;
  GtkWidget *notebook;
  GtkWidget *page;
  GtkWidget *label;
  GtkWidget *scroll;
  GtkWidget *frame;
  
  config_page=MakeNewPage("Config");

  cfgbox=gtk_vbox_new(FALSE,3);

  notebook=gtk_notebook_new();

  page=gtk_frame_new(NULL);

  vbox=gtk_vbox_new(FALSE,3);

  entry=MakeStrEntry(NULL,mp3exename,"MP3 executable",256,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,TRUE,TRUE,0);
  gtk_widget_show(entry);

  gtk_container_add(GTK_CONTAINER(page),vbox);
  gtk_widget_show(vbox);

  label=gtk_label_new("MP3");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook),page,label);
  gtk_widget_show(page);

  page=gtk_frame_new(NULL);

  vbox=gtk_vbox_new(FALSE,3);

  entry=MakeStrEntry(NULL,sql_host,"SQL host",60,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,TRUE,TRUE,0);
  gtk_widget_show(entry);

  entry=MakeStrEntry(NULL,sql_dbname,"SQL database",32,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,TRUE,TRUE,0);
  gtk_widget_show(entry);

  entry=MakeStrEntry(NULL,sql_user,"SQL username",16,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,TRUE,TRUE,0);
  gtk_widget_show(entry);

  entry=MakeStrEntry(NULL,sql_pswd,"SQL password",16,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,TRUE,TRUE,0);
  gtk_widget_show(entry);

  gtk_container_add(GTK_CONTAINER(page),vbox);
  gtk_widget_show(vbox);

  label=gtk_label_new("Client");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook),page,label);
  gtk_widget_show(page);

  page=gtk_frame_new(NULL);

  vbox=gtk_vbox_new(FALSE,3);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Root Password:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  root_pswd_entry=gtk_entry_new_with_max_length(16);
  gtk_box_pack_start(GTK_BOX(hbox),root_pswd_entry,TRUE,TRUE,0);
  gtk_widget_show(root_pswd_entry);

  button=gtk_button_new_with_label("Log in");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(LogInAsRoot),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,FALSE,FALSE,0);
  gtk_widget_show(button);

  button=gtk_button_new_with_label("Initialize SQL database");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(DoInitSQL),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,FALSE,FALSE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox2=gtk_hbox_new(TRUE,3);

  vbox2=gtk_vbox_new(FALSE,3);

  user_clist=gtk_clist_new_with_titles(2,user_titles);

  gtk_clist_set_selection_mode(GTK_CLIST(user_clist),GTK_SELECTION_SINGLE);

  gtk_clist_column_titles_passive(GTK_CLIST(user_clist));

  gtk_clist_set_column_width(GTK_CLIST(user_clist),0,125);
  gtk_clist_set_column_width(GTK_CLIST(user_clist),1,125);

  gtk_signal_connect(GTK_OBJECT(user_clist),"select_row",
		     GTK_SIGNAL_FUNC(SelectUser),
		     NULL);

#if (GTK_MINOR_VERSION == 0)
  gtk_clist_set_policy(GTK_CLIST(user_clist),GTK_POLICY_AUTOMATIC,
		       GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(vbox2),user_clist,TRUE,TRUE,0);
  gtk_widget_show(user_clist);
#else
  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),user_clist);
  gtk_widget_show(user_clist);

  gtk_box_pack_start(GTK_BOX(vbox2),scroll,TRUE,TRUE,0);
  gtk_widget_show(scroll);
#endif

  gtk_box_pack_start(GTK_BOX(hbox2),vbox2,TRUE,TRUE,0);
  gtk_widget_show(vbox2);

  frame=gtk_frame_new(NULL);

  vbox2=gtk_vbox_new(FALSE,3);
  gtk_container_border_width(GTK_CONTAINER(vbox2),3);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Username:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  tmp_user_name_entry=gtk_entry_new_with_max_length(16);
  gtk_box_pack_start(GTK_BOX(hbox),tmp_user_name_entry,TRUE,TRUE,0);
  gtk_widget_show(tmp_user_name_entry);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,TRUE,TRUE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Host:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  tmp_user_host_entry=gtk_entry_new_with_max_length(60);
  gtk_box_pack_start(GTK_BOX(hbox),tmp_user_host_entry,TRUE,TRUE,0);
  gtk_widget_show(tmp_user_host_entry);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,TRUE,TRUE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Password:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  tmp_user_pswd_entry=gtk_entry_new_with_max_length(16);
  gtk_box_pack_start(GTK_BOX(hbox),tmp_user_pswd_entry,TRUE,TRUE,0);
  gtk_widget_show(tmp_user_pswd_entry);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,TRUE,TRUE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  button=gtk_button_new_with_label("Add user account");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(CreateUser),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);
 
  button=gtk_button_new_with_label("Delete user account");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(DeleteUser),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  gtk_container_add(GTK_CONTAINER(frame),vbox2);
  gtk_widget_show(vbox2);

  gtk_box_pack_start(GTK_BOX(hbox2),frame,TRUE,TRUE,0);
  gtk_widget_show(frame);

  gtk_box_pack_start(GTK_BOX(vbox),hbox2,TRUE,TRUE,0);
  gtk_widget_show(hbox2);

  gtk_container_add(GTK_CONTAINER(page),vbox);
  gtk_widget_show(vbox);

  label=gtk_label_new("Server");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook),page,label);
  gtk_widget_show(page);

  gtk_box_pack_start(GTK_BOX(cfgbox),notebook,TRUE,TRUE,0);
  gtk_widget_show(notebook);

  gtk_container_add(GTK_CONTAINER(config_page),cfgbox);
  gtk_widget_show(cfgbox);

  DBScan();
}

void Homepage(void)
{
  system("netscape -remote openURL\\(http://www.nostatic.org/ddj\\)");
}

void MakeAboutPage(void)
{
  GtkWidget *vbox,*vbox2,*hbox;
  GtkWidget *label;
  GtkWidget *logo;
  GtkWidget *ebox;
  GtkWidget *button;
  char versionbuf[20];

  about_page=MakeNewPage("About");

  ebox=gtk_event_box_new();
  gtk_widget_set_style(ebox,style_wb);

  vbox=gtk_vbox_new(TRUE,5);
  gtk_container_border_width(GTK_CONTAINER(vbox),3);

  logo=Loadxpm(ddj_xpm);

  gtk_box_pack_start(GTK_BOX(vbox),logo,FALSE,FALSE,0);
  gtk_widget_show(logo);

  vbox2=gtk_vbox_new(TRUE,0);

  label=gtk_label_new("DigitalDJ");
  gtk_widget_set_style(label,style_wb);
  gtk_box_pack_start(GTK_BOX(vbox2),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  sprintf(versionbuf,"Version %s",VERSION);
  label=gtk_label_new(versionbuf);
  gtk_widget_set_style(label,style_wb);
  gtk_box_pack_start(GTK_BOX(vbox2),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  label=gtk_label_new("Copyright (c) 1999, Mike Oliphant");
  gtk_widget_set_style(label,style_wb);
  gtk_box_pack_start(GTK_BOX(vbox2),label,FALSE,FALSE,0);
  gtk_widget_show(label);


  hbox=gtk_hbox_new(TRUE,0);

  button=gtk_button_new_with_label("http://www.nostatic.org/ddj");
  gtk_widget_set_style(button,style_dark_grey);
  gtk_widget_set_style(GTK_BUTTON(button)->child,style_wb);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(Homepage),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,FALSE,FALSE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);
  

  gtk_container_add(GTK_CONTAINER(vbox),vbox2);
  gtk_widget_show(vbox2);

  gtk_container_add(GTK_CONTAINER(ebox),vbox);
  gtk_widget_show(vbox);

  gtk_container_add(GTK_CONTAINER(about_page),ebox);
  gtk_widget_show(ebox);
}

GtkWidget *MakeNewPage(char *name)
{
  GtkWidget *page;
  GtkWidget *label;

  page=gtk_frame_new(NULL);
  gtk_widget_show(page);

  label=gtk_label_new(name);
  gtk_notebook_append_page(GTK_NOTEBOOK(main_notebook),page,label);

  return page;
}

GtkWidget *MakeControls(void)
{
  GtkWidget *vbox,*vbox3,*hbox,*imagebox,*hbox2;
  GtkWidget *button;
  GtkWidget *ebox,*ebox2,*lcdbox;
  GtkWidget *image;

  ebox=gtk_event_box_new();
  gtk_widget_set_style(ebox,style_wb);

  vbox=gtk_vbox_new(FALSE,0);
  gtk_container_border_width(GTK_CONTAINER(vbox),0);

  vbox3=gtk_vbox_new(FALSE,2);
  gtk_container_border_width(GTK_CONTAINER(vbox3),2);

  lcdbox=gtk_event_box_new();
  gtk_signal_connect(GTK_OBJECT(lcdbox),"button_press_event",
		     GTK_SIGNAL_FUNC(ToggleControlButtons),NULL);
  gtk_widget_set_style(lcdbox,style_LCD);

  hbox2=gtk_hbox_new(FALSE,0);

  imagebox=gtk_vbox_new(FALSE,0);

  image=Loadxpm(upleft_xpm);
  gtk_box_pack_start(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  image=Loadxpm(lowleft_xpm);
  gtk_box_pack_end(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  gtk_box_pack_start(GTK_BOX(hbox2),imagebox,FALSE,FALSE,0);
  gtk_widget_show(imagebox);
  
  hbox=gtk_hbox_new(FALSE,0);
  gtk_container_border_width(GTK_CONTAINER(hbox),0);

  current_song_label=gtk_label_new("");
  gtk_widget_set_style(current_song_label,style_LCD);
  gtk_box_pack_start(GTK_BOX(hbox),current_song_label,TRUE,TRUE,0);
  gtk_widget_show(current_song_label);

  imagebox=gtk_vbox_new(FALSE,0);

  image=Loadxpm(upright_xpm);
  gtk_box_pack_start(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  image=Loadxpm(lowright_xpm);
  gtk_box_pack_end(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  gtk_box_pack_start(GTK_BOX(hbox),imagebox,FALSE,FALSE,0);
  gtk_widget_show(imagebox);

  ebox2=gtk_event_box_new();
  gtk_widget_set_usize(ebox2,2,0);
  gtk_widget_set_style(ebox2,style_wb);
  gtk_box_pack_start(GTK_BOX(hbox),ebox2,FALSE,FALSE,0);
  gtk_widget_show(ebox2);

  imagebox=gtk_vbox_new(FALSE,0);

  image=Loadxpm(upleft_xpm);
  gtk_box_pack_start(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  image=Loadxpm(lowleft_xpm);
  gtk_box_pack_end(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  gtk_box_pack_start(GTK_BOX(hbox),imagebox,FALSE,FALSE,0);
  gtk_widget_show(imagebox);

  play_time_label=gtk_label_new("00:00");
  gtk_widget_set_style(play_time_label,style_LCD);
  gtk_box_pack_start(GTK_BOX(hbox),play_time_label,FALSE,FALSE,0);
  gtk_widget_show(play_time_label);

  gtk_container_add(GTK_CONTAINER(hbox2),hbox);
  gtk_widget_show(hbox);

  imagebox=gtk_vbox_new(FALSE,0);

  image=Loadxpm(upright_xpm);
  gtk_box_pack_start(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  image=Loadxpm(lowright_xpm);
  gtk_box_pack_end(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  gtk_box_pack_start(GTK_BOX(hbox2),imagebox,FALSE,FALSE,0);
  gtk_widget_show(imagebox);
  
  gtk_container_add(GTK_CONTAINER(lcdbox),hbox2);
  gtk_widget_show(hbox2);

  gtk_box_pack_start(GTK_BOX(vbox3),lcdbox,FALSE,FALSE,0);
  gtk_widget_show(lcdbox);

  gtk_box_pack_start(GTK_BOX(vbox),vbox3,FALSE,FALSE,0);
  gtk_widget_show(vbox3);

  control_button_box=gtk_vbox_new(TRUE,0);

  hbox=gtk_hbox_new(TRUE,0);

  button=ImageButton(playpaus_xpm);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(PlayCB),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Play track / Pause play",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=ImageButton(stop_xpm);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(StopPlay),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Stop play",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=ImageButton(prevtrk_xpm);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(PrevSong),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Go to previous song",NULL);
  gtk_widget_show(button);

  button=ImageButton(nexttrk_xpm);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(NextSong),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Go to next song",NULL);
  gtk_widget_show(button);

  button=ImageButton(minmax_xpm);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(MinMax),0);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Toggle playlist display",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=ImageButton(quit_xpm);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Exit DDJ",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(ShutDown),0);
  gtk_widget_show(button);
  
  gtk_box_pack_start(GTK_BOX(control_button_box),hbox,TRUE,TRUE,0);
  gtk_widget_show(hbox);

  gtk_box_pack_start(GTK_BOX(vbox),control_button_box,TRUE,TRUE,0);
  gtk_widget_show(control_button_box);


  gtk_container_add(GTK_CONTAINER(ebox),vbox);
  gtk_widget_show(vbox);

  return ebox;
}

GdkColor *MakeColor(int red,int green,int blue)
{
  GdkColor *c;

  c=(GdkColor *)g_malloc(sizeof(GdkColor));
  c->red=red;
  c->green=green;
  c->blue=blue;

  gdk_color_alloc(gdk_colormap_get_system(),c);

  return c;
}

static gfloat style_color_mods[5]={0.0,-0.1,0.2,-0.2};

GtkStyle *MakeStyle(GdkColor *fg,GdkColor *bg,gboolean do_grade)
{
  GtkStyle *def;
  GtkStyle *sty;
  int state;

  def=gtk_widget_get_default_style();
  sty=gtk_style_copy(def);

  for(state=0;state<5;state++) {
    sty->fg[state]=*fg;

    sty->bg[state]=*bg;

    if(do_grade) {
      sty->bg[state].red+=sty->bg[state].red*style_color_mods[state];
      sty->bg[state].green+=sty->bg[state].green*style_color_mods[state];
      sty->bg[state].blue+=sty->bg[state].blue*style_color_mods[state];
    }
  }

  return sty;
}

void MakeStyles(void)
{
  gdk_color_white(gdk_window_get_colormap(window->window),&gdkwhite);
  gdk_color_black(gdk_window_get_colormap(window->window),&gdkblack);

  color_LCD=MakeColor(33686,38273,29557);
  color_dark_grey=MakeColor(0x4444,0x4444,0x4444);
  
  style_wb=MakeStyle(&gdkwhite,&gdkblack,FALSE);
  style_LCD=MakeStyle(&gdkblack,color_LCD,FALSE);
  style_LCD->font=gdk_font_load
    ("-b&h-lucidatypewriter-medium-r-normal-sans-12-120-75-75-m-70-iso8859-1");
  style_dark_grey=MakeStyle(&gdkwhite,color_dark_grey,TRUE);
}

int main(int argc,char *argv[])
{
  char buf[256];

  srand(time(NULL));

  snprintf(buf,256,"%s/.ddj",getenv("HOME"));

  LoadConfig(buf,"DDJ",1,1,cfg_entries);

  gtk_init(&argc,&argv);

  if((GTK_MINOR_VERSION!=gtk_minor_version)) {
    printf("Warning: This version of DigitalDJ was compiled using gtk+ "
	   "version %d.%d.%d, and you are running gtk+ version %d.%d.%d...\n",
	   GTK_MAJOR_VERSION,GTK_MINOR_VERSION,GTK_MICRO_VERSION,
	   gtk_major_version,gtk_minor_version,gtk_micro_version);
  }

  if(!ConnectSQL()) {
    DisplayMsg("Unable to connect to SQL Server.\nPerhaps you need to initialize the database?");
  }
  else {
    DisconnectSQL();
    have_db=TRUE;
  }

  gtk_rc_parse("~/.gtkrc");

  window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_realize(window);

  gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,FALSE);

  sprintf(buf,"%s %s",PROGRAM,VERSION);
  gtk_window_set_title(GTK_WINDOW(window),buf);

  gtk_signal_connect(GTK_OBJECT(window),"delete_event",
		     GTK_SIGNAL_FUNC(ShutDown),NULL);

  gtk_signal_connect_after(GTK_OBJECT(window),"size_allocate",
			   GTK_SIGNAL_FUNC(CheckWindowPosition),NULL);
    
  winbox=gtk_vbox_new(FALSE,3);
  if(!minimized) gtk_container_border_width(GTK_CONTAINER(winbox),3);

  MakeStyles();

  main_notebook=gtk_notebook_new();

  MakePlayPage();
  MakeQueryPage();
  MakeEditPage();
  MakeConfigPage();
  MakeAboutPage();

  gtk_box_pack_start(GTK_BOX(winbox),main_notebook,TRUE,TRUE,0);
  gtk_widget_show(main_notebook);
  
  control_box=MakeControls();
  gtk_box_pack_start(GTK_BOX(winbox),control_box,FALSE,FALSE,0);
  gtk_widget_show(control_box);

  gtk_container_add(GTK_CONTAINER(window),winbox);
  gtk_widget_show(winbox);

  gtk_widget_show(window);

  GeneratePlaylist();

  UpdateScrollDisplay(TRUE);

  gtk_timeout_add(250,TimeOut,0);

  gtk_main();

  DoSaveConfig();

  return 0;
}
