/*
 * Detect a Library for hardware detection
 *
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#include "detect.h"
#include "serial.h"
#include "utils.h"


typedef enum {
  MODEM_V32,            /* Modem can do V.32 (9600 bps)               */
  MODEM_V32B,           /* Modem can do V.32bis (14400 bps)           */
  MODEM_V34,            /* Modem can do V.34 (33600 bps)              */
  MODEM_V34S,           /* Modem can do V.34S (symmetrical-only)      */
  MODEM_V34B,           /* Modem can do V.34B (extended asymmetric)   */
  MODEM_V34BS,          /* Modem can do V.34BS (extended symmetric)   */
  MODEM_VX2,            /* Modem can do V.X2 (older 56k)              */
  MODEM_V90,            /* Modem can do V.90 (56k)                    */
  MODEM_V110,           /* Modem can do V.110 (isdn japan / USA)      */
  MODEM_V120,           /* Modem can do V.120 (isdn data calls USA)   */
  MODEM_FAX1,           /* Modem can do FAX Class 1                   */
  MODEM_FAX2,           /* Modem can do FAX Class 2                   */
  MODEM_MNP5,           /* Modem can do MNP5                          */
  MODEM_V42BIS          /* Modem can do V.42bis                       */
} ModemType;



/* Copyrigt (c) RedHat Sotfware */
extern char *modem_response(int fd, char *send){
  char done;
  int respindex, starttime, temp;
  char *resp = (char *) my_malloc(128);
  struct timeval timo;
  char buf[2];
  
  if(write(fd, send, strlen(send)) != strlen(send)){
    return NULL;
  }/*endif*/
  done = 0;
  respindex = 0;
  starttime = time(NULL);
  memset(resp, 0, 128);
  while(!done){
    timo.tv_sec = 0;
    timo.tv_usec = 250000;
    if(wait_for_input(fd, &timo) > 0){
      temp = read(fd, buf, 1);
      if(temp < 0){
        if(errno != EAGAIN){
          return NULL;
	}/*endif*/
      }else{
        resp[respindex++] = buf[0];
      }/*endif*/
    }else{
      done = 1;
    }/*endif*/
    if(time(NULL) - starttime > 5){
      done = 1;
    }/*endif*/
    if(respindex > 127){
      done = 1;
    }/*endif*/
  }/*endwhile*/
  return resp;
}/*endfunc modem_response*/


/* Copyrigt (c) RedHat Sotfware */
/* Modified by bero and alex */
extern int find_legacy_modem(int fd){
  int temp, done;
  int respindex;
  int starttime;
  unsigned char resp[10], buf[2];
  struct timeval timo;

  free(init_serial_port(fd));
  usleep(200000);

  if(write(fd, "AT\r", 3) != 3){
   /* We can write to a modem - so this is not one. */
    return 2;
  }/*endif*/
  done = 0;
  respindex = 0;
  starttime = time(NULL);
  memset(resp, 0, sizeof(resp));
  while(!done){
    timo.tv_sec = 0;
    timo.tv_usec = 250000;
    if(wait_for_input(fd, &timo) > 0){
      temp = read(fd, buf, 1);
      if(temp < 0){
        if(errno != EAGAIN){
          return 1;
        }/*endif*/
      }else{
        resp[respindex++] = buf[0];
      }/*endif*/
    }else{
      done = 1;
    }/*endif*/
    if(time(NULL) - starttime > 5){
      done = 1;
    }/*endif*/
    if(respindex > 9){
      done = 1;
    }/*endif*/
  }/*endwhile*/
  if(strstr(resp, "OK") || strchr(resp, '0')){
    return 3;
  }else{
    return 2;
  }/*endif*/
}/*endfunc find_legacy_modem */


/* Determine maximum port speed for the modem */
#define PORTSPEED(x) cfsetospeed(attr, x); cfsetispeed(attr, x); set_serial_attr(fd, attr);


/*Added by bero */
extern long modem_speed(int fd){
  /* Port speed... */
  struct termios *attr;
  
  long speed = -1;
  
  attr = init_serial_port(fd);
  PORTSPEED(B230400)
  if(find_legacy_modem(fd) == 3){
    speed = 230400;
  }else{
    PORTSPEED(B115200)
    if(find_legacy_modem(fd) == 3){
      speed = 115200;
    }else{
      PORTSPEED(B57600)
      if(find_legacy_modem(fd) == 3){
        speed = 57600;
      }else{
        PORTSPEED(B38400)
        if(find_legacy_modem(fd) == 3){
          speed = 38400;
        }else{
          PORTSPEED(B19200)
          if(find_legacy_modem(fd) == 3){
            speed = 19200;
          }else{
            PORTSPEED(B9600)
            if(find_legacy_modem(fd) == 3){
              speed = 9600;
            }else{
              PORTSPEED(B4800)
              if(find_legacy_modem(fd) == 3){
                speed = 4800;
             }else{
                PORTSPEED(B2400)
                if(find_legacy_modem(fd) == 3){
                  speed = 2400;
                }else{
                  speed = 1200;
                }/*endif*/
              }/*endif*/
            }/*endif*/
          }/*endif*/
        }/*endif*/
      }/*endif*/
    }/*endif*/
  }/*endif*/
  free(attr);
  return speed;
}/*endfunc modem_speed*/

/*Added by bero */
extern short modem_capabilities(int fd){
  char *s = 0;
  short capabilities = 0;

  s = modem_response(fd, "ATV1+MS=V32,1,0,0\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_V32;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1+MS=V32B,1,0,0\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_V32B;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1+MS=V34,1,0,0\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_V34;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1+MS=V34S,1,0,0\r");
  if(strstr(s, "OK")){
   capabilities |= MODEM_V34S;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1+MS=V34B,1,0,0\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_V34B;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1+MS=V34BS,1,0,0\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_V34BS;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1+MS=VX2,1,0,0\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_VX2;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1+MS=V90,1,0,0\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_V90;
  }/*endif*/
 free(s);
  s = modem_response(fd, "ATV1+FCLASS=1\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_FAX1;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1+FCLASS=2\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_FAX2;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1%C1\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_MNP5;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1\\N3\r");
  if(strstr(s, "OK")){
    capabilities |= MODEM_V42BIS;
  }/*endif*/
  free(s);
  s = modem_response(fd, "ATV1\\N4\r");
 if(strstr(s, "OK")){
    capabilities |= MODEM_V42BIS;
  }/*endif*/
  free(s);
  /* Don't leave the modem screwed... */
  free(modem_response(fd, "ATZ\r"));
  return capabilities;
}/*endfunc modem_capabilities*/


extern struct modem_info *modem_detect(struct bus_lst *bus){
  struct isa_info *isa = (struct isa_info *)NULL;
  struct pci_info *pci = (struct pci_info *)NULL;
  struct usb_info *usb = (struct usb_info *)NULL;
  struct serial_info *serial = (struct serial_info *)NULL;
  struct modem_info *result = (struct modem_info *)NULL;
  static struct modem_info *first = (struct modem_info *)NULL;


  if(first){
    return first;
  }/*endif*/
  
  if(debug)
    fprintf(stdout, "\nProbing modems...\n");
  
  /********************************************************************/
  /************************** PCI MODEM  ******************************/
  /********************************************************************/
 if(debug)
   fprintf(stdout, "\tProbing PCI modem...\n");

  for(pci = bus->pci; pci; pci = pci->next){
    if(pci->type == MODEM){
      if(!first){
        first = result = (struct modem_info *)
                                   my_malloc(sizeof(struct modem_info));
      }else{
        result->next = (struct modem_info *)
                                   my_malloc(sizeof(struct modem_info));
        result = result->next;
      }/*endif*/
      result->next = (struct modem_info *)NULL;
      result->vendor = pci->vendor;
      result->model = pci->model;
      result->device = s_not_available;
      result->module = pci->modulename;
      result->bus = PCI;
      result->io = 0;
      result->irq = 0;
      result->long_id = pci->id;
      result->speed = -1;
      if(debug)
	fprintf(stdout, "\t\tFound %s", result->model);
    }/*endif*/
  }/*next pci*/
  
  
  /********************************************************************/
  /************************** USB MODEM  ******************************/
  /********************************************************************/
  if(debug)
    fprintf(stdout, "\tProbing USB modem...\n");

  for(usb = bus->usb; usb; usb = usb->next){
    if(usb->type == MODEM){
      if(!first){
        first = result = (struct modem_info *)
	  my_malloc(sizeof(struct modem_info));
      }else{
        result->next = (struct modem_info *)
	  my_malloc(sizeof(struct modem_info));
        result = result->next;
      }/*endif*/
      result->next = (struct modem_info *)NULL;
      result->vendor = usb->vendor;
      result->model = usb->model;
      result->device = s_not_available;
      result->module = usb->modulename;
      result->bus = USB;
      result->io = 0;
      result->irq = 0;
      result->long_id = usb->id;
      result->speed = -1;
      if(debug)
	fprintf(stdout, "\t\tFound %s", result->model);
    }/*endif*/
  }/*next usb*/
  
  
  /********************************************************************/
  /********************* ISA MODEM DETECTION **************************/
  /********************************************************************/
  if(debug)
    fprintf(stdout, "\tProbing ISA modem...\n");

  for(isa = bus->isa; isa; isa = isa->next){
    if(isa->type == MODEM){
      if(!first){
	first = result = (struct modem_info *) 
	  my_malloc(sizeof(struct modem_info));
      }else{
	result->next = (struct modem_info *) 
	  my_malloc(sizeof(struct modem_info));
      result = result->next;
      }/*endif*/
      result->next = (struct modem_info *)NULL;
      
      result->board_num = isa->board_num;
      result->board_id = isa->board_id;
      result->dev_num = isa->dev_num;
      result->dev_id = isa->dev_id;
      result->vendor = isa->vendor;
      result->model = isa->model;
      result->device = s_not_available;
      result->module = isa->modulename;
      result->bus = ISA;
      result->io = isa->io;
      result->irq = isa->irq;
      result->speed = -1;
      if(debug)
	fprintf(stdout, "\t\tFound %s %s on %s\n", 
		result->vendor, result->model, result->device);
    }/*endif*/
  }/*next isa*/


  /********************************************************************/
  /********************* SERIAL MODEM DETECTION ***********************/
  /********************************************************************/
  if(debug)
    fprintf(stdout, "\tProbing serial modem...\n");
  if(!first){
    for(serial = bus->serial; serial; serial=serial->next){
      if(serial->type == MODEM){
        if(!first){
          first = result = (struct modem_info *) 
                                   my_malloc(sizeof(struct modem_info));
        }else{
          result->next = (struct modem_info *) 
                                   my_malloc(sizeof(struct modem_info));
          result = result->next;
        }/*endif*/
        result->next = (struct modem_info *)NULL;
        
        result->board_num = 0;
        result->board_id = 0;
        result->dev_num = 0;
        result->dev_id = serial->dev_id;
        result->vendor = serial->vendor;
        result->model = serial->model;
        result->module = serial->modulename;
        result->bus = SERIAL;
        result->device = serial->device;
        result->io = -1;
        result->irq = -1;
        result->speed = serial->speed;
	if(debug)
	  fprintf(stdout, "\t\tFound %s %s on %s\n", 
		  result->vendor, result->model, result->device);
      }/*endif*/
    }/*next serial*/
  }/*endif*/
  return first;
}/*endfunc modem_detect*/
