/* exportmudela.c
 * Functions for actually exporting what Denemo's working on to a mudela file
 *
 * AJT 14/3/2000 Parametised for quick midi playback
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 2000, 2001 Matthew Hiller
 */

/* Yes -- if you're curious, this is a very straightforward adaptation
 * of what I used to export mudela in my extensions to kooBase. */

#include "config.h"
#include "datastructures.h"
#include "utils.h"
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>

static void
determinekey (gint number, gchar ** keyname)
{
  switch (number)
    {
    case -7:
      *keyname = "ces";
      break;
    case -6:
      *keyname = "ges";
      break;
    case -5:
      *keyname = "des";
      break;
    case -4:
      *keyname = "aes";
      break;
    case -3:
      *keyname = "ees";
      break;
    case -2:
      *keyname = "bes";
      break;
    case -1:
      *keyname = "f";
      break;
    case 0:
      *keyname = "c";
      break;
    case 1:
      *keyname = "g";
      break;
    case 2:
      *keyname = "d";
      break;
    case 3:
      *keyname = "a";
      break;
    case 4:
      *keyname = "e";
      break;
    case 5:
      *keyname = "b";
      break;
    case 6:
      *keyname = "fis";
      break;
    case 7:
      *keyname = "cis";
      break;
    case 8:
      *keyname = "gis";
      break;
    case 9:
      *keyname = "dis";
      break;
    case 10:
      *keyname = "ais";
      break;
    default:
      *keyname = _("%{error. defaulting to%}c");
      break;
    }
}

static void
determineclef (gint type, gchar ** clefname)
{
  switch (type)
    {
    case TREBLE:
      *clefname = "treble";
      break;
    case BASS:
      *clefname = "bass";
      break;
    case ALTO:
      *clefname = "alto";
      break;
    case G_8:
      *clefname = "\"G_8\"";
      break;
    case TENOR:
      *clefname = "tenor";
      break;
    case SOPRANO:
      *clefname = "soprano";
      break;
    default:
      *clefname = _("%{error. defaulting to%}treble");
      break;
    }
  /* I've found the quotes are necessary for ^ and _ clefs
   * to parse correctly */
}

static gint
internaltomuduration (gint internalduration)
{
  return 1 << internalduration;
}

/* Actually export the mudela. This could've been done with lots
 * of g_list_foreach'es, but for the most part I consider those things
 * to be cumbersome.
 *
 * The function works in four passes: the first pass writes out all the
 * voices; the second pass groups them into staffs; the third pass
 * writes out the score block. I'm also planning to add a fourth pass
 * that will write out additional score blocks for instrumental parts,
 * as instructed by the user, but this is not implemented yet.
 *
 * The loading routines, in contrast, glean all the information
 * they need about the score from the information written in the first
 * pass.
 */

void
exportmudela (gchar * thefilename, struct scoreinfo *si, gint start, gint end)
{
  gchar *clef;
  /* clef name */
  gchar *keyname;
  /* key signature name */
  gboolean empty_measure;
  gint cur_stime1;
  gint cur_stime2;
  FILE *fp;
  staffnode *curstaff;
  staff *curstaffstruct;
  measurenode *curmeasure;
  objnode *curobjnode;
  mudelaobject *curobj;
  gint curmeasurenum;
  chord chordval;
  gint duration, numdots;
  gint prevduration, prevnumdots;
  gint octave, enshift;
  gint noteheadtype;
  gchar mid_c_offset;
  GList *curtone;
  GString *filename = g_string_new (thefilename);
  gint i, j, last, k;
  gboolean first_staff = TRUE;
  gboolean is_normalnotehead = TRUE;
  gboolean is_chordmode = FALSE;
  gchar temp[50];
  /*gchar dynamic_string[50];*/
  GString *dynamic_string =NULL;

  /* Append .ly onto the filename if necessary */
  if (strcmp (filename->str + filename->len - 3, ".ly"))
    g_string_append (filename, ".ly");

  /* Now open the file */
  fp = fopen (filename->str, "w");

  /* And cut off the filename extension for use in the file itself */
  g_string_truncate (filename, filename->len - 3);

  fprintf (fp, _("%% Mudela file generated by Denemo version "));
  fprintf (fp, VERSION "\n\n");
  fprintf (fp, "%%http://www.gnu.org/software/denemo/denemo.html\n\n");
  /*Print out lilypond syntax version */
  /*fprintf (fp, _("\\version \"1.3.148\"\n"));*/
  /* header stuff */
  fprintf (fp, "\\header{\n");
  fprintf (fp, "\ttitle = \"%s\"\n", si->title->str);
  fprintf (fp, "\tsubtitle = \"%s\"\n", si->subtitle->str);
  fprintf (fp, "\tcomposer = \"%s\"\n", si->composer->str);
  fprintf (fp, "}\n\n");

  fprintf (fp, "\\include \"paper16.ly\"\n");

  /* First pass */
  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
    {
      curstaffstruct = curstaff->data;
      prevduration = 0;
      prevnumdots = -1;
      fprintf (fp, "%s = \\notes \\context Voice = $%s {\n",
	       curstaffstruct->lily_name->str,
	       curstaffstruct->lily_name->str);
      /* When Lilypond changes such that it no longer accepts
       * '$', fix this. Of course, it's problematic that numerals
       * are really useful in staff names... */

      if (curstaffstruct->transposition != 0)
	fprintf (fp, "\n\t\\property Staff.transposing = #%d\n",
		 curstaffstruct->transposition);
      /* Write a comment for Denemo to recognize later if this is not
       * a primary voice */
      if (curstaffstruct->voicenumber == 2)
	fprintf (fp, "%%!Nonprimary Voice\n");
      /* I ought to get rid of Mr. Magic Numeric Constant there */

      /* The midi instrument */
      fprintf (fp, "\t\\property Staff.midiInstrument = \"%s\"\n",
	       curstaffstruct->midi_instrument->str);

      /* Time signature */
      fprintf (fp, "\t\\time %d/%d\n", curstaffstruct->stime1,
	       curstaffstruct->stime2);

      cur_stime1 = curstaffstruct->stime1;
      cur_stime2 = curstaffstruct->stime2;

      /* Determine the key signature */

      determinekey (curstaffstruct->skey_isminor ?
		    curstaffstruct->skey + 3 :
		    curstaffstruct->skey, &keyname);
      fprintf (fp, "\t\\key %s", keyname);
      if (curstaffstruct->skey_isminor)
	fprintf (fp, " \\minor");
      else
	fprintf (fp, " \\major");
      fprintf (fp, "\n");

      /* Determine the clef */

      determineclef (curstaffstruct->sclef, &clef);
      fprintf (fp, "\t\\clef %s\n", clef);
      curmeasurenum = 0;
      curmeasure = curstaffstruct->measures;

      if (end)
	last = end;
      else	
	last = g_list_length(curmeasure);

      /* Now each measure */
      if (start)
	curmeasure = g_list_nth (curmeasure, start - 1);

      for (i = MAX (start, 1); curmeasure && i <= last;
	   curmeasure = curmeasure->next, i++)
	{
	  empty_measure = TRUE;

	  if ((++curmeasurenum % 5) == 0)
	    fprintf (fp, "%%%d\n", curmeasurenum);
	  fprintf (fp, "\t");

	  for (curobjnode = curmeasure->data; curobjnode;
	       curobjnode = curobjnode->next)
	    {
	      curobj = curobjnode->data;
	      /* I'm marching off the end of the page - put me
	       * in another function */
	      switch (curobj->type)
		{
		case CHORD:
		  empty_measure = FALSE;
		  chordval = curobj->u.chordval;
		  duration = internaltomuduration (chordval.baseduration);
		  numdots = chordval.numdots;
		  is_chordmode = FALSE;
		  if (!chordval.tones)
		    {		/* A rest */
		      fprintf (fp, "r");
		      /* Duplicated code follows. I ought to fix that */
		      if (duration != prevduration || numdots != prevnumdots)
			{
			  /* only in this case do we explicitly note the duration */
			  fprintf (fp, "%d", duration);
			  prevduration = duration;
			  prevnumdots = numdots;
			  for (j = 0; j < numdots; j++)
			    fprintf (fp, ".");
			}
		      fprintf (fp, " ");
		    }
		  else
		    {
		      if (chordval.slur_end_p)
			fprintf (fp, ")");
		      if (chordval.tones->next)
			{
			  is_chordmode = TRUE;
			  fprintf (fp, "<");
			}
		      for (curtone = chordval.tones; curtone;
			   curtone = curtone->next)
			{
			  noteheadtype =
			    ((note *) curtone->data)->noteheadtype;

			  switch (noteheadtype)
			    {
			    case NORMAL:
			      if (!is_normalnotehead)
				{
				  fprintf (fp,
					   "\n\t\\property Voice.noteHeadStyle = #\'default ");
				  is_normalnotehead = !is_normalnotehead;
				}
			      break;
			    case CROSS:
			      fprintf (fp,
				       "\n\t\\property Voice.noteHeadStyle = #\'cross ");
			      is_normalnotehead = FALSE;
			      break;
			    case HARMONIC:
			      fprintf (fp,
				       "\n\t\\property Voice.noteHeadStyle = #\'harmonic ");
			      is_normalnotehead = FALSE;
			      break;
			    case DIAMOND:
			      fprintf (fp,
				       "\n\t\\property Voice.noteHeadStyle = #\'diamond ");
			      is_normalnotehead = FALSE;
			      break;
			    default:
			      fprintf (fp,
				       "\n\t\\property Voice.noteHeadStyle = #\'default ");
			      break;
			    }

			  mid_c_offset =
			    ((note *) curtone->data)->mid_c_offset;
			  fprintf (fp, "%c",
				   mid_c_offsettoname (mid_c_offset));
			  enshift = ((note *) curtone->data)->enshift;
			  if (enshift < 0)
			    for (k = enshift; k; k++)
			      fprintf (fp, "es");
			  else
			    for (k = enshift; k; k--)
			      fprintf (fp, "is");
			  octave = mid_c_offsettooctave (mid_c_offset);
			  if (octave < 0)
			    for (; octave; octave++)
			      fprintf (fp, ",");
			  else
			    for (; octave; octave--)
			      fprintf (fp, "\'");
			  if (duration != prevduration
			      || numdots != prevnumdots)
			    {
			      /* only in this case do we explicitly note the duration */
			      fprintf (fp, "%d", duration);
			      prevduration = duration;
			      prevnumdots = numdots;
			      for (j = 0; j < numdots; j++)
				fprintf (fp, ".");
			    }
			  if (curtone->next)
			    fprintf (fp, " ");
			}	/* End chord loop */
		      
		      if(chordval.dynamics) {
		         dynamic_string = (GString *)chordval.dynamics->data;
		         if (is_chordmode)
			   fprintf (fp, "-\\%s", dynamic_string->str);
			 else
			   fprintf (fp, "-\\%s ", dynamic_string->str);
		      }
		      
		      
		      if (chordval.has_stacatto_p)
			fprintf (fp, " -\\staccato");
		      if (chordval.is_accented_p)
			fprintf (fp, "-\\accent");
		      if (chordval.has_fermata_p)
			fprintf (fp, " -\\fermata");
		      if (chordval.has_tenuto_p)
			fprintf (fp, " -\\tenuto");
		      if (chordval.has_trill_p)
			fprintf (fp, " -\\trill");
		      if (chordval.has_turn_p)
			fprintf (fp, " -\\turn");
		      if (chordval.has_mordent_p)
			fprintf (fp, " -\\mordent");
		      if (chordval.has_staccatissimo_p)
			fprintf (fp, " -\\staccatissimo");
		      if (chordval.crescendo_begin_p)
			fprintf (fp, " -\\cr");
		      else if (chordval.diminuendo_begin_p)
			fprintf (fp, " -\\decr");
		      if (chordval.crescendo_end_p)
			fprintf (fp, " -\\rc");
		      else if (chordval.diminuendo_end_p)
			fprintf (fp, " -\\rced");
		      if (chordval.tones->next && chordval.has_dynamic)
			{
			  sprintf (temp, ">");
			}
		      else if (chordval.tones->next)
			fprintf (fp, ">");

		      if (chordval.slur_begin_p)
			fprintf (fp, "(");
		      if (chordval.is_tied)
			fprintf (fp, " ~");
		      fprintf (fp, " ");
		    }		/* End code for dealing with chord */
		  break;
		case CLEF:
		  determineclef (curobj->u.clefval.type, &clef);
		  fprintf (fp, "\\clef %s ", clef);
		  break;
		case KEYSIG:
		  determinekey (curobj->u.keyval.isminor ?
				curobj->u.keyval.number + 3 :
				curobj->u.keyval.number, &keyname);
		  fprintf (fp, "\\key %s", keyname);
		  if (curobj->u.keyval.isminor)
		    fprintf (fp, " \\minor");
		  else
		    fprintf (fp, " \\major");
		  fprintf (fp, " ");
		  break;
		case TIMESIG:
		  fprintf (fp, "\\time %d/%d ", curobj->u.timeval.time1,
			   curobj->u.timeval.time2);
		  cur_stime1 = curobj->u.timeval.time1;
		  cur_stime2 = curobj->u.timeval.time2;
		  break;
		case TUPOPEN:
		  /* added by Yu CHeung "toby" Ho 3 Jun 00, adapted by Hiller
		   * 8 Jun 00 (happy birthday to me...) :) */
		  fprintf (fp, "\\times %d/%d { ", curobj->u.tupval.numerator,
			   curobj->u.tupval.denominator);
		  break;
		case TUPCLOSE:
		  fprintf (fp, "} ");
		  break;
		case GRACE_START:
		  fprintf (fp, "\\grace { ");
		  break;
		case GRACE_END:
		  fprintf (fp, "} ");
		case STEMDIRECTIVE:
		  switch (curobj->u.stemval.type)
		    {
		    case STEMDOWN:
		      fprintf (fp, "\\stemDown ");
		      break;
		    case STEMBOTH:
		      fprintf (fp, "\\stemBoth ");
		      break;
		    case STEMUP:
		      fprintf (fp, "\\stemUp ");
		      break;
		    }
		  break;
		case DYNAMIC:
		  /*if (is_chordmode)
		    {
		      sprintf (dynamic_string, "-\\%s ",
			       curobj->u.dynval.type->str);
		      strcat (dynamic_string, temp);
		      fprintf (fp, "%s", dynamic_string);
		    }
		  else
		    fprintf (fp, "-\\%s ", curobj->u.dynval.type->str);*/
		  break;
		default:
		  break;
		}
	    }			/* Done with this measure */
	  if (empty_measure)
	    {
	      fprintf (fp, "s1*%d/%d ", cur_stime1, cur_stime2);
	      prevduration = 0;
	    }
	  if (curmeasure->next)
	    fprintf (fp, "|\n");
	  else
	    fprintf (fp, "\\bar \"|.\"\n");
	}			/* Done with this staff */

      fprintf (fp, "}\n");
    }				/* Done with pass one */

  /* Now to create staffs; second pass. This pass will also put
   * score blocks in for each staff -- useful for part extraction --
   * but it'll comment them out with Lilypond block-comment markers
   * first */

  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
    {
      curstaffstruct = curstaff->data;
      if (curstaffstruct->voicenumber != 2)
	{
	  /* Print the stuff that indicates the start of a new staff */
	  if (!first_staff)
	    /* Close out the preceding staff first */
	    fprintf (fp, ">\n\n");
	  fprintf (fp, "\n%sStaff = \\context Staff = %sStaff <\n",
		   curstaffstruct->lily_name->str,
		   curstaffstruct->lily_name->str);
	}
      fprintf (fp, "\t\\%s\n", curstaffstruct->lily_name->str);
      first_staff = FALSE;
    }
  fprintf (fp, ">\n\n");
  /* End second pass */

  /* Now make the master score; third pass */

  fprintf (fp, "\\score {\n" "\t<\n");
  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
    {
      curstaffstruct = curstaff->data;
      if (curstaffstruct->voicenumber != 2)
	fprintf (fp, "\t\t\\%sStaff\n", curstaffstruct->lily_name->str);
    }
  fprintf (fp,
	   "\t>\n"
	   "\t\\paper {\n"
	   "\t}\n"
	   "\t\\midi {\n" "\t\t\\tempo 4 = %d\n" "\t}\n" "}\n", si->tempo);

  /* Third pass finished */

  fclose (fp);
  g_string_free (filename, FALSE);
}
