/* chordops.c
 *
 * functions which manipulate chords
 * For denemo, a gtk+ frontend to Lilypond, the GNU music typesetter
 *
 * (c) 2000, 2001 Matthew Hiller <matthew.hiller@yale.edu>
 *
 */

#include <math.h>
#include <string.h>
#include "chordops.h"
#include "datastructures.h"	/* includes <glib.h> */
#include "utils.h"

/* Calculates the height of a notehead */

static void
calcheight (gpointer data, gpointer user_data)
{
  note *thenote = data;
  gint dclef = GPOINTER_TO_INT (user_data);

  thenote->y = calculateheight (thenote->mid_c_offset, dclef);
}

void
newclefify (mudelaobject * thechord, gint dclef)
{
  g_list_foreach (thechord->u.chordval.tones, calcheight,
		  GINT_TO_POINTER (dclef));
  thechord->u.chordval.highesty =
    calculateheight (thechord->u.chordval.highestpitch, dclef);
  thechord->u.chordval.lowesty =
    calculateheight (thechord->u.chordval.lowestpitch, dclef);
}

/* This function goes through a chord and checks to see how its notes are
 * laid out, and if it will have to push the display of any tones to the
 * "wrong" side of the staff as a result */

void
findreversealigns (mudelaobject * thechord)
{
  GList *current;
  GList *previous;
  note *curnote;
  note *prevnote;

  thechord->u.chordval.is_reversealigned = FALSE;
  if (thechord->u.chordval.tones) 
    {
      if (thechord->u.chordval.is_stemup)
	{
	  /* note clusters are painted left-right from bottom to top */
	  previous = thechord->u.chordval.tones;
	  current = previous->next;
	  prevnote = (note *) previous->data;
	  prevnote->reversealign = FALSE;
	  for (;
	       current;
	       previous = current, prevnote = curnote, current = current->next)
	    {
	      curnote = (note *) current->data;
	      if (prevnote->mid_c_offset == curnote->mid_c_offset - 1)
		{
		  thechord->u.chordval.is_reversealigned = TRUE;
		  curnote->reversealign = !prevnote->reversealign;
		}
	      else
		curnote->reversealign = FALSE;
	    }			/* End for */
	}
      else
	{
	  /* the stem's down
	   * note clusters are painted right-left from top to bottom */
	  previous = g_list_last (thechord->u.chordval.tones);
	  current = previous->prev;
	  prevnote = (note *) previous->data;
	  prevnote->reversealign = FALSE;
	  for (;
	       current;
	       previous = current, prevnote = curnote, current = current->prev)
	    {
	      curnote = (note *) current->data;
	      if (prevnote->mid_c_offset == curnote->mid_c_offset + 1)
		{
		  curnote->reversealign = !prevnote->reversealign;
		  thechord->u.chordval.is_reversealigned = TRUE;
		}
	      else
		curnote->reversealign = FALSE;
	    }			/* End for */
	}				/* End else */
    } /*End if*/
  setpixelmin (thechord);
}

mudelaobject *
newchord (gint baseduration, gint numdots)
{
  mudelaobject *thechord = g_malloc (sizeof (mudelaobject));

  thechord->type = CHORD;
  thechord->u.chordval.tones = NULL;
  thechord->u.chordval.dynamics = NULL;
  thechord->u.chordval.highestpitch = G_MININT;
  thechord->u.chordval.lowestpitch = G_MAXINT;
  thechord->u.chordval.baseduration = baseduration;
  thechord->u.chordval.numdots = numdots;
  thechord->u.chordval.sum_mid_c_offset = 0;
  thechord->u.chordval.numtones = 0;
  thechord->u.chordval.is_tied = FALSE;
  thechord->u.chordval.is_reversealigned = FALSE;
  thechord->u.chordval.slur_begin_p = FALSE;
  thechord->u.chordval.slur_end_p = FALSE;
  thechord->u.chordval.crescendo_begin_p = FALSE;
  thechord->u.chordval.crescendo_end_p = FALSE;
  thechord->u.chordval.diminuendo_begin_p = FALSE;
  thechord->u.chordval.diminuendo_end_p = FALSE;
  thechord->u.chordval.hasanacc = FALSE;
  thechord->u.chordval.has_stacatto_p = FALSE;
  thechord->u.chordval.has_fermata_p = FALSE;
  thechord->u.chordval.is_accented_p = FALSE;
  thechord->u.chordval.has_tenuto_p = FALSE;
  thechord->u.chordval.has_turn_p = FALSE;
  thechord->u.chordval.has_trill_p = FALSE;
  thechord->u.chordval.has_mordent_p = FALSE;
  thechord->u.chordval.has_staccatissimo_p = FALSE;
  thechord->u.chordval.has_marcato_p = FALSE;
  thechord->u.chordval.has_ubow_p = FALSE;
  thechord->u.chordval.has_dbow_p = FALSE;
  thechord->u.chordval.has_rheel_p = FALSE;
  thechord->u.chordval.has_lheel_p = FALSE;
  thechord->u.chordval.has_rtoe_p = FALSE;
  thechord->u.chordval.has_ltoe_p = FALSE;
  thechord->u.chordval.is_grace = FALSE;
  thechord->u.chordval.struck_through = FALSE;
  thechord->u.chordval.has_dynamic = FALSE;
  thechord->u.chordval.is_highlighted = FALSE;
  set_basic_numticks (thechord);
  return thechord;
}

static gint
findcomparefunc (gconstpointer a, gconstpointer b)
{
  const note *anote = a;
  const int bnum = GPOINTER_TO_INT (b);

  if (anote->mid_c_offset == bnum)
    return 0;			/* Identical */
  else
    return 1;			/* Not identical */
}

static gint
insertcomparefunc (gconstpointer a, gconstpointer b)
{
  const note *anote = a;
  const note *bnote = b;

  return anote->mid_c_offset - bnote->mid_c_offset;
}

void
addtone (mudelaobject * thechord, gint mid_c_offset, gint enshift, gint dclef)
{
  note *newnote;

  if (!g_list_find_custom
      (thechord->u.chordval.tones, GINT_TO_POINTER (mid_c_offset),
       findcomparefunc))
    {
      /* A-ha! The note isn't already in the chord */
      newnote = g_malloc (sizeof (note));
      newnote->mid_c_offset = mid_c_offset;
      newnote->enshift = enshift;
      newnote->reversealign = FALSE;
      newnote->y = calculateheight (mid_c_offset, dclef);
      newnote->noteheadtype = NORMAL;
      thechord->u.chordval.tones =
	g_list_insert_sorted (thechord->u.chordval.tones, newnote,
			      insertcomparefunc);
      if (mid_c_offset > thechord->u.chordval.highestpitch)
	{
	  thechord->u.chordval.highestpitch = mid_c_offset;
	  thechord->u.chordval.highesty =
	    calculateheight (mid_c_offset, dclef);
	}
      if (mid_c_offset < thechord->u.chordval.lowestpitch)
	{
	  thechord->u.chordval.lowestpitch = mid_c_offset;
	  thechord->u.chordval.lowesty =
	    calculateheight (mid_c_offset, dclef);
	}
      thechord->u.chordval.sum_mid_c_offset += mid_c_offset;
      thechord->u.chordval.numtones++;
    }
}

/* This function finds the node of the closest chord tone to n; in the
 * case of equally distant chord tones, it'll select the higher tones
 * of the two */

/* I don't think that I could have quite done this with a g_list_find_custom */

GList *
findclosest (GList * tones, gint n)
{
  GList *cur_tnode = tones;
  GList *next_tnode;
  note *cur_tone;
  note *next_tone;
  gint distance_from_cur;
  gint distance_from_next;

  if (!cur_tnode)
    return NULL;
  for (next_tnode = cur_tnode->next;;
       cur_tnode = next_tnode, next_tnode = cur_tnode->next)
    {
      cur_tone = cur_tnode->data;
      if (n <= cur_tone->mid_c_offset || !next_tnode)
	/* Aha! We have no other options */
	return cur_tnode;
      else
	{
	  next_tone = next_tnode->data;
	  if (cur_tone->mid_c_offset < n && n < next_tone->mid_c_offset)
	    {
	      distance_from_cur = n - cur_tone->mid_c_offset;
	      distance_from_next = next_tone->mid_c_offset - n;
	      if (distance_from_cur < distance_from_next)
		return cur_tnode;
	      else
		return next_tnode;
	    }
	}
    }				/* End for loop */
}

void
removetone (mudelaobject * thechord, gint mid_c_offset, gint dclef)
{
  GList *tnode;			/* Tone node to remove */
  note *tone;

  tnode = findclosest (thechord->u.chordval.tones, mid_c_offset);
  if (tnode)
    {
      tone = tnode->data;
      if (!tnode->next)	{	/* That is, we're removing the highest pitch */
	if (tnode->prev)
	  {
	    thechord->u.chordval.highestpitch =
	      ((note *) tnode->prev->data)->mid_c_offset;
	    thechord->u.chordval.highesty =
	      calculateheight (thechord->u.chordval.highestpitch, dclef);
	  }
	else
	  {
	    thechord->u.chordval.highestpitch = G_MININT;
	    /* Had to take care of this somewhere */
	    thechord->u.chordval.is_tied = FALSE;   
	  }
      }
      if (!tnode->prev) {	/* That is, we're removing the lowest pitch */
	if (tnode->next)
	  {
	    thechord->u.chordval.lowestpitch =
	      ((note *) tnode->next->data)->mid_c_offset;
	    thechord->u.chordval.lowesty =
	      calculateheight (thechord->u.chordval.lowestpitch, dclef);
	  }
	else
	  thechord->u.chordval.lowestpitch = G_MAXINT;
      }
      thechord->u.chordval.sum_mid_c_offset -= tone->mid_c_offset;

      /* Now that we no longer need any info in tnode or tone, 
       * actually free stuff */

      g_free (tone);
      thechord->u.chordval.tones =
	g_list_remove_link (thechord->u.chordval.tones, tnode);
      g_list_free_1 (tnode);
    }
}


void
shiftpitch (mudelaobject * thechord, gint mid_c_offset, gint is_sharpening)
{
  GList *tnode;			/* Tone node to inflect */
  note *tone;

  tnode = findclosest (thechord->u.chordval.tones, mid_c_offset);
  if (tnode)
    {
      tone = tnode->data;
      if (is_sharpening)
	tone->enshift = MIN (tone->enshift + 1, 2);
      else
	tone->enshift = MAX (tone->enshift - 1, -2);
    }
}

void
changedur (mudelaobject * thechord, gint baseduration, gint numdots)
{
  thechord->u.chordval.baseduration = baseduration;
  thechord->u.chordval.numdots = numdots;
  set_basic_numticks (thechord);
}

void
changenumdots (mudelaobject * thechord, gint number)
{
  thechord->u.chordval.numdots =
    MAX (thechord->u.chordval.numdots + number, 0);
  set_basic_numticks (thechord);
}

void
freechord (mudelaobject * thechord)
{
  g_list_foreach (thechord->u.chordval.tones, freeit, NULL);
  g_list_free (thechord->u.chordval.tones);
  g_free (thechord);
}

mudelaobject *
clone_chord (mudelaobject * thechord)
{
  mudelaobject *ret = g_malloc (sizeof (mudelaobject));
  GList *curtone;
  note *newnote;

  /* I'd use a g_list_copy here, only that won't do the deep copy of
   * the list data that I'd want it to */
  memcpy (ret, thechord, sizeof (mudelaobject));
  ret->u.chordval.tones = NULL;
  for (curtone = thechord->u.chordval.tones; curtone; curtone = curtone->next)
    {
      newnote = g_malloc (sizeof (note));
      memcpy (newnote, curtone->data, sizeof (note));
      ret->u.chordval.tones = g_list_append (ret->u.chordval.tones, newnote);
    }
  return ret;
}
