# debpartial_mirror - partial debian mirror package tool
# (c) 2004 Otavio Salvador <otavio@debian.org>, Marco Presi <zufus@debian.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# TODO:
# Add a control on md5sum to check wich files has to be updated

import os
import os.path
import time

import apt_pkg

from cdd import Package
from cdd import PackageList
from cdd import FileSystem

from debpartial_mirror import Config
from debpartial_mirror import Download

class Dists:
    """
    This class provides methods to manage dists on partial-mirrors
    """

    def __init__ (self, backend):
        self._backend = backend
        self._filesystem = FileSystem.FileSystem(backend["mirror_dir"],
                                                 backend["name"])
        self._files = []
        self._already_loaded = False

class MirrorDists(Dists):

    def __init__(self, backend):
        Dists.__init__(self, backend)

        # Partial package lists
        self.__bin = PackageList.PackageList()
        self.__source = PackageList.PackageList()

        # Full package lists
        self.__full_bin    = PackageList.PackageList()
        self.__full_source = PackageList.PackageList()

    def _fill_files(self):
        # If we already have it doesn't rerun
        if self._files:
            return
        
        for component in self._backend["components"]:
            for dist in self._backend["distributions"]:
                dist='dists/'+dist
                for arch in self._backend["architectures"]:
                    arch='binary-'+arch
                    mirror_name = self._backend["mirror_dir"] +\
                                  self._backend["name"]
                    
                    self._files.append("%s/%s/%s/Packages.gz" %
                                       (dist, component, arch))
                    self._files.append("%s/Release" % dist )
                    if not component.endswith("/debian-installer"):
                        # debian-installer components don't have Release files
                        self._files.append("%s/%s/%s/Release" %
                                           (dist, component, arch))
            if self._backend["get_sources"] and not \
               component.endswith("/debian-installer"):
                self._files.append("%s/%s/source/Sources.gz" %
                                   (dist, component))

    def get_index(self):
        """
        Get only index files (on Debian, this mean: Packages.gz
        and Sources.gz)
        """
        # TODO: Put checking using Release file when available.
        files = []
        self._fill_files()
        for file in self._files:
            if os.path.basename(file) != "Release":
                files.append(file.split('.gz')[0])
        return files

    def get_binary_list(self):
        return self.__bin

    def get_source_list(self):
        return self.__source

    def get_full_binary_list(self):
        return self.__full_bin

    def get_full_source_list(self):
        return self.__full_source

    def load(self):
        if self._already_loaded:
            return
        
        for file in self.get_index():
            # Choose object type
            if os.path.basename(file) == 'Packages':
                pkg = Package.Package
                pkglist = self.__full_bin
            elif os.path.basename(file) == 'Sources':
                pkg = Package.SourcePackage
                pkglist = self.__full_source
            # Load file on list
            parse_in = open(os.path.join(self._filesystem.base(), file), "r")
            parse = apt_pkg.ParseTagFile(parse_in)
            print "Loading", file
            while parse.Step() == 1:
                package = pkg(parse.Section)
                pkglist.add(package)
            parse_in.close()
        self._already_loaded = True
        
    def filter(self):
        
        pkgfilter = []
        try:
            pkgfilter = self._backend['filter']
        except Config.InvalidOption:
            pass

        # to load indexes
        self.load()
        # Apply filter or use as final list
        if pkgfilter:
            self.__bin = self.__full_bin.filter(pkgfilter)
            self.__source = self.__full_source.filter(pkgfilter)
        else:
            self.__bin = self.__full_bin
            self.__source = self.__full_source

    def resolve(self):

        # Collect the lists to be used to resolve dependencies
        pkglists = []
        backends = []
        backend_names = [self._backend._name]

        try:
            backend_names = self._backend['resolve_deps_using'] + backend_names
        except Config.InvalidOption:
            pass # Is possible to don't have this option

        # Build our list of PackageLists to be used for resolve_depends
        for backend in self._backend.backends:
            if backend._name in backend_names:
                backends.append(backend)
                pkglists.append(backend.get_full_binary_list())

        # Resolve our dependencies using pkglists
        try:
            self.__bin.resolve_depends(pkglists)
        except PackageList.BrokenDependencies, exception:
            print "ERROR: Can't resolve dependencies:"
            for dependency in exception.deps:
                print "       -", dependency['Package']
        
    def process(self):
        self.filter()
        self.resolve()

class RemoteDists (MirrorDists):
    """
    This class provides methods to fill dists dir downloading remote files
    """

    def update (self):
        """ Get only files that need updates """
        self._fill_files()
        download = Download.Download(name="Dist_" + self._backend["name"])
        for file in self._files:
            self._filesystem.mkdir(os.path.dirname(file))
            server = "%s/%s" % (self._backend["server"], file)
            filename = "%s/%s" % (self._filesystem.base(), file)
            download.get(server, filename)

        download.wait_mine()

        for file in self._files:
            if os.path.basename(file) != "Release":
                try:
                    self._filesystem.uncompress(file)
                except IOError:
                    return False
            
class LocalDists (MirrorDists):
    """
    This class provides methods to fill dists dir downloading local files
    """

    def update (self):
        """ Get only files that need updates """
        self._fill_files()
        for server, filename, dirname in self._files:
            orig, filename = file
            self._filesystem.mkdir(dirname)
            os.link (orig.split('file://')[1], filename)

class MergeDists (Dists):
    """
    This class provides methods to fill dists dir when merging backends
    """
    def __init__(self, backend):
        Dists.__init__(self, backend)

        self.__mirrors = []
        # Collect the related mirrors reference
        for backend_name in self._backend['backends']:
            for backend in self._backend.backends:
                if backend._name == backend_name:
                    self.__mirrors.append(backend)
                    break
        
    def merge(self):

        components = []
        architectures = []
        
        # Create one empty pkg list for each index
        pkglists = {}
        from string import join
        for mirror in self.__mirrors:
            for index in mirror._dists.get_index():
                index = index.split("/")
                index[1] = self._backend['name']
                index = join(index,"/")
                if index not in pkglists.keys():
                    pkglists[index] = PackageList.PackageList()

        # Fill package lists
        for mirror in self.__mirrors:
            for pkg in mirror.get_binary_list().values():
                component = pkg['Filename'].split("/")[1]
                if pkg['Filename'].endswith("udeb"):
                    component += "/debian-installer"
                tmp = pkg['Filename'].split("_")
                tmp = tmp[len(tmp) -1]
                arch = tmp.split(".")[0]
                
                indices = []
                    
                dist='dists/'+self._backend['name']
                if pkg['Filename'].endswith("deb"):
                    if arch == "all":
                        for arch in mirror["architectures"]:
                            indices.append("%s/%s/binary-%s/Packages" %
                                           (dist, component, arch))
                    else:
                        indices.append("%s/%s/binary-%s/Packages" %
                                       (dist, component, arch))
                        
                if pkg['Filename'].endswith(".dsc") or \
                   pkg['Filename'].endswith(".gz"):
                    indices.append("%s/%s/source/Packages" % (dist, component))

                for index in indices:
                    if not pkglists[index].has_key(pkg['Package']):
                        pkglists[index].add(pkg)

        # Write package lists and release files
        components = []
        architectures = []
        
        from string import join
        
        release = {}
        
        for key in ['origin', 'label', 'description', 'suite', 'codename']:
            if self._backend.has_key(key):
                release[key] = self._backend[key]
            else:
                release[key] = "DebPartialMirror"

        for index in pkglists.keys():

            if not self._filesystem.exists(os.path.dirname(index)):
                self._filesystem.mkdir(os.path.dirname(index))

            if index.endswith("Packages"):

                component = index.split("/")[2]
                if index.split("/")[3] == "debian-installer":
                    component += "/debian-installer"
                if index.split("/")[3] == "source":
                    architecture = "source"
                else:
                    architecture = (index.split("/")[-2]).split("-")[1]
                
                # Write Release
                if not component.endswith("/debian-installer"):
                        # debian-installer components don't have Release files

                        if component not in components:
                            components.append(component)

                        if architecture not in architectures:
                            architectures.append(architecture)

                        file = os.path.dirname(index) + "/Release"
                        self._files.append(join(file.split("/")[2:],"/"))
                        out = open(os.path.join(self._filesystem.base(), file),
                                                "w+")
                        out.write("Archive: %s\n" % (self._backend._name))
                        out.write("Component: %s\n" % (component))
                        out.write("Origin: %s\n"    % (release['origin']))
                        out.write("Label: %s\n"     % (release['label']))
                        out.write("Architecture: %s\n" % (architecture))
                        out.close()

            # Write index
            out = open(os.path.join(self._filesystem.base(), index), "w+")
            for pkg in pkglists[index].values():
                out.write(pkg.dump() + "\n")
            out.close()
            self._filesystem.compress(index)
            filename = join(index.split("/")[2:],"/")
            self._files.append(filename)
            self._files.append(filename + ".gz")

	out = open(os.path.join(self._filesystem.base(),
                            "dists/" + self._backend._name + "/Release"), "w+")

        out.write("Origin: %s\n" % (release['origin']))
        out.write("Label: %s\n"  % (release['label']))
	out.write("Suite: %s\n"  % (release['suite']))
	out.write("Codename: %s\n"  % (release['codename']))
	out.write("Date: %s\n" % (time.strftime("%a, %d %b %Y %H:%M:%S UTC",
                                            time.gmtime(time.time()))))
	out.write("Architectures: %s\n" % (join(architectures)))
        out.write("Components: %s\n" % (join(components)))
	out.write("Description: %s\n" % (release['description']))
	out.write("MD5Sum:\n")
        for filename in self._files:
            fullpath = "dists/%s/%s" % (self._backend._name, filename)
	    out.write(" %s         %8d %s\n" %
                  (self._filesystem.md5sum(fullpath),
                   self._filesystem.size(fullpath), filename))
	out.write("SHA1:\n")
        for filename in self._files:
	    out.write(" %s         %8d %s\n" %
                  (self._filesystem.sha1sum(fullpath),
                   self._filesystem.size(fullpath), filename))
                    
    def get_mirrors (self):
        return self.__mirrors
