#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <limits.h>
#include <sys/wait.h>
#include <string.h>
#include <dirent.h>
#include <errno.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <gnome.h>

#include "do_connect.h"
#include "main.h"
#include "misc_gtk.h"
#include "dctc_process.h"
#include "init_fnc.h"
#include "timed_out_string.h"

static void reset_interface(void)
{
	char *clst[]={	"download_clist",
						"upload_clist",
						"queue_clist",
						"user_clist",
						"gdl_ctree",		/* a CTREE is also a CLIST */
						"uaddr_clist",
						NULL};

	int i;

	i=0;
	while(clst[i]!=NULL)
	{
		gtk_clist_clear(GTK_CLIST(get_widget_by_widget_name(clst[i])));
		i++;
	}
}

typedef struct
{
	char *var_name;
	char *var_value;
} VAR_ENTRY;

static GArray *vars=NULL;

/***********************************/
/* add a new var to the vars array */
/***********************************/
void add_var(char *var_name, char *var_value)
{
	VAR_ENTRY nw;
	VAR_ENTRY *tmp;
	int i;
	int fnd=0;

	if(vars==NULL)
		vars=g_array_new(FALSE,FALSE,sizeof(VAR_ENTRY));

	for(i=0;i<vars->len;i++)
	{
		tmp=&(g_array_index(vars,VAR_ENTRY,i));

		if((tmp->var_name!=NULL)&&(!strcmp(tmp->var_name,var_name)))
		{
			/* a var with the same name exists, update its value */
			free(tmp->var_value);
			tmp->var_value=strdup(var_value);
			fnd=1;
			break;
		}
	}

	if(!fnd)
	{
		nw.var_name=strdup(var_name);
		nw.var_value=strdup(var_value);
		vars=g_array_append_val(vars,nw);
	}

	fix_pref_window();
}

/***********************************/
/* get a value from the vars array */
/***********************************/
const char *get_var(char* var_name)
{
	int i;

	if(vars==NULL)
		return NULL;

	for(i=0;i<vars->len;i++)
	{
		VAR_ENTRY *v;

		v=&(g_array_index(vars,VAR_ENTRY,i));
		if((v->var_name!=NULL)&&(!strcmp(v->var_name,var_name)))
			return v->var_value;
	}
	return NULL;
}


static void clear_var_array(void)
{
	if(vars==NULL)
	{
		vars=g_array_new(FALSE,FALSE,sizeof(VAR_ENTRY));
		fix_pref_window();
		return;
	}

	/* free all vars entry */
	while(vars->len!=0)
	{
		VAR_ENTRY *v;

		v=&(g_array_index(vars,VAR_ENTRY,0));
		if(v->var_name)
			free(v->var_name);
		if(v->var_value)
			free(v->var_value);
		vars=g_array_remove_index_fast(vars,0);
	}
	fix_pref_window();
}

static guint timeout_hdl=0;

/**************************************/
/* get the size of the file dir/fname */
/**********************************************/
/* output: file size. On error, 0 is returned */
/**********************************************/
unsigned long get_file_size(const char *dir, char *fname)
{
	char buf[51200];
	struct stat st;

	if(fname[0]!='/')
	{
		sprintf(buf,"%s/%s",dir,fname);
	}
	else
	{
		strcpy(buf,fname);
	}

	if(stat(buf,&st))
		return 0;
	return st.st_size;
}

/******************************************************************************************/
/* update the line i of the clist (clst) with the name 't' in the dir (cur_dir="dl_path") */
/******************************************************************************************/
void update_dl_clist_size(GtkCList *clst, int i, char *t, const char *cur_dir)
{
	unsigned long full_size;

	if(cur_dir==NULL)
		return;

	gtk_clist_get_text(clst,i,2,&t);
	full_size=strtoul(t,NULL,10);

	gtk_clist_get_text(clst,i,5,&t);
	if((t!=NULL)&&(strlen(t)))
	{
		char buf[5120];
		unsigned long cur_size;
		DL_XTRA *xtra;
					
		cur_size=get_file_size(cur_dir,t);

		xtra=gtk_clist_get_row_data(clst,i);
		if(xtra==NULL)
			sprintf(buf,"%15lu (%.2f%%)",cur_size,100.0*(double)cur_size/(double)full_size);
		else
		{
			double spd;
			time_t ttl_time;
			sprintf(buf,"%15lu (%.2f%%)",cur_size,100.0*(double)cur_size/(double)full_size);

			ttl_time=time(NULL)-xtra->start_time;
			
			spd=((double)cur_size-(double)xtra->start_pos)/((double)ttl_time);

			if(spd<1024.0)
			{
				sprintf(buf+strlen(buf)," %.2lfB/s",spd);
			}
			else if(spd<(1024.0*1024.0))
			{
				sprintf(buf+strlen(buf)," %.2lfKB/s",spd/1024.0);
			}
		}
	
		gtk_clist_set_text(clst,i,3,buf);
	}
}

/* this function is called 2 times per second */
static gint update_counter(gpointer data)
{
	const char *cur_dir;
	GtkWidget *w;

	if(current_dctc==NULL)
		goto eofunc;

	/* update GDL status */
	{
		static int counter=0;

		if(counter!=20)
			counter++;
		else
		{
			counter=0;
			send_data_to_dctc("/SLOWGDLLST\n");		/* force a refresh of the GDL list */
		}
	}

	/* update download status */
	timeout_tos();				/* purge TOS string */

	if((vars==NULL)||(vars->len==0))
		goto eofunc;

	w=get_widget_by_widget_name("download_clist");
	if(w==NULL)
		goto eofunc;

	if(GTK_CLIST(w)->rows==0)
		goto eofunc;

	cur_dir=get_var("dl_path");
	if(cur_dir==NULL)
		goto eofunc;

	{
		GtkCList *clst;
		char *t;
		int i;

		clst=GTK_CLIST(w);
		gtk_clist_freeze(clst);
		for(i=0;i<clst->rows;i++)
		{
			gtk_clist_get_text(clst,i,0,&t);
			if((t!=NULL)&&(strlen(t)))
			{
				update_dl_clist_size(clst,i,t,cur_dir);
			}
		}
		gtk_clist_thaw(clst);
	}
	eofunc:
	return TRUE;		/* keep this function alive */
}

/*********************************************************/
/* find a DCTC client already connected to the given hub */
/***********************************************************************************/
/* if a client still exists, connect_to_a_running_dctc is called and 1 is returned */
/***********************************************************************************/
/* if no_wait is not set, if a client already exist, the GUI switches to it, else */
/* the client remains unchanged, only the status is returned                      */
/**********************************************************************************/
static int already_connected(char *hub_address, int no_wait)
{
	DIR *dir;
	struct dirent *obj;

	dir=opendir(dctc_dir->str);
	if(dir==NULL)
		return 0;

	while((obj=readdir(dir))!=NULL)
	{
		if(strlen(obj->d_name)<=(5+8+1))		/* size should be at least enough for "dctc-xxxxxxxx-" */
			continue;

		if(!strcmp(hub_address,obj->d_name+(5+8+1)))
		{
			if(no_wait==0)
				connect_to_a_running_dctc(obj->d_name);
			closedir(dir);
			return 1;
		}

	}
	closedir(dir);
	return 0;
}

/*********************************************************/
/* add the given opt_name according to widget_name value */
/*********************************************************/
static void add_cmd_line_opt(GPtrArray *cmd_line, GStringChunk **sc,char *widget_name, char *opt_name)
{
	GtkWidget *w;
	char *t;

	w=get_widget_by_widget_name(widget_name);
	if(w==NULL)
	{
		fprintf(stderr,"add_cmd_line_opt: aborting, unknown widget: %s\n",widget_name);
		exit(1);
	}

	t=gtk_entry_get_text(GTK_ENTRY(w));
	if((t==NULL)||(strlen(t)==0))
		return;

	g_ptr_array_add(cmd_line,g_string_chunk_insert((*sc),opt_name));
	g_ptr_array_add(cmd_line,g_string_chunk_insert((*sc),t));
}

/*********************************************************/
/* add the given opt_name according to widget_name value */
/* same version as before except a --precmd prefix is put */
/*********************************************************/
static void add_cmd_line_opt_precmd(GPtrArray *cmd_line, GStringChunk **sc,char *widget_name, char *opt_name)
{
	GtkWidget *w;
	char *t;

	w=get_widget_by_widget_name(widget_name);
	if(w==NULL)
	{
		fprintf(stderr,"add_cmd_line_opt_precmd: aborting, unknown widget: %s\n",widget_name);
		exit(1);
	}

	t=gtk_entry_get_text(GTK_ENTRY(w));
	if((t==NULL)||(strlen(t)==0))
		return;

	g_ptr_array_add(cmd_line,g_string_chunk_insert((*sc),"--precmd"));

	{
		GString *nw;

		nw=g_string_new(opt_name);
		nw=g_string_append_c(nw,' ');
		nw=g_string_append(nw,t);
		g_ptr_array_add(cmd_line,g_string_chunk_insert((*sc),nw->str));
		g_string_free(nw,TRUE);
	}
}


/*********************************************************/
/* add the given opt_name according to widget_name value */
/***************************************************************************************************************************/
/* if the GTK_BUTTON is true, on_true1 is appended and if on_true2 is not null, it is also appended (as a 2nd argument)    */
/* if the GTK_BUTTON is false, on_false1 is appended and if on_false2 is not null, it is also appended (as a 2nd argument) */
/***************************************************************************************************************************/
static void add_cmd_line_opt_radio(GPtrArray *cmd_line, GStringChunk **sc,char *widget_name,
												const char *on_true1, const char *on_true2,
												const char *on_false1, const char *on_false2)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(widget_name);
	if(w==NULL)
	{
		fprintf(stderr,"add_cmd_line_opt_radio: aborting, unknown widget: %s\n",widget_name);
		exit(1);
	}

	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==TRUE)
	{
		if(on_true1)
		{
			g_ptr_array_add(cmd_line,g_string_chunk_insert((*sc),on_true1));
			if(on_true2)
				g_ptr_array_add(cmd_line,g_string_chunk_insert((*sc),on_true2));
		}
	}
	else
	{
		if(on_false1)
		{
			g_ptr_array_add(cmd_line,g_string_chunk_insert((*sc),on_false1));
			if(on_false2)
				g_ptr_array_add(cmd_line,g_string_chunk_insert((*sc),on_false2));
		}
	}
}

/********************/
/* start a new DCTC */
/***********************************************************************************************/
/* if no_wait is set, the -l flag is not added and DCTC starts without waiting a UI connection */
/***********************************************************************************************/
void start_a_new_dctc(char *hub_address, int no_wait)
{
	GPtrArray *cmd_line;
	GStringChunk *sc;
	int son;
	char *start_path;

	if(already_connected(hub_address,no_wait))
	{
		if(no_wait==0)
			gnome_app_error(GNOME_APP(main_window),_("Using client already connected to this hub"));
		return;
	}

	/* we must create a new dctc with the following arguments */
	/* -n (GTK_ENTRY(nickname_entry)) [if not empty] */
	/* -i (GTK_ENTRY(user_description_entry))  [if not empty] */
	/* -c (GTK_ENTRY(cnx_type_entry)) */
	/* -e (GTK_ENTRY(e_mail_entry)) [if not empty] */
	/* -d (GTK_HSCALE(sim_dl_hscale)) */
	/* -s (one for each clistentry of GTK_CLIST(shared_dir_clist)) [if not empty] */
	/* -o (GTK_ENTRY(size_offset_entry) x GTK_ENTRY(size_offset_unit_entry)) [if size_offset_entry is not empty or null] */
	/* -a	(GTK_ENTRY(xfer_host_ip_entry)) [if not empty] */
	/* -p (GTK_ENTRY(incoming_port_number_entry)) [if not empty] */
	/* -g hub_address */
	/* -f [if GTK_RADIO_BUTTON(passive_mode_radio_button) is set] */
	/* -x [if GTK_CHECK_BUTTON(enable_upload_checkbutton) is not set] */
	/* -v VERSION [if GTK_RADIO_BUTTON(override_version_check_button) is set and version_number_entry is not empty */
	/* -u UBL [if GTK_RADIO_BUTTON(ubl_check_button) is set and ubl_entry is not empty */
	cmd_line=g_ptr_array_new();
	sc=g_string_chunk_new(64);

	/* argv[0] */
	g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"dctc"));

	/* argv[1] */
	g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"dctc"));

	add_cmd_line_opt(cmd_line,&sc,"nickname_entry","-n");
	add_cmd_line_opt(cmd_line,&sc,"user_description_entry","-i");
	add_cmd_line_opt(cmd_line,&sc,"cnx_type_entry","-c");
	add_cmd_line_opt(cmd_line,&sc,"e_mail_entry","-e");
	
	add_cmd_line_opt(cmd_line,&sc,"socks_address_entry","-S");
	add_cmd_line_opt(cmd_line,&sc,"socks_port_entry","-P");
	add_cmd_line_opt(cmd_line,&sc,"socks_userid_entry","-U");

	add_cmd_line_opt(cmd_line,&sc,"vshare_dir_entry","-D");
	add_cmd_line_opt_precmd(cmd_line,&sc,"unodeport_entry","/UNODEPORT");

	/* -d */
	{
		GtkWidget *w;
		w=get_widget_by_widget_name("sim_dl_hscale");
		if(w==NULL)
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","sim_dl_hscale");
		else
		{
			char bf[512];

			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"-d"));

			sprintf(bf,"%u",(unsigned int)gtk_range_get_adjustment(GTK_RANGE(w))->value);
			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,bf));
		}
		
	}

	/* -s */
	{
		GtkWidget *w;
		w=get_widget_by_widget_name("shared_dir_clist");
		if(w==NULL)
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","shared_dir_clist");
		else
		{
			int i;

			char *t;

			for(i=0;i<GTK_CLIST(w)->rows;i++)
			{
				gtk_clist_get_text(GTK_CLIST(w),i,0,&t);

				if((t!=NULL)&&(strlen(t)!=0))
				{
					g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"-s"));
					g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,t));
				}
			}
		}
	}

	/* -o */
	{
		double value;

		GtkWidget *w;
		GtkWidget *w1;

		w=get_widget_by_widget_name("size_offset_entry");
		if(w==NULL)
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","size_offset_entry");
		else
		{
			w1=get_widget_by_widget_name("size_offset_unit_entry");
			if(w1==NULL)
				fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","size_offset_unit_entry");
			else
			{
				char *t;

				t=gtk_entry_get_text(GTK_ENTRY(w));
				if((t!=NULL)&&(strlen(t)!=0))
				{
					value=strtod(t,NULL);

					t=gtk_entry_get_text(GTK_ENTRY(w1));
					if((t!=NULL)&&(strlen(t)!=0))
					{
						char bf[512];
						if(!strcmp(t,_("Bytes")))
						{
							value*=1.0;
						}
						else if(!strcmp(t,_("KBytes")))
						{
							value*=1024.0;
						}
						else if(!strcmp(t,_("MBytes")))
						{
							value*=1024.0*1024.0;
						}
						else if(!strcmp(t,_("GBytes")))
						{
							value*=1024.0*1024.0*1024.0;
						}
						else
						{
							printf("unknown unit in size_offset_unit_entry\n");
						}

						g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"-o"));

						sprintf(bf,"%.0lf",value);
						g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,bf));
					}
				}
			}
		}
	}

	{
		GtkWidget *w;

		w=get_widget_by_widget_name("ignore_ipcheckbutton");
		if(w!=NULL)
		{	/* add the IP only if ignore ip is not set */
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==FALSE)
				add_cmd_line_opt(cmd_line,&sc,"xfer_host_ip_entry","-a");
		}
	}

	add_cmd_line_opt(cmd_line,&sc,"incoming_port_number_entry","-p");

	g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"-g"));
	g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,hub_address));

	{
		struct
		{
			char *widget_name;
			const char *on_true1;
			const char *on_true2;
			const char *on_false1;
			const char *on_false2;
		} tbl_opt_radio[]={	{"passive_mode_radio_button", "-f",NULL,NULL,NULL},
									{"enable_upload_checkbutton", NULL,NULL,"-x",NULL},
									{"use_done_dir_checkbutton", "-w",NULL,NULL,NULL},
									{"md5sum_computation_checkbutton", NULL,NULL,"-5",NULL},
									{"follow_forcemove_checkbutton", "--precmd","/FOLLOWFORCE","--precmd","/UNFOLLOWFORCE"},
									{"ddl_checkbutton", "--precmd","/DDL","--precmd","/NODDL"},
									{"dctclink_checkbutton", "--precmd","/LINK","--precmd","/NOLINK"},
									{"away_togglebutton", "--precmd","/AWAY","--precmd","/HERE"},
									{"grabip_checkbutton", "--precmd","/GBANIP","--precmd","/NOGBANIP"},
									{"force_dl_checkbutton", "--precmd","/DLFORCE","--precmd","/NODLFORCE"},
									{"hide_abscheckbutton", "--precmd","/HIDE_ABS","--precmd","/SHOW_ABS"},
									{"abort_upload_checkbutton", "--precmd","/ABORTLEAVED","--precmd","/NOABORTLEAVED"},
									{"hide_kick_checkbutton", "--precmd","/HIDE_KICK","--precmd","/SHOW_KICK"},
									{"lazykc_checkbutton", "--precmd","/LAZYKC","--precmd","/NOLAZYKC"},
									{"incoming_wake_up_checkbutton", "--precmd","/DFLAG with_incoming_wake_up 1","--precmd","/DFLAG with_incoming_wake_up 0"},
									{"sr_wake_up_checkbutton", "--precmd","/DFLAG with_sr_wake_up 1","--precmd","/DFLAG with_sr_wake_up 0"},
									{NULL,NULL,NULL,NULL,NULL}};
		int i;

		i=0;
		while(tbl_opt_radio[i].widget_name!=NULL)
		{
			add_cmd_line_opt_radio(cmd_line, &sc,tbl_opt_radio[i].widget_name,
												tbl_opt_radio[i].on_true1,tbl_opt_radio[i].on_true2,
												tbl_opt_radio[i].on_false1,tbl_opt_radio[i].on_false2);
			i++;
		}
	}

	/* -v VERSION [if GTK_RADIO_BUTTON(override_version_check_button) is set and version_number_entry is not empty */
	{
		GtkWidget *w;
		w=get_widget_by_widget_name("override_version_checkbutton");
		if(w==NULL)
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","override_version_checkbutton");
		else
		{
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==TRUE)
			{
				add_cmd_line_opt(cmd_line,&sc,"version_number_entry","-v");
			}
		}
	}

	/* -u UBL [if GTK_RADIO_BUTTON(ubl_checkbutton) is set and ubl_entry is not empty */
	{
		GString *xbl_string;
		GtkWidget *w;
		char *t;

		xbl_string=g_string_new("");

		/* add UBL value */
		w=get_widget_by_widget_name("ubl_checkbutton");
		if(w==NULL)
		{
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","ubl_checkbutton");
			g_string_sprintfa(xbl_string,"%d",INT_MAX);
		}
		else
		{
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==TRUE)
			{
				t=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name("ubl_entry")));
				if((t==NULL)||(strlen(t)==0))
					g_string_sprintfa(xbl_string,"%d",INT_MAX);
				else
					xbl_string=g_string_append(xbl_string,t);
			}
			else
				g_string_sprintfa(xbl_string,"%d",INT_MAX);
		}
		xbl_string=g_string_append_c(xbl_string,',');

		/* add DBL value */
		w=get_widget_by_widget_name("dbl_checkbutton");
		if(w==NULL)
		{
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","dbl_checkbutton");
			g_string_sprintfa(xbl_string,"%d",INT_MAX);
		}
		else
		{
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==TRUE)
			{
				t=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name("dbl_entry")));
				if((t==NULL)||(strlen(t)==0))
					g_string_sprintfa(xbl_string,"%d",INT_MAX);
				else
					xbl_string=g_string_append(xbl_string,t);
			}
			else
				g_string_sprintfa(xbl_string,"%d",INT_MAX);
		}
		xbl_string=g_string_append_c(xbl_string,',');


		/* add GBL value */
		w=get_widget_by_widget_name("gbl_checkbutton");
		if(w==NULL)
		{
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","gbl_checkbutton");
			g_string_sprintfa(xbl_string,"%d",INT_MAX);
		}
		else
		{
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==TRUE)
			{
				t=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name("gbl_entry")));
				if((t==NULL)||(strlen(t)==0))
					g_string_sprintfa(xbl_string,"%d",INT_MAX);
				else
					xbl_string=g_string_append(xbl_string,t);
			}
			else
				g_string_sprintfa(xbl_string,"%d",INT_MAX);
		}

		g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"-u"));
		g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,xbl_string->str));
		g_string_free(xbl_string,TRUE);
	}

	/* --precmd "/RECOND xxx" */
	{
		GtkWidget *w;
		w=get_widget_by_widget_name("reconnect_delay_scale");
		if(w==NULL)
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","reconnect_delay_scale");
		else
		{
			char bf[512];

			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"--precmd"));

			sprintf(bf,"/RECOND %u",(unsigned int)gtk_range_get_adjustment(GTK_RANGE(w))->value);
			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,bf));
		}
	}

	/* --precmd "/REBUILD xxx" */
	{
		GtkWidget *w;
		w=get_widget_by_widget_name("rebuild_delay_scale");
		if(w==NULL)
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","rebuild_delay_scale");
		else
		{
			char bf[512];

			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"--precmd"));

			sprintf(bf,"/REBUILD %u",60*(unsigned int)gtk_range_get_adjustment(GTK_RANGE(w))->value);
			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,bf));
		}
	}

	/* --precmd "/MAXRUNGDLSRC xxx" */
	{
		GtkWidget *w;
		w=get_widget_by_widget_name("maxrunspinbutton");
		if(w==NULL)
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","maxrunspinbutton");
		else
		{
			char bf[512];

			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"--precmd"));
			sprintf(bf,"/MAXRUNGDLSRC %u",(unsigned int)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w)));
			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,bf));
		}
	}

	/* --precmd "/GDLASOFFAFT xxx" */
	{
		GtkWidget *w;
		w=get_widget_by_widget_name("maxasoffspinbutton");
		if(w==NULL)
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","maxasoffspinbutton");
		else
		{
			char bf[512];

			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"--precmd"));
			sprintf(bf,"/GDLASOFFAFT %u",(unsigned int)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w)));
			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,bf));
		}
	}


	/* detach from tty */
	g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"-t"));

	/* wait GUI connection */
	if(no_wait==0)
		g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"-l"));

	/* --precmd "/DFLAG min_gdl_wake_up_delay xxx" */
	{
		GtkWidget *w;
		w=get_widget_by_widget_name("min_gdl_wake_up_delay_entry");
		if(w==NULL)
			fprintf(stderr,"cmd_line_opt: aborting, unknown widget: %s\n","min_gdl_wake_up_delay_entry");
		else
		{
			char bf[512];
			unsigned int v;
			char *t;
			t=gtk_entry_get_text(GTK_ENTRY(w));
			v=strtoul(t,NULL,10);
			if(v<30)
				v=30;

			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,"--precmd"));
			sprintf(bf,"/DFLAG min_gdl_wake_up_delay %u",v);
			g_ptr_array_add(cmd_line,g_string_chunk_insert(sc,bf));
		}
	}


	/* ... */
	g_ptr_array_add(cmd_line,NULL);		/* end the command line */

	/* get the path from where dctc must be started */
	{
		GtkWidget *w;

		w=get_widget_by_widget_name("dl_dir_entry");
		if(w==NULL)
		{
			fprintf(stderr,"add_cmd_line_opt: aborting, unknown widget: %s\n","dl_dir_entry");
			exit(1);
		}

		start_path=gtk_entry_get_text(GTK_ENTRY(w));
		if((start_path==NULL)||(strlen(start_path)==0))
			start_path=".";
		printf("start_path: %s\n",start_path);
	}

	/* reset the away toggle */
	{
		GtkWidget *w;

		w=get_widget_by_widget_name("away_togglebutton");
		if(w!=NULL)
      	gtk_toggle_button_set_active(
                  	GTK_TOGGLE_BUTTON(
                        	get_widget_by_widget_name("away_togglebutton")),
                  	FALSE);
	}

	switch(son=fork())
	{
		case -1:
					gnome_app_error(GNOME_APP(main_window),_("Fork fails. Unable to start a new client."));
					break;

		case 0:	/* the son */
					{
						chdir(start_path);
						execvp("dctc",(char**)cmd_line->pdata);
					}
					_exit(1);
					break;

		default: /* the GUI */
					{	
						GString *wanted;
						int i;
						char dc_name[5120];
						int have=0;
						int on_error=0;
						struct stat st;

						sprintf(dc_name,"dctc-%08X-%s",son,hub_address);
						wanted=g_string_new(dctc_dir->str);
						g_string_sprintfa(wanted,"/%s",dc_name);

						for(i=0;i<180;i++)
						{
							if(stat(wanted->str,&st)==0)
							{
								have=1;
								break;
							}

							/* child has exited ? */
							if(waitpid(son,NULL,WNOHANG)==-1)
							{
								on_error=1;
								break;
							}

							usleep( (1000000/180)*20);	/* 20 seconds sliced into 180 parts */
						}

						g_string_free(wanted,TRUE);
						if(have)
						{
							if(no_wait==0)
								connect_to_a_running_dctc(dc_name);
						}
						else
						{
							kill(son,SIGKILL);

							if(!on_error)
								gnome_app_error(GNOME_APP(main_window),_("new DCTC client fails to connect in 20 seconds."));
							else
								gnome_app_error(GNOME_APP(main_window),_("hub has closed its connection."));
						}
					}
					break;
				
	}
	{
		int i;
		/* bug fixed by Glen Koundry: the NULL at the end of the array is no more displayed */
		for(i=0;i<cmd_line->len-1;i++)
		{
			printf("%s\n",(char*)g_ptr_array_index(cmd_line,i));
		}
	}
	g_ptr_array_free(cmd_line,TRUE);
	g_string_chunk_free(sc);
}

char last_known_running_dctc_name[512]={'\0'};

/**************************************/
/* connect to an already running DCTC */
/**************************************/
void connect_to_a_running_dctc(char *dctc_entry)
{
	GString *fpath;
	int a;
	struct sockaddr_un name;
	int nw_dctc_fd;

	if(current_dctc!=NULL)
	{	/* close the connection to the current dctc if exists */
		close_current_dctc();
	}

	reset_interface();

	nw_dctc_fd=socket(AF_UNIX,SOCK_STREAM,0);
	if(nw_dctc_fd==-1)
	{
		perror("connect_to_running_dctc - socket");
		return;
	}
	current_dctc=create_new_dctc_com(nw_dctc_fd);

	strcpy(last_known_running_dctc_name,dctc_entry);

	fpath=g_string_new(dctc_dir->str);
	g_string_sprintfa(fpath,"/%s",dctc_entry);
	name.sun_family=AF_UNIX;
	strcpy(name.sun_path,fpath->str);

	a=connect(current_dctc->dctc_fd,(void *)&name,sizeof(struct sockaddr_un));
	if(a==-1)
	{
		int err=errno;

		GString *emsg;

		emsg=g_string_new(_("Unable to contact this client: "));
		g_string_sprintfa(emsg,_("%s\nThis client is probably already dead."),strerror(err));
		
		gnome_app_error(GNOME_APP(main_window),emsg->str);
		g_string_free(emsg,TRUE);
		close_current_dctc();
		return;
	}
	
	current_dctc->tag_read=gdk_input_add(current_dctc->dctc_fd, GDK_INPUT_READ, process_data_from_dctc, NULL);

	clear_var_array();
	clear_op_array();

	/* get the current hubname, the user list and the transfer list */
	send_data_to_dctc("/HUBNAME\n/VARS\n/ULIST\n/XFER\n/GDLLST\n/UADDRLST\n");

	if(timeout_hdl==0)
		timeout_hdl=gtk_timeout_add(500,update_counter,NULL);

	g_string_sprintfa(fpath,".done");
	load_done_xfer(fpath->str,0);
	g_string_free(fpath,TRUE);
}

/*************************************************************************/
/* send the redirection message to force the client to go to another hub */
/*************************************************************************/
void go_to_another_hub(char *hubname)
{
	gchar *c;

	c=g_strconcat("/GOTO ",hubname,"\n",NULL);
	send_data_to_dctc(c);
	g_free(c);

	/* reset all previous hub parameters */
	reset_interface();
	clear_var_array();		/* this should not be necessary because we remain on the same client */
	clear_op_array();

	/* get the current hubname, the user list and the transfer list */
	send_data_to_dctc("/HUBNAME\n/VARS\n/ULIST\n/XFER\n/SLOWGDLLST\n/UADDRLST\n");

}

