/* Copyright (C) 2004 - 2007  db4objects Inc.  http://www.db4o.com

This file is part of the db4o open source object database.

db4o is free software; you can redistribute it and/or modify it under
the terms of version 2 of the GNU General Public License as published
by the Free Software Foundation and as clarified by db4objects' GPL 
interpretation policy, available at
http://www.db4o.com/about/company/legalpolicies/gplinterpretation/
Alternatively you can write to db4objects, Inc., 1900 S Norfolk Street,
Suite 350, San Mateo, CA 94403, USA.

db4o is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */
using System;
using System.Diagnostics;
using Cecil.FlowAnalysis.CecilUtilities;
using Db4objects.Db4o.Internal.Query;
using Db4objects.Db4o.Nativequery.Expr;
using Db4objects.Db4o.Query;
using Db4objects.Db4o.Tools.NativeQueries;
using Mono.Cecil;
using Mono.Cecil.Cil;
using MethodAttributes=Mono.Cecil.MethodAttributes;

namespace Db4oAdmin
{
	public class PredicateOptimizer : AbstractAssemblyInstrumentation
	{
		int _predicateCount;

		protected override void BeforeAssemblyProcessing()
		{
			_predicateCount = 0;
		}
		
		protected override void  AfterAssemblyProcessing()
		{
			string format = _predicateCount == 1
			                	? "{0} predicate class processed."
			                	: "{0} predicate classes processed.";
			TraceInfo(format, _predicateCount);
		}
		
		protected override void ProcessType(TypeDefinition type)
		{
			if (IsPredicateClass(type))
			{
				InstrumentPredicateClass(type);
			}
		}

		private void InstrumentPredicateClass(TypeDefinition type)
		{
			++_predicateCount;
			
			MethodDefinition match = GetMatchMethod(type);
			IExpression e = GetExpression(match);
			if (null == e) return;

			OptimizePredicate(type, match, e);
		}

		private void OptimizePredicate(TypeDefinition type, MethodDefinition match, IExpression e)
		{
			TraceInfo("Optimizing '{0}' ({1})", type, e);
			
			MethodDefinition optimizeQuery = CreateOptimizeQueryMethod();

			TypeReference extent = match.Parameters[0].ParameterType;
			EmitPrologue(optimizeQuery, extent);

			e.Accept(new SodaEmitterVisitor(_context, optimizeQuery));

			EmitEpilogue(optimizeQuery);

			type.Methods.Add(optimizeQuery);
			type.Interfaces.Add(Import(typeof(IDb4oEnhancedFilter)));

			DumpMethodBody(optimizeQuery);
		}

		private void DumpMethodBody(MethodDefinition m)
		{
			if (_context.TraceSwitch.TraceVerbose)
			{
				TraceVerbose(CecilFormatter.FormatMethodBody(m));
			}
		}

		private IExpression GetExpression(MethodDefinition match)
		{
			try
			{
				return QueryExpressionBuilder.FromMethodDefinition(match);
			}
			catch (Exception x)
			{	
				TraceWarning("WARNING: Predicate '{0}' could not be optimized. {1}", match.DeclaringType, x.Message);
                TraceVerbose("{0}", x);
			}
			return null;
		}

		private static void EmitEpilogue(MethodDefinition method)
		{
			method.Body.CilWorker.Emit(OpCodes.Ret);
		}

		private void EmitPrologue(MethodDefinition method, TypeReference extent)
		{
			CilWorker worker = method.Body.CilWorker;
			
			// query.Constrain(extent);
			worker.Emit(OpCodes.Ldarg_1);
			worker.Emit(OpCodes.Ldtoken, extent);
			worker.Emit(OpCodes.Call, Import(typeof(Type).GetMethod("GetTypeFromHandle")));
			worker.Emit(OpCodes.Callvirt, Import(typeof(IQuery).GetMethod("Constrain")));
			worker.Emit(OpCodes.Pop);
		}

		private MethodDefinition CreateOptimizeQueryMethod()
		{
			// TODO: make sure importing typeof(void) is ok here for the
			// following scenario: CF 1.0 assembly being instrumented by
			// Db4oAdmin running under .net 2.0
			MethodDefinition method = new MethodDefinition("OptimizeQuery",
			                                               MethodAttributes.Virtual|MethodAttributes.Public,
														   Import(typeof(void)));
			method.Parameters.Add(new ParameterDefinition(Import(typeof(IQuery))));
			
			return method;
		}

		private MethodDefinition GetMatchMethod(TypeDefinition type)
		{
			MethodDefinition[] methods = type.Methods.GetMethod("Match");
			Debug.Assert(1 == methods.Length);
			return methods[0];
		}

		private bool IsPredicateClass(TypeReference typeRef)
		{
			TypeDefinition type = typeRef as TypeDefinition;
			if (null == type) return false;
			TypeReference baseType = type.BaseType;
			if (null == baseType) return false;
			if (typeof(Db4objects.Db4o.Query.Predicate).FullName == baseType.FullName) return true;
			return IsPredicateClass(baseType);
		}
	}
}
