/*
 * $Id: nt_applet.c,v 1.35 2000/11/28 22:39:51 syatskevich Exp $
 *
 * GNOME applet for WebDownloader for X.
 *
 * Author: Sergey N. Yatskevich <syatskevich@mail.ru>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <signal.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include "gnome-pixmap-button.h"
#include "nt.h"
#include "nt_applet.h"
#include "nt_error.xpm"

#include "interface.h"

#define GOTO_FRAME(na, frame)                                \
	    gnome_pixmap_button_goto_frame                   \
	          (GNOME_PIXMAP_BUTTON (na->animator), frame)

#define TOTAL_FRAMES(na)                              \
	    gnome_pixmap_button_get_total_frames      \
	          (GNOME_PIXMAP_BUTTON (na->animator))

enum {
	CLICKED,
	URL_DROPPED,
	PROPERTIES,
	LAST_SIGNAL
};

static void nt_applet_init (NTApplet *na);
static void nt_applet_class_init (NTAppletClass *klass);
static void nt_applet_destroy (GtkObject *na);

static void nt_applet_change_pixel_size (AppletWidget *aw, int size);

static void nt_applet_url_dropped (gchar *name, GtkObject *na);
static void nt_applet_drag_data_received (GtkObject        *na,
			                  GdkDragContext   *context,
			                  gint             x,
			                  gint             y,
			                  GtkSelectionData *selection_data,
			                  guint            info,
			                  guint            time,
			                  GtkObject        *animator);

static void nt_applet_properties (AppletWidget *applet, gpointer data);

static gint nt_applet_animator_blink (NTApplet *na);
static gint nt_applet_animator_play  (NTApplet *na);

static GtkObjectClass *parent_class = NULL;
static guint nt_applet_signals[LAST_SIGNAL] = { 0 };

GtkType
nt_applet_get_type (void)
{
	static GtkType nt_applet_type = 0;

	if (!nt_applet_type)
	{
		static const GtkTypeInfo nt_applet_info = {
			"NTApplet",
			sizeof (NTApplet),
			sizeof (NTAppletClass),
			(GtkClassInitFunc) nt_applet_class_init,
			(GtkObjectInitFunc) nt_applet_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL,
			(GtkClassInitFunc) NULL
		};

		nt_applet_type = gtk_type_unique (applet_widget_get_type (), &nt_applet_info);
	}

	return nt_applet_type;
}

static void
nt_applet_init (NTApplet *na)
{
	na->animator = NULL;

	na->animation_fname = NULL;
	na->tile_up_fname   = NULL;
	na->tile_down_fname = NULL;

	na->direction    = NTA_DIR_PINGPONG;
	na->mode         = NTA_MODE_BLINK;
	na->download_cmd = PACKET_ADD_OPEN;

	na->timer_id    = FALSE;
	na->cur_frame   = 1;
	na->next_frame  = 1;
	na->blink_state = FALSE;

	na->buffer[0]  = '\0';
	na->prev_state = NTA_STATE_UNDEFINED;
	na->prev_speed = 0;
}

static void
nt_applet_class_init (NTAppletClass *klass)
{
	parent_class = GTK_OBJECT_CLASS (gtk_type_class (applet_widget_get_type ()));

	nt_applet_signals[CLICKED] =
		gtk_signal_new ("clicked",
    			0, GTK_TYPE_NT_APPLET,
			0, gtk_marshal_NONE__NONE,
			GTK_TYPE_NONE, 0);

	nt_applet_signals[URL_DROPPED] =
		gtk_signal_new ("url_dropped",
    			0, GTK_TYPE_NT_APPLET,
			0, gtk_marshal_NONE__POINTER_INT,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_POINTER,
			GTK_TYPE_INT);

	nt_applet_signals[PROPERTIES] =
		gtk_signal_new ("properties",
    			0, GTK_TYPE_NT_APPLET,
			0, gtk_marshal_NONE__NONE,
			GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (GTK_OBJECT_CLASS (klass),
	                              nt_applet_signals,
				      LAST_SIGNAL);

	GTK_OBJECT_CLASS (klass)->destroy = nt_applet_destroy;
	APPLET_WIDGET_CLASS (klass)->change_pixel_size = nt_applet_change_pixel_size;
}

static void
nt_applet_properties (AppletWidget *na, gpointer data)
{
	gtk_signal_emit (GTK_OBJECT (na), nt_applet_signals[PROPERTIES]);
}

static void
nt_applet_press (GtkObject *na, GtkObject *gpb)
{
	gtk_signal_emit (na, nt_applet_signals[CLICKED]);
}

static void
nt_applet_url_dropped (gchar *name, GtkObject *na)
{
	g_return_if_fail (name != NULL);

	gtk_signal_emit (na, nt_applet_signals[URL_DROPPED], name, NT_APPLET (na)->download_cmd);
}

enum {
  TARGET_URL_LIST
};

static void
nt_applet_drag_data_received (GtkObject        *na,
			      GdkDragContext   *context,
			      gint             x,
			      gint             y,
			      GtkSelectionData *selection_data,
			      guint            info,
			      guint            time,
			      GtkObject        *animator)
{
	GList *names;

	g_return_if_fail (selection_data != NULL);

	switch (info)
	{
	  case TARGET_URL_LIST:
		  names = gnome_uri_list_extract_uris ((char *)selection_data->data);
		  g_list_foreach (names, (GFunc)nt_applet_url_dropped, na);
		  gnome_uri_list_free_strings (names);
		  break;
	}
}

static void
nt_applet_about (AppletWidget *app, gpointer data)
{
	static GtkWidget *about_dlg = NULL;

	if(about_dlg)
		gdk_window_raise (about_dlg->window);
	else
	{
		about_dlg = create_nt_applet_about ();

		gtk_signal_connect (GTK_OBJECT (about_dlg), "destroy",
                                    GTK_SIGNAL_FUNC (gtk_widget_destroyed),
                                    &about_dlg);

		gtk_widget_show (about_dlg);
	}
}

void
nt_applet_construct (NTApplet *na, const gchar *goad_id)
{
	static GtkTargetEntry drop_types [] = { 
		{ "x-url/http",     0, TARGET_URL_LIST },
		{ "x-url/ftp",      0, TARGET_URL_LIST },
		{ "_NETSCAPE_URL",  0, TARGET_URL_LIST }
	};
	static gint n_drop_types = sizeof (drop_types) / sizeof(drop_types[0]);

	int size;

	applet_widget_construct (APPLET_WIDGET (na), goad_id);
	size = applet_widget_get_panel_pixel_size (APPLET_WIDGET (na));
	gtk_widget_set_usize (GTK_WIDGET (na), size, size);

	na->animator = gnome_pixmap_button_new_with_size (size, size);

	applet_widget_register_stock_callback (APPLET_WIDGET (na),
					       "properties",
					       GNOME_STOCK_MENU_PROP,
					       _("Properties..."),
					       nt_applet_properties,
					       NULL);
	applet_widget_register_stock_callback (APPLET_WIDGET (na),
					       "about",
					       GNOME_STOCK_MENU_ABOUT,
					       _("About..."),
					       nt_applet_about,
					       NULL);

	gtk_signal_connect_object (GTK_OBJECT (na->animator), "clicked",
                                   GTK_SIGNAL_FUNC (nt_applet_press),
			           GTK_OBJECT (na));
	gtk_drag_dest_set  (GTK_WIDGET (na->animator),
			    GTK_DEST_DEFAULT_MOTION |
			    GTK_DEST_DEFAULT_HIGHLIGHT |
			    GTK_DEST_DEFAULT_DROP,
			    drop_types, n_drop_types,
			    GDK_ACTION_COPY);
	gtk_signal_connect_object (GTK_OBJECT (na->animator), "drag_data_received",
			           GTK_SIGNAL_FUNC (nt_applet_drag_data_received),
			           GTK_OBJECT (na));

	gtk_signal_connect_object (GTK_OBJECT (na), "back_change",
				   GTK_SIGNAL_FUNC (gnome_pixmap_button_repaint),
				   GTK_OBJECT (na->animator));

	gtk_widget_show (na->animator);
	applet_widget_add (APPLET_WIDGET (na), na->animator);
}

static void
nt_applet_destroy (GtkObject *na)
{
	g_free (NT_APPLET (na)->animation_fname);
	g_free (NT_APPLET (na)->tile_up_fname);
	g_free (NT_APPLET (na)->tile_down_fname);

	parent_class->destroy (na);
}

static gchar *
get_full_pixmap_file (const gchar *fname)
{
	if (fname && (fname[0] != '\0'))
	{
		if (fname[0] != '/')
			return gnome_pixmap_file (fname);
		else
		{
			if (g_file_exists (fname))
				return g_strdup (fname);
		}
	}

	return NULL;
}

void
nt_applet_config_reload (NTApplet *na, NTProps *ntp)
{
	gchar             *animation_fname  = NULL;

	gboolean          tiles_reload     = FALSE;
	gchar             *tile_up_fname   = NULL;
	gchar             *tile_down_fname = NULL;

	g_return_if_fail (na != NULL);
	g_return_if_fail (IS_NT_APPLET (na));
	g_return_if_fail (ntp != NULL);
	g_return_if_fail (IS_NT_PROPS (ntp));
	g_return_if_fail (na->animator != NULL);

	animation_fname = get_full_pixmap_file (ntp->nta_animation_fname);
	if (!animation_fname || !na->animation_fname ||
	    strcmp (animation_fname, na->animation_fname))
	{
		g_free (na->animation_fname);
		na->animation_fname = (animation_fname) ? g_strdup (animation_fname) : NULL;

		if (na->animation_fname)
			gnome_pixmap_button_animation_load_from_file
				(GNOME_PIXMAP_BUTTON (na->animator), na->animation_fname);
		else
			gnome_pixmap_button_animation_load_from_xpm_d
				(GNOME_PIXMAP_BUTTON (na->animator), (const gchar **)nt_error_xpm);

		nt_applet_state_set (na, NTA_STATE_UNDEFINED, 0);
	}
	g_free (animation_fname);

	tile_up_fname = get_full_pixmap_file (ntp->nta_tile_up_fname);
	if (!tile_up_fname || !na->tile_up_fname ||
	    strcmp (tile_up_fname, na->tile_up_fname))
	{
		g_free (na->tile_up_fname);
		na->tile_up_fname = (tile_up_fname) ? g_strdup (tile_up_fname) : NULL;

		tiles_reload = TRUE;
	}
	g_free (tile_up_fname);

	tile_down_fname = get_full_pixmap_file (ntp->nta_tile_down_fname);
	if (!tile_down_fname || !na->tile_down_fname ||
	    strcmp (tile_down_fname, na->tile_down_fname))
	{
		g_free (na->tile_down_fname);
		na->tile_down_fname = (tile_down_fname) ? g_strdup (tile_down_fname) : NULL;

		tiles_reload = TRUE;
	}
	g_free (tile_down_fname);

	if (tiles_reload)
		gnome_pixmap_button_tiles_load_from_file
			(GNOME_PIXMAP_BUTTON (na->animator),
			 na->tile_up_fname,
			 na->tile_down_fname);

	if (ntp->nta_tiles_enabled != GNOME_PIXMAP_BUTTON (na->animator)->tiles_enabled)
	{
		GNOME_PIXMAP_BUTTON (na->animator)->tiles_enabled = ntp->nta_tiles_enabled;
		gnome_pixmap_button_repaint (GNOME_PIXMAP_BUTTON (na->animator));
	}

	GNOME_PIXMAP_BUTTON (na->animator)->depth = ntp->nta_depth;

	GNOME_PIXMAP_BUTTON (na->animator)->prelight_enabled = ntp->nta_prelight_button_on_mouseover;

	if (na->direction != ntp->nta_direction)
	{
		na->direction = ntp->nta_direction;

		if (na->direction == NTA_DIR_BACKWARD)
			na->next_frame = -1;
		else if (na->direction == NTA_DIR_FORWARD)
			na->next_frame = 1;
	}

	if (na->mode != ntp->nta_mode)
	{
		na->mode = ntp->nta_mode;
		nt_applet_state_set (na, NTA_STATE_UNDEFINED, 0);
	}

	na->download_cmd = ntp->nta_download_cmd;
}

GtkWidget *
nt_applet_new (const gchar *goad_id)
{
	GtkWidget *na = gtk_type_new (GTK_TYPE_NT_APPLET);

	nt_applet_construct (NT_APPLET (na), goad_id);

	return na;
}

void
nt_applet_tooltip_set (NTApplet *na, gchar *tip)
{
	g_return_if_fail (na != NULL);
	g_return_if_fail (IS_NT_APPLET (na));
	g_return_if_fail (tip != NULL);

	if (!g_str_equal (na->buffer, tip))
	{
		strncpy (na->buffer, tip, PATH_MAX - 1);
		applet_widget_set_tooltip (APPLET_WIDGET (na), tip);
	}
}

static gint
nt_applet_animator_blink (NTApplet *na)
{
	GDK_THREADS_ENTER ();

	if (na->blink_state)
	{
		GOTO_FRAME (na, na->cur_frame);
		na->blink_state = FALSE;
	}
	else
	{
		GOTO_FRAME (na, 0);
		na->blink_state = TRUE;
	}

	GDK_THREADS_LEAVE ();

	return TRUE;
}

static gint
nt_applet_animator_play (NTApplet *na)
{
	GDK_THREADS_ENTER ();

	na->cur_frame += na->next_frame;

	if ((na->direction == NTA_DIR_FORWARD) &&
	    (na->cur_frame >= TOTAL_FRAMES (na)))
		na->cur_frame = 3;

	if ((na->direction == NTA_DIR_BACKWARD) && (na->cur_frame < 3))
		na->cur_frame = TOTAL_FRAMES (na) - 1;

	if (na->direction == NTA_DIR_PINGPONG)
	{
		if (na->cur_frame >= (TOTAL_FRAMES (na) - 1))
		{
			na->cur_frame = TOTAL_FRAMES (na) - 1;
			na->next_frame = -1;
		}

		if (na->cur_frame <= 3)
		{
			na->cur_frame = 3;
			na->next_frame = 1;
		}
	}

	GOTO_FRAME (na, na->cur_frame);

	GDK_THREADS_LEAVE ();

	return TRUE;
}

void
nt_applet_state_set (NTApplet *na, NTAppletState state, gint speed)
{
	g_return_if_fail (na != NULL);
	g_return_if_fail (IS_NT_APPLET (na));

	if (TOTAL_FRAMES (na) < 4)
	{
		GOTO_FRAME (na, 0);
		return;
	}

	if (speed <= 0)
		speed = 0;
	else if (speed <= 500)
		speed = 500;
	else if (speed <= 1000)
		speed = 400;
	else if (speed <= 2000)
		speed = 300;
	else if (speed <= 4000)
		speed = 200;
	else
		speed = 150;

	if ((na->prev_state == state) &&
	    ((state == NTA_STATE_NOT_RUN)   ||
	     (state == NTA_STATE_NO_FILES)  ||
	     (state == NTA_STATE_ERROR)     ||
	     (state == NTA_STATE_UNDEFINED) ||
	     ((state == NTA_STATE_LOAD) && (na->prev_speed == speed))))
		return;

	if (na->timer_id)
	{
    		gtk_timeout_remove (na->timer_id);
		na->timer_id = FALSE;
	}

	if ((state == NTA_STATE_NOT_RUN) || (state == NTA_STATE_UNDEFINED))
		na->cur_frame = 1;
	else if (state == NTA_STATE_ERROR)
		na->cur_frame = 2;
	else if ((na->prev_state != NTA_STATE_LOAD) ||
		 (state == NTA_STATE_NO_FILES))
		na->cur_frame = 3;

	if (state == NTA_STATE_LOAD)
	{
		if (speed)
			na->timer_id = gtk_timeout_add (speed,
					(GtkFunction)nt_applet_animator_play,
					na);
		else
		{
			switch (na->mode)
			{
			    case NTA_MODE_CONTINUE:
				na->timer_id = gtk_timeout_add (500,
					(GtkFunction)nt_applet_animator_play,
					na);
				break;

			    case NTA_MODE_SWITCH:
				na->cur_frame = 0;
				break;

			    case NTA_MODE_BLINK:
				na->blink_state = FALSE;
				na->timer_id = gtk_timeout_add (900,
					(GtkFunction)nt_applet_animator_blink,
					na);
				break;

			    default:
				break;
			}
		}
	}

	GOTO_FRAME (na, na->cur_frame);

	na->prev_state = state;
	na->prev_speed = speed;
}

static void
nt_applet_change_pixel_size (AppletWidget *aw, int size)
{
	NTApplet *na;

	g_return_if_fail (size > 0);

	na = NT_APPLET (aw);

	gnome_pixmap_button_set_size (GNOME_PIXMAP_BUTTON (na->animator),
				      size, size);
	gtk_widget_set_usize (GTK_WIDGET (na), size, size);

	gnome_pixmap_button_animation_load_from_file
		(GNOME_PIXMAP_BUTTON (na->animator),
		 na->animation_fname);

	gnome_pixmap_button_tiles_load_from_file
		(GNOME_PIXMAP_BUTTON (na->animator),
		 na->tile_up_fname,
		 na->tile_down_fname);

	nt_applet_state_set (na, NTA_STATE_UNDEFINED, 0);
}
