/*
 * $Id: nt.c,v 1.18 2001/01/19 22:34:43 syatskevich Exp $
 *
 * GNOME applet for WebDownloader for X.
 *
 * Author: Sergey N. Yatskevich <syatskevich@mail.ru>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <signal.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>

#include "nt_applet.h"
#include "nt.h"

enum {
	TIP_CHANGED,
	STATE_CHANGED,
	LAST_SIGNAL
};

static void nt_init (NT *nt);
static void nt_class_init (NTClass *klass);
static void nt_destroy (GtkObject *nt);

static gboolean nt_run (NT *nt);
static gint     nt_connect_to_downloader (void);
static gint     nt_send_command (NT *nt, gint cmd, gchar *data, gint len);
static gint     nt_status_update (gpointer _nt);
static gint     nt_update_download (gpointer _nt);

static GtkObjectClass *parent_class = NULL;
static guint nt_signals[LAST_SIGNAL] = { 0 };

GtkType
nt_get_type (void)
{
	static GtkType nt_type = 0;

	if (!nt_type)
	{
		static const GtkTypeInfo nt_info = {
			"NT",
			sizeof (NT),
			sizeof (NTClass),
			(GtkClassInitFunc) nt_class_init,
			(GtkObjectInitFunc) nt_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL,
			(GtkClassInitFunc) NULL
		};

		nt_type = gtk_type_unique (GTK_TYPE_OBJECT, &nt_info);
	}

	return nt_type;
}

static void
nt_init (NT *nt)
{
	nt->update_download_id = FALSE;
	nt->download_list      = NULL;

	g_snprintf (nt->lock_file, PATH_MAX - 1, "%s/downloader_for_x_lock_%s", g_get_tmp_dir (), g_get_user_name ());

	nt->update_id = gtk_timeout_add (1000, nt_status_update, nt);
}

static void
nt_class_init (NTClass *klass)
{
	parent_class = GTK_OBJECT_CLASS (gtk_type_class (GTK_TYPE_OBJECT));

	nt_signals[TIP_CHANGED] =
		gtk_signal_new ("tip_changed",
    			0, NT_TYPE,
			0, gtk_marshal_NONE__POINTER,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_POINTER);

	nt_signals[STATE_CHANGED] =
		gtk_signal_new ("state_changed",
    			0, NT_TYPE,
			0, gtk_marshal_NONE__INT_INT,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_INT,
			GTK_TYPE_INT);

	gtk_object_class_add_signals (GTK_OBJECT_CLASS (klass),
	                              nt_signals, LAST_SIGNAL);

	GTK_OBJECT_CLASS (klass)->destroy = nt_destroy;
}

void
nt_destroy (GtkObject *_nt)
{
	NT *nt = NT_ (_nt);

	gtk_timeout_remove(nt->update_id);
	nt->update_id = FALSE;

	if (nt->update_download_id)
	{
		gtk_timeout_remove(nt->update_download_id);
		nt->update_download_id = FALSE;
	}

	if (nt->download_list)
	{
		g_list_foreach (nt->download_list, (GFunc)g_free, NULL);
		g_list_free (nt->download_list);
	}

	parent_class->destroy (_nt);
}

#define nt_is_running(nt) g_file_exists ((nt)->lock_file)

static gboolean
nt_run (NT *nt)
{
	gchar cmd[] = "nt &";

	if(system (cmd) != 0)
	{
		gtk_signal_emit (GTK_OBJECT (nt),
				 nt_signals[TIP_CHANGED],
				 _("Can't start Downloader for X"));
		return FALSE;
	}

	return TRUE;
}

static gint
nt_connect_to_downloader (void)
{
	gint SOCKET_ID = -1;

	SOCKET_ID = socket (AF_UNIX, SOCK_STREAM, 0);

	if (SOCKET_ID > 0)
	{
		uid_t stored_uid, euid;
		struct sockaddr_un saddr;

		saddr.sun_family = AF_UNIX;
		stored_uid = getuid ();
		euid = geteuid ();
		setuid (euid);
		g_snprintf (saddr.sun_path, 107,
		            "%s/downloader_for_x_sock_%s",
			    g_get_tmp_dir (),
			    g_get_user_name ());
		setreuid (stored_uid, euid);

		if (connect (SOCKET_ID,
		             (struct sockaddr *) &saddr,
			     sizeof (saddr)) < 0)
		{
			close (SOCKET_ID);
			SOCKET_ID = -1;
		}
	}
	return SOCKET_ID;
}

struct NTPacket {
	gint type;
	gint len;
};

enum {
	ERROR_RUN_DOWNLOADER              = -1,
	ERROR_CONNECT_TO_DOWNLOADER       = -2,
	ERROR_COMMUNICATE_WITH_DOWNLOADER = -3
};

static gint
nt_send_command (NT *nt, gint cmd, gchar *data, gint len)
{
	gint SOCKET_ID = -1;
	int  result    =  0;
	struct NTPacket packet;

	if (!nt_is_running (nt) && !nt_run (nt))
		return ERROR_RUN_DOWNLOADER;

	packet.type = cmd;
	packet.len  = data ? len : 0;

	SOCKET_ID = nt_connect_to_downloader ();

	if (SOCKET_ID < 0)
		return ERROR_CONNECT_TO_DOWNLOADER;

	/* send command */
	write (SOCKET_ID, &packet, sizeof (packet));
	if (packet.len)
		write (SOCKET_ID, data, packet.len);

	/* recieve answer */
	if ((read (SOCKET_ID, &packet, sizeof(packet)) <= 0))
	{
		g_warning ("Error read answer from Downloader");

		close (SOCKET_ID);
		return ERROR_COMMUNICATE_WITH_DOWNLOADER;
	}

	if ((packet.type != PACKET_ACK))
	{
		g_warning ("Downloader can't execute command");

		close (SOCKET_ID);
		return ERROR_COMMUNICATE_WITH_DOWNLOADER;
	}

	/* read data in answer */
	if ((read (SOCKET_ID, &result, sizeof (int)) <= 0))
	{
		g_warning ("Error read data from Downloader");

		close (SOCKET_ID);
		return ERROR_COMMUNICATE_WITH_DOWNLOADER;
	}

	close (SOCKET_ID);
	return (gint)result;
}

static gint
nt_status_update (gpointer _nt)
{
	NT *nt = NT_ (_nt);
	gchar buffer[PATH_MAX];

	GDK_THREADS_ENTER ();

	if (nt_is_running (nt))
	{
		gint run_downloads      = 0;
		gint left_downloads     = 0;
		gint error_downloads    = 0;
		gint pause_downloads    = 0;
		gint complete_downloads = 0;

		run_downloads = nt_send_command (nt, PACKET_ASK_RUN, NULL, 0);

		left_downloads = nt_send_command (nt, PACKET_ASK_FULLAMOUNT, NULL, 0);

		if ((run_downloads < 0) || (left_downloads < 0))
		{
			GDK_THREADS_LEAVE ();
			return TRUE;
		}

		error_downloads = nt_send_command (nt, PACKET_ASK_STOP, NULL, 0);
		if (error_downloads > 0)
			left_downloads -= error_downloads;

		pause_downloads = nt_send_command (nt, PACKET_ASK_PAUSE, NULL, 0);
		if (pause_downloads > 0)
			left_downloads -= pause_downloads;

		complete_downloads = nt_send_command (nt, PACKET_ASK_COMPLETE, NULL, 0);
		if (complete_downloads > 0)
			left_downloads -= complete_downloads;

		if (left_downloads > 0)
		{
		    gint speed = nt_send_command (nt, PACKET_ASK_SPEED, NULL, 0);

		    gtk_signal_emit (GTK_OBJECT (nt), nt_signals[STATE_CHANGED], NTA_STATE_LOAD, speed);

		    g_snprintf(buffer, PATH_MAX - 1, _("Downloading %d/%d file(s)"), run_downloads, left_downloads);
		}
		else
		{
		    if (error_downloads > 0)
		    {
			    gtk_signal_emit (GTK_OBJECT (nt), nt_signals[STATE_CHANGED], NTA_STATE_ERROR, 0);

			    g_snprintf(buffer, PATH_MAX - 1, _("There are %d error(s)"), error_downloads);
		    }
		    else
		    {
			    gtk_signal_emit (GTK_OBJECT (nt), nt_signals[STATE_CHANGED], NTA_STATE_NO_FILES, 0);

			    g_snprintf(buffer, PATH_MAX - 1, _("No files for download"));
		    }
		}

		gtk_signal_emit (GTK_OBJECT (nt), nt_signals[TIP_CHANGED], buffer);
	}
	else
	{
		gtk_signal_emit (GTK_OBJECT (nt), nt_signals[TIP_CHANGED], _("Downloader for X is not running"));
		gtk_signal_emit (GTK_OBJECT (nt), nt_signals[STATE_CHANGED], NTA_STATE_NOT_RUN, 0);
	}

	GDK_THREADS_LEAVE ();
	return TRUE;
}

static gint
nt_update_download (gpointer _nt)
{
	NT *nt = NT_ (_nt);
	gchar *file_name = NULL;

	GDK_THREADS_ENTER ();

	while (nt->download_list)
	{
		file_name = (nt->download_list)->data;

		if (nt_send_command (nt, file_name[0], (file_name + 1), strlen (file_name + 1)) < 0)
			break;

		nt->download_list = g_list_remove (nt->download_list, file_name);
		g_free (file_name);
	}

	if (nt->download_list)
	{
		GDK_THREADS_LEAVE ();
		return TRUE;
	}
	else
	{
		nt_status_update (nt);
		nt->update_download_id = FALSE;

		GDK_THREADS_LEAVE ();
		return FALSE;
	}
}

void
nt_download_uri (NT *nt, gchar *file_name, NT_CMDS cmd)
{
	gint answer = 0;

	g_return_if_fail (nt != NULL);
	g_return_if_fail (IS_NT (nt));
	g_return_if_fail (cmd != PACKET_NOP);

	if ((file_name == NULL) || (strlen (file_name) == 0))
		return;

	answer = nt_send_command (nt, cmd, file_name, strlen (file_name));

	if ((answer < 0) && (answer != ERROR_RUN_DOWNLOADER))
	{
		gchar cmd_s[2]; cmd_s[0] = cmd; cmd_s[1] = '\0';

		nt->download_list = g_list_append (nt->download_list, g_strconcat (cmd_s, file_name, NULL));
		if (nt->update_download_id == FALSE)
			nt->update_download_id = gtk_timeout_add(2000, nt_update_download, nt);
	}
	else
		nt_status_update (nt);
}

void
nt_open_window (NT *nt)
{
	g_return_if_fail (nt != NULL);
	g_return_if_fail (IS_NT (nt));

	if (!nt_is_running (nt))
		nt_run (nt);
	else
		nt_send_command (nt, PACKET_POPUP, NULL, 0);
}

GtkObject *
nt_new (void)
{
	return gtk_type_new (NT_TYPE);
}
