/*
 * $Id: gnome-pixmap-button.c,v 1.11 2000/11/28 22:39:51 syatskevich Exp $
 *
 * GNOME applet for WebDownloader for X.
 *
 * Author: Sergey N. Yatskevich <syatskevich@mail.ru>
 */

#include <config.h>

#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <libgnome/gnome-defs.h>
#include <gtk/gtk.h>
#include "gnome-pixmap-button.h"

enum {
	PRESSED,
	RELEASED,
	CLICKED,
	ENTER,
	LEAVE,
	LAST_SIGNAL
};

static void do_colorshift (GdkPixbuf *dest, GdkPixbuf *src, int shift);

static void gnome_pixmap_button_class_init (GnomePixmapButtonClass *klass);
static void gnome_pixmap_button_init       (GnomePixmapButton      *gpb);
static void gnome_pixmap_button_destroy    (GtkObject              *object);

static void gnome_pixmap_button_realize        (GtkWidget *widget);
static void gnome_pixmap_button_size_allocate  (GtkWidget *widget, GtkAllocation    *allocation);
static void gnome_pixmap_button_draw           (GtkWidget *widget, GdkRectangle     *area);
static gint gnome_pixmap_button_expose         (GtkWidget *widget, GdkEventExpose   *event);
static gint gnome_pixmap_button_button_press   (GtkWidget *widget, GdkEventButton   *event);
static gint gnome_pixmap_button_button_release (GtkWidget *widget, GdkEventButton   *event);
static gint gnome_pixmap_button_enter_notify   (GtkWidget *widget, GdkEventCrossing *event);
static gint gnome_pixmap_button_leave_notify   (GtkWidget *widget, GdkEventCrossing *event);

static void real_button_pressed  (GnomePixmapButton *button);
static void real_button_released (GnomePixmapButton *button);
static void real_button_enter    (GnomePixmapButton *button);
static void real_button_leave    (GnomePixmapButton *button);

static void free_cache     (GnomePixmapButton *gpb);
static void new_cache      (GnomePixmapButton *gpb);
static void free_button    (GnomePixmapButton *gpb);
static void free_animation (GnomePixmapButton *gpb);

static GtkWidgetClass *parent_class;
static guint button_signals[LAST_SIGNAL] = { 0 };

guint
gnome_pixmap_button_get_type (void)
{
	static guint pixmap_button_type = 0;

	if (!pixmap_button_type)
	{
		static const GtkTypeInfo pixmap_button_info = {
			"GnomePixmapButton",
			sizeof (GnomePixmapButton),
			sizeof (GnomePixmapButtonClass),
			(GtkClassInitFunc) gnome_pixmap_button_class_init,
			(GtkObjectInitFunc) gnome_pixmap_button_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL,
			(GtkClassInitFunc) NULL
		};

		pixmap_button_type = gtk_type_unique (GTK_TYPE_WIDGET, &pixmap_button_info);
	}

	return pixmap_button_type;
}

static void
gnome_pixmap_button_init (GnomePixmapButton *gpb)
{
	GTK_WIDGET_SET_FLAGS (gpb, GTK_CAN_FOCUS | GTK_RECEIVES_DEFAULT);
	GTK_WIDGET_UNSET_FLAGS (gpb, GTK_NO_WINDOW);

	gpb->in_button   = FALSE;
	gpb->button_down = FALSE;

	gpb->depth            = -2;
	gpb->tiles_enabled    = TRUE;
	gpb->prelight_enabled = TRUE;

	gpb->repaint = TRUE;
	gpb->pixmap  = NULL;
	gpb->bitmap  = NULL;
	gpb->gc      = NULL;

	gpb->im_up     = NULL;
	gpb->im_up_p   = NULL;
	gpb->im_down   = NULL;
	gpb->im_down_p = NULL;

	gpb->num_frames           = 0;
	gpb->current_frame_number = 0;
	gpb->im_animation         = NULL;
	gpb->im_animation_p       = NULL;
}

/* make compiler happy */
typedef void (*StyleSetFunc) (GtkWidget *, GtkStyle *);
#define STYLE_SET_FUNC(f) ((StyleSetFunc) (f))

static void
gnome_pixmap_button_class_init (GnomePixmapButtonClass *klass)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (klass);
	widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = gtk_type_class (GTK_TYPE_WIDGET);

	button_signals[PRESSED] =
		gtk_signal_new ("pressed",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnomePixmapButtonClass, pressed),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);
	button_signals[RELEASED] =
		gtk_signal_new ("released",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnomePixmapButtonClass, released),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);
	button_signals[CLICKED] =
		gtk_signal_new ("clicked",
                    GTK_RUN_FIRST | GTK_RUN_ACTION,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnomePixmapButtonClass, clicked),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);
	button_signals[ENTER] =
		gtk_signal_new ("enter",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnomePixmapButtonClass, enter),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);
	button_signals[LEAVE] =
		gtk_signal_new ("leave",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnomePixmapButtonClass, leave),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, button_signals, LAST_SIGNAL);

	widget_class->activate_signal      = button_signals[CLICKED];

	widget_class->realize              = gnome_pixmap_button_realize;
	widget_class->draw                 = gnome_pixmap_button_draw;
	widget_class->size_allocate        = gnome_pixmap_button_size_allocate;
	widget_class->expose_event         = gnome_pixmap_button_expose;
	widget_class->button_press_event   = gnome_pixmap_button_button_press;
	widget_class->button_release_event = gnome_pixmap_button_button_release;
	widget_class->enter_notify_event   = gnome_pixmap_button_enter_notify;
	widget_class->leave_notify_event   = gnome_pixmap_button_leave_notify;
	widget_class->style_set            = STYLE_SET_FUNC (gnome_pixmap_button_repaint);
	object_class->destroy              = gnome_pixmap_button_destroy;

	klass->pressed  = real_button_pressed;
	klass->released = real_button_released;
	klass->clicked  = NULL;
	klass->enter    = real_button_enter;
	klass->leave    = real_button_leave;
}

static void
gnome_pixmap_button_destroy (GtkObject *object)
{
	GnomePixmapButton *gpb;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (object));

	gpb = GNOME_PIXMAP_BUTTON (object);

	free_cache     (gpb);
	free_button    (gpb);
	free_animation (gpb);

	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
gnome_pixmap_button_realize (GtkWidget *widget)
{
	GdkWindowAttr attributes;
	gint attributes_mask;
	GnomePixmapButton *gpb;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (widget));

	gpb = GNOME_PIXMAP_BUTTON (widget);
	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.x           = widget->allocation.x;
	attributes.y           = widget->allocation.y;
	attributes.width       = widget->allocation.width;
	attributes.height      = widget->allocation.height;
	attributes.wclass      = GDK_INPUT_OUTPUT;
	attributes.visual      = gtk_widget_get_visual (widget);
	attributes.colormap    = gtk_widget_get_colormap (widget);
	attributes.event_mask  = gtk_widget_get_events (widget);
	attributes.event_mask |= (GDK_EXPOSURE_MASK |
				  GDK_BUTTON_PRESS_MASK |
				  GDK_BUTTON_RELEASE_MASK |
				  GDK_ENTER_NOTIFY_MASK |
				  GDK_LEAVE_NOTIFY_MASK);
	attributes_mask        = (GDK_WA_X |
				  GDK_WA_Y |
				  GDK_WA_VISUAL |
				  GDK_WA_COLORMAP);

	widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
	gdk_window_set_user_data (widget->window, gpb);

	widget->style = gtk_style_attach (widget->style, widget->window);
	gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);

	new_cache (gpb);
}

static void
gnome_pixmap_button_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	GnomePixmapButton *button;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (widget));
	g_return_if_fail (allocation != NULL);

	button = GNOME_PIXMAP_BUTTON (widget);

	widget->allocation = *allocation;

	if (GTK_WIDGET_REALIZED (widget))
		gdk_window_move_resize (widget->window,
			    widget->allocation.x,
			    widget->allocation.y,
			    widget->allocation.width,
			    widget->allocation.height);
}

static void
paint (GnomePixmapButton *gpb, GdkRectangle *area)
{
	GtkWidget *widget = GTK_WIDGET(gpb);

	guint x, y, width, height;

	g_return_if_fail (gpb != NULL);
	g_return_if_fail (gpb->pixmap != NULL);
	g_return_if_fail (gpb->bitmap != NULL);
	g_return_if_fail (gpb->gc != NULL);

	if (gpb->repaint)
	{
		gpb->repaint = FALSE;

		if (widget->style->bg_pixmap[0])
			gdk_draw_pixmap (gpb->pixmap, gpb->gc, widget->style->bg_pixmap[0],
		            		 0, 0, 0, 0, gpb->width, gpb->height);
		else
		{
			gdk_gc_set_foreground (gpb->gc, &(widget->style->bg[0]));
			gdk_draw_rectangle (gpb->pixmap, gpb->gc,
					    TRUE, 0, 0, gpb->width, gpb->height);
		}

		if (gpb->tiles_enabled)
		{
			if (GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE)
			{
				if (gpb->im_down)
				{
					gdk_pixbuf_render_threshold_alpha (gpb->im_down, gpb->bitmap,
									   0, 0, 0, 0, gpb->width, gpb->height,
									   1);

					gdk_gc_set_clip_mask (gpb->gc, gpb->bitmap);

					if (gpb->im_down_p && gpb->prelight_enabled)
						gdk_pixbuf_render_to_drawable (gpb->im_down_p, gpb->pixmap, gpb->gc,
									       0, 0, 0, 0, gpb->width, gpb->height,
									       GDK_RGB_DITHER_NORMAL, 0, 0);
					else
						gdk_pixbuf_render_to_drawable (gpb->im_down, gpb->pixmap, gpb->gc,
									       0, 0, 0, 0, gpb->width, gpb->height,
									       GDK_RGB_DITHER_NORMAL, 0, 0);
				}
			}
			else
			{
				if (gpb->im_up)
				{
					gdk_pixbuf_render_threshold_alpha (gpb->im_up, gpb->bitmap,
									   0, 0, 0, 0, gpb->width, gpb->height,
									   1);

					gdk_gc_set_clip_mask (gpb->gc, gpb->bitmap);

					if (gpb->im_up_p && gpb->prelight_enabled && (GTK_WIDGET_STATE (widget) == GTK_STATE_PRELIGHT))
						gdk_pixbuf_render_to_drawable (gpb->im_up_p, gpb->pixmap, gpb->gc,
									       0, 0, 0, 0, gpb->width, gpb->height,
									       GDK_RGB_DITHER_NORMAL, 0, 0);
					else
						gdk_pixbuf_render_to_drawable (gpb->im_up, gpb->pixmap, gpb->gc,
									       0, 0, 0, 0, gpb->width, gpb->height,
									       GDK_RGB_DITHER_NORMAL, 0, 0);
				}
			}
		}

		if (gpb->im_animation)
		{
			gint dd   = (GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE) ? gpb->depth : 0;
			gint xsrc = gpb->current_frame_number * gpb->width;

			gdk_pixbuf_render_threshold_alpha (gpb->im_animation, gpb->bitmap,
							   xsrc, 0, dd, dd, gpb->width, gpb->height,
							   1);

			gdk_gc_set_clip_mask (gpb->gc, gpb->bitmap);

			if (gpb->im_animation_p && gpb->prelight_enabled &&
			    ((GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE) ||
			     (GTK_WIDGET_STATE (widget) == GTK_STATE_PRELIGHT)))
				gdk_pixbuf_render_to_drawable (gpb->im_animation_p, gpb->pixmap, gpb->gc,
							       xsrc, 0, dd, dd, gpb->width, gpb->height,
							       GDK_RGB_DITHER_NORMAL, 0, 0);
			else
				gdk_pixbuf_render_to_drawable (gpb->im_animation, gpb->pixmap, gpb->gc,
							       xsrc, 0, dd, dd, gpb->width, gpb->height,
							       GDK_RGB_DITHER_NORMAL, 0, 0);
		}

		gdk_gc_set_clip_mask (gpb->gc, NULL);
	}

	if (area)
	{
		x      = area->x;
		y      = area->y;
		width  = area->width;
		height = area->height;
	}
	else
	{
		x      = widget->allocation.x;
		y      = widget->allocation.y;
		width  = widget->allocation.width;
		height = widget->allocation.height;
	}

	gdk_draw_pixmap (widget->window, widget->style->black_gc, gpb->pixmap, x, y, x, y, width, height);
}

static void
gnome_pixmap_button_draw (GtkWidget *widget, GdkRectangle *area)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (widget));
	g_return_if_fail (area != NULL);

	if (GTK_WIDGET_DRAWABLE (widget))
		paint (GNOME_PIXMAP_BUTTON (widget), area);
}

static gint
gnome_pixmap_button_expose (GtkWidget *widget, GdkEventExpose *event)
{
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GNOME_IS_PIXMAP_BUTTON (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	if (GTK_WIDGET_DRAWABLE (widget))
		paint (GNOME_PIXMAP_BUTTON (widget), &event->area);

	return FALSE;
}

static gint
gnome_pixmap_button_button_press (GtkWidget *widget, GdkEventButton *event)
{
	GnomePixmapButton *button;

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GNOME_IS_PIXMAP_BUTTON (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	if (event->type == GDK_BUTTON_PRESS)
	{
		button = GNOME_PIXMAP_BUTTON (widget);

		if (!GTK_WIDGET_HAS_FOCUS (widget))
			gtk_widget_grab_focus (widget);

		if (event->button == 1)
		{
			gtk_grab_add (GTK_WIDGET (button));
			gnome_pixmap_button_pressed (button);
		}
	}

	return TRUE;
}

static gint
gnome_pixmap_button_button_release (GtkWidget *widget, GdkEventButton *event)
{
	GnomePixmapButton *button;

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GNOME_IS_PIXMAP_BUTTON (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	if (event->button == 1)
	{
		button = GNOME_PIXMAP_BUTTON (widget);
		gtk_grab_remove (GTK_WIDGET (button));
		gnome_pixmap_button_released (button);
	}

	return TRUE;
}

static gint
gnome_pixmap_button_enter_notify (GtkWidget *widget, GdkEventCrossing *event)
{
	GnomePixmapButton *button;
	GtkWidget *event_widget;

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GNOME_IS_PIXMAP_BUTTON (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	button = GNOME_PIXMAP_BUTTON (widget);
	event_widget = gtk_get_event_widget ((GdkEvent*) event);

	if ((event_widget == widget) && (event->detail != GDK_NOTIFY_INFERIOR))
	{
		button->in_button = TRUE;
		gnome_pixmap_button_enter (button);
	}

	return FALSE;
}

static gint
gnome_pixmap_button_leave_notify (GtkWidget *widget, GdkEventCrossing *event)
{
	GnomePixmapButton *button;
	GtkWidget *event_widget;

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GNOME_IS_PIXMAP_BUTTON (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	button = GNOME_PIXMAP_BUTTON (widget);
	event_widget = gtk_get_event_widget ((GdkEvent*) event);

	if ((event_widget == widget) && (event->detail != GDK_NOTIFY_INFERIOR))
	{
		button->in_button = FALSE;
		gnome_pixmap_button_leave (button);
	}

	return FALSE;
}

static void
real_button_pressed (GnomePixmapButton *button)
{
	GtkStateType new_state;

	g_return_if_fail (button != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (button));

	button->button_down = TRUE;

	new_state = (button->in_button) ? GTK_STATE_ACTIVE : GTK_STATE_NORMAL;

	if (GTK_WIDGET_STATE (button) != new_state)
	{
		button->repaint = TRUE;
		gtk_widget_set_state (GTK_WIDGET (button), new_state);
		gtk_widget_queue_draw (GTK_WIDGET (button));
	}
}

static void
real_button_released (GnomePixmapButton *button)
{
	g_return_if_fail (button != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (button));

	if (button->button_down)
	{
		GtkStateType new_state;

		button->button_down = FALSE;

		if (button->in_button)
			gnome_pixmap_button_clicked (button);

		new_state = (button->in_button) ? GTK_STATE_PRELIGHT : GTK_STATE_NORMAL;

		if (GTK_WIDGET_STATE (button) != new_state)
		{
			button->repaint = TRUE;
			gtk_widget_set_state (GTK_WIDGET (button), new_state);
			gtk_widget_queue_draw (GTK_WIDGET (button));
		}
	}
}

static void
real_button_enter (GnomePixmapButton *button)
{
	GtkStateType new_state;

	g_return_if_fail (button != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (button));

	new_state = (button->button_down) ? GTK_STATE_ACTIVE : GTK_STATE_PRELIGHT;

	if (GTK_WIDGET_STATE (button) != new_state)
	{
		button->repaint = TRUE;
		gtk_widget_set_state (GTK_WIDGET (button), new_state);
		gtk_widget_queue_draw (GTK_WIDGET (button));
	}
}

static void
real_button_leave (GnomePixmapButton *button)
{
	g_return_if_fail (button != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (button));

	if (GTK_WIDGET_STATE (button) != GTK_STATE_NORMAL)
	{
		button->repaint = TRUE;
		gtk_widget_set_state (GTK_WIDGET (button), GTK_STATE_NORMAL);
		gtk_widget_queue_draw (GTK_WIDGET (button));
	}
}

static void
free_cache (GnomePixmapButton *gpb)
{
	if (gpb->pixmap)
	{
		gdk_pixmap_unref (gpb->pixmap);
		gpb->pixmap = NULL;
	}
	if (gpb->bitmap)
	{
		gdk_bitmap_unref (gpb->bitmap);
		gpb->bitmap = NULL;
	}
	if (gpb->gc)
	{
		gdk_gc_unref (gpb->gc);
		gpb->gc = NULL;
	}
}

static void
new_cache (GnomePixmapButton *gpb)
{
	free_cache (gpb);

	gpb->pixmap = gdk_pixmap_new (GTK_WIDGET (gpb)->window, gpb->width, gpb->height, -1);
	gpb->bitmap = gdk_pixmap_new (GTK_WIDGET (gpb)->window, gpb->width, gpb->height, 1);
	gpb->gc     = gdk_gc_new (GTK_WIDGET (gpb)->window);
}

static void
free_button (GnomePixmapButton *gpb)
{
	g_return_if_fail (gpb != NULL);

	if (gpb->im_up)
	{
		gdk_pixbuf_unref (gpb->im_up);
		gpb->im_up = NULL;
	}

	if (gpb->im_up_p)
	{
		gdk_pixbuf_unref (gpb->im_up_p);
		gpb->im_up_p = NULL;
	}

	if (gpb->im_down)
	{
		gdk_pixbuf_unref (gpb->im_down);
		gpb->im_down = NULL;
	}

	if (gpb->im_down_p)
	{
		gdk_pixbuf_unref (gpb->im_down_p);
		gpb->im_down_p = NULL;
	}
}

static void
free_animation (GnomePixmapButton *gpb)
{
	g_return_if_fail (gpb != NULL);

	gpb->num_frames           = 0;
	gpb->current_frame_number = 0;

	if (gpb->im_animation)
	{
		gdk_pixbuf_unref (gpb->im_animation);
		gpb->im_animation = NULL;
	}

	if (gpb->im_animation_p)
	{
		gdk_pixbuf_unref (gpb->im_animation_p);
		gpb->im_animation_p = NULL;
	}
}

/*
 * Public class interface
 */

GtkWidget*
gnome_pixmap_button_new_with_size (gint width, gint height)
{
	GtkWidget *button = GTK_WIDGET (gtk_type_new (gnome_pixmap_button_get_type ()));

	g_return_val_if_fail ((button != NULL), NULL);

	GNOME_PIXMAP_BUTTON (button)->width  = width;
	GNOME_PIXMAP_BUTTON (button)->height = height;

	return button;
}

void
gnome_pixmap_button_set_size (GnomePixmapButton *gpb, gint width, gint height)
{
	gpb->width  = width;
	gpb->height = height;

	new_cache (gpb);
}

gboolean
gnome_pixmap_button_tiles_load_from_pixbuf (GnomePixmapButton *gpb, GdkPixbuf *im_up, GdkPixbuf *im_down)
{
	g_return_val_if_fail ((gpb != NULL), FALSE);

	free_button (gpb);

	if (im_up)
	{
		if ((gdk_pixbuf_get_width  (im_up) == gpb->width) &&
		    (gdk_pixbuf_get_height (im_up) == gpb->height))
			gpb->im_up = gdk_pixbuf_ref (im_up);
		else
			gpb->im_up = gdk_pixbuf_scale_simple (im_up, gpb->width, gpb->height, GDK_INTERP_HYPER);

		gpb->im_up_p = gdk_pixbuf_new (gdk_pixbuf_get_colorspace (gpb->im_up),
					       gdk_pixbuf_get_has_alpha (gpb->im_up),
					       gdk_pixbuf_get_bits_per_sample (gpb->im_up),
					       gdk_pixbuf_get_width (gpb->im_up),
					       gdk_pixbuf_get_height (gpb->im_up));
		do_colorshift (gpb->im_up_p, gpb->im_up, 30);
	}

	if (im_down)
	{
		if ((gdk_pixbuf_get_width  (im_down) == gpb->width) &&
		    (gdk_pixbuf_get_height (im_down) == gpb->height))
			gpb->im_down = gdk_pixbuf_ref (im_down);
		else
			gpb->im_down = gdk_pixbuf_scale_simple (im_down, gpb->width, gpb->height, GDK_INTERP_HYPER);


		gpb->im_down_p = gdk_pixbuf_new (gdk_pixbuf_get_colorspace (gpb->im_down),
						 gdk_pixbuf_get_has_alpha (gpb->im_down),
						 gdk_pixbuf_get_bits_per_sample (gpb->im_down),
						 gdk_pixbuf_get_width (gpb->im_down),
						 gdk_pixbuf_get_height (gpb->im_down));
		do_colorshift (gpb->im_down_p, gpb->im_down, 30);
	}

	gpb->repaint = TRUE;

	if (GTK_WIDGET_DRAWABLE (gpb))
    		gtk_widget_queue_draw (GTK_WIDGET (gpb));

	return TRUE;
}

gboolean
gnome_pixmap_button_tiles_load_from_file (GnomePixmapButton *gpb, const gchar *file_up, const gchar *file_down)
{
	GdkPixbuf *im_up, *im_down;
	gboolean result;

	g_return_val_if_fail ((gpb != NULL), FALSE);
	g_return_val_if_fail ((GNOME_IS_PIXMAP_BUTTON (gpb)), FALSE);

	im_up   = (file_up)   ? gdk_pixbuf_new_from_file (file_up)   : NULL;
	im_down = (file_down) ? gdk_pixbuf_new_from_file (file_down) : NULL;

	result = gnome_pixmap_button_tiles_load_from_pixbuf (gpb, im_up, im_down);

	gdk_pixbuf_unref (im_up);
	gdk_pixbuf_unref (im_down);

	return result;
}

gboolean
gnome_pixmap_button_tiles_load_from_xpm_d (GnomePixmapButton *gpb, const gchar **xpm_up, const gchar **xpm_down)
{
	GdkPixbuf *im_up, *im_down;
	gboolean result;

	g_return_val_if_fail ((gpb != NULL), FALSE);
	g_return_val_if_fail ((GNOME_IS_PIXMAP_BUTTON (gpb)), FALSE);

	im_up   = (xpm_up)   ? gdk_pixbuf_new_from_xpm_data (xpm_up)   : NULL;
	im_down = (xpm_down) ? gdk_pixbuf_new_from_xpm_data (xpm_down) : NULL;

	result = gnome_pixmap_button_tiles_load_from_pixbuf (gpb, im_up, im_down);

	gdk_pixbuf_unref (im_up);
	gdk_pixbuf_unref (im_down);

	return result;
}

gboolean
gnome_pixmap_button_animation_load_from_pixbuf (GnomePixmapButton *gpb, GdkPixbuf *im)
{
	guint     rest;

	g_return_val_if_fail ((gpb != NULL), FALSE);
	g_return_val_if_fail (GNOME_IS_PIXMAP_BUTTON (gpb), FALSE);
	g_return_val_if_fail ((im != NULL), FALSE);

	rest = gdk_pixbuf_get_width (im) % gdk_pixbuf_get_height (im);
	if (rest) return FALSE;

	free_animation (gpb);

	gpb->num_frames = gdk_pixbuf_get_width (im) / gdk_pixbuf_get_height (im);

	if ((gdk_pixbuf_get_width  (im) == (gpb->num_frames * gpb->width)) &&
	    (gdk_pixbuf_get_height (im) == gpb->height))
		gpb->im_animation = gdk_pixbuf_ref (im);
	else
		gpb->im_animation = gdk_pixbuf_scale_simple (im, gpb->num_frames * gpb->width, gpb->height, GDK_INTERP_HYPER);

	gpb->im_animation_p = gdk_pixbuf_new (gdk_pixbuf_get_colorspace (gpb->im_animation),
					      gdk_pixbuf_get_has_alpha (gpb->im_animation),
					      gdk_pixbuf_get_bits_per_sample (gpb->im_animation),
					      gdk_pixbuf_get_width (gpb->im_animation),
					      gdk_pixbuf_get_height (gpb->im_animation));
	do_colorshift (gpb->im_animation_p, gpb->im_animation, 30);

	gpb->repaint = TRUE;

	if (GTK_WIDGET_DRAWABLE (gpb))
    		gtk_widget_queue_draw (GTK_WIDGET (gpb));

	return TRUE;
}

gboolean
gnome_pixmap_button_animation_load_from_file (GnomePixmapButton *gpb, const gchar *fname)
{
	GdkPixbuf *im;
	gboolean result;

	g_return_val_if_fail ((gpb != NULL), FALSE);
	g_return_val_if_fail ((GNOME_IS_PIXMAP_BUTTON (gpb)), FALSE);
	g_return_val_if_fail ((fname != NULL), FALSE);

	im = gdk_pixbuf_new_from_file (fname);

	result = gnome_pixmap_button_animation_load_from_pixbuf (gpb, im);

	gdk_pixbuf_unref (im);

	return result;
}

gboolean
gnome_pixmap_button_animation_load_from_xpm_d (GnomePixmapButton *gpb, const gchar **xpm)
{
	GdkPixbuf *im;
	gboolean result;

	g_return_val_if_fail ((gpb != NULL), FALSE);
	g_return_val_if_fail ((GNOME_IS_PIXMAP_BUTTON (gpb)), FALSE);
	g_return_val_if_fail ((xpm != NULL), FALSE);

	im = gdk_pixbuf_new_from_xpm_data (xpm);

	result = gnome_pixmap_button_animation_load_from_pixbuf (gpb, im);

	gdk_pixbuf_unref (im);

	return result;
}

void
gnome_pixmap_button_goto_frame (GnomePixmapButton *gpb, guint frame_number)
{
	g_return_if_fail (gpb != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (gpb));

	if (frame_number == gpb->current_frame_number)
		return;

	gpb->current_frame_number = (frame_number < gpb->num_frames) ? frame_number : (gpb->num_frames - 1);

	gpb->repaint = TRUE;

	if (GTK_WIDGET_DRAWABLE (gpb))
    		gtk_widget_queue_draw (GTK_WIDGET (gpb));
}

void
gnome_pixmap_button_repaint (GnomePixmapButton *gpb)
{
	gpb->repaint = TRUE;

	if (GTK_WIDGET_DRAWABLE (gpb))
    		gtk_widget_queue_draw (GTK_WIDGET (gpb));
}

void
gnome_pixmap_button_pressed (GnomePixmapButton *button)
{
	g_return_if_fail (button != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (button));

	gtk_signal_emit (GTK_OBJECT (button), button_signals[PRESSED]);
}

void
gnome_pixmap_button_released (GnomePixmapButton *button)
{
	g_return_if_fail (button != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (button));

	gtk_signal_emit (GTK_OBJECT (button), button_signals[RELEASED]);
}

void
gnome_pixmap_button_clicked (GnomePixmapButton *button)
{
	g_return_if_fail (button != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (button));

	gtk_signal_emit (GTK_OBJECT (button), button_signals[CLICKED]);
}

void
gnome_pixmap_button_enter (GnomePixmapButton *button)
{
	g_return_if_fail (button != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (button));

	gtk_signal_emit (GTK_OBJECT (button), button_signals[ENTER]);
}

void
gnome_pixmap_button_leave (GnomePixmapButton *button)
{
	g_return_if_fail (button != NULL);
	g_return_if_fail (GNOME_IS_PIXMAP_BUTTON (button));

	gtk_signal_emit (GTK_OBJECT (button), button_signals[LEAVE]);
}

/* Next function is copied from gnome-core/panel/button-widget.c
 */

/* colorshift a pixbuf */
static void
do_colorshift (GdkPixbuf *dest, GdkPixbuf *src, int shift)
{
	gint i, j;
	gint width, height, has_alpha, srcrowstride, destrowstride;
	guchar *target_pixels;
	guchar *original_pixels;
	guchar *pixsrc;
	guchar *pixdest;
	int val;
	guchar r,g,b;

	has_alpha = gdk_pixbuf_get_has_alpha (src);
	width = gdk_pixbuf_get_width (src);
	height = gdk_pixbuf_get_height (src);
	srcrowstride = gdk_pixbuf_get_rowstride (src);
	destrowstride = gdk_pixbuf_get_rowstride (dest);
	target_pixels = gdk_pixbuf_get_pixels (dest);
	original_pixels = gdk_pixbuf_get_pixels (src);

	for (i = 0; i < height; i++) {
		pixdest = target_pixels + i*destrowstride;
		pixsrc = original_pixels + i*srcrowstride;
		for (j = 0; j < width; j++) {
			r = *(pixsrc++);
			g = *(pixsrc++);
			b = *(pixsrc++);
			val = r + shift;
			*(pixdest++) = CLAMP(val, 0, 255);
			val = g + shift;
			*(pixdest++) = CLAMP(val, 0, 255);
			val = b + shift;
			*(pixdest++) = CLAMP(val, 0, 255);
			if (has_alpha)
				*(pixdest++) = *(pixsrc++);
		}
	}
}
