/* blue6.h					-*- C++ -*-
   $Id: blue6.h,v 1.19 2003/05/27 16:30:25 elf Exp $
   
   written by Marc Singer
   22 Feb 1998

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1998 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   with your Debian GNU/Linux system, in
   /usr/share/common-licenses/GPL, or with the Debian GNU/Linux hello
   source package as the file COPYING. If not, write to the Free
   Software Foundation, Inc., 59 Temple Place -Suite 330, MA
   02111-1307, USA.

   Simple that classs to interface to termcap.

*/

#if !defined (__BLUE6_H__)
#    define   __BLUE6_H__

/* ----- Includes */

#include "termcap.h"
#include <list>			// STL list
//using namespace stl;

inline void _memset (void* pv, unsigned char ch, int cb) \
			{ memset (pv, ch, cb); }

#if !defined (ZERO_BASED)
# define ZERO_BASED(t,cb) _memset ((char*) (t) + (cb), 0, sizeof (*(t)) - (cb))
#endif

/* ----- Types */

/* ----- Classes */

class Attribute {
public:
  static Attribute normal;	// Used to reset attributes

  enum eAttr {
    // *** The preceeding underscores are to work around a CPP bug
    // that doesn't permit parameter pasting to a class specifier.  We
    // cannot append Attribute:: to one of these identifiers.
    _Plain		= 0x0000,
    _Bold		= 0x0001,
    _Underline		= 0x0002,
    _Inverse		= 0x0004,
    _Reverse		= 0x0004,
    _Standout		= 0x0008,
    _Blink		= 0x0010,
    //    Dim		= 0x0020,
    _AllMode		= 0x003f,
    _FgBright		= 0x0080,
    _FgDefault		= 0x0000,	// Foreground colors
    _FgMask		= 0x0f00,
    _FgShift		= 8,
    _FgFirst		= 0x0100,
    _FgBlack		= 0x0100,
    _FgRed		= 0x0200,
    _FgGreen		= 0x0300,
    _FgYellow		= 0x0400,
    _FgBlue		= 0x0500,
    _FgMagenta		= 0x0600,
    _FgCyan		= 0x0700,
    _FgWhite		= 0x0800,
    _BgMask		= 0xf000, 	// Background colors
    _BgShift		= 12,
    _BgDefault		= 0x0000,
    _BgFirst		= 0x1000,
    _BgBlack		= 0x1000,
    _BgRed		= 0x2000,
    _BgGreen		= 0x3000,
    _BgYellow		= 0x4000,
    _BgBlue		= 0x5000,
    _BgMagenta		= 0x6000,
    _BgCyan		= 0x7000,
    _BgWhite		= 0x8000,
    _All		= 0xff3f,
  };

protected:
  int m_value;
  int m_mask;

public:
  Attribute () {
    zero (); init (); }
  Attribute (const Attribute& attr) {
    m_value = attr.m_value; m_mask = attr.m_mask; }
  Attribute (int value, int mask) {
    m_value = value; m_mask = mask; }
  void zero (void) {
    memset (this, 0, sizeof (*this)); }
  void init (void) {
    /* m_mask = (AllMode | BgMask | FgMask); */ }

  int reconcile (const Termcap& termcap, Attribute attr, char* sz, int cbMax);
  void set (int attr) {
    m_value = attr; m_mask = (_AllMode | _BgMask | _FgMask); }
  void clear (void) {
    m_value = m_mask = 0; }
  int value (void) const {
    return m_value; }
  void ignore (int attr) {
    m_value &= ~attr;
    m_mask &= ~attr; }
				// I need more help here.
  void invert (int attr) {
    m_value = (m_value & ~attr) | ((m_value & attr) ? 0 : attr);
    m_mask |= attr; }

  Attribute merge (const Attribute& attr) const;

};

class Rectangle;

class Position {
public:
  int m_x;
  int m_y;

public:
  Position () { zero (); }
  Position (int x, int y) {
    init (x, y); }
  void init (int x, int y) {
    m_x = x; m_y = y; }
  void zero (void) {
    _memset (this, 0, sizeof (*this)); }

  void constrain (Rectangle& m_rc);

};

class Rectangle {
public:
  int m_x;
  int m_y;
  int m_dx;
  int m_dy;

public:
  Rectangle () { zero (); }
  Rectangle (int x, int y, int dx, int dy) {
    m_x = x; m_y = y; m_dx = dx; m_dy = dy; }
  Rectangle (const Rectangle& rc) {
    m_x = rc.m_x; m_y = rc.m_y; m_dx = rc.m_dx; m_dy = rc.m_dy; }
  void zero (void) {
    _memset (this, 0, sizeof (*this)); }

  bool is_empty (void) {
    return !(m_dx && m_dy); }
  Rectangle& intersect (const Rectangle& rc1, const Rectangle& rc2);
  Rectangle& intersect (const Rectangle& rc);

  Rectangle& offset (const Position& pos) {
    m_x += pos.m_x; m_y += pos.m_y; return *this; }
  Rectangle& offset (int dx, int dy) {
    m_x += dx; m_y += dy; return *this; }

  bool operator== (const Rectangle& rc) {
    return (   m_x  == rc.m_x
	    && m_y  == rc.m_y 
	    && m_dx == rc.m_dx
	    && m_dy == rc.m_dy); }

};

class Op {
public:
  enum eOp {
    opNul = 0,
    opErase,
    opDrawText,
    opDrawLine,
    opMoveTo,
  };

protected:
  Position m_pos;		// Position or 
  Rectangle m_rc;		//   Rectangle for operation
  Attribute m_attr;		// Attributes
  eOp m_op;			// Operation
  char* m_sz;			// String for draw_text

public:
  Op () {
    zero (); }
  Op (eOp op, const Rectangle& rc, const Attribute& attr) {
    zero (); init (op, rc, attr); }
  Op (eOp op, const Position& pos, const Attribute& attr,
      const char* sz, int cb) {
    zero (); init (op, pos, attr, sz, cb); }
  Op (eOp op, const Position& pos) {
    zero (); init (op, pos); }
  void zero (void) {
    memset (this, 0, sizeof (*this)); }
  void init (eOp op, const Rectangle& rc, const Attribute& attr);
  void init (eOp op, const Position& pos, const Attribute& attr,
	     const char* sz, int cb);
  void init (eOp op, const Position& pos);
  void init (eOp op, const Attribute& attr);
  
  eOp op (void) {
    return m_op; }
  const Rectangle& rc (void) {
    return m_rc; }
  const Position& pos (void) {
    return m_pos; }
  const Attribute& attr (void) {
    return m_attr; }
  const char* sz (void) {
    return m_sz; }
};

class Drawable {
protected:
  Position m_position;		// Current position
  Attribute m_attribute;	// Attributes, color and style
  Attribute m_attributeBg;	// Background or ambient attribute

public:
  Drawable () {
    zero (); init (); }
  void zero () {
    ZERO_BASED (this, sizeof (void*)); }
  void init (void) {
    m_attributeBg.set (0); }

  virtual Attribute& attr (void);
  virtual Attribute& attrBg (void);
  virtual Position& pos (void);
  virtual void _draw_text (const char* sz) = 0;
  virtual void _draw_text (const Position& pos, const char* sz);
  virtual void draw_text (const char* sz, ...);
  virtual void draw_text (const Position& pos, const char* sz, ...);
  virtual void draw_vertical_line (int x, int y, int dy) = 0;
  virtual void erase (const Rectangle& rc) = 0;
  virtual void erase (void) = 0;
  virtual void move_to (const Position& pos) = 0;
  virtual int prompt (Position pos, const char* szPrompt,
		      char* szReposonse, size_t cchResponse,
		      const char* szDefault) = 0;
  virtual bool scroll (int c) = 0;
};

class Frame {
protected:
  Rectangle m_rc;

public:
  Frame () {
    zero (); }
  Frame (Rectangle& rc) {
    zero (); init (rc); }
  void zero (void) {
    memset (this, 0, sizeof (*this)); }
  void init (const Rectangle& rc) {
    m_rc = rc; }

  int dx (void) const {
    return m_rc.m_dx; }
  int dy (void) const {
    return m_rc.m_dy; }
};

class Window;

class Pane : public Drawable, public Frame {
protected:
  Window* m_pWindow;		// Pointer to parent window

  Termcap& termcap (void);

public:
  Pane () {
    zero (); }
  Pane (Window& window, const Rectangle& rc) {
    zero (); init (window); Frame::init (rc); }
  void init (Window& window) {
    m_pWindow = &window; }
  void zero (void) {
    ZERO_BASED (this, sizeof (Drawable) + sizeof (Frame)); }

		// -- Implement Drawable
  void _draw_text (const char* sz);
  void erase (const Rectangle& rc);
  void draw_vertical_line (int x, int y, int dy);

  void erase (void);
  void move_to (const Position& pos);
  int prompt (Position pos, const char* szPrompt, 
	      char* szResponse, size_t cchResponse, 
	      const char* szDefault);
  bool scroll (int /* c */) { return false; }

  int getch (const Position& pos);

};

class Window : public Pane {
protected:
  Termcap* m_pTermcap;		// Termcap that we use
  //  Attribute m_attr;		// Default attribute for next character
  Position m_pos;		// Position of real cursor
  Attribute m_attrSave;		// Saved attribute for pushing

  char* m_rgb;			// Window buffer
  char m_rgbInput[10];		// Input buffer for multi-byte keystrokes
  int m_cbInput;		// Valid characters in the input buffer
  bool m_fNoGraphicCharset;	// Inhibit alt charset for line drawing

  std::list<Op*>* m_pListOp;	// List of pending operations

  //  list<Pane*> m_listPane;	// List of panes

public:
  Window () {
    zero (); Pane::init (*this); }
  virtual ~Window () {
    release_this (); }
  void zero (void) {
    ZERO_BASED (this, sizeof (Pane)); }
  void init (Termcap* pTermcap);
  void release_this (void);
  
  Termcap& termcap (void) {
    return *m_pTermcap; }

  void inhibit_alt_charset (bool fInhibit) {
    m_fNoGraphicCharset = fInhibit; }
  void queue (Op* pOp);
  void flush (void);		// Flush output queue
  int getch (void);
  void refresh (void) {}	// Repaint window
  void reset (void);
  void pop_attribute (void);
  void push_attribute (void);
  void activate (bool fActivate);

  void do_erase (Op* pOp);
  void do_draw_text (Op* pOp);
  void do_draw_line (Op* pOp);
  void do_move_to (Op* pOp);

		// -- Implement Drawable
  void erase (const Rectangle& rc);
  void erase (void) {
    erase (m_rc); }
};

inline Termcap& Pane::termcap (void) {
  return m_pWindow->termcap (); }


/* ----- Globals */



#endif  /* __BLUE6_H__ */
