# axes.rb: A module to deal with axes customization.
# Copyright (c) 2007, 2008 by Vincent Fourmond
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/utils'
require 'CTioga/layout'
require 'CTioga/log'
require 'CTioga/boundaries'
require 'CTioga/partition'

module CTioga

  Version::register_svn_info('$Revision: 828 $', '$Date: 2008-08-07 11:29:21 +0200 (Thu, 07 Aug 2008) $')

  # A module to be included by the main PlotMaker instance. It
  # deals with various aspects of axes manipulations.
  module Axes

    LabelSpecification = {
      /^x(label)?$/i => :xlabel,
      /^y(label)?$/i => :ylabel,
      /^t(itle)?$/i => :title,
      /^xticks?$/i => :xticks,
      /^yticks?$/i => :yticks,
    }

    AxisSpecification = {
      /^x$/i => 'xaxis',
      /^y?$/i => 'yaxis',
    }

    LabelSide = {
      /bottom/i => Tioga::FigureConstants::BOTTOM,
      /top/i => Tioga::FigureConstants::TOP,
      /left/i => Tioga::FigureConstants::LEFT,
      /right/i => Tioga::FigureConstants::RIGHT,
    }

    Alignment = {
      /B/ => Tioga::FigureConstants::ALIGNED_AT_BASELINE,
      /baseline/i => Tioga::FigureConstants::ALIGNED_AT_BASELINE,
      /b/ => Tioga::FigureConstants::ALIGNED_AT_BOTTOM,
      /bottom/i => Tioga::FigureConstants::ALIGNED_AT_BOTTOM,
      /t(op)?/i => Tioga::FigureConstants::ALIGNED_AT_TOP,
    }

    Justification = {
      /c(enter)?/i => Tioga::FigureConstants::CENTERED,
      /l(eft)?/i => Tioga::FigureConstants::LEFT_JUSTIFIED,
      /r(ight)?/i => Tioga::FigureConstants::RIGHT_JUSTIFIED,
    }
    

    # TODO: all the following attributes should go to PlotStyle!
    # This would allow to remove the dirty ugly funcall hack for
    # log values...

    # Whether to have X or Y log axes
    attr_accessor :x_log, :y_log
    # Scaling of data axes
    attr_accessor :x_factor, :y_factor
    # Offsets
    attr_accessor :x_offset, :y_offset
    # Decimal separator
    attr_accessor :decimal_separator
    
    # Initializes variables pertaining to axes
    def init_axes
      # TODO: same as above
      @x_log = false
      @y_log = false
      @x_factor = false
      @y_factor = false
      @x_offset = false
      @y_offset = false
      @decimal_separator = false
    end

    # Turns a String into a [ left, right, top, bottom ]
    # edge visibility specification
    def parse_edge_visibility(str)
      # First and simplest : a series of v (for visible) and i
      # (for invisible)
      if str =~ /^\s*([vi]{4})\s*$/i
        val = $1.split(//).map { |s| s =~ /v/i ? true : false }
        return val
      end
      # By default, everything is visible
      return [ true, true, true, true ]
    end

    # Runs a block safely with a label specification 
    def run_with_label(name)
      w = Utils::interpret_arg(name, LabelSpecification) {false}
      if w
        yield current_object.plot_style.send(w)
      elsif current_plot_style.edges.axes.key? name
        yield current_plot_style.edges.axes[name].ticks
      else
        error "The label/axis specification #{w} was not understood, ignoring"
      end
    end

    # Runs a block safely with an axis specification.
    #
    # This function relies on the fact that it is being used
    # from within a PlotMaker instance !!
    def run_with_axis(name)
      w = Utils::interpret_arg(name, AxisSpecification) {name}
      if current_plot_style.edges.axes.key? w
        yield current_plot_style.edges.axes[w]
      else
        error "The axis specification #{w} was not understood, ignoring"
        error "Valid axis specifications are " +
          current_plot_style.edges.axes.keys.join(', ')
      end
    end

    # Prepare the option parser for axes options
    def axes_options(parser)

      parser.separator "\nVarious axes manipulations:"
      parser.on("--xfact FACT",
                "Multiply all x values by FACT") do |f|
        @x_factor = safe_float(f)
      end
      parser.on("--yfact FACT",
                "Multiply all y values by FACT") do |f|
        @y_factor = safe_float(f)
      end
      parser.on("--xoffset OFFSET",
                "Add OFFSET to all X values (after multiplication)") do |f|
        @x_offset = safe_float(f)
      end
      parser.on("--yoffset OFFSET",
                "Add OFFSET to all X values (after multiplication)") do |f|
        @y_offset = safe_float(f)
      end

      parser.on("--[no-]xlog",
                "Uses logarithmic scale for X axis") do |v|
        @x_log = v
        add_elem_funcall(:xaxis_log_values=, v)
      end
      parser.on("--[no-]ylog",
                "Uses logarithmic scale for Y axis") do |v|
        @y_log = v
        add_elem_funcall(:yaxis_log_values=, v)
      end

      parser.on("--reset-transformations",
                "Reset axes transformations") do |v|
        @y_log = false
        @x_log = false
        # TODO: remove that when this is move to PlotStyle:
        # it should be setup automatically by the Container
        # at the beginning of a plot...
        add_elem_funcall(:xaxis_log_values=, false)
        add_elem_funcall(:yaxis_log_values=, false)
        @x_factor = nil
        @y_factor = nil
        @x_offset = nil
        @y_offset = nil
      end

      

      parser.on("--comma",
                "Uses a comma for the decimal separator") do 
        @decimal_separator = ','
      end
      parser.on("--decimal SEP",
                "Uses SEP for the decimal separator") do |s|
        @decimal_separator = s
      end

      parser.separator "\nLabels and titles"
      parser.on("-x","--[no-]xlabel [LABEL]",
                "Label of the x axis") do |l|
        current_object.plot_style.xlabel.label = l
      end
      parser.on("-y","--[no-]ylabel [LABEL]",
                 "Label of the y axis") do |l|
        current_object.plot_style.ylabel.label = l
      end
      parser.on('-t', "--[no-]title [TITLE]",
                 "Sets the title of the plot") do |l|
        current_object.plot_style.title.label = l
      end
      parser.on("--side WHAT ALIGN",
                "Sets the side for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle)") do |w|
        run_with_label(w) do |label|
          a = Utils::interpret_arg(@args.shift, LabelSide) {false}
          label.side = a if a
        end
      end

      parser.on("--lcolor WHAT COLOR",
                "Sets the color for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle)") do |w|
        run_with_label(w) do |label|
          a = CTioga.get_tioga_color(@args.shift)
          label.color = a if a
        end
      end

      parser.on("--position WHAT WHERE",
                "Sets the position for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle)",
                "and WHERE a number between 0 and 1, 0.5 = centered") do |w|
        run_with_label(w) do |label|
          a = safe_float(@args.shift)
          label.position = a if a
        end
      end

      parser.on("--angle WHAT ANGLE",
                "Sets the angle for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).") do |w|
        run_with_label(w) do |label|
          a = safe_float(@args.shift)
          label.angle = a if a
        end
      end

      parser.on("--scale WHAT SCALE",
                "Sets the scale for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).") do |w|
        run_with_label(w) do |label|
          a = safe_float(@args.shift)
          label.scale = a if a
        end
      end

      parser.on("--shift WHAT SHIFT",
                "Sets the shift for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).",
                "The shift is the distance of the label from the plot") do |w|
        run_with_label(w) do |label|
          a = safe_float(@args.shift)
          label.shift = a if a
        end
      end

      parser.on("--align WHAT ALIGN",
                "Sets the 'vertical' alignment for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).") do |w|
        run_with_label(w) do |label|
          a = Utils::interpret_arg(@args.shift, Alignment) {false}
          label.alignment = a if a
        end
      end

      parser.on("--just WHAT JUST",
                "Sets the 'horizontal' alignment for the WHAT label, ",
                "where WHAT = x(label),y(label) or t(itle).") do |w|
        run_with_label(w) do |label|
          a = Utils::interpret_arg(@args.shift, Justification) {false}
          label.justification = a if a
        end
      end

      parser.separator "\nEdges and axes look:"

      parser.on("--edges-visibility SPEC",
                "Chooses which edges of the current graph will be",
                "visible. Will not act on edges that are also axes.") do |w|
        current_object.plot_style.edges.edge_visibility = 
          parse_edge_visibility(w)
      end

      parser.on("--axis-pos AXIS POS",
                "Sets the position of AXIS to POS") do |w|
        run_with_axis(w) do |axis|
          axis.loc = EdgesAndAxes.parse_axis_position(@args.shift)
        end
      end


      parser.on("--xaxis STYLE",
                "Sets the style of the X axis") do |w|
        current_object.plot_style.set_axis_style(:x, w)
      end

      parser.on("--yaxis STYLE",
                "Sets the style of the Y axis") do |w|
        current_object.plot_style.set_axis_style(:y, w)
      end

      parser.on("--edge-style WHICH STYLE",
                "Sets the style of the X axis") do |w|
        current_object.plot_style.set_axis_style(w.to_sym,
                                                 @args.shift)
      end

      parser.on("--no-axes",
                "No title, labels and axes") do
        ps = current_plot_style
        ps.set_axis_style(:x, "none")
        ps.set_axis_style(:y, "none")
        ps.xlabel.label = false
        ps.ylabel.label = false
        ps.title.label = false
      end

      parser.on("--lines-color AXIS COLOR",
                "Sets the color for lines perpendicular to AXIS",
                "'none' for no lines") do |w|
        run_with_axis(w) do |axis|
          color = @args.shift
          if color =~ /no(ne)?/
            color = false
          else
            color = CTioga.get_tioga_color(color)
          end
          axis.lines_color = color
        end
      end

      parser.on("--new-axis NAME POSITION",
                "Creates a new axis named NAME for this plot") do |name|
        current_plot_style.edges.axes[name] = 
          EdgesAndAxes::Axis.new(@args.shift)
      end

      parser.on("--axis-function AXIS TO FROM",
                "Sets AXIS to use a non-linear mapping represented",
                "by 2 mathematical functions (of x): TO (to convert from",
                "real to axis) and FROM that does the opposite") do |w|
        run_with_axis(w) do |axis|
          axis.real_to_axis = eval "proc { |x| #{@args.shift} }"
          axis.axis_to_real = eval "proc { |x| #{@args.shift} }"
        end
      end



    end
    
  end

  # A class that describes the various attributes of a label,
  # either for the X or Y axis, or even the plot title.
  class Label

    include Log

    # Include the figure constants, else it gets really painful...
    include Tioga::FigureConstants
    
    # The various attributes
    Attributes = [:alignment, :angle, :color, :justification,
                  :position, :scale, :shift, :side, :visible]
    
    attr_accessor *Attributes

    # Which axis ?
    attr_accessor :which

    # The actual text to be displayed
    attr_accessor :label

    # Creates a label specification for the _which_ axis.
    # _other_values_ is a hash that can be used to provide default
    # values for the various Attributes.
    #
    # _which_ can be :xlabel, :ylabel or :title
    def initialize(which, label = nil, other_values = {})
      @which = which
      @label = label
      for key,val in other_values
        self.send("#{key}=", val)
      end
    end

    # Shows the given label on the figure _t_
    def show(t)
      # We don't do anything unless we have a label to display !
      return unless @label
      # We wrap the call in a context so we don't pollute subpictures
      t.context do 
        for attr in Attributes
          if instance_variables.include?("@#{attr}")
            val = instance_variable_get("@#{attr}")
            t.send("#{which}_#{attr}=", val)
          end
        end
        t.send("show_#{which}", @label) 
      end
    end

    # Guesses the extension of the label on the side of the plot.
    # The output is only garanteed when called with _t_ exactly in the
    # context that will be used for the plot. An external _scale_ factor
    # can be given in the case we have more information about the actual
    # overall scale. Of course, just multiplying the result by it does
    # give the same thing.
    #
    # The value returned is an array left,right, top, bottom
    # containing only zeroes but for the place where the axis
    # does extend.
    def extension(t, scale = 1)
      ret_val = [0,0,0,0]
      if @label
        ext = (@scale || tioga_value(t,:scale)) * 
          (1 + (@shift || tioga_value(t,:shift))) * 
        t.default_text_scale * t.default_font_size * scale
        # Set to 0 if label doesn't extend.
        ext = 0 if ext < 0
        case (@side || tioga_value(t,:side)) 
        when LEFT
          ret_val[0] = ext
        when BOTTOM
          ret_val[3] = ext
        when TOP
          ret_val[2] = ext
        when RIGHT
          ret_val[1] = ext
        end
      end
      debug "Axis #{self.inspect} extension #{ret_val.inspect}"
      return ret_val
    end
  
  
    # Returns the value of the _what_ attribute
    # corresponding to the value of which.
    def tioga_value(t, what)
      return t.send("#{@which}_#{what}")
    end
  end

  # A tick label is basically the same thing as a label
  class TickLabels < Label

    # Extension currently does not work really well.
    def extension(t)
      return [0,0,0,0]
    end

    def show(t)
      for attr in Attributes
        if instance_variables.include?("@#{attr}")
          val = instance_variable_get("@#{attr}")
          t.send("#{which}_#{attr}=", val)
        end
      end
    end

    undef :color=
    def color=(*a)
      error "There is no attribute color for tick labels"
    end

    
  end

  # A class that holds the information about how to plot the edges.
  # And the size they potentially take.
  #
  # TODO: make sure the TickLabels and EdgesAndAxes
  # talk to each other (for the #extension, essentially). They should be
  # connected from the SubPlot class.
  class EdgesAndAxes

    # This function parses an axes position specification
    # from the command-line.
    def self.parse_axis_position(str)
      case str
      when /xori?g(in)?/i, /x=0/i
        return Tioga::FigureConstants::AT_X_ORIGIN
      when /yori?g(in)?/i, /y=0/i
        return Tioga::FigureConstants::AT_Y_ORIGIN
      when /l(eft)?/i
        return Tioga::FigureConstants::LEFT
      when /r(ight)?/i
        return Tioga::FigureConstants::RIGHT
      when /t(op)?/i
        return Tioga::FigureConstants::TOP
      when /b(ottom)?/i
        return Tioga::FigureConstants::BOTTOM
      else
        raise "Unkown axis position: #{str}"
      end
    end

    # A class that represents an axis.
    class Axis

      include Log

      # Include the figure constants, else it gets really painful...
      include Tioga::FigureConstants

      # For an easy use of Boundaries:
      include Utils

    
      # The various attributes
      Attributes = [:line_width, :loc, :log_values, :type,
                   :ticks_inside, :ticks_outside, :visible]
      
      attr_accessor *Attributes

      # Which axis ?
      attr_accessor :which

      # The corresponding TickLabels
      attr_accessor :ticks

      # The color of perpendicular lines, or false if no lines are
      # to be drawn
      attr_accessor :lines_color

      # The line style of perpendicular lines
      attr_accessor :lines_style

      # The line width of perpendicular lines
      attr_accessor :lines_width

      # The coordinate conversions blocks
      attr_accessor :real_to_axis, :axis_to_real

      # _which_ can be :x or :y
      def initialize(which, ticks = nil)
        @which = which
        case which
        when :x
          @loc = BOTTOM
        when :y
          @loc = LEFT
        when String
          # We have a named position, in which case a little
          # more tweaking needs to be done.
          if which =~ /(\w+)([+-].*)?/
            # OK, $1 is the position, and $2 is a position shift
            loc = EdgesAndAxes::parse_axis_position($1)
            if $2
              dim = Dimension.new($2)
            else
              dim = false
            end
            if dim
              @loc = [loc, dim]
            else
              @loc = loc
            end
          end
        end
        @ticks = ticks || TickLabels.new(:none)

        # Background lines.
        @lines_color = false
        @lines_style = Styles::Dashes
        @lines_width = false
      end
      
      # Setup attributes for the axis.
      def setup(t, container)
        # We readjust the position of the axis if the X or Y origin is
        # out of bounds
        case @loc
        when AT_X_ORIGIN
          bounds = Boundaries.new(container.internal_get_boundaries)
          case bounds.where_x?(0)
          when :left
            @loc = LEFT
          when :right
            @loc = RIGHT
          end
        when AT_Y_ORIGIN
          bounds = Boundaries.new(container.internal_get_boundaries)
          case bounds.where_y?(0)
          when :top
            @loc = TOP
          when :bottom
            @loc = BOTTOM
          end
        end
        for attr in Attributes
          if instance_variables.include?("@#{attr}")
            # The visible attribute cannot be explicitly set to true.
            next if attr == :visible && @visible
            val = instance_variable_get("@#{attr}")
            t.send("#{which}axis_#{attr}=", val)
          end
        end
      end

      # Returns the boundaries of the axis corresponding to the
      # location of the axis
      def axis_coordinate_boundaries(t, container)
        loc = (@loc.is_a?(Array) ? @loc.first : @loc)
        case loc
        when TOP, BOTTOM, AT_Y_ORIGIN
          return [t.bounds_left, t.bounds_right]
        when LEFT, RIGHT, AT_X_ORIGIN
          return [t.bounds_bottom, t.bounds_top]
        end
      end

      # Prepares the major ticks/tick labels, in the case of a
      # non-linear axis. These are returned as a
      # [ [tick position], [tick values]]
      #
      # Returns nil if no tick changes are applicable
      def prepare_major_ticks(t, container)
        if @real_to_axis
          # First, the naive version:
          
          x1, x2 = axis_coordinate_boundaries(t, container)

          y1 = @real_to_axis.call(x1)
          y2 = @real_to_axis.call(x2)

          # Now the y1, y2 segment is our mapping.
          # We make sure y1 < y2:
          if y1 > y2 
            y1, y2 = y2, y1
          end

          # We get information about what the axis would have been
          loc = (@loc.is_a?(Array) ? @loc.first : @loc)
          info = t.axis_information(loc)

          # First: we divide relatively naively
          number = info['major'].size

          values = Utils::partition_nonlinear(@real_to_axis, @axis_to_real,
                                              x1, x2, number + 1)

          # Now, we have the positions (or nearly so), we need to
          # convert that back to original
          
          positions = values.map(&@axis_to_real)

          return [positions, values]
          
        else
          return nil
        end
      end

      # Gets the argument to give to FigureMaker#show_axis and
      # FigureMaker#axis_information.
      def axis_argument(t, container)
        if @which == :x or @which == :y
          return @loc
        end
        spec = {}
        if @loc.is_a? Array
          spec['location'] = @loc.first
        else
          spec['location'] = @loc
        end
        
        spec['type'] = @type if @type
        if val = prepare_major_ticks(t, container)
          spec['major_ticks'] = val[0]
          spec['labels'] = val[1].map { |v| v.to_s }
        end
        return spec
      end

      # Draws the background lines associated with this axis, if
      # applicable.
      #
      # This function requires Tioga SVN -r 482
      def draw_lines(t, container)

        return unless @lines_color
        # First, getting major ticks location from tioga
        info = t.axis_information(axis_argument(t, container))

        if info['vertical']
          x0 = t.bounds_left
          x1 = t.bounds_right
        else
          y0 = t.bounds_bottom
          y1 = t.bounds_top
        end

        t.context do
          t.stroke_color = @lines_color
          t.line_width = @lines_width || info['major_tick_width']
          t.line_type = @lines_style
          for val in info['major']
            if info['vertical']
              t.stroke_line(x0, val, x1, val)
            else
              t.stroke_line(val, y0, val, y1)
            end
          end
        end
      end

      def label_shift(t)
        shift = nil
        if @ticks
          shift = @ticks.shift
        end
        shift = t.send("#{@which}axis_numeric_label_shift") unless shift
        return shift
      end

      def label_scale(t)
        scale = nil
        if @ticks
          scale = @ticks.scale
        end
        scale = t.send("#{@which}axis_numeric_label_scale") unless scale
        return scale
      end
      
      def extension(t, scale = 1)
        ext = 0
        f = [0,0,0,0]
        # First, we estimate the size:
        case @type
        when AXIS_WITH_MAJOR_TICKS_AND_NUMERIC_LABELS, 
          AXIS_WITH_TICKS_AND_NUMERIC_LABELS
          ext = (1 + (label_shift(t))) * label_scale(t) *
            t.default_text_scale * t.default_font_size
        else
          ext = 0
        end
        loc_idx = Utils::location_index(@loc)
        if loc_idx  # That is, left,right, top, bottom
          f[loc_idx] = ext
        end
        return f
      end

      # Manually shows the given element.
      def show(t, container)
        t.show_axis(axis_argument(t, container))
      end
    
    end

    include Tioga::FigureConstants

    include Log

    Edges = [:left, :right, :top, :bottom]


    # A hash containing the edge type for each edge.
    attr_accessor :edge_specs

    # A hash containing the default visibility for
    # the edges.
    attr_reader :edge_visibility

    # The default values for the edges
    Defaults = {
      :left => AXIS_WITH_TICKS_AND_NUMERIC_LABELS,
      :right => AXIS_WITH_TICKS_ONLY,
      :top => AXIS_WITH_TICKS_ONLY,
      :bottom => AXIS_WITH_TICKS_AND_NUMERIC_LABELS
    }

    # The main axes
    attr_accessor :xaxis, :yaxis

    # Named axes:
    attr_accessor :axes


    def initialize(xticks = nil, yticks = nil)
      @edge_specs = Defaults.dup
      @xaxis = Axis.new(:x, xticks)
      @xaxis.type = AXIS_WITH_TICKS_AND_NUMERIC_LABELS
      @yaxis = Axis.new(:y, yticks)
      @yaxis.type = AXIS_WITH_TICKS_AND_NUMERIC_LABELS

      # A handy shortcut
      @axis = {:x => @xaxis, :y => @yaxis}

      @edge_visibility = {}

      @axes = {
        'xaxis' => @xaxis,
        'yaxis' => @yaxis,
      }
    end

    def set_edges(value, *which)
      which = Edges if which.empty?
      for edge in which
        @edge_specs[edge] = value
      end
    end

    # Disable the given edges, or all if no arguments
    def disable(*which)
      set_edges(AXIS_HIDDEN, *which)
    end

    # Show only one line for the given edges.
    def line_only(*which)
      set_edges(AXIS_LINE_ONLY, *which)
    end

    # Returns either #xaxis or  #yaxis depending on the value of
    # which.
    def axis(which)
      return @axis[which]
    end
    

    # Sets up the edges for the given plot. Should probably be called
    # just before the exit of the show_plot_with_legend block argument
    def setup(t, container)
      # Send axes informations.
      @xaxis.setup(t, container)
      @yaxis.setup(t, container)
      for edge in Edges
        t.send("#{edge}_edge_type=", @edge_specs[edge]) if 
          @edge_specs.key? edge
        # Can only disable edges
        if (@edge_visibility.key?(edge) && !@edge_visibility[edge])
          t.send("#{edge}_edge_visible=", false)
        end
          
      end
    end


    # Sets the style for both the axis and the edges of the given
    # side.
    def set_axis_and_edges_style(axis, style)
      case axis
      when :x
        @xaxis.type = style
        set_edges(style, :top,:bottom)
      when :y
        @yaxis.type = style
        set_edges(style, :left,:right)
      else                      # for :left, :right, :top, :bottom
        set_edges(style, axis)
      end
    end

    # This does not take ticks into account yet. No need !
    def extension(t, scale = 1)
      f = [0,0,0,0]
      Dimension.update_extensions(f, @xaxis.extension(t))
      Dimension.update_extensions(f, @yaxis.extension(t))
      debug "Edges and axis #{self.inspect} extension #{f.inspect}"
      return f
    end

    # Choosing which edges are visible
    def edge_visibility=(ar)
      if ar.is_a?(Array) and ar.size == 4
        4.times do |i|
          @edge_visibility[Edges[i]] = ar[i]
        end
      else
        if ar
          @edge_visibility = h.dup
        end
      end
    end

    # Sets the edge visibility for the given sides
    def set_edges_visibility(which, what)
      case which
      when :y
        @edge_visibility[:left] = @edge_visibility[:right] = what
      when :x
        @edge_visibility[:top] = @edge_visibility[:bottom] = what
      end
    end

    # Called from PlotStyle#show_background to display
    # axis lines.
    def show_axis_lines(t, container)
      for axis in @axes.keys.sort
        @axes[axis].draw_lines(t, container)
      end
    end

    # Draw all named non-standard axes
    def show_additional_axes(t, container)
      for name in @axes.keys - %w(xaxis yaxis)
        @axes[name].show(t, container)
      end
    end

  end
end
    
