# math.rb :This files provides the Math backend, which provides data based on
# mathematical formulas.
# Copyright (C) 2006 Vincent Fourmond

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA


require 'SciYAG/Backends/backend'
require 'Dobjects/Dvector'
require 'Dobjects/Function'

module SciYAG

  module Backends

    class MathBackend < Backend

      include Dobjects
      include Math
      
      describe 'math', 'Mathematical functions', <<EOD
This backend returns computations of mathematical formulas.
EOD

      param_accessor :samples, 'samples', "Samples",
      {:type => :integer}, "The number of points"
      param_accessor :x_range, 'xrange',  "X Range", 
      {:type => :float_range}, "X range (a:b)"
      param_accessor :t_range, 'trange',  "T Range", 
      {:type => :float_range}, "T range (a:b) (parametric plot)"

      param_accessor :log, 'log',  "Logarithmic scale", 
      {:type => :boolean}, "Space samples logarithmically"
      
      def initialize
        super()
        @samples = 100
        @x_range = -10.0..10.0
        @t_range = -10.0..10.0
        @log = false
      end

      # This is called by the architecture to get the data. It splits
      # the set name into func@range, reads the file if necessary and
      # calls get_data.
      def query_xy_data(set)
        if set =~ /(.*)@(.*)/
          @x_range = ParamRange.new($2)
          set = $1
        end
        return Function.new(*get_data(set))
      end

      
      def get_data(set)
        # Parametric plot with a : in the middle
        if set =~ /:/
          x_block, y_block = set.split(/:/).map { |s|
            eval "proc {|t| #{s}}"
          }

          t_v = make_dvector(@t_range, @samples)
          x_v = t_v.collect(&x_block)
          y_v = t_v.collect(&y_block)
        else
          y_block = eval "proc {|x| #{set} }"
          x_v = make_dvector(@x_range, @samples)
          y_v = x_v.collect(&y_block)
        end
        return [x_v,y_v]
      end
      
      # Turns a Range and a number of points into a Dvector
      def make_dvector(range, nb_points, log = @log)
        n = nb_points -1
        a = Dvector.new(nb_points) { |i| 
          i.to_f/(n.to_f)
        }
        # a is in [0:1] inclusive...
        if log
          delta = range.last/range.first
          # delta is positive necessarily
          a *= delta.log
          a.exp!
          a *= range.first
        else
          delta = range.last - range.first
          a *= delta
          a += range.first
        end
        return a
      end

    end
  end
end
