/*
 *  declarations for encryption/security mechanisms for cryptmount
 *  $Revision: 202 $, $Date: 2008-06-07 10:29:31 +0100 (Sat, 07 Jun 2008) $
 *  (C)Copyright 2005-2008, RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */


#ifndef _ARMOUR_H
#define _ARMOUR_H

struct keyinfo;
struct cment;
struct cm_testinfo;
struct km_pw_context;

typedef struct keymanager {
    const char *ident;

    int initialized;
    int (*init_algs)(void);
    int (*free_algs)(void);

    /* replace NULL algo-fields in keyinfo with suitable defaults: */
    void (*mk_default)(struct keyinfo *keyinfo);

    /* determine whether this keymanager can handle given key: */
    int (*is_compat)(const struct keyinfo *keyinfo, FILE *fp_key);

    /* check if a password is needed to access the key: */
    int (*needs_pw)(const struct keyinfo *keyinfo);

    /* extract encrypted key from file: */
    int (*get_key)(const char *ident, const struct keyinfo *keyinfo,
                    const struct km_pw_context *pw_ctxt,
                    unsigned char **key, int *keylen, FILE *fp_key);

    /* write encrypted key into file: */
    int (*put_key)(const char *ident, const struct keyinfo *keyinfo,
                    const struct km_pw_context *pw_ctxt,
                    const unsigned char *key, const int keylen, FILE *fp_key);

    /* message-digest computation (used as source of entropy): */
    void* (*md_prepare)(void);
    void (*md_block)(void *state, unsigned char *buff, size_t len);
    void (*md_final)(void *state, unsigned char **mdval, size_t *mdlen);
    void (*md_release)(void *state);

    /* linked-list scaffolding: */
    struct keymanager *next;

#ifdef TESTING
    void (*install_testctxt)(struct cm_testinfo *context);
    int (*run_tests)(void);
#endif
} keymanager_t;


const char **get_keymgr_list(void);
keymanager_t *get_keymanager(const struct keyinfo *keyinfo, FILE *fp_key);


keymanager_t *kmblti_gethandle(void);
keymanager_t *kmgcry_gethandle(void);
keymanager_t *init_keymanager(keymanager_t *km);
int free_keymanagers(void);

int is_keysafe(const struct keyinfo *keyinfo);
int get_key(const char *ident, const struct keyinfo *keyinfo,
            const struct km_pw_context *pw_ctxt,
            unsigned char **key, int *keylen);
int put_key(const char *ident, const struct keyinfo *keyinfo,
            const struct km_pw_context *pw_ctxt,
            const unsigned char *key, const int keylen, FILE *fp_key);
size_t mk_key_string(const unsigned char *key, const size_t keylen,
            char *buff);

int sycheck_directory(const char *dirname);
int sycheck_cmtab(const char *cmtab);
int sycheck_target(const struct cment *ent);

int cm_mutex_lock(void);
int cm_mutex_unlock(void);


/* loadable keymanager-module helper macro: */
#define KM_GETHANDLE(MOD_PATH, SYMBOL)                                      \
    void *dlptr=NULL, *table=NULL;                                          \
    dlptr = dlopen(MOD_PATH, RTLD_LAZY);                                    \
    if (dlptr == NULL) {                                                    \
        fprintf(stderr, "failed to open keymanager-module \"%s\" - %s\n",   \
                MOD_PATH, dlerror());                                        \
        return NULL;                                                        \
    }                                                                       \
    table = dlsym(dlptr, SYMBOL);                                           \
    return (keymanager_t*)table;


#endif  /* _ARMOUR_H */

/*
 *  (C)Copyright 2005-2008, RW Penney
 */
