/*
 *  methods for encryption/security mechanisms for cryptmount
 *  $Revision: 130 $, $Date: 2006-10-15 11:42:36 +0100 (Sun, 15 Oct 2006) $
 *  Copyright 2005-2006 RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exemption, permission is granted to link cryptmount
    with the OpenSSL project's "OpenSSL" library and distribute
    the linked code without invoking clause 2(b) of the GNU GPL version 2.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with cryptmount; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/times.h>

#include <linux/major.h>

#include "armour.h"
#include "cryptmount.h"
#include "tables.h"
#ifdef TESTING
#  include "cmtesting.h"
#endif


#ifdef TESTING
extern const char *argpassword[];
#endif

int km_get_passwd(const char *ident, char **passwd, int new, int verify);


/*
 *  ==== OpenSSL key-management routines ====
 */

#if HAVE_OPENSSL
#  include <openssl/bio.h>
#  include <openssl/err.h>
#  include <openssl/evp.h>
#  include <openssl/objects.h>


const char ssl_saltstr[]="Salted__";


int kmssl_get_algos(const keyinfo_t *keyinfo,
            const EVP_CIPHER **cipher, const EVP_MD **digest)
    /* get SSL algorithms for encoding key */
{   const char *ciphernameP,*digestnameP;

    *cipher = NULL;
    *digest = NULL;

    ciphernameP = (keyinfo->cipheralg != NULL
                        ? keyinfo->cipheralg : SN_bf_cbc);
    digestnameP = (keyinfo->digestalg != NULL
                        ? keyinfo->digestalg : SN_md5);

    *cipher = EVP_get_cipherbyname(ciphernameP);
    if (*cipher == NULL) {
        fprintf(stderr,
            _("couldn't find OpenSSL cipher \"%s\"\n"), ciphernameP);
        return ERR_BADALGORITHM;
    }

    *digest = EVP_get_digestbyname(digestnameP);
    if (*digest == NULL) {
        fprintf(stderr,
            _("couldn't find OpenSSL digest \"%s\"\n"), digestnameP);
        return ERR_BADALGORITHM;
    }

    return ERR_NOERROR;
}


#  ifdef TESTING

int kmssl_test_getalgos()
{   keyinfo_t keyinfo;
    const EVP_CIPHER *cipher=NULL;
    const EVP_MD *digest=NULL;

    CM_TEST_START("OpenSSL algorithm-identification");

    keyinfo.cipheralg = NULL;
    keyinfo.digestalg = NULL;
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmssl_get_algos(&keyinfo, &cipher, &digest));
    CM_ASSERT_DIFFERENT(NULL, cipher);
    CM_ASSERT_DIFFERENT(NULL, digest);

    keyinfo.cipheralg = "bf-ecb";
    keyinfo.digestalg = "sha1";
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmssl_get_algos(&keyinfo, &cipher, &digest));
    CM_ASSERT_EQUAL(EVP_bf_ecb(), cipher);
    CM_ASSERT_EQUAL(EVP_sha1(), digest);

    keyinfo.cipheralg = "aes-192-cbc";
    keyinfo.digestalg = "md5";
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmssl_get_algos(&keyinfo, &cipher, &digest));
    CM_ASSERT_EQUAL(EVP_aes_192_cbc(), cipher);
    CM_ASSERT_EQUAL(EVP_md5(), digest);

    CM_TEST_OK();
}

#  endif    /* TESTING */


int kmssl_init_algs()
    /* initialize cipher + hash algorithms */
{
    OpenSSL_add_all_ciphers();
    OpenSSL_add_all_digests();

    return 0;
}


int kmssl_free_algs()
    /* remove all cipher + hash algorithms */
{
    EVP_cleanup();

    return 0;
}


void kmssl_mk_default(keyinfo_t *keyinfo)
{
    if (keyinfo == NULL) return;

    if (keyinfo->digestalg == NULL) {
        keyinfo->digestalg = cm_strdup("md5");
    }

    if (keyinfo->cipheralg == NULL) {
        keyinfo->cipheralg = cm_strdup("bf-cbc");
    }
}


int kmssl_is_compat(const keyinfo_t *keyinfo)
{
    /* be prepared to act as default keymanager if format is unspecified: */
    return (keyinfo->format == NULL || strcmp(keyinfo->format, "openssl") == 0);
}


static int kmssl_get_key(const char *ident, const keyinfo_t *keyinfo,
            unsigned char **key, int *keylen, FILE *fp_key)
    /* extract key from openssl-encrypted file */
{   enum { BUFFSZ=8192 };
    BIO *keybio=NULL,*blkenc=NULL;
    char passwd[256],buff[BUFFSZ];
    unsigned char salt[PKCS5_SALT_LEN], cryptkey[EVP_MAX_KEY_LENGTH],
                    cryptiv[EVP_MAX_IV_LENGTH];
    int i,len,lmt,eflag=ERR_NOERROR;
    const EVP_CIPHER *cipher=NULL;
    const EVP_MD *digest=NULL;

    /*
     *  this routine is strongly influenced by apps/enc.c in openssl-0.9.8
     */

    *key = NULL; *keylen = 0;

    eflag = kmssl_get_algos(keyinfo, &cipher, &digest);
    if (eflag != ERR_NOERROR) goto bail_out;


    BIO_snprintf(buff, sizeof(buff),
                _("enter password for target \"%s\": "), ident);
#ifdef TESTING
    strncpy(passwd, (argpassword[0] != NULL ? argpassword[0] : ""),
            sizeof(passwd));
#else
    if (EVP_read_pw_string(passwd, sizeof(passwd), buff, 0) != 0) {
        fprintf(stderr, _("bad password for target \"%s\"\n"), ident);
        eflag = ERR_BADPASSWD;
        goto bail_out;
    }
#endif

    keybio = BIO_new_fp(fp_key, BIO_NOCLOSE);
    if (keybio == NULL) {
        fprintf(stderr,
                _("failed to connect to OpenSSL keyfile for target \"%s\"\n"),
                ident);
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    len = sizeof(ssl_saltstr) - 1;
    i = BIO_read(keybio, buff, len);
    if (i != len || strncmp(buff,ssl_saltstr,len) != 0) {
        fprintf(stderr, _("bad keyfile \"%s\"\n"), keyinfo->filename);
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    i = BIO_read(keybio, (void*)salt, sizeof(salt));
    EVP_BytesToKey(cipher, digest, salt,
            (unsigned char*)passwd, strlen(passwd), 1, cryptkey, cryptiv);
    OPENSSL_cleanse(passwd, sizeof(passwd));

    blkenc = BIO_new(BIO_f_cipher());
    BIO_set_cipher(blkenc, cipher, cryptkey, cryptiv, 0);

    BIO_push(blkenc, keybio);

    /* read and decrypt data from keyfile: */
    for (;;) {
        lmt = (keyinfo->maxlen > 0 && (*keylen + BUFFSZ) > keyinfo->maxlen
                ? keyinfo->maxlen - *keylen : BUFFSZ);
        len = BIO_read(blkenc, (void*)buff, lmt);
        if (len <= 0) break;

        /* copy new block of data onto end of current key: */
        *key = (unsigned char*)sec_realloc((void*)*key, (size_t)(*keylen+len));
        memcpy((void*)(*key + *keylen), (const void*)buff, len);
        *keylen += len;
    }

    if ((i = ERR_peek_last_error()) != 0) {
        fprintf(stderr, _("key-extraction failed [%x] for \"%s\"\n"),
                i, keyinfo->filename);
        eflag = ERR_BADDECRYPT;
    }

    if (blkenc != NULL) BIO_free_all(blkenc);

  bail_out:

    return eflag;
}


static int kmssl_put_key(const char *ident, const keyinfo_t *keyinfo,
            const unsigned char *key, const int keylen, FILE *fp_key)
    /* store key in openssl-encrypted file */
{   enum { BUFFSZ=8192 };
    BIO *keybio=NULL,*blkenc=NULL;
    char passwd[256],buff[BUFFSZ];
    unsigned char salt[PKCS5_SALT_LEN], cryptkey[EVP_MAX_KEY_LENGTH],
                    cryptiv[EVP_MAX_IV_LENGTH];
    int i,len,lmt,pos,eflag=ERR_NOERROR;
    const EVP_CIPHER *cipher=NULL;
    const EVP_MD *digest=NULL;

    eflag = kmssl_get_algos(keyinfo, &cipher, &digest);
    if (eflag != ERR_NOERROR) goto bail_out;

    BIO_snprintf(buff, sizeof(buff),
                _("enter new password for target \"%s\": "), ident);
#ifdef TESTING
    strncpy(passwd, (argpassword[1] != NULL ? argpassword[1] : ""),
            sizeof(passwd));
#else
    if (EVP_read_pw_string(passwd, sizeof(passwd), buff, 1) != 0) {
        eflag = ERR_BADPASSWD;
        goto bail_out;
    }
#endif

    keybio = BIO_new_fp(fp_key, BIO_NOCLOSE);
    if (keybio == NULL) {
        fprintf(stderr, _("failed to create file handle\n"));
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    len = sizeof(ssl_saltstr) - 1;
    if (BIO_write(keybio, ssl_saltstr, len) != len) {
        fprintf(stderr, _("bad keyfile \"%s\"\n"), keyinfo->filename);
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    get_randkey(salt, sizeof(salt));
    BIO_write(keybio, (const void*)salt, sizeof(salt));

    EVP_BytesToKey(cipher, digest, salt,
            (unsigned char*)passwd, strlen(passwd), 1, cryptkey, cryptiv);
    OPENSSL_cleanse(passwd, sizeof(passwd));

    blkenc = BIO_new(BIO_f_cipher());
    BIO_set_cipher(blkenc, cipher, cryptkey, cryptiv, 1);

    BIO_push(blkenc, keybio);

    /* encrypt and write data into keyfile: */
    for (pos=0; pos<keylen; ) {
        lmt = ((pos + BUFFSZ) > keylen ? (keylen - pos) : BUFFSZ);
        len = BIO_write(blkenc, (const void*)(key + pos), lmt);
        if (len <= 0) break;

        pos += len;
    }

    if ((i = BIO_flush(blkenc)) == 0) {
        fprintf(stderr, _("key-writing failed [%d] for \"%s\"\n"),
                i, keyinfo->filename);
        eflag = ERR_BADENCRYPT;
    }

  bail_out:

    if (blkenc != NULL) BIO_free_all(blkenc);

    return eflag;
}


void *kmssl_md_prepare()
{   EVP_MD_CTX *mdcontext;

    mdcontext = (EVP_MD_CTX*)malloc(sizeof(EVP_MD_CTX));
    EVP_MD_CTX_init(mdcontext);
    EVP_DigestInit(mdcontext, EVP_sha1());

    return (void*)mdcontext;
}


void kmssl_md_block(void *state, unsigned char *buff, unsigned len)
{   EVP_MD_CTX *mdcontext=(EVP_MD_CTX*)state;

    EVP_DigestUpdate(mdcontext, buff, len);
}


void kmssl_md_final(void *state, unsigned char **mdval, unsigned *mdlen)
{   EVP_MD_CTX *mdcontext=(EVP_MD_CTX*)state;

    *mdval = (unsigned char*)malloc((size_t)(EVP_MAX_MD_SIZE*sizeof(unsigned char)));
    EVP_DigestFinal(mdcontext, *mdval, mdlen);
}


void kmssl_md_release(void *state)
{   EVP_MD_CTX *mdcontext=(EVP_MD_CTX*)state;

    EVP_MD_CTX_cleanup(mdcontext);
    free((void*)mdcontext);
}


#  ifdef TESTING

int kmssl_test_hash()
{   void *mdcontext;
    unsigned char *mdval=NULL;
    unsigned mdlen,i,q;
    const char *str="random\n";
    const char *hash="5d64b71392b1e00a3ad893db02d381d58262c2d6";

    CM_TEST_START("OpenSSL hashing");

    mdcontext = kmssl_md_prepare();
    CM_ASSERT_DIFFERENT(NULL, mdcontext);
    kmssl_md_block(mdcontext, (unsigned char*)str, strlen(str));
    kmssl_md_final(mdcontext, &mdval, &mdlen);
    CM_ASSERT_DIFFERENT(NULL, mdval);
    CM_ASSERT_EQUAL(strlen(hash)/2, mdlen);
    for (i=0; i<mdlen; ++i) {
        sscanf(hash+2*i, "%2x", &q);
        CM_ASSERT_EQUAL(q, mdval[i]);
    }

    kmssl_md_release(mdcontext);

    CM_TEST_OK();
}

#  endif    /* TESTING */

#endif  /* HAVE_OPENSSL */



/*
 *  ==== libgcrypt key-management routines (EXPERIMENTAL) ====
 */

#if HAVE_LIBGCRYPT
#  include <gcrypt.h>

/*
 *  keyfile format is:
 *      char magic[7]="cm-gcry";
 *      char version;
 *      uint16{LSB-first} cipher_blocklength, keylength;
 *      char salt[kmgcry_saltlen];
 *      [block][block][block];
 *      (last block ends with uint32 xor-checksum of key
 *      (post-padded with zeros to next 4-byte boundary),
 *      post-padded with zeros to next cipher_blocklength boundary);
 */

const char kmgrcy_magstr[]="cm-gcry";
const char kmgcry_version = (char)0;
enum {
    kmgcry_maglen = 7,      /* = strlen(kmgcry_magstr) */
    kmgcry_saltlen = 12
};

static struct kmgcry_mode {
    char *name;
    unsigned mode; } kmgcry_modes[] = {
    { "ecb",    GCRY_CIPHER_MODE_ECB },
    { "cfb",    GCRY_CIPHER_MODE_CFB },
    { "cbc",    GCRY_CIPHER_MODE_CBC },
    { "ofb",    GCRY_CIPHER_MODE_OFB },
    { "cfb",    GCRY_CIPHER_MODE_CFB },
    { NULL, GCRY_CIPHER_MODE_NONE }
};


int kmgcry_get_algos(const keyinfo_t *keyinfo,
                    int *cipher, int *ciphermode, int *digest)
    /* get libgcrypt algorithms for encoding key */
{   char *buff=NULL,*pos;
    struct kmgcry_mode *cmd;
    const char *algstr="blowfish",*mdstr="cbc";

    if (keyinfo->cipheralg != NULL) {
        buff = (char*)malloc(strlen(keyinfo->cipheralg) + 1);
        strcpy(buff, keyinfo->cipheralg);
        algstr = buff;
        if ((pos = strchr(buff,'-')) != NULL) {
            *pos = '\0';
            mdstr = pos + 1;
            for (++pos; *pos!='\0'; ++pos) {
                *pos = tolower((int)*pos);
            }
        }
    }

    *cipher = gcry_cipher_map_name(algstr);
    for (cmd=kmgcry_modes; cmd->name!=NULL; ++cmd) {
        if (strcmp(cmd->name,mdstr) == 0) break;
    }
    *ciphermode = cmd->mode;

    *digest = gcry_md_map_name((keyinfo->digestalg != NULL
                                    ? keyinfo->digestalg : "md5"));

    if (buff != NULL) free((void*)buff);

    return (cipher != 0 && digest != 0 ? ERR_NOERROR : ERR_BADALGORITHM);
}


#  ifdef TESTING

int kmgcry_test_getalgos()
{   keyinfo_t keyinfo;
    int cipher=0, mode=0, digest=0;

    CM_TEST_START("libgcrypt algorithm-identification");

    keyinfo.cipheralg = NULL;
    keyinfo.digestalg = NULL;
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmgcry_get_algos(&keyinfo, &cipher, &mode, &digest));
    CM_ASSERT_DIFFERENT(0, cipher);
    CM_ASSERT_DIFFERENT(0, mode);
    CM_ASSERT_DIFFERENT(0, digest);

    keyinfo.cipheralg = "";
    keyinfo.digestalg = "";
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmgcry_get_algos(&keyinfo, &cipher, &mode, &digest));

    keyinfo.cipheralg = "twofish";
    keyinfo.digestalg = "sha1";
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmgcry_get_algos(&keyinfo, &cipher, &mode, &digest));
    CM_ASSERT_EQUAL(GCRY_CIPHER_TWOFISH, cipher);
    CM_ASSERT_EQUAL(GCRY_CIPHER_MODE_CBC, mode);
    CM_ASSERT_EQUAL(GCRY_MD_SHA1, digest);

    keyinfo.cipheralg = "CAST5-CFB";
    keyinfo.digestalg = "ripemd160";
    CM_ASSERT_EQUAL(ERR_NOERROR,
        kmgcry_get_algos(&keyinfo, &cipher, &mode, &digest));
    CM_ASSERT_EQUAL(GCRY_CIPHER_CAST5, cipher);
    CM_ASSERT_EQUAL(GCRY_CIPHER_MODE_CFB, mode);
    CM_ASSERT_EQUAL(GCRY_MD_RMD160, digest);

    CM_TEST_OK();
}

#  endif    /* TESTING */


int kmgcry_initcipher(int cipher, int ciphermode, int digest,
            const unsigned char *salt, const char *pass, size_t passlen,
            gcry_cipher_hd_t *hd)
{   gcry_md_hd_t md_hand;
    size_t ckeysz,cblksz,kpos,ivpos,pos,mdlen;
    unsigned char *ckey=NULL,*civ=NULL,*buff;
    int eflag=ERR_BADALGORITHM;

    if (gcry_cipher_open(hd, cipher, ciphermode, 0) != 0) {
        fprintf(stderr, "cannot open libgcrypt cipher[%d,%d]\n",
                cipher, ciphermode);
        goto bail_out;
    }

    (void)gcry_cipher_algo_info(cipher, GCRYCTL_GET_KEYLEN, NULL, &ckeysz);
    ckey = (unsigned char*)sec_realloc(ckey, ckeysz);
    gcry_cipher_algo_info(cipher, GCRYCTL_GET_BLKLEN, NULL, &cblksz);
    civ = (unsigned char*)sec_realloc(civ, cblksz);

    /* generate cipher key & iv by hashing password: */
    kpos = ivpos = 0;
    if (gcry_md_open(&md_hand, digest, 0) != 0) {
        fprintf(stderr, "cannot open libgcrypt digest[%d]\n", digest);
        goto bail_out;
    }
    mdlen = gcry_md_get_algo_dlen(digest);
    do {
        gcry_md_reset(md_hand);

        gcry_md_write(md_hand, (const void*)salt, (size_t)kmgcry_saltlen);
        gcry_md_write(md_hand, (const void*)pass, passlen);
        if (kpos > 0) {
            gcry_md_write(md_hand, (const void*)ckey, kpos); }
        if (ivpos > 0) {
            gcry_md_write(md_hand, (const void*)civ, ivpos); }

        buff = gcry_md_read(md_hand, digest);
        pos = 0;
        while (kpos < ckeysz && pos < mdlen) {
            ckey[kpos++] = buff[pos++]; }
        while (ivpos < cblksz && pos < mdlen) {
            civ[ivpos++] = buff[pos++]; }
    } while (kpos < ckeysz || ivpos < cblksz);
    gcry_md_close(md_hand);

    /* setup cipher initial state: */
    if (gcry_cipher_setkey(*hd, (void*)ckey, ckeysz) != 0
      || gcry_cipher_setiv(*hd, (void*)civ, cblksz) != 0) {
        fprintf(stderr, "failed to setup libgcrypt cipher iv[%d,%d]\n",
                (int)ckeysz, (int)cblksz);
        goto bail_out;
    }
    sec_free(ckey);
    sec_free(civ);

    eflag = ERR_NOERROR;

  bail_out:

    return eflag;
}


int kmgcry_init_algs()
{
    /* nothing needed */
    return 0;
}


int kmgcry_free_algs()
{
    /* nothing needed */
    return 0;
}


void kmgcry_mk_default(keyinfo_t *keyinfo)
{
    if (keyinfo == NULL) return;

    if (keyinfo->digestalg == NULL) {
        keyinfo->digestalg = cm_strdup("md5");
    }

    if (keyinfo->cipheralg == NULL) {
        keyinfo->cipheralg = cm_strdup("blowfish");
    }
}


int kmgcry_is_compat(const keyinfo_t *keyinfo)
{
    /* be prepared to act as default keymanager if format is unspecified: */
    return (keyinfo->format == NULL || strcmp(keyinfo->format, "libgcrypt") == 0);
}


static int kmgcry_get_key(const char *ident, const keyinfo_t *keyinfo,
            unsigned char **key, int *keylen, FILE *fp_key)
    /* extract key from unencrypted (plain) file */
{   gcry_cipher_hd_t chd;
    char *passwd=NULL;
    unsigned char salt[kmgcry_saltlen];
    enum { BUFFSZ=512 };
    unsigned char buff[BUFFSZ],*bptr;
    size_t cblksz;
    uint32_t chksum,*kptr;
    int cnt,cipher,ciphermode,digest,eflag=ERR_NOERROR;

    *key = NULL; *keylen = 0;

    eflag = kmgcry_get_algos(keyinfo, &cipher, &ciphermode, &digest);
    if (eflag != ERR_NOERROR) goto bail_out;
    gcry_cipher_algo_info(cipher, GCRYCTL_GET_BLKLEN, NULL, &cblksz);

#ifdef TESTING
    passwd = (char*)malloc((size_t)BUFFSZ);
    strncpy(passwd, (argpassword[1] != NULL ? argpassword[1] : ""),
            BUFFSZ);
#else
    if (km_get_passwd(ident, &passwd, 0, 0) != ERR_NOERROR) goto bail_out;
#endif

    /* read key header: */
    fread((void*)buff, (size_t)1, kmgcry_maglen, fp_key);
    if (strncmp((const char*)buff, kmgrcy_magstr, kmgcry_maglen) != 0) {
        fprintf(stderr, "bad keyfile format\n");
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    fread((void*)buff, (size_t)1, (size_t)1, fp_key);
    if (buff[0] != '\0') {
        fprintf(stderr, "bad keyfile version [%d]\n", (int)buff[0]);
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    fread((void*)buff, (size_t)4, (size_t)1, fp_key);
    if (buff[0] != (cblksz & 0xff) || buff[1] != ((cblksz & 0xff00) >> 8)) {
        fprintf(stderr, "mismatched cipher block size\n");
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    *keylen = (int)buff[2] | ((int)buff[3] << 8);

    /* read salt from keyfile: */
    fread((void*)salt, (size_t)1, sizeof(salt), fp_key);

    /* read encrypted key from keyfile: */
    kmgcry_initcipher(cipher, ciphermode, digest,
                    salt, passwd, strlen(passwd), &chd);
    cnt = ((*keylen + 8 + cblksz - 1) / cblksz);
    bptr = buff;
    while (cnt--) {
        fread((void*)bptr, cblksz, (size_t)1, fp_key);
        gcry_cipher_decrypt(chd, (void*)bptr, cblksz, NULL, 0);
        bptr += cblksz;
    }
    gcry_cipher_close(chd);

    /* verify checksum: */
    cnt = (*keylen + sizeof(chksum) - 1) / sizeof(chksum);
    chksum = 0;
    kptr = (uint32_t*)buff;
    while (cnt--) {
        chksum ^= *kptr;
        ++kptr;
    }
    if (chksum != *kptr) {
        fprintf(stderr, "checksum mismatch in keyfile (%x != %x)\n",
            chksum, *kptr);
        eflag = ERR_BADFILE;
    }

    if (keyinfo->maxlen > 0 && *keylen > keyinfo->maxlen) {
        *keylen = keyinfo->maxlen;
    }
    *key = (unsigned char*)sec_realloc((void*)*key, (size_t)*keylen);
    memcpy(*key, buff, *keylen);

    if (ferror(fp_key) != 0) {
        fprintf(stderr, _("key-extraction failed for \"%s\"\n"),
                keyinfo->filename);
        eflag = ERR_BADDECRYPT;
    }

  bail_out:

    return eflag;
}


static int kmgcry_put_key(const char *ident, const keyinfo_t *keyinfo,
            const unsigned char *key, const int keylen, FILE *fp_key)
    /* store key in unencrypted (plain) file */
{   gcry_cipher_hd_t chd;
    char *passwd=NULL;
    unsigned char salt[kmgcry_saltlen],*buff=NULL,*bptr;
    size_t idx,buffsz,cblksz;
    uint32_t chksum,*kptr;
    int cnt,cipher,ciphermode,digest,eflag=ERR_NOERROR;

    eflag = kmgcry_get_algos(keyinfo, &cipher, &ciphermode, &digest);
    if (eflag != ERR_NOERROR) goto bail_out;
    gcry_cipher_algo_info(cipher, GCRYCTL_GET_BLKLEN, NULL, &cblksz);

    buffsz = cblksz * ((keylen + 8 + cblksz - 1) / cblksz);
    buff = (unsigned char*)sec_realloc(buff, buffsz);

#ifdef TESTING
    passwd = sec_realloc((void*)passwd, (size_t)1024);
    strncpy(passwd, (argpassword[1] != NULL ? argpassword[1] : ""),
            1024);
#else
    if (km_get_passwd(ident, &passwd, 1, 1) != ERR_NOERROR) goto bail_out;
#endif

    /* write key header: */
    fwrite((const void*)kmgrcy_magstr, (size_t)1, kmgcry_maglen, fp_key);
    fwrite((const void*)&kmgcry_version, (size_t)1, (size_t)1, fp_key);
    buff[0] = (cblksz & 0xff); buff[1] = (cblksz & 0xff00) >> 8;
    buff[2] = (keylen & 0xff); buff[3] = (keylen & 0xff00) >> 8;
    fwrite((const void*)buff, (size_t)4, (size_t)1, fp_key);

    /* generate salt & record in keyfile: */
    get_randkey(salt, sizeof(salt));
    fwrite((const void*)salt, (size_t)1, sizeof(salt), fp_key);

    /* copy key & generate crude checksum: */
    memset(buff, 0, buffsz);
    memcpy(buff, key, keylen);
    cnt = (keylen + sizeof(chksum) - 1) / sizeof(chksum);
    chksum = 0;
    kptr = (uint32_t*)buff;
    for (idx=0; idx<(size_t)cnt; ++idx) {
        chksum ^= *kptr;
        ++kptr;
    }
    *kptr = chksum;     /* invariant with respect to byte-ordering */
    idx = (idx + 1) * sizeof(chksum);
    if (idx < buffsz) {
        get_randkey((buff + idx), (buffsz - idx));
    }

    /* write encrypted key into keyfile: */
    kmgcry_initcipher(cipher, ciphermode, digest,
                    salt, passwd, strlen(passwd), &chd);
    cnt = buffsz / cblksz;
    bptr = buff;
    while (cnt--) {
        gcry_cipher_encrypt(chd, (void*)bptr, cblksz, NULL, 0);
        fwrite((const void*)bptr, cblksz, (size_t)1, fp_key);
        bptr += cblksz;
    }
    gcry_cipher_close(chd);

  bail_out:
    if (buff != NULL) sec_free(buff);
    if (passwd != NULL) sec_free(passwd);

    return eflag;
}


void *kmgcry_md_prepare()
{   gcry_md_hd_t *handle;

    handle = (gcry_md_hd_t*)malloc(sizeof(gcry_md_hd_t));
    (void)gcry_md_open(handle, GCRY_MD_SHA1, 0);

    return (void*)handle;
}


void kmgcry_md_block(void *state, unsigned char *buff, unsigned len)
{   gcry_md_hd_t *handle=(gcry_md_hd_t*)state;

    gcry_md_write(*handle, (const void*)buff, (size_t)len);
}


void kmgcry_md_final(void *state, unsigned char **mdval, unsigned *mdlen)
{   gcry_md_hd_t *handle=(gcry_md_hd_t*)state;
    unsigned char *buff;
    int algo;

    gcry_md_final(*handle);
    algo = gcry_md_get_algo(*handle);
    *mdlen = gcry_md_get_algo_dlen(algo);

    buff = gcry_md_read(*handle, algo);
    *mdval = (unsigned char*)malloc((size_t)*mdlen);
    memcpy((void*)*mdval, (const void*)buff, (size_t)*mdlen);
}


void kmgcry_md_release(void *state)
{   gcry_md_hd_t *handle=(gcry_md_hd_t*)state;

    gcry_md_close(*handle);
    free((void*)handle);
}

#  ifdef TESTING

int kmgcry_test_hash()
{   void *mdcontext;
    unsigned char *mdval=NULL;
    unsigned mdlen,i,q;
    const char *str="noisy\n";
    const char *hash="7c1c9261fa774475ec1c0d887eaf00c19b0eb218";

    CM_TEST_START("libgcrypt hashing");

    mdcontext = kmgcry_md_prepare();
    CM_ASSERT_DIFFERENT(NULL, mdcontext);
    kmgcry_md_block(mdcontext, (unsigned char*)str, strlen(str));
    kmgcry_md_final(mdcontext, &mdval, &mdlen);
    CM_ASSERT_DIFFERENT(NULL, mdval);
    CM_ASSERT_EQUAL(strlen(hash)/2, mdlen);
    for (i=0; i<mdlen; ++i) {
        sscanf(hash+2*i, "%2x", &q);
        CM_ASSERT_EQUAL(q, mdval[i]);
    }

    kmgcry_md_release(mdcontext);

    CM_TEST_OK();
}

#  endif    /* TESTING */

#endif  /* HAVE_LIBGCRYPT */



/*
 *  ==== raw key-management routines ====
 */

int kmraw_init_algs()
{
    return 0;
}


int kmraw_free_algs()
{
    return 0;
}


void kmraw_mk_default(keyinfo_t *keyinfo)
{
    if (keyinfo == NULL) return;

    if (keyinfo->digestalg == NULL) {
        keyinfo->digestalg = cm_strdup("none");
    }

    if (keyinfo->cipheralg == NULL) {
        keyinfo->cipheralg = cm_strdup("none");
    }
}


int kmraw_is_compat(const keyinfo_t *keyinfo)
{
    return (keyinfo != NULL && strcmp(keyinfo->cipheralg, "none") == 0);
}


static int kmraw_get_key(const char *ident, const keyinfo_t *keyinfo,
            unsigned char **key, int *keylen, FILE *fp_key)
    /* extract key from unencrypted (plain) file */
{   enum { BUFFSZ=512 };
    char buff[BUFFSZ];
    int len,lmt,eflag=ERR_NOERROR;

    *key = NULL; *keylen = 0;

    if (fp_key == NULL) {
        eflag = ERR_BADFILE;
        goto bail_out;
    }


    /* read data directly from keyfile: */
    for (;;) {
        lmt = (keyinfo->maxlen > 0 && (*keylen + BUFFSZ) > keyinfo->maxlen
                ? keyinfo->maxlen - *keylen : BUFFSZ);
        len = fread((void*)buff, (size_t)1, (size_t)lmt, fp_key);
        if (len <= 0) break;

        /* copy new block of data onto end of current key: */
        *key = (unsigned char*)sec_realloc((void*)*key, (size_t)(*keylen+len));
        memcpy((void*)(*key + *keylen), (const void*)buff, len);
        *keylen += len;
    }

    if (ferror(fp_key) != 0) {
        fprintf(stderr, _("key-extraction failed for \"%s\"\n"),
                keyinfo->filename);
        eflag = ERR_BADDECRYPT;
    }

  bail_out:

    return eflag;
}


static int kmraw_put_key(const char *ident, const keyinfo_t *keyinfo,
            const unsigned char *key, const int keylen, FILE *fp_key)
    /* store key in unencrypted (plain) file */
{   int eflag=ERR_NOERROR;

    /* write data directly into keyfile: */
    if (fwrite((const void*)key, (size_t)keylen, (size_t)1, fp_key) != 1
      || ferror(fp_key) != 0) {
        fprintf(stderr, _("key-writing failed for \"%s\"\n"),
                keyinfo->filename);
        eflag = ERR_BADENCRYPT;
    }

    return eflag;
}


void *kmraw_md_prepare()
{   unsigned long *mdcontext;

    /* you're in trouble if this ends up being your default hashing alg */
    mdcontext = (unsigned long*)malloc(sizeof(unsigned long));
    *mdcontext = 0xc0895ca3;
    *mdcontext = (*mdcontext << 16) ^ 0x306a5498;

    return (void*)mdcontext;
}


void kmraw_md_block(void *state, unsigned char *buff, unsigned len)
{   unsigned long *mdcontext=(unsigned long*)state;

    while (len > 0) {
        *mdcontext = (*mdcontext << 1) ^ ((*mdcontext >> 5) & 1);
        *mdcontext ^= (unsigned long)*buff;
        ++buff; --len;
    }
}


void kmraw_md_final(void *state, unsigned char **mdval, unsigned *mdlen)
{
    *mdval = (unsigned char*)malloc(sizeof(unsigned long));
    memcpy((void*)*mdval, (const void*)state, sizeof(unsigned long));
    *mdlen = sizeof(unsigned long);
}


void kmraw_md_release(void *state)
{
    free(state);
}




/*
 *  ==== abstract key-management interfaces ====
 */

/* This table contains abstract interfaces to cryptographic functions.
   Early entries in the list represent good quality encryption functions.
   The list is terminated by an entry whose .ident field is NULL */
static keymanager_t keymgrs[] = {
#if HAVE_OPENSSL
    { "openssl", 0,     kmssl_init_algs, kmssl_free_algs,
                        kmssl_mk_default, kmssl_is_compat,
                        kmssl_get_key, kmssl_put_key,
                        kmssl_md_prepare, kmssl_md_block,
                        kmssl_md_final, kmssl_md_release },
#endif
#if HAVE_LIBGCRYPT
    { "libgcrypt", 0,   kmgcry_init_algs, kmgcry_free_algs,
                        kmgcry_mk_default, kmgcry_is_compat,
                        kmgcry_get_key, kmgcry_put_key,
                        kmgcry_md_prepare, kmgcry_md_block,
                        kmgcry_md_final, kmgcry_md_release },
#endif
    { "raw", 0,         kmraw_init_algs, kmraw_free_algs,
                        kmraw_mk_default, kmraw_is_compat,
                        kmraw_get_key, kmraw_put_key,
                        kmraw_md_prepare, kmraw_md_block,
                        kmraw_md_final, kmraw_md_release },
    { NULL, 0,   NULL, NULL,   NULL, NULL,
                NULL, NULL,   NULL, NULL, NULL, NULL }
};


keymanager_t *get_keymanager(const struct keyinfo *keyinfo)
    /* search for keymanager that supports given key */
{   keymanager_t *km;
    int is_raw;

    if (keyinfo == NULL) return NULL;

    is_raw = (strcmp(keyinfo->cipheralg, "none") == 0);

    for (km=keymgrs; km->ident!=NULL; ++km) {
        if (is_raw && strcmp(km->ident, "raw") != 0) continue;

        if (km->is_compat(keyinfo)) {
            return init_keymanager(km);
        }
    }

    return NULL;
}


keymanager_t *init_keymanager(keymanager_t *km)
{
    if (km != NULL) {
        if (!km->initialized) {
            km->init_algs();    /* FIXME - check return status */
            km->initialized = 1;
        }
    }

    return km;
}


int free_keymanagers()
{   keymanager_t *km;

    for (km=keymgrs; km->ident!=NULL; ++km) {
        if (km->initialized) {
            km->free_algs();
            km->initialized = 0;
        }
    }

    return 0;
}


int km_get_passwd(const char *ident, char **passwd, int new, int verify)
    /* read password from terminal, possibly asking for confirmation */
{   enum { BUFFSZ=2048 };
    char *tmppass,buff[BUFFSZ];
    size_t plen;
    int eflag=ERR_NOERROR;

    snprintf(buff, sizeof(buff),
            (new ? _("enter new password for target \"%s\": ")
                : _("enter password for target \"%s\": ")),
            ident);

    tmppass = getpass(buff);
    plen = strlen(tmppass);
    *passwd = (char*)sec_realloc((void*)*passwd, (plen + 1));
    strcpy(*passwd, tmppass);
    mem_cleanse((unsigned char*)tmppass, plen + 1);

    if (verify) {
        snprintf(buff, sizeof(buff), _("confirm password: "));
        tmppass = getpass(buff);
        plen = strlen(tmppass);
        if (strcmp(*passwd, tmppass) != 0) {
            fprintf(stderr, _("password mismatch\n"));
            sec_free(*passwd);
            *passwd = NULL;
            eflag = ERR_BADPASSWD;
        }
        mem_cleanse((unsigned char*)tmppass, plen);
    }

    return eflag;
}


int get_key(const char *ident, const keyinfo_t *keyinfo,
            unsigned char **key, int *keylen)
{   keymanager_t *km;
    FILE *fp=NULL;
    int eflag=ERR_NOERROR;

    if (keyinfo == NULL || keyinfo->filename == NULL) {
        fprintf(stderr, _("missing key-file for target \"%s\"\n"), ident);
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    if ((km = get_keymanager(keyinfo)) != NULL) {
        if ((fp=fopen(keyinfo->filename, "rb")) == NULL) {
            fprintf(stderr,
                _("failed to open keyfile \"%s\" for target \"%s\"\n"),
                keyinfo->filename, ident);
            eflag = ERR_BADFILE;
            goto bail_out;
        }
        eflag = km->get_key(ident, keyinfo, key, keylen, fp);
    } else {
        fprintf(stderr,
                _("unrecognized key format for target \"%s\"\n"), ident);
        eflag = ERR_BADKEYFORMAT;
    }

  bail_out:
    if (fp != NULL) fclose(fp);

    return eflag;
}


int put_key(const char *ident, const keyinfo_t *keyinfo,
            const unsigned char *key, const int keylen, FILE *fp_key)
{   keymanager_t *km;

    if (fp_key == NULL) {
        fprintf(stderr, _("missing output stream for target \"%s\"\n"), ident);
        return ERR_BADFILE;
    }

    if ((km = get_keymanager(keyinfo)) != NULL) {
        return km->put_key(ident, keyinfo, key, keylen, fp_key);
    }

    return ERR_BADKEYFORMAT;
}


int is_keysafe(const keyinfo_t *keyinfo)
    /* check whether key is stored in encrypted form */
{
    return (keyinfo != NULL && keyinfo->cipheralg != NULL && strcmp(keyinfo->cipheralg, "none") != 0);
}


int get_randkey(unsigned char *buff, unsigned len)
    /* generate random string of bytes */
{   struct rnddev {
        const char *name;
        unsigned devmaj, devmin;
        FILE *fp; };
    struct rnddev devs[] = {
        { "/dev/hwrng",     MISC_MAJOR, 183,    NULL },
        { "/dev/random",    MEM_MAJOR, 8,       NULL },
        { "/dev/urandom",   MEM_MAJOR, 9,       NULL },
        { NULL, 0, 0, NULL } };
    keymanager_t *kmgr=NULL;
    void *mdcontext=NULL;
    unsigned char *mdval,*devbuff=NULL;
    unsigned pos,sz,mdlen;
    int i,cnt,eflag=ERR_NOERROR;
    struct stat sbuff;
    pid_t pid;
    struct tms tbuff;
    clock_t clk;
    static unsigned seed=17;

    /* assume first entry in key-manager list is decent quality: */
    kmgr = init_keymanager(keymgrs);

    /* try to find good-quality random-number devices: */
    for (i=0,cnt=0; devs[i].name!=NULL; ++i) {
        if (stat(devs[i].name, &sbuff) != 0) continue;
        if ((unsigned)major(sbuff.st_rdev) != devs[i].devmaj
          || (unsigned)minor(sbuff.st_rdev) != devs[i].devmin) continue;
        if ((devs[i].fp = fopen(devs[i].name,"rb")) != NULL) ++cnt;
    }
    if (cnt == 0) {
        fprintf(stderr, _("no random-number devices found\n"));
        eflag = WRN_LOWENTROPY;
    }
    devbuff = (unsigned char*)sec_realloc(NULL, (size_t)len);


    pid = getpid();

    /*  combine multiple sources of entropy
        (should still be effective if subset aren't viable): */
    pos = 0;
    while (pos < len) {
        mdcontext = kmgr->md_prepare();

        /* fold-in entropy from random-number devices: */
        for (i=0; devs[i].name!=NULL; ++i) {
            if (devs[i].fp == NULL) continue;
            (void)fread((void*)devbuff, 1, (size_t)len, devs[i].fp);
            kmgr->md_block(mdcontext, devbuff, len);
        }

        /* fold-in existing bytes from key: */
        if (pos > 0) {
            kmgr->md_block(mdcontext, buff, pos);
        }

        /* fold-in some (weak) sources of entropy: */
        kmgr->md_block(mdcontext, (unsigned char*)&pid, sizeof(pid));
        clk = times(&tbuff);
        kmgr->md_block(mdcontext, (unsigned char*)&clk, sizeof(clk));
        kmgr->md_block(mdcontext, (unsigned char*)&seed, sizeof(seed));
        kmgr->md_block(mdcontext, (unsigned char*)&tbuff, sizeof(tbuff));

        kmgr->md_final(mdcontext, &mdval, &mdlen);

        sz = ((pos + mdlen) > len ? (len - pos) : mdlen);
        memcpy((void*)(buff + pos),(const void*)mdval, (size_t)sz);

        pos += sz;
        seed = seed * 20 + 1;

        kmgr->md_release(mdcontext);
    }

    sec_free((void*)devbuff);
    for (i=0; devs[i].name!=NULL; ++i) {
        if (devs[i].fp != NULL) fclose(devs[i].fp);
    }

    return eflag;
}


#ifdef TESTING

int km_test_keyrw()
    /* test key read-writing (creation/extraction) */
{   enum { MAXKEY=256 };
    keymanager_t *km;
    keyinfo_t keyinfo;
    int i,keylen,keylen1;
    char str[256];
    unsigned char key0[MAXKEY],*key1=NULL;
    FILE *fp;

    CM_TEST_START("key read-write");

    for (km=keymgrs; km->ident!=NULL; ++km) {
        keyinfo.format = NULL;
        keyinfo.filename = "NOWHERE";
        keyinfo.digestalg = NULL;
        keyinfo.cipheralg = NULL;
        keyinfo.maxlen = -1;
        km->mk_default(&keyinfo);
        km->init_algs();

        for (keylen=1; keylen<=MAXKEY; keylen<<=2) {
            sprintf(str, "key read-write, %s, keylen=%d",
                km->ident, keylen);
            CM_TEST_IDENT(str);

            /* generate (low-entropy) key: */
            for (i=0; i<keylen; ++i) {
                key0[i] = (i * 0x9d) ^ ((keylen * (unsigned long)km) % 253);
            }

            /* write key to file: */
            fp = tmpfile();
            if (fp == NULL) CM_TEST_ABORT();
            i = km->put_key("TEST-OUT", &keyinfo, key0, keylen, fp);
            CM_ASSERT_EQUAL(ERR_NOERROR, i);

            key1 = NULL; keylen1 = -keylen;

            /* try reading key back from file: */
            rewind(fp);
            i = km->get_key("TEST-IN", &keyinfo, &key1, &keylen1, fp);
            CM_ASSERT_EQUAL(ERR_NOERROR, i);
            CM_ASSERT_EQUAL(keylen, keylen1);
            CM_ASSERT_DIFFERENT(key0, key1);
            CM_ASSERT_DIFFERENT(NULL, key1);
            for (i=0; i<keylen; ++i) {
                CM_ASSERT_EQUAL(key0[i], key1[i]);
            }

            fclose(fp);
            sec_free(key1);
        }

        km->free_algs();
        free((void*)keyinfo.cipheralg);
        free((void*)keyinfo.digestalg);
    }


    CM_TEST_OK();
}

#endif  /* TESTING */



/*
 *  ==== miscellaneous routines ====
 */

void *sec_realloc(void *ptr, size_t size)
    /* slightly more secure version of realloc() */
{   unsigned char *addr=NULL;

    addr = (unsigned char*)malloc(size + sizeof(size_t));
    if (addr == NULL) {
        fprintf(stderr, _("unable to allocated memory\n"));
        abort();
        return NULL;
    }

    /* prepend usable memory chunk with record of size of chunk: */
    *((size_t*)addr) = size;
    addr += sizeof(size_t);

    if (ptr != NULL) {
        unsigned char *old;
        unsigned i,oldsz;

        /* copy (usable) part of old memory block into new: */
        old = (unsigned char*)ptr;
        oldsz = *((size_t*)(old - sizeof(size_t)));
        if (oldsz > size) oldsz = size;
        for (i=0; i<oldsz; ++i) addr[i] = old[i];

        /* dispose of old memory block: */
        sec_free(ptr);
    }

    return (void*)addr;
}


void mem_cleanse(unsigned char *addr, size_t sz)
    /* overwrite memory with (weak) pseudo-random numbers */
{   size_t i;
    static unsigned long salt=0x917c;

    salt ^= (unsigned long)addr;

    for (i=0; i<sz; ++i) {
        addr[i] = (i % 21) ^ (salt % 221);
        salt += 4;
    }
}


void sec_free(void *ptr)
    /* slightly more secure version of free() */
{   unsigned char *addr;
    size_t sz;

    if (ptr == NULL) return;

    addr = (unsigned char*)ptr;
    sz = *((size_t*)(addr - sizeof(size_t)));

    mem_cleanse(addr, sz);

    free((void*)(addr - sizeof(size_t)));
}


int mk_key_string(const unsigned char *key, const int keylen, char *buff)
    /* create text version of crypto key */
{   int i;

    for (i=0; i<keylen; ++i) {
        sprintf(buff+2*i, "%02x", key[i]);
    }

    return (2 * keylen);
}


int sycheck_cmtab(const char *cmtab)
    /* check that permissions on ${sysconfdir}/cryptmount/cmtab are sensible */
{   struct stat sfile,sdir;
    char *dirname=NULL;
    int pos,eflag=ERR_NOERROR;

    /* extract directory name from cmtab filename: */
    pos = strlen(cmtab);
    dirname = (char*)malloc((size_t)(pos + 1));
    for ( ; pos>0 && cmtab[pos-1] != '/'; --pos) dirname[pos] = '\0';
    while (--pos >= 0) dirname[pos] = cmtab[pos];

    if (stat(cmtab,&sfile) != 0 || stat(dirname,&sdir) != 0) {
        fprintf(stderr, "cannot open \"%s\" or \"%s\"\n", cmtab, dirname);
        eflag = ERR_INSECURE;
        goto bail_out;
    }

    /* check file/directory ownerships: */
    if (sfile.st_uid != (uid_t)0 || sdir.st_uid != (uid_t)0) {
        fprintf(stderr, "\"%s\" and \"%s\" must be owned by root\n",
                cmtab, dirname);
        eflag = ERR_INSECURE;
        goto bail_out;
    }

    /* check that file isn't globally writable: */
    if (!S_ISREG(sfile.st_mode) || (sfile.st_mode & S_IWOTH) != 0) {
        fprintf(stderr, "lax permissions on \"%s\"\n", cmtab);
        eflag = ERR_INSECURE;
        goto bail_out;
    }

    /* check that directory isn't globally writable: */
    if (!S_ISDIR(sdir.st_mode) || (sdir.st_mode & S_IWOTH) != 0) {
        fprintf(stderr, "lax permissions on \"%s\"\n", dirname);
        eflag = ERR_INSECURE;
        goto bail_out;
    }


  bail_out:

    if (dirname != NULL) free((void*)dirname);

    return eflag;
}


static int sy_path(const char *path)
    /* check whether pathname is considered secure */
{
    if (path == NULL) return ERR_NOERROR;
    if (path[0] == '/') return ERR_NOERROR;

    return ERR_INSECURE;
}

int sycheck_target(const cment_t *cment)
    /* check that paths within target-specification are sensible */
{   int eflag=ERR_NOERROR;

    if (cment == NULL) return 0;

    eflag |= sy_path(cment->dev);
    eflag |= sy_path(cment->dir);
    eflag |= sy_path(cment->key.filename);

    if (eflag != ERR_NOERROR) {
        fprintf(stderr, "specification for target \"%s\" contains non-absolute pathname\n", cment->ident);
    }

    return eflag;
}



/*
 *  ==== mutex-locking on configuration directory ====
 */

int cm_mutex_lock()
    /* try to acquire lock on configuration directory (via symlink marker) */
{   char *fname=NULL,ident[64];
    int tries=3,eflag=1;

    (void)cm_path(&fname, "_lock_");
    sprintf(ident, "%u-%u", (unsigned)getpid(), (unsigned)getuid());

    while (tries-->0) {
        if (symlink(ident, fname) == 0) {
            /* lock acquired */
            eflag = 0; break;
        } else {
            if (errno == EEXIST) sleep(1);  /* try again later */
            else break;     /* failed to make link for more peculiar reason */
        }
    }

    if (eflag != 0) {
        fprintf(stderr, "failed to create lock-file \"%s\"\n", fname);
    }

    free((void*)fname);

    return eflag;
}

int cm_mutex_unlock()
    /* release lock on configuration directory */
{   char *fname=NULL;
    struct stat sbuff;
    int eflag=0;

    (void)cm_path(&fname, "_lock_");

    if (lstat(fname, &sbuff) != 0
      || !S_ISLNK(sbuff.st_mode)
      || unlink(fname) != 0) {
        fprintf(stderr, "failed to remove lock-file \"%s\"\n", fname);
        eflag = 1;
    }

    free((void*)fname);

    return eflag;
}


/*
 *  (C)Copyright 2005-2006, RW Penney
 */
