// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// history.cpp
//
// History:
//  05-06-2001 - created
////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "history.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : HistEvent::HistEvent
// DESCRIPTION: Create a new event for the turn history.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

HistEvent::HistEvent( void ) {
  type = HIST_DONE;
}

////////////////////////////////////////////////////////////////////////
// NAME       : HistEvent::HistEvent
// DESCRIPTION: Load an event from a file.
// PARAMETERS : file - SDL_RWops file descriptor
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

HistEvent::HistEvent( SDL_RWops *file ) {
  SDL_RWread( file, &type, 1, 1 );
  for ( int i = 0; i < 4; i++ ) data[i] = SDL_ReadLE16( file );
}

////////////////////////////////////////////////////////////////////////
// NAME       : HistEvent::Save
// DESCRIPTION: Save an event to a file.
// PARAMETERS : file - SDL_RWops file descriptor
// RETURNS    : 0 on success, non-0 otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int HistEvent::Save( SDL_RWops *file ) const {
  SDL_RWwrite( file, &type, 1, 1 );
  for ( int i = 0; i < 4; i++ ) SDL_WriteLE16( file, data[i] );
  return 0;
}


////////////////////////////////////////////////////////////////////////
// NAME       : History::History
// DESCRIPTION: Load a History from a file.
// PARAMETERS : file - open SDL_RWops descriptor
//              num  - number of recorded events in file
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

History::History( SDL_RWops *file, unsigned short num ) {
  unsigned short num_units, i;

  num_units = SDL_ReadLE16( file );

  for ( i = 0; i < num; i++ ) {
    events.AddTail( new HistEvent( file ) );
  }

  for ( i = 0; i < num_units; i++ ) {
    Unit *u = Gam->LoadUnit( file );
    if ( u ) units.AddTail( u );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : History::History
// DESCRIPTION: Create a new (empty) History from the current game.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

History::History( void ) {
  // create the unit copies
  Unit *dummy, *u = static_cast<Unit *>( Gam->UnitsList()->Head() );
  while ( u ) {
    dummy = new Unit( *u );
    dummy->SetFlags( U_DUMMY );
    units.AddTail( dummy );
    u = static_cast<Unit *>( u->Next() );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : History::~History
// DESCRIPTION: If the History is destroyed, all HIST_TILE events must
//              still be executed so that we get the original map back.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

History::~History( void ) {
  while ( !events.IsEmpty() ) {
    HistEvent *he = static_cast<HistEvent *>( events.RemHead() );
    if ( he->type == HIST_TILE )
      Gam->SetHexType( he->data[1], he->data[2], he->data[0] );
    delete he;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : History::Save
// DESCRIPTION: Save a history to file.
// PARAMETERS : file - open SDL_RWops file descriptor
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int History::Save( SDL_RWops *file ) const {
  unsigned short num = events.CountNodes();
  SDL_WriteLE16( file, num );
  if ( num == 0 ) return 0;

  num = units.CountNodes();
  SDL_WriteLE16( file, num );

  // save events
  HistEvent *he = static_cast<HistEvent *>( events.Head() );
  while ( he ) {
    he->Save( file );
    he = static_cast<HistEvent *>( he->Next() );
  }

  // save units
  Unit *u = static_cast<Unit *>( units.Head() );
  while ( u ) {
    u->Save( file );
    u = static_cast<Unit *>( u->Next() );
  }
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : History::Replay
// DESCRIPTION: Play all events that have been stored during the last
//              turn.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void History::Replay( void ) {
  if ( events.IsEmpty() ) return;

  List units_bak;
  Gam->BeginReplay( units, units_bak );

  View *view = Gam->GetView();
  unsigned short count = events.CountNodes();
  bool abort = false;

  // undo the map tile changes which have been recorded this turn
  HistEvent *he = static_cast<HistEvent *>( events.Head() );
  while ( he && (he->type == HIST_TILE_INTERNAL) ) {
    Gam->SetHexType( he->data[1], he->data[2], he->data[0] );
    he->type = HIST_DONE;
    he = static_cast<HistEvent *>( he->Next() );
    --count;
  }

  Gam->Draw();
  Gam->Show();
  ProgressWindow *progwin = new ProgressWindow( 5,
                  Gam->Height() - view->SmallFont()->Height() - 30,
                  Gam->Width() / 2, view->SmallFont()->Height() + 20,
                  0, count, 0, true, view );

  count = 0;
  Unit *lastunit = NULL;
  Point lastpos = { -1, -1 };

  while ( he && !abort ) {
    Unit *u;
    progwin->progress->SetLevel( ++count );

    switch ( he->type ) {
      case HIST_MOVE:              // move unit
        u = Gam->GetUnitByID( he->data[0] );
        if ( u ) {
          if ( u != lastunit ) {
            SDL_Delay( DEFAULT_DELAY );

            if ( lastunit && lastunit->IsTransport() &&
                 !Gam->IsBuilding( lastpos.x, lastpos.y ) ) {
              // we must adjust the position of carried units manually because
              // those pseudo-transports can't handle this themselves
              for ( Unit *carry = static_cast<Unit *>( Gam->UnitsList()->Head() );
                    carry; carry = static_cast<Unit *>( carry->Next() ) ) {
                if ( carry->Position() == lastpos )
                  carry->SetPosition( lastunit->Position().x, lastunit->Position().y );
              }
            }
            lastunit = u;
            lastpos = u->Position();
          }

          if ( !Gam->HexOnScreen( u->Position().x, u->Position().y ) ) {
            Gam->CenterOnHex( u->Position().x, u->Position().y );
            view->Refresh();
          }
          Gam->MoveUnit( u, (Direction)he->data[1] );
        }
        break;
      case HIST_ATTACK:
        u = Gam->GetUnitByID( he->data[0] );
        if ( u ) {
          if ( u != lastunit ) SDL_Delay( DEFAULT_DELAY );
          if ( !Gam->HexOnScreen( u->Position().x, u->Position().y ) ||
               !Gam->HexOnScreen( he->data[1], he->data[2] ) ) {
            Gam->CenterOnHex( (u->Position().x + he->data[1]) / 2,
                              (u->Position().y + he->data[2]) / 2 );
            view->Refresh();
          }
          Gam->SetCursorImage( IMG_CURSOR_HIGHLIGHT );
          Gam->SetCursor( u->Position().x, u->Position().y, false );
          Gam->SetCursorImage( IMG_CURSOR_ATTACK );
          Gam->SetCursor( he->data[1], he->data[2], false );
          Gam->CursorOff();
          SDL_Delay( DEFAULT_DELAY );
          Gam->UpdateHex( u->Position().x, u->Position().y );
          Gam->UpdateHex( he->data[1], he->data[2] );
          lastunit = u;
        }
        break;
      case HIST_COMBAT: {
        Combat *cmb = new Combat( Gam->GetUnitByID( he->data[0] ),
                                  Gam->GetUnitByID( he->data[1] ) );
        cmb->amod = (char)he->data[3];
        cmb->dmod = (char)he->data[2];
        CombatWindow *cwin = cmb->Resolve( Gam, Gam->GetView() );
        if ( cwin ) {
          cwin->EventLoop();
          view->CloseWindow( cwin );
        }
        break; }
      case HIST_TILE:
        Gam->SetHexType( he->data[1], he->data[2], he->data[0] );
        Gam->UpdateHex( he->data[1], he->data[2] );
        SDL_Delay( DEFAULT_DELAY );
        break;
      default:
        break;
    }
    he->type = HIST_DONE;
    he = static_cast<HistEvent *>( he->Next() );
    abort = progwin->Aborted();
  }

  Gam->EndReplay( units_bak );
  if ( !abort ) SDL_Delay( DEFAULT_DELAY * 2 );
  view->CloseWindow( progwin );
}

////////////////////////////////////////////////////////////////////////
// NAME       : History::RecordMoveEvent
// DESCRIPTION: Record the movement of a unit from one hex to another,
//              neighboring hex.
// PARAMETERS : u   - unit to be moved
//              dir - direction to move the unit in
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int History::RecordMoveEvent( const Unit &u, Direction dir ) {
  HistEvent *he = new HistEvent;

  he->type = HIST_MOVE;
  he->data[0] = u.ID();
  he->data[1] = dir;

  events.AddTail( he );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : History::RecordAttackEvent
// DESCRIPTION: Record an attack being inititated.
// PARAMETERS : u      - unit to inititate the attack
//              target - unit being attacked
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int History::RecordAttackEvent( const Unit &u, const Unit &target ) {
  HistEvent *he = new HistEvent;

  he->type = HIST_ATTACK;
  he->data[0] = u.ID();
  he->data[1] = target.Position().x;
  he->data[2] = target.Position().y;

  events.AddTail( he );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : History::RecordCombatEvent
// DESCRIPTION: Record the results of a combat.
// PARAMETERS : combat - combat data
//              loss1  - attackers' casualties
//              loss2  - defenders' casualties
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int History::RecordCombatEvent( const Combat &combat, unsigned char loss1,
                                unsigned char loss2 ) {
  HistEvent *he = new HistEvent;

  he->type = HIST_COMBAT;
  he->data[0] = combat.c_att->ID();
  he->data[1] = combat.c_def->ID();
  he->data[2] = loss1;
  he->data[3] = loss2;

  events.AddTail( he );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : History::RecordTileEvent
// DESCRIPTION: Record a modification of the map. This event is special
//              in so far as two separate events are created for each
//              event. One allows for the initial map to be created from
//              the map at the end of the turn, the other represents the
//              actual map change.
// PARAMETERS : tile - tile type being set
//              old  - tile type being replaced
//              dx   - x hex being modified
//              dy   - y hex being modified
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int History::RecordTileEvent( unsigned short tile, unsigned short old,
                              short dx, short dy ) {
  HistEvent *he = new HistEvent;

  // first record the actual change
  he->type = HIST_TILE;
  he->data[0] = tile;
  he->data[1] = dx;
  he->data[2] = dy;
  events.AddTail( he );

  // now create the map state inititalzer event and put it at the
  // head of the list
  he = new HistEvent;
  he->type = HIST_TILE_INTERNAL;
  he->data[0] = old;
  he->data[1] = dx;
  he->data[2] = dy;
  events.AddHead( he );
  return 0;
}

