# configlet.py -- Python module for configlet 

# $Progeny: configlet.py,v 1.118 2002/03/26 06:30:15 jlicquia Exp $

# Copyright (C) 2001  Progeny Linux Systems, Inc.
# AUTHORS: Jeff Licquia <jlicquia@progeny.com>
#          John Daily <jdaily@progeny.com>
#          Eric Gillespie, Jr. <epg@progeny.com>

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License, version 2,  as
# published by the Free Software Foundation.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Main Python interface to configlet infrastructure.
"""

import copy
import sys
import os
import re
import string
import time
import types
import traceback

configlet_classes = {}
configlet_global_namespaces = {}
started_configlets = {}
last_configlet_class = None
gnome_loaded = 0
cfglet_dir = ""
privileged_runner = None

# Internal-use-only functions.

# Not only does this make the debconf-communicate code cleaner and
# simpler, it protects us from a bug discovered during PGI development
# wherein debconf-communicate doesn't always react well to more than
# one line of input.  See its man page synopsis for a hint.
def _debconf_communicate(package, s):
    debug("DebConf", "Saying %s with a %s accent." % (s, package))

##     pipe = os.popen("debconf-communicate %s" % (package,), "w")
##     pipe.write(s)
##     return pipe.close()
    
    cmdline = "echo '%s' | debconf-communicate %s" % (s, package)
    privileged_run(cmdline)

def load_gnome():
    global gnome_loaded
    global gtk
    global libglade

    if not gnome_loaded:
        #from gtk import *
        #from libglade import *
        import gtk
        import libglade

        # Do we need this?  It would be really nice if we didn't.
        #from gnome.ui import *

        gnome_loaded = 1

def make_temp_dir():
    success = 0
    for index in range(1, 4):
        temp_path = "/tmp/configlet.%d.%d.%d" \
                    % (os.getpid(), time.time(), index)
        try:
            os.mkdir(temp_path, 0700)
        except:
            continue

        success = 1
        break

    if not success:
        raise RuntimeError, "could not create temporary directory"

    return temp_path

def check_attributes(attributes):
    if not isinstance(attributes["packages"], types.ListType):
        raise TypeError, "packages must be a list"
    if not isinstance(attributes["display_title"], types.StringType):
        raise TypeError, "display_title must be a string"
    if not isinstance(attributes["description"], types.StringType):
        raise TypeError, "description must be a string"

    try:
        if not isinstance(attributes["page_names"], types.ListType):
            raise TypeError, "page_names must be a list"
    except KeyError:
        attributes["page_names"] = ["mainwidget"]

    if len(attributes["page_names"]) > 1:
        # Only required for multi-page configlets
        if not isinstance(attributes["page_display_titles"], types.DictType):
            raise TypeError, "page_display_titles must be a dictionary"
        for page_name in attributes["page_names"]:
            if not attributes["page_display_titles"].has_key(page_name):
                raise TypeError, "no title supplied for page %s" % page_name

    if not attributes.has_key("priority"):
        attributes["priority"] = 50
    else:
        if not isinstance(attributes["priority"], types.IntType):
            raise TypeError, "priority must be an integer"

    try:
        if not isinstance(attributes["shared_questions"], types.ListType):
            raise TypeError, "shared_questions must be a list"
    except:
        attributes["shared_questions"] = []

def load_configlet(path):
    """Loads a configlet's class into the system
    
    Takes one argument: the directory that contains main.py
    """
    global cfglet_dir
    global configlet_global_namespaces

    if not os.path.exists(path):
        raise RuntimeError, "path %s does not exist" % (path,)
    script_path = "%s/main.py" % (path,)
    if not os.path.exists(script_path):
        raise RuntimeError, "path %s does not contain a configlet" % (path,)

    cfglet_dir = path
    configlet_global_namespaces[path] = {}
    execfile(script_path, configlet_global_namespaces[path])

def print_traceback():
    tb_info = sys.exc_info()
    traceback.print_exception(tb_info[0], tb_info[1], tb_info[2])

# End of internal-use-only functions.

# Log/debug functions.

def debug(identifier, message):
    if os.environ.has_key("DEBUG"):
        warn(identifier, message)

def warn(identifier, message):
    sys.stderr.write(identifier + ": " + message + "\n")

def error(identifier, message):
    warn(identifier, message)

# Register a new configlet.
    
def register_configlet(configlet, attributes):
    "Registers the configlet class with the configlet system."
    
    global configlet_classes
    global last_configlet_class
    global last_configlet_attributes
    global cfglet_dir

    if not attributes.has_key("dir"):
        attributes["dir"] = cfglet_dir

    if attributes.has_key("name"):
        cfglet_name = attributes["name"]
    else:
        cfglet_name = attributes["dir"]

    try:
        check_attributes(attributes)
    except TypeError, e:
        sys.stderr.write("%s: attribute error: %s\n" % (cfglet_name, e))
        return
    except KeyError, e:
        sys.stderr.write("%s: attribute not defined: %s\n" % (cfglet_name, e))
        return

    last_configlet_class = configlet
    last_configlet_attributes = attributes

    dirpath = attributes["dir"]
    configlet_classes[dirpath] = (configlet, attributes)

# Start a configlet.

def start_configlet(path):
    """Return the configlet object found at the particular path.

    This ensures that only one of each configlet will be started.
    It takes the path where the configlet's files are stored.  If
    the configlet has already been started, it will return the
    current instance.  If not, it will start it.  If there's a
    problems (such as the attributes being invalid), it will return
    None.
    """

    try:
        if path not in configlet_classes.keys():
            load_configlet(path)
        if path in configlet_classes.keys() and \
           configlet_classes[path] is not None and \
           path not in started_configlets.keys():
            started_configlets[path] = \
                configlet_classes[path][0](configlet_classes[path][1])
    except:
        configlet_classes[path] = None
        started_configlets[path] = None
        warn("start_configlet", "configlet in path %s is bad" % (path,))
        if os.environ.has_key("DEBUG"):
            print_traceback()

    return started_configlets[path]

# Privileged run functions.

def set_privileged_runner(runner):
    global privileged_runner

    privileged_runner = runner

def privileged_run(command):
    """Run a command with privilege.

    This runs a command using the current privileged runner class,
    or just tries to run the command if no privileged runner class
    has been set.
    """

    global privileged_runner

    if privileged_runner:
        privileged_runner.run(command)
    else:
        debug("privileged_run", "no privileged runner set - running command")
        debug("privileged_run", "  %s" % (command,))
        retval = os.system(command)
        if retval:
            raise RuntimeError, "Command %s failed to execute, status %X" \
                  % (command, retval)

# Simple default privileged run class

class SimplePrivilegedRunner:
    """Default privileged runner implementation.

    This class defines an interface for running a command with
    privilege on the system.  The implementation of this class is
    kept to a bare minimum; it assumes that we already have privilege.
    Don't use this class directly; instead, use its interface and
    set the privileged runner in your config group to your class.
    """

    def run(self, command):
        "Run the given command with privilege."

        debug("SimplePrivilegedRunner", "running %s" % (command,))
        retval = os.system(command)
        if retval:
            raise RuntimeError, "Command %s failed to execute, status %X" \
                  % (command, retval)

# Privileged run interface based on gnome-sudo

class GnomeSudoPrivilegedRunner:
    "This is a privileged runner class that uses gnome-sudo."

    def run(self, command):
        "Run the given command with privilege."

        debug("GnomeSudoPrivilegedRunner", "running %s" % (command,))

        if os.getuid() == 0:
            retval = os.system(command)
        else:
            retval = os.system("gnome-sudo -- %s" % (command,))

        if retval:
            raise RuntimeError, "Command %s failed, exit status %X" \
                  % (command, retval)

# Privileged run interface based on xsu.

class XsuPrivilegedRunner:
    "This is a privileged runner class that uses xsu."

    def run(self, command):
        "Run the given command with privilege."

        message = "Please type the root password to gain access."
        title = "Configlet Root Access"
        
        debug("XsuPrivilegedRunner", "running %s" % (command,))

        if os.getuid() == 0:
            retval = os.system(command)
        else:
            temp_path = make_temp_dir()

            script = open("%s/script" % (temp_path,), "w")
            script.write("%s\n" % (command,))
            script.close()

            retval = os.system("xsu -e -u root -m '%s' -t '%s' -c '/bin/sh %s/script'"
                               % (message, title, temp_path))

            os.unlink("%s/script" % (temp_path,))
            os.rmdir(temp_path)

        if retval:
            raise RuntimeError, "Command %s failed, exit status %X" \
                  % (command, retval)

class DebConf:
    """Debconf interface base class.

    This is the base class that defines how the configlet system
    will talk to debconf.  Most people won't need to mess with this
    directly; it's only needed in order to keep the configlets working
    as debconf changes.  The base config group class defines a
    function for getting the right DebConf instance for the system
    you're on."""

    def __init__(self, pkglist):
        self.package_list = pkglist
        #self.set_privileged_runner(SimplePrivilegedRunner())

    def get(self):
        """Get all debconf data.

        Return the entire debconf database, formatted in the standard
        manner used by the configlets.  The default implementation
        does nothing; this needs to be overriden by a derived class.
        """

        return []

    def set(self, debconf_info):
        """Set debconf data.

        Set all debconf values according to the passed-in list of
        debconf information.  The default implementation does nothing;
        this needs to be overriden by a derived class.
        """

        pass

    def commit(self):
        "Commit all previously set() debconf information."

        os.environ["DEBIAN_FRONTEND"] = "noninteractive"
        for package in self.package_list:
            if package == "shared":
                continue

            postinst = "/var/lib/dpkg/info/%s.postinst" % (package,)
            if os.path.exists(postinst):
                print postinst
                self.privileged_run(postinst + " configure")

        self.package_list = []

    def set_privileged_runner(self, runner):
        "This sets the privileged runner class used for running debconf."

        set_privileged_runner(runner)

    def privileged_run(self, command):
        """Run a command with privilege.

        This runs a command using the current privileged runner class,
        or just tries to run the command if no privileged runner class
        has been set.
        """

        privileged_run(command)

class DebConf_1_0(DebConf):
    """DebConf implementation for debconf 1.0.

    This is the implementation of the DebConf class that works with
    debconf 1.0, the currently shipping version of debconf in Debian.
    """

    def __init__(self, pkglist):
        DebConf.__init__(self, pkglist)
        
        # Open the debconf config file and figure out which database
        # is the main config db.
        self.config_db = ""
        debconf_config = open("/etc/debconf.conf")
        line = debconf_config.readline()
        while line:
            results = re.match(r"\s*Config:\s+(\S+)", line)
            if results:
                self.config_db = results.group(1)
                break
            line = debconf_config.readline()
        debconf_config.close()

        # If we couldn't find a line in debconf's config, assume
        # a default value.
        if not self.config_db:
            self.config_db = "configdb"
    
    def get(self):
        temp_path = make_temp_dir()

        privileged_run("debconf-copydb %s configletdb -c Name:configletdb -c Driver:File -c Filename:%s/configletdb"
                            % (self.config_db, temp_path))
        privileged_run("chmod 644 %s/configletdb" % (temp_path,))

        configletdb = open("%s/configletdb" % (temp_path,))
        line = configletdb.readline()
        itemhash = {}
        debconf_data = []
        while line:
            if re.match(r'\S+', line):
                results = re.match(r'\s*(\w+):\s*(.+)', line)
                if results:
                    itemhash[results.group(1)] = results.group(2)
            else:
                if len(itemhash.keys()):
                    if not itemhash.has_key("Value"):
                        itemhash["Value"] = '""'
                    debconf_str = "%s %s %s" % \
                                  (itemhash["Template"], itemhash["Name"],
                                   itemhash["Value"])
                    debconf_data.append(debconf_str)
                    itemhash = {}

            line = configletdb.readline()

        configletdb.close()
        os.unlink("%s/configletdb" % (temp_path,))
        os.rmdir(temp_path)

        return debconf_data

    def set(self, debconf_info):
        # Setting debconf info is a bit tricky.  There's a bug in
        # debconf-communicate where it doesn't always run right if you
        # pipe lots of debconf commands to it at once, so for safety's
        # sake, we need to send one command at a time.  However, we
        # need to run it with privilege, and the privileged runner
        # interface doesn't provide a way to do pipes (yet).  So, we
        # run a little shell script on the other side of the
        # privileged runner, piping a temp file into it.

        command_str = "while read line; do pkg=`echo $line | cut -d: -f1`; dc=`echo $line | cut -d: -f2-`; echo $dc | debconf-communicate $pkg %s; done"

        if os.environ.has_key("DEBUG"):
            command_str = command_str % ("",)
        else:
            command_str = command_str % ("> /dev/null",)

        temp_path = make_temp_dir()
        temp_file = open("%s/debconf_output" % (temp_path,), "w")

        for line in debconf_info:
            fields = string.split(line, None, 2)
            (template, name) = fields[:2]
            package = string.split(template, "/")[0]
            debug("DebConf", "variable %s" % (name,))
            if package == "shared":
                package = ""
                debug("DebConf", " shared template")
            else:
                debug("DebConf", " package is %s" % (package,))

            if len(fields) == 3:
                value = fields[2]
                
                if value[:6] == "FLAGS:":
                    debug("DebConf", " flags found (value is %s)" % (value,))
                    flag_fields = string.split(value[6:], None, 1)
                    flags = string.split(flag_fields[0], ",")
                    debug("DebConf", " flags are %s" % flag_fields[0])
                    if len(flag_fields) > 1:
                        value = flag_fields[1]
                    else:
                        value = ""
                else:
                    flags = []
            else:
                value = ""

            if value:
                if template != name:
                    debug("DebConf", " registering from %s" % (template,))
                    temp_file.write("%s:register %s %s\n"
                                    % (package, template, name))
                debug("DebConf", " setting to %s" % (value,))
                temp_file.write("%s:set %s %s\n" % (package, name, value))
                temp_file.write("%s:fset %s seen true\n" % (package, name))
                for flag in flags:
                    temp_file.write("%s:fset %s %s true\n"
                                    % (package, name, flag))
            else:
                debug("DebConf", " unsetting")
                temp_file.write("%s:reset %s\n" % (package, name))
                for flag in flags:
                    temp_file.write("%s:fset %s %s true\n"
                                    % (package, name, flag))

        temp_file.close()

        privileged_run("/bin/sh -c 'cat %s/debconf_output | (%s)'"
                       % (temp_path, command_str))

        os.unlink("%s/debconf_output" % (temp_path,))
        os.rmdir(temp_path)
        
def get_debconf(pkglist):
    """Instantiate the proper DebConf object.

    This should be called by the config group; it's recommended to
    just ask your config group for the debconf object if you need it.
    """

    return DebConf_1_0(pkglist)

class Configlet:
    """Main base class for configlets.

    All configlets must inherit from this class.  It provides various
    support functionality for hooking the configlet into the system,
    as well as helper functions and default values.

    The procedure for writing a configlet involves inheriting from
    this class, and is documented separately.
    """

    def __init__(self, attr):
        "Initialize the configlet located in the provided directory."
        
        self.attributes = attr
        self.dirpath = self.attributes["dir"]
        self.can_assign_packages = 1

        path_components = [dir]
        if not self.attributes.has_key("name"):
            while self.attributes["name"] == "":
                path_components = os.path.split(path_components[0])
                self.attributes["name"] = path_components[1]

        self.priv_runner = GnomeSudoPrivilegedRunner()

    ###################################################################
    # Methods the inheriting configlet should (or may) override

    def load_debconf(self, data):
        """Harvest appropriate info from the debconf database.

        This function takes the debconf data passed and extracts
        appropriate items for use in setting the default values
        for the configlet.  It should be overriden by derived
        classes to grab the appropriate data.
        """

        pass

    def gnome_setup(self):
        """Gnome-related initialization.

        Any graphical configuration (e.g. initializing widgets) should be
        done here, rather than in __init__.

        If subclasses override this method, they must either replicate the
        functionality or call it themselves.
        """

        load_gnome()

        self.wtree = libglade.GladeXML("%s/main.glade" % (self.dirpath,))

        self.widgets = {}
        for pagename in self.get_page_names():
            self.widgets[pagename] = self.wtree.get_widget(pagename)


    def validate(self):
        """Validate the current information in the configlet.

        This function validates all pages in the configlet at once.
        Single-page configlets can override this function or
        validate_page(); multi-page configlets should not override
        this.
        """

        for page in self.get_page_names():
            if not self.validate_page(page):
                return 0

        return 1

    def validate_page(self, pagename):
        """Validate a single page of information in the configlet.

        Make sure that the data displayed in the given configlet
        page is valid.  Multi-page configlets should override this
        method to implement real validity tests if possible.
        """

        return 1

    def on_gnome_close(self):
        """Notify the configlet that the front end is closing it.

        This function gets called by the front end when the window
        containing the configlet is closing.  The idea is that the
        configlet gets a chance to grab the current values from the
        UI before it goes away.
        """

        pass

    def report_debconf(self):
        """Returns the debconf information as modified by the user.

        Once the configlet has been displayed, it should return the
        appropriate changes to the debconf values as the return value
        of this function.

        One form of additional information may be returned in the
        list: Debconf flags.  This is done by passing a string of the
        form "package/template package/question FLAGS:flag1,flag2"
        where flag1 and flag2 are the flags that should be set to
        true.  Any number of flags may be set in this manner.
        """
        
        return []


    ###################################################################
    # Methods that the front-end will invoke

    def get_name(self):
        "Returns the name of the configlet."

        return self.attributes["name"]

    def get_path(self):
        "Returns the path to the directory containing the configlet."

        return self.dirpath

    def get_widget(self, pagename = "mainwidget"):
        "Returns the GTK widget for the particular page."
        
        if pagename not in self.attributes["page_names"]:
            raise IndexError, "page name %s not found" % (pagename,)
        return self.widgets[pagename]

    def get_display_title(self):
        "Returns the configlet display title."
        
        return self.attributes["display_title"]

    def get_description(self):
        "Returns the description of the configlet."
        
        return self.attributes["description"]

    def get_priority(self):
        "Returns the configlet priority."
        
        return self.attributes["priority"]

    def get_page_names(self):
        "Returns the list of page names."
        
        return tuple(self.attributes["page_names"])

    def get_page_display_title(self, name):
        "Returns the display title associated with a page."

        if name not in self.attributes["page_names"]:
            raise IndexError, "page name %s not found" % (name,)
        if self.attributes["page_display_titles"].has_key(name):
            return self.attributes["page_display_titles"][name]
        else:
            return ""

    def get_packages(self):
        "Returns the list of packages the configlet configures."
        
        return self.attributes["packages"]

    def get_shared_questions(self):
        "Returns the list of shared questions the configlet answers."

        return self.attributes["shared_questions"]

    ###################################################################
    # Logging and debugging functions.

    def debug(self, message):
        debug(self.get_name(), message)

    def warn(self, message):
        warn(self.get_name(), message)

    def error(self, message):
        error(self.get_name(), message)

class BasicConfigGroup:
    """Base class for a configlet container class.

    This class can be used by front ends to keep track of the
    configlets.  It contains helper functions for doing operations
    on all of the configlets we know about.  It should be usable as
    it is, or one can inherit from it to change or add functionality.
    """
    
    def __init__(self, dir = "/usr/share/configlets"):
        """Initialize the config group.

        Sets things up, and load the configlets in the (optional)
        directory specified.
        """

        self.configlets = []
        self.broken_configlets = []
        pkglist = []
        for configlet_dir in os.listdir(dir):
            curdir = "%s/%s" % (dir, configlet_dir)
            if not os.path.isdir(curdir) or \
               not os.path.exists("%s/main.glade" % (curdir,)) or \
               not os.path.exists("%s/main.py" % (curdir,)):
                continue

            configlet = start_configlet(curdir)
            if configlet is not None:
                self.configlets.append(configlet)
                pkglist.extend(configlet.get_packages())

        # Priority sort the configlet list.
        self.configlets.sort(
            lambda x,y: cmp(x.get_priority(), y.get_priority()))

        # Start the debconf subsystem
        self.dc = get_debconf(pkglist)

    def __len__(self):
        "Return the number of configlets in the group."

        return len(self.configlets)

    def __getitem__(self, key):
        "Retrieve an individual configlet by index."
        
        return self.configlets[key]

    def __getslice__(self, i, j):
        "Retrieve several adjacent configlets in the group."
        
        return self.configlets[i:j]

    def validate(self):
        "Validate all configlets in the group."
        
        for configlet in self.configlets:
            if not configlet.validate():
                return 0
        return 1

    def on_gnome_close(self):
        "Notify all configlets that they will be closing."
        
        for configlet in self.configlets:
            if configlet not in self.broken_configlets:
                try:
                    configlet.on_gnome_close()
                except:
                    self.broken_configlets.append(configlet)
                    self.warn("on_gnome_close failed for %s" % configlet.get_name())
                    if os.environ.has_key("DEBUG"):
                        print_traceback()

    def gnome_setup(self):
        "Cause all configlets to set themselves up."

        for configlet in self.configlets:
            if configlet not in self.broken_configlets:
                try:
                    configlet.gnome_setup()
                except:
                    self.broken_configlets.append(configlet)
                    self.warn("gnome_setup failed for %s" % configlet.get_name())
                    if os.environ.has_key("DEBUG"):
                        print_traceback()

    def load_debconf(self, data):
        "Pass the debconf data to all configlets for use."
        
        for configlet in self.configlets:
            if configlet not in self.broken_configlets:
                try:
                    configlet.load_debconf(data)
                except:
                    self.broken_configlets.append(configlet)
                    self.warn("load_debconf failed for %s" % configlet.get_name())
                    if os.environ.has_key("DEBUG"):
                        print_traceback()

    def report_debconf(self):
        "Retrieve all changed debconf information from all configlets."
        
        results = []
        for configlet in self.configlets:
            if configlet not in self.broken_configlets:
                try:
                    results = results + configlet.report_debconf()
                except:
                    self.broken_configlets.append(configlet)
                    self.warn("report_debconf failed for %s" % configlet.get_name())
                    if os.environ.has_key("DEBUG"):
                        print_traceback()

        return results

    def get_debconf(self):
        "Get the debconf interface object in use by the group."
        
        return self.dc

    def load_all_debconf(self):
        "Load debconf info and send it to the configlets in one shot."
        
        self.load_debconf(self.dc.get())

    def save_and_commit_all_debconf(self):
        "Get debconf info from the configlets, save it, and commit."
        
        self.dc.set(self.report_debconf())
        self.dc.commit()

    def set_privileged_runner(self, runner):
        "Set privileged runner class for the debconf object."

        set_privileged_runner(runner)

    def privileged_run(self, command):
        "Run a command with privilege using the current runner class."

        privileged_run(command)

    def debug(self, message):
        debug("BasicConfigGroup", message)

    def warn(self, message):
        warn("BasicConfigGroup", message)

    def error(self, message):
        error("BasicConfigGroup", message)
