/*
	$Id: keyboard_directfb.cpp,v 1.3 2001/12/22 17:03:33 sphair Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

#include "Core/precomp.h"

#ifdef USE_DIRECTFB

#include <directfb.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <iostream>

#include "keyboard_directfb.h"

#include <API/Display/Input/input.h>
#include <API/Display/Input/inputaxis.h>
#include <API/Display/Input/key.h>
#include <API/Display/Input/inputbutton.h>
#include <API/Display/Input/inputcursor.h>
#include <API/Display/Input/inputhat.h>
#include <API/Display/Input/keyboard.h>
#include <API/Core/System/error.h>
#include "API/Core/System/cl_assert.h"

#include "Core/System/Generic/string_asm.h"

/******************
  Keyboard
******************/

CL_DirectFBKeyboard::CL_DirectFBKeyboard (unsigned int id)
{
  DFBResult ret;

  ret = DirectFBCreate (&dfb);
  if (ret)
    {
      DirectFBError ("CL_DirectFBKeyboard - DirectFBCreate", ret);
      throw new CL_Error ("CL_DirectFBKeyboard - DirectFBCreate failed");
    }

  ret = dfb->GetInputDevice (dfb, id, &keyboard);
  if (ret)
    {
      DirectFBError ("CL_DirectFBKeyboard - GetInputDevice", ret);
      dfb->Release (dfb);
      throw new CL_Error ("CL_DirectFBKeyboard - GetInputDevice failed");
    }

  ret = keyboard->CreateInputBuffer (keyboard, &keybuffer);
  if (ret)
    {
      DirectFBError ("CL_DirectFBKeyboard - CreateInputBuffer", ret);
      keyboard->Release (keyboard);
      dfb->Release (dfb);
      throw new CL_Error ("CL_DirectFBKeyboard - CreateInputBuffer failed");
    }

  buttons = new CL_InputButton_DirectFBKeyboard*[CL_NUM_KEYS];
  for (int i=0; i<CL_NUM_KEYS; i++) buttons[i] = NULL;

  this->id = id;
}

CL_DirectFBKeyboard::~CL_DirectFBKeyboard()
{
  for (int i=0; i<CL_NUM_KEYS; i++)
    if (buttons[i])
      delete buttons[i];

  delete[] buttons;

  keybuffer->Release (keybuffer);
  keyboard->Release (keyboard);
  dfb->Release (dfb);
}

void CL_DirectFBKeyboard::keep_alive()
{
  DFBInputEvent evt;

  while (keybuffer->GetEvent (keybuffer, &evt) == DFB_OK)
    {
      if (evt.type == DIET_KEYPRESS  ||  evt.type == DIET_KEYRELEASE)
	{
	  CL_Key key;
		
	  key.x = key.y = -1; // since the keyboard is no pointing device

	  key.id = map_directfb_to_cl (evt.keycode);
	  key.ascii = evt.key_ascii;
		
	  if (evt.type == DIET_KEYRELEASE)
	    {
	      key.state = CL_Key::Released;
	      CL_Input::sig_button_release(this, key);
	    }
	  else
	    {
	      key.state = CL_Key::Pressed;
	      CL_Input::sig_button_press(this, key);
	    }
	}
    }
}

char *CL_DirectFBKeyboard::get_name() const
{
  if (id == DIDID_KEYBOARD)
    return "DirectFB Primary Keyboard";
  else
    return "DirectFB Additional Keyboard";
}

CL_InputDevice::InputDeviceType CL_DirectFBKeyboard::get_type() const
{
	return CL_InputDevice::type_keyboard;
}

int CL_DirectFBKeyboard::get_num_buttons() const
{
  return CL_NUM_KEYS;
}

CL_InputButton *CL_DirectFBKeyboard::get_button(int button_num)
{
  if (buttons[button_num] != NULL) return buttons[button_num];

  DFBInputDeviceKeyIdentifier translate_key = map_cl_to_directfb (button_num);
  if (translate_key == DIKC_UNKNOWN) return NULL;

  buttons[button_num] = new CL_InputButton_DirectFBKeyboard (keyboard, translate_key);

  return buttons[button_num];
}

int CL_DirectFBKeyboard::get_num_axes() const
{
	return 0;
}

CL_InputAxis *CL_DirectFBKeyboard::get_axis(int axis_num)
{
	return NULL;
}

int CL_DirectFBKeyboard::get_num_hats() const
{
	return 0;
}

CL_InputHat *CL_DirectFBKeyboard::get_hat(int hat_num)
{
	return NULL;
}

int CL_DirectFBKeyboard::get_num_cursors() const
{
	return 0;
}

CL_InputCursor *CL_DirectFBKeyboard::get_cursor(int cursor_num)
{
	return NULL;
}

int CL_DirectFBKeyboard::map_directfb_to_cl (DFBInputDeviceKeyIdentifier key)
{
  switch (key)
    {
    case DIKC_A:   return CL_KEY_A;
    case DIKC_B:   return CL_KEY_B;
    case DIKC_C:   return CL_KEY_C;
    case DIKC_D:   return CL_KEY_D;
    case DIKC_E:   return CL_KEY_E;
    case DIKC_F:   return CL_KEY_F;
    case DIKC_G:   return CL_KEY_G;
    case DIKC_H:   return CL_KEY_H;
    case DIKC_I:   return CL_KEY_I;
    case DIKC_J:   return CL_KEY_J;
    case DIKC_K:   return CL_KEY_K;
    case DIKC_L:   return CL_KEY_L;
    case DIKC_M:   return CL_KEY_M;
    case DIKC_N:   return CL_KEY_N;
    case DIKC_O:   return CL_KEY_O;
    case DIKC_P:   return CL_KEY_P;
    case DIKC_Q:   return CL_KEY_Q;
    case DIKC_R:   return CL_KEY_R;
    case DIKC_S:   return CL_KEY_S;
    case DIKC_T:   return CL_KEY_T;
    case DIKC_U:   return CL_KEY_U;
    case DIKC_V:   return CL_KEY_V;
    case DIKC_W:   return CL_KEY_W;
    case DIKC_X:   return CL_KEY_X;
    case DIKC_Y:   return CL_KEY_Y;
    case DIKC_Z:   return CL_KEY_Z;

    case DIKC_0:   return CL_KEY_0;
    case DIKC_1:   return CL_KEY_1;
    case DIKC_2:   return CL_KEY_2;
    case DIKC_3:   return CL_KEY_3;
    case DIKC_4:   return CL_KEY_4;
    case DIKC_5:   return CL_KEY_5;
    case DIKC_6:   return CL_KEY_6;
    case DIKC_7:   return CL_KEY_7;
    case DIKC_8:   return CL_KEY_8;
    case DIKC_9:   return CL_KEY_9;

    case DIKC_F1:  return CL_KEY_F1;
    case DIKC_F2:  return CL_KEY_F2;
    case DIKC_F3:  return CL_KEY_F3;
    case DIKC_F4:  return CL_KEY_F4;
    case DIKC_F5:  return CL_KEY_F5;
    case DIKC_F6:  return CL_KEY_F6;
    case DIKC_F7:  return CL_KEY_F7;
    case DIKC_F8:  return CL_KEY_F8;
    case DIKC_F9:  return CL_KEY_F9;
    case DIKC_F10: return CL_KEY_F10;
    case DIKC_F11: return CL_KEY_F11;
    case DIKC_F12: return CL_KEY_F12;

    case DIKC_ESCAPE:    return CL_KEY_ESCAPE;
    case DIKC_LEFT:      return CL_KEY_LEFT;
    case DIKC_RIGHT:     return CL_KEY_RIGHT;
    case DIKC_UP:        return CL_KEY_UP;
    case DIKC_DOWN:      return CL_KEY_DOWN;
    case DIKC_CTRL:      return CL_KEY_LCTRL;
    case DIKC_SHIFT:     return CL_KEY_LSHIFT;
    case DIKC_ALT:       return CL_KEY_ALT;
    case DIKC_ALTGR:     return CL_KEY_ALTGR;
    case DIKC_TAB:       return CL_KEY_TAB;
    case DIKC_ENTER:     return CL_KEY_ENTER;
    case DIKC_SPACE:     return CL_KEY_SPACE;
    case DIKC_BACKSPACE: return CL_KEY_BACKSPACE;
    case DIKC_INSERT:    return CL_KEY_INSERT;
    case DIKC_DELETE:    return CL_KEY_DELETE;
    case DIKC_HOME:      return CL_KEY_HOME;
    case DIKC_END:       return CL_KEY_END;
    case DIKC_PAGEUP:    return CL_KEY_PAGEUP;
    case DIKC_PAGEDOWN:  return CL_KEY_PAGEDOWN;
    case DIKC_CAPSLOCK:  return CL_KEY_CAPSLOCK;
    case DIKC_NUMLOCK:   return CL_KEY_NUMLOCK;
    case DIKC_SCRLOCK:   return CL_KEY_SCRLOCK;
    case DIKC_PRINT:     return CL_KEY_PRINT;
    case DIKC_PAUSE:     return CL_KEY_PAUSE;
    case DIKC_KP_DIV:    return CL_KEY_KP_DIV;
    case DIKC_KP_MULT:   return CL_KEY_KP_MULT;
    case DIKC_KP_MINUS:  return CL_KEY_KP_MINUS;
    case DIKC_KP_PLUS:   return CL_KEY_KP_PLUS;
    case DIKC_KP_ENTER:  return CL_KEY_KP_ENTER;

    default:
      break;
    }

  return CL_KEY_NONE_OF_THE_ABOVE;
}

DFBInputDeviceKeyIdentifier CL_DirectFBKeyboard::map_cl_to_directfb(int id)
{
  switch (id)
    {
    case CL_KEY_A:   return DIKC_A;
    case CL_KEY_B:   return DIKC_B;
    case CL_KEY_C:   return DIKC_C;
    case CL_KEY_D:   return DIKC_D;
    case CL_KEY_E:   return DIKC_E;
    case CL_KEY_F:   return DIKC_F;
    case CL_KEY_G:   return DIKC_G;
    case CL_KEY_H:   return DIKC_H;
    case CL_KEY_I:   return DIKC_I;
    case CL_KEY_J:   return DIKC_J;
    case CL_KEY_K:   return DIKC_K;
    case CL_KEY_L:   return DIKC_L;
    case CL_KEY_M:   return DIKC_M;
    case CL_KEY_N:   return DIKC_N;
    case CL_KEY_O:   return DIKC_O;
    case CL_KEY_P:   return DIKC_P;
    case CL_KEY_Q:   return DIKC_Q;
    case CL_KEY_R:   return DIKC_R;
    case CL_KEY_S:   return DIKC_S;
    case CL_KEY_T:   return DIKC_T;
    case CL_KEY_U:   return DIKC_U;
    case CL_KEY_V:   return DIKC_V;
    case CL_KEY_W:   return DIKC_W;
    case CL_KEY_X:   return DIKC_X;
    case CL_KEY_Y:   return DIKC_Y;
    case CL_KEY_Z:   return DIKC_Z;

    case CL_KEY_0:   return DIKC_0;
    case CL_KEY_1:   return DIKC_1;
    case CL_KEY_2:   return DIKC_2;
    case CL_KEY_3:   return DIKC_3;
    case CL_KEY_4:   return DIKC_4;
    case CL_KEY_5:   return DIKC_5;
    case CL_KEY_6:   return DIKC_6;
    case CL_KEY_7:   return DIKC_7;
    case CL_KEY_8:   return DIKC_8;
    case CL_KEY_9:   return DIKC_9;

    case CL_KEY_F1:  return DIKC_F1;
    case CL_KEY_F2:  return DIKC_F2;
    case CL_KEY_F3:  return DIKC_F3;
    case CL_KEY_F4:  return DIKC_F4;
    case CL_KEY_F5:  return DIKC_F5;
    case CL_KEY_F6:  return DIKC_F6;
    case CL_KEY_F7:  return DIKC_F7;
    case CL_KEY_F8:  return DIKC_F8;
    case CL_KEY_F9:  return DIKC_F9;
    case CL_KEY_F10: return DIKC_F10;
    case CL_KEY_F11: return DIKC_F11;
    case CL_KEY_F12: return DIKC_F12;

    case CL_KEY_ESCAPE:    return DIKC_ESCAPE;
    case CL_KEY_LEFT:      return DIKC_LEFT;
    case CL_KEY_RIGHT:     return DIKC_RIGHT;
    case CL_KEY_UP:        return DIKC_UP;
    case CL_KEY_DOWN:      return DIKC_DOWN;
    case CL_KEY_LCTRL:     return DIKC_CTRL;
    case CL_KEY_RCTRL:     return DIKC_CTRL;
    case CL_KEY_LSHIFT:    return DIKC_SHIFT;
    case CL_KEY_RSHIFT:    return DIKC_SHIFT;
    case CL_KEY_ALT:       return DIKC_ALT;
    case CL_KEY_ALTGR:     return DIKC_ALTGR;
    case CL_KEY_TAB:       return DIKC_TAB;
    case CL_KEY_ENTER:     return DIKC_ENTER;
    case CL_KEY_SPACE:     return DIKC_SPACE;
    case CL_KEY_BACKSPACE: return DIKC_BACKSPACE;
    case CL_KEY_INSERT:    return DIKC_INSERT;
    case CL_KEY_DELETE:    return DIKC_DELETE;
    case CL_KEY_HOME:      return DIKC_HOME;
    case CL_KEY_END:       return DIKC_END;
    case CL_KEY_PAGEUP:    return DIKC_PAGEUP;
    case CL_KEY_PAGEDOWN:  return DIKC_PAGEDOWN;
    case CL_KEY_CAPSLOCK:  return DIKC_CAPSLOCK;
    case CL_KEY_NUMLOCK:   return DIKC_NUMLOCK;
    case CL_KEY_SCRLOCK:   return DIKC_SCRLOCK;
    case CL_KEY_PRINT:     return DIKC_PRINT;
    case CL_KEY_PAUSE:     return DIKC_PAUSE;
    case CL_KEY_KP_DIV:    return DIKC_KP_DIV;
    case CL_KEY_KP_MULT:   return DIKC_KP_MULT;
    case CL_KEY_KP_MINUS:  return DIKC_KP_MINUS;
    case CL_KEY_KP_PLUS:   return DIKC_KP_PLUS;
    case CL_KEY_KP_ENTER:  return DIKC_KP_ENTER;
    }
	
  return DIKC_UNKNOWN;
}

/***************************
  CL_InputButton_XKeyboard
***************************/

CL_InputButton_DirectFBKeyboard::CL_InputButton_DirectFBKeyboard(IDirectFBInputDevice *keyboard,
								 DFBInputDeviceKeyIdentifier key)
{
  keyboard->AddRef (keyboard);

  this->key      = key;
  this->keyboard = keyboard;
}

CL_InputButton_DirectFBKeyboard::~CL_InputButton_DirectFBKeyboard()
{
  keyboard->Release (keyboard);
}

bool CL_InputButton_DirectFBKeyboard::is_pressed()
{
  DFBInputDeviceKeyState state;

  keyboard->GetKeyState (keyboard, key, &state);

  return (state == DIKS_DOWN);
}

#endif
