;;; -*- Mode: Lisp -*-
;;;; MaiSQL --- Common Lisp Interface Layer to SQL Databases
;;;; This is copyrighted software.  See documentation for terms.
;;;; 
;;;; sql.cl --- The SQL-Interface itself.
;;;; 
;;;; Checkout Tag: $Name:  $
;;;; $Id: sql.lisp,v 1.67 2002/03/11 23:32:52 craig Exp $

(in-package :MAISQL-SYS)


(eval-when (:compile-toplevel :load-toplevel)


;;;; %File Description:
;;;; 
;;;; 
;;;; 

;;;; Simple implementation of SQL along the lines of Harlequin's Common SQL

;;; Database Types

(defvar *loaded-database-types* nil
  "Contains a list of database types which have been defined/loaded.")

(defun reload-database-types ()
  "Reloads any foreign code for the loaded database types after a dump."
  (mapc #'database-type-load-foreign *loaded-database-types*))

#+nil(defgeneric database-type-load-foreign (database-type)
  (:documentation
   "The internal generic implementation of reload-database-types.")
  (:method :after (database-type)
	   (pushnew database-type *loaded-database-types*)))

(defvar *default-database-type* nil
  "Specifies the default type of database.  Supported types include
:postgresql, :mysql, and :oracle")

(defvar *initialized-database-types* nil
  "Contains a list of database types which have been initialized by calls
to initialize-database-type.")

(defun initialize-database-type (&key (database-type *default-database-type*))
  "Initialize the given database-type, if it is not already
initialized, as indicated by `*initialized-database-types*'."
  (if (member database-type *initialized-database-types*)
      t
      (when (database-initialize-database-type database-type)
	(push database-type *initialized-database-types*)
	t)))

;;; Database handling

(defvar *connect-if-exists* :error
  "Default value for the if-exists parameter of connect calls.")

(defvar *connected-databases* nil
  "List of active database objects.")

(defun connected-databases ()
  "Return the list of active database objects."
  *connected-databases*)

(defclass database ()
  ((name :initarg :name :reader database-name)
   (spec :initarg :spec)
   (db-type :initarg :db-type :reader database-type
	    :initform :unknown)
   (command-recording-stream
    :accessor command-recording-stream
    :initform nil)
   (result-recording-stream
    :accessor result-recording-stream
    :initform nil)
   (view-classes
    :accessor database-view-classes
    :initform nil)
   (transaction-level
    :accessor transaction-level
    :initform 0))
  (:documentation
   "This class is the supertype of all databases handled by MaiSQL."))
   
(defclass closed-database ()
  ((name :initarg :name :reader database-name))
  (:documentation
   "This class represents all databases after they are closed via
`disconnect'."))

(defun signal-closed-database-error (database)
  (cerror "Ignore this error and return nil."
	  'maisql-closed-error
	  :database database))

(defun signal-no-database-error ()
  (cerror "Ignore this error and return nil."
	  'maisql-nodb-error))

(defmethod database-disconnect ((database closed-database))
  (signal-closed-database-error database))

(defmethod database-disconnect ((database t))
  (signal-no-database-error))

(defmethod database-query (query-expression (database closed-database))
  	   (declare (ignore query-expression))
	   (signal-closed-database-error database))

(defmethod database-query (query-expression (database t))
  	   (declare (ignore query-expression))
	   (signal-no-database-error))

(defmethod database-execute-command (sql-expression (database closed-database))
  (declare (ignore sql-expression))
  (signal-closed-database-error database))

(defmethod database-execute-command (sql-expression (database t))
  (declare (ignore sql-expression))
  (signal-no-database-error))

(defmethod database-query-result-set (expr (database closed-database)
                                           &optional full-set)
  (declare (ignore expr full-set))
  (signal-closed-database-error database)
  (values nil nil nil))

(defmethod database-query-result-set (expr (database t)
                                           &optional full-set)
  (declare (ignore expr full-set))
  (signal-no-database-error))


(defmethod database-query-result-set ((expr %sql-expression) database
                                      &optional full-set)
  (database-query-result-set (sql-output expr database) database full-set))

(defmethod database-dump-result-set (result-set (database closed-database))
  (declare (ignore result-set))
  (signal-closed-database-error database))

(defmethod database-store-next-row (result-set (database closed-database) list)
  (declare (ignore result-set list))
  (signal-closed-database-error database))

(defmethod database-start-transaction ((database closed-database))
  (signal-closed-database-error database))

(defmethod database-start-transaction (database)
  (with-accessors ((level transaction-level))
    database
    (incf level)
    (when (= level 1)
      (execute-command "BEGIN" :database database))))

(defmethod database-commit-transaction ((database closed-database))
  (signal-closed-database-error database))

(defmethod database-commit-transaction (database)
  (with-accessors ((level transaction-level))
    database
    (if (< 0 level)
        (progn
          (decf level)
          (when (= level 0)
            (execute-command "COMMIT" :database database)))
        (warn "Continue without commit."
              'maisql-simple-error
              :format-control "Cannot commit transaction against ~A because there is no transaction in progress."
              :format-arguments (list database)))))

(defmethod database-abort-transaction ((database closed-database))
  (signal-closed-database-error database))

(defmethod database-abort-transaction (database)
  (with-accessors ((level transaction-level))
    database
  (if (< 0 level)
      (progn
        (setf level 0)
        (execute-command "ROLLBACK" :database database))
      (warn "Continue without abort."
              'maisql-simple-error
              :format-control "Cannot abort transaction against ~A because there is no transaction in progress."
              :format-arguments (list database)))))

(defun find-database (database &key (errorp t) (db-type nil))
  (etypecase database
    (database
     ;; Return the database object itself
     database)
    (string
     (or (find-if #'(lambda (db1)
			 (and
			  (string= (database-name db1)
				   database)
			  (if db-type
			      (equal (database-type db1)
				     db-type)
			    t)))
		  (connected-databases))

	 (when errorp
	   (cerror "Return nil."
		   'maisql-simple-error
		   :format-control "There exists no database called ~A."
		   :format-arguments (list database)))))))

(defun connect (connection-spec
		&key (if-exists *connect-if-exists*)
		(make-default t)
		(database-type *default-database-type*))
  "Connects to a database of the given database-type, using the
type-specific connection-spec.  The value of if-exists determineswhat
happens if a connection to that daabase is already established.  A
value of :new means create a new connection.  A value of :warn-new
means warn the user and create a new connect.  A value of :warn-old
means warn the user and use the old connection.  A value of :error
means fail, notifying the user.  A value of :old means return the old
connection.  If make-default is true, then *default-database* is set
to the new connection, otherwise *default-database is not changed."
  (let* ((db-name (database-name-from-spec connection-spec database-type))
	 (old-db (find-database db-name :db-type database-type
				:errorp nil))
	 (result nil))
    (if old-db
	(ecase if-exists
	  (:new
	   (setq result
		 (database-connect connection-spec database-type)))
	  (:warn-new
	   (setq result
		 (database-connect connection-spec database-type))
	   (warn 'maisql-exists-warning :old-db old-db :new-db result))
	  (:error
	   (restart-case
	       (error 'maisql-exists-error :old-db old-db)
	     (create-new ()
	       :report "Create a new connection."
	       (setq result
		     (database-connect connection-spec database-type)))
	     (use-old ()
	       :report "Use the existing connection."
	       (setq result old-db))))
	  (:warn-old
	   (setq result old-db)
	   (warn 'maisql-exists-warning :old-db old-db :new-db old-db))
	  (:old
	   (setq result old-db)))
	(setq result
	      (database-connect connection-spec database-type)))
    (when result
      (pushnew result *connected-databases*)
      (when make-default (setq *default-database* result))
      result)))

(defun disconnect (&key (database *default-database*))
  "Closes the connection to database. Resets *default-database* if that
database was disconnected and only one other connection exists."
  (when (database-disconnect database)
    (setq *connected-databases* (delete database *connected-databases*))
    (when (eq database *default-database*)
      (setq *default-database* (car *connected-databases*)))
    (change-class database 'closed-database)
    t))

;;; Basic operations on databases

(defgeneric query (query-expression &key (database *default-database*))
  (:documentation
   "Execute the SQL query expression query-expression on the given database.
Returns a list of lists of values of the result of that expression."))

(defmethod query ((expr %sql-expression) &key (database *default-database*))
;  (format t "~&;; ~s~%" (sql-output expr database))
;  (execute-command (concatenate 'string "explain "  (sql-output expr database)))
;  (break)
  (query (sql-output expr database) :database database))

(defun print-query (query-exp &key titles formats sizes stream
			      (database *default-database*))
  (declare (ignore titles formats sizes stream))
  (query (sql-output query-exp) :database database))

(defmethod execute-command ((expr %sql-expression)
                            &key (database *default-database*))
  (execute-command (sql-output expr database) :database database))

(defun create-sequence (name &key (database *default-database*))
  (database-create-sequence name database))

(defun drop-sequence (name &key (database *default-database*))
  (database-drop-sequence name database))

(defun sequence-next (name &key (database *default-database*))
  (database-sequence-next name database))

(defun insert-records (&key (into nil)
			    (attributes nil)
			    (values nil)
			    (av-pairs nil)
			    (query nil)
			    (database *default-database*))
  (let ((stmt (make-sql-insert :into into :attrs attributes
			       :vals values :av-pairs av-pairs
			       :subquery query)))
    (execute-command stmt :database database)))

(defun make-sql-insert (&key (into nil)
			    (attrs nil)
			    (vals nil)
			    (av-pairs nil)
			    (subquery nil))
  (if (null into)
      (error 'maisql-sql-syntax-error :reason ":into keyword not supplied"))

  (let ((ins (make-instance 'sql-insert :into into)))
    (with-slots (attributes values query)
      ins
      (cond ((and vals (not attrs) (not query) (not av-pairs))
	     (setf values vals))
	    ((and vals attrs (not query) (not av-pairs))
	     (setf attributes attrs)
	     (setf values vals))
	    ((and av-pairs (not vals) (not attrs) (not query))
	     (setf attributes (mapcar #'car av-pairs))
	     (setf values (mapcar #'cadr av-pairs)))
	    ((and query (not vals) (not attrs) (not av-pairs))
	     (setf query subquery))
	    ((and query attrs (not vals) (not av-pairs))
	     (setf attributes attrs)
	     (setf query subquery))
	    (t
	     (error 'maisql-sql-syntax-error :reason "bad or ambiguous keyword combination.")))
      ins)))
    

(defun delete-records (&key (from nil)
			   (where nil)
			   (database *default-database*))
  (let ((stmt (make-instance 'sql-delete :from from :where where)))
    (execute-command stmt :database database)))

(defun update-records (&key (table nil)
			   (attributes nil)
			   (values nil)
			   (av-pairs nil)
			   (where nil)
			   (database *default-database*))
  (if av-pairs
      (setq attributes (mapcar #'car av-pairs)
	    values (mapcar #'cadr av-pairs)))
  (let ((stmt (make-instance 'sql-update :table table
			     :attributes attributes
			     :values values
			     :where where)))
    (execute-command stmt :database database)))

(defun create-table (name description &key (database *default-database*)
                      (constraints nil))
  "Create a table with the given NAME, containing the attributes in DESCRIPTION."
  (let ((stmt (make-instance 'sql-create-table
			     :name name
			     :columns description
			     :modifiers constraints)))
    (execute-command stmt :database database)))


(defmacro with-database ((database)
			 &rest body)
  "Perform BODY with DATABASE bound as *default-database*."
  `(progv '(*default-database*)
       (list ,database)
     ,@body))

(defvar *transaction-level* 0)
(defvar *transaction-id* nil)

(defvar *transaction-aborts* (make-hash-table))
(defvar *transaction-completes* (make-hash-table))

(defun on-txn-abort (fn)
  (push (cons *transaction-level* fn) (gethash *transaction-id* *transaction-aborts*)))

(defun on-txn-complete (fn)
  (if (> *transaction-level* 0)
      (push fn (gethash *transaction-id* *transaction-completes*))
      (warn "Cannot define on-txn-complete actions outside of transactions.")))

(defun run-abort-hooks ()
  (let ((remainder (remove-if (lambda (hook)
                                (< (car hook) *transaction-level*))
                              (gethash *transaction-id* *transaction-aborts*))))
    (mapcar #'(lambda (hook)
                (funcall (cdr hook)))
            (gethash *transaction-id* *transaction-aborts*))
    (setf (gethash *transaction-id* *transaction-aborts*) remainder)))
    

(defmacro with-transaction ((&key database)
			    &rest body)
  (let ((dbsym (gensym "db"))
	(transym (gensym "tran")))
    `(let ((,dbsym (or ,database *default-database*))
	   (,transym nil)
           (*transaction-id* (or *transaction-id*
                                 (gensym "txn")))
           (*transaction-level* (1+ *transaction-level*)))
      (unwind-protect
        (progn
          (database-start-transaction ,dbsym)
          (setf ,transym t)
          ,@body
          (database-commit-transaction ,dbsym)
          (setf ,transym nil))
        (if ,transym
            (progn                      ; was aborted
              (database-abort-transaction ,dbsym)
              ;; (format t "~&;; Transaction Abort, level ~d~%" *transaction-level*)
              (run-abort-hooks)
              (when (= 1 *transaction-level*)
                (remhash  *transaction-id* *transaction-aborts*)))
            (when (= 1 *transaction-level*)
              (let ((completes (gethash *transaction-id* *transaction-completes*)))
                ;; (format t "~&;; Running ~d post actions.~%" (length completes))
                (mapcar #'funcall completes)
                (remhash  *transaction-id* *transaction-completes*))))))))
              

(defun map-query (output-type-spec function query-expression
				   &key (database *default-database*))
  "Map the function over all tuples that are returned by the query in
query-expression.  The results of the function are collected as
specified in output-type-spec and returned like in MAP."
  ;; DANGER Will Robinson: Parts of the code for implementing
  ;; map-query (including the code below and the helper functions
  ;; called) are highly CMU CL specific.
  (macrolet ((type-specifier-atom (type)
	       `(if (atom ,type) ,type (car ,type))))
    (case (type-specifier-atom output-type-spec)
      ((nil) (map-query-for-effect function query-expression database))
      (list (map-query-to-list function query-expression database))
      ((simple-vector simple-string vector string array simple-array
                      bit-vector simple-bit-vector base-string
		      simple-base-string)
       (map-query-to-simple output-type-spec
			    function query-expression database))
      (t
       (funcall #'map-query
		#+cmu (lisp::result-type-or-lose output-type-spec t)
		#+sbcl (sb-impl::result-type-or-lose output-type-spec t)
              function query-expression :database database)))))

(defun map-query-for-effect (function query-expression database)
  (multiple-value-bind (result-set columns)
      (database-query-result-set query-expression database)
    (when result-set
      (unwind-protect
	   (do ((row (make-list columns)))
	       ((not (database-store-next-row result-set database row))
		nil)
	     (apply function row))
	(database-dump-result-set result-set database)))))
		     
(defun map-query-to-list (function query-expression database)
  (multiple-value-bind (result-set columns)
      (database-query-result-set query-expression database)
    (when result-set
      (unwind-protect
	   (let ((result (list nil)))
	     (do ((row (make-list columns))
		  (current-cons result (cdr current-cons)))
		 ((not (database-store-next-row result-set database row))
		  (cdr result))
	       (rplacd current-cons (list (apply function row)))))
	(database-dump-result-set result-set database)))))

(defun map-query-to-simple
    (output-type-spec function query-expression database)
  (multiple-value-bind (result-set columns rows)
      (database-query-result-set query-expression database t)
    (when result-set
      (unwind-protect
	   (do ((result
		 #+cmu (lisp::make-sequence-of-type output-type-spec rows)
		 #+sbcl (sb-impl::make-sequence-of-type output-type-spec rows))
		(row (make-list columns))
		(index 0 (1+ index)))
	       ((not (database-store-next-row result-set database row))
		result)
	     (declare (fixnum index))
	     (setf (aref result index)
		   (apply function row)))
	(database-dump-result-set result-set database)))))

(defmacro do-query (((&rest args) query-expression
		     &key (database '*default-database*))
		    &body body)
  (let ((result-set (gensym))
	(columns (gensym))
	(row (gensym))
	(db (gensym)))
    `(let ((,db ,database))
       (multiple-value-bind (,result-set ,columns)
	   (database-query-result-set ,query-expression ,db)
	 (when ,result-set
	   (unwind-protect
		(do ((,row (make-list ,columns)))
		    ((not (database-store-next-row ,result-set ,db ,row))
		     nil)
		  (destructuring-bind ,args ,row
		    ,@body))
	     (database-dump-result-set ,result-set ,db)))))))


;;;; MIT-LOOP extension

#+cmu
(defun loop-record-iteration-path (variable data-type prep-phrases)
  (let ((in-phrase nil)
	(from-phrase nil))
    (loop for (prep . rest) in prep-phrases
	  do
	  (case prep
	    ((:in :of)
	     (when in-phrase
	       (ansi-loop::loop-error
		"Duplicate OF or IN iteration path: ~S." (cons prep rest)))
	     (setq in-phrase rest))
	    ((:from)
	     (when from-phrase
	       (ansi-loop::loop-error
		"Duplicate FROM iteration path: ~S." (cons prep rest)))
	     (setq from-phrase rest))
	    (t
	     (ansi-loop::loop-error
	      "Unknown preposition: ~S." prep))))
    (unless in-phrase
      (ansi-loop::loop-error "Missing OF or IN iteration path."))
    (unless from-phrase
      (setq from-phrase '(*default-database*)))
    (cond
      ((consp variable)
       (let ((query-var (ansi-loop::loop-gentemp 'loop-record-))
	     (db-var (ansi-loop::loop-gentemp 'loop-record-database-))
	     (result-set-var (ansi-loop::loop-gentemp
			      'loop-record-result-set-))
	     (step-var (ansi-loop::loop-gentemp 'loop-record-step-)))
	 (push `(when ,result-set-var
		 (database-dump-result-set ,result-set-var ,db-var))
	       ansi-loop::*loop-epilogue*)
	 `(((,variable nil ,data-type) (,query-var ,(first in-phrase))
	    (,db-var ,(first from-phrase))
	    (,result-set-var nil)
	    (,step-var nil))
	   ((multiple-value-bind (%rs %cols)
		(database-query-result-set ,query-var ,db-var)
	      (setq ,result-set-var %rs ,step-var (make-list %cols))))
	   ()
	   ()
	   (not (database-store-next-row ,result-set-var ,db-var ,step-var))
	   (,variable ,step-var)
	   (not ,result-set-var)
	   ()
	   (not (database-store-next-row ,result-set-var ,db-var ,step-var))
	   (,variable ,step-var))))
      (t
       (let ((query-var (ansi-loop::loop-gentemp 'loop-record-))
	     (db-var (ansi-loop::loop-gentemp 'loop-record-database-))
	     (result-set-var (ansi-loop::loop-gentemp
			      'loop-record-result-set-)))
	 (push `(when ,result-set-var
		 (database-dump-result-set ,result-set-var ,db-var))
	       ansi-loop::*loop-epilogue*)
	 `(((,variable nil ,data-type) (,query-var ,(first in-phrase))
	    (,db-var ,(first from-phrase))
	    (,result-set-var nil))
	   ((multiple-value-bind (%rs %cols)
		(database-query-result-set ,query-var ,db-var)
	      (setq ,result-set-var %rs ,variable (make-list %cols))))
	   ()
	   ()
	   (not (database-store-next-row ,result-set-var ,db-var ,variable))
	   ()
	   (not ,result-set-var)
	   ()
	   (not (database-store-next-row ,result-set-var ,db-var ,variable))
	   ()))))))

#+cmu
(ansi-loop::add-loop-path '(record records tuple tuples)
			  'loop-record-iteration-path
			  ansi-loop::*loop-ansi-universe*
			  :preposition-groups '((:of :in) (:from))
			  :inclusive-permitted nil)

;; 1-800-WANT-REGEXP

(defmethod database-output-sql ((str string) database)
  (declare (ignore database)
           (optimize (speed 3) (safety 1) #+cmu (extensions:inhibit-warnings 3))
           (type (simple-array * (*)) str))
  (let ((len (length str)))
    (declare (type fixnum len))
    (cond ((= len 0)
           +empty-string+)
          ((and (null (position #\' str))
                (null (position #\\ str)))
           (concatenate 'string "'" str "'"))
          (t
           (let ((buf (make-string (+ (* len 2) 2) :initial-element #\')))
             (do* ((i 0 (incf i))
                   (j 1 (incf j)))
                  ((= i len) (subseq buf 0 (1+ j)))
               (declare (type integer i j))
               (let ((char (aref str i)))
                 (cond ((eql char #\')
                        (setf (aref buf j) #\\)
                        (incf j)
                        (setf (aref buf j) #\'))
                       ((eql char #\\)
                        (setf (aref buf j) #\\)
                        (incf j)
                        (setf (aref buf j) #\\))
                       (t
                        (setf (aref buf j) char))))))))))

(let ((keyword-package (symbol-package :foo)))
  (defmethod database-output-sql ((sym symbol) database)
    (declare (ignore database))
    (if (eql (symbol-package sym) keyword-package)
        (concatenate 'string "'" (string sym) "'")
        (symbol-name sym))))

(defmethod database-output-sql ((num number) database)
  (declare (ignore database))
  (princ-to-string num))

(defmethod output-sql ((expr list) &optional (database *default-database*))
  (if (null expr)
      (write-string +null-string+ *sql-stream*)
      (progn
        (write-char #\( *sql-stream*)
        (do ((item expr (cdr item)))
            ((null (cdr item))
             (output-sql (car item) database))
          (output-sql (car item) database)
          (write-char #\, *sql-stream*))
        (write-char #\) *sql-stream*)))
  t)

(defmethod database-output-sql ((arg list) database)
  (if (null arg)
      "NULL"
      (format nil "(~{~A~^,~})" (mapcar #'(lambda (val)
                                            (sql-output val database))
                                        arg))))

(defmethod database-output-sql ((arg vector) database)
  (format nil "~{~A~^,~}" (map 'list #'(lambda (val)
					 (sql-output val database))
			       arg)))

(defmethod output-sql-hash-key ((arg vector) &optional database)
  (list 'vector (map 'list (lambda (arg)
                             (or (output-sql-hash-key arg database)
                                 (return-from output-sql-hash-key nil)))
                     arg)))

(defmethod database-output-sql (thing database)
  (if (or (null thing)
	  (eq 'null thing))
      "NULL"
    (error 'maisql-simple-error
           :format-control "No type conversion to SQL for ~A is defined for DB ~A."
           :format-arguments (list (type-of thing) (type-of database)))))

(defmethod output-sql (expr &optional (database *default-database*))
  (write-string (database-output-sql expr database) *sql-stream*)
  t)

(defmacro sql (&rest args)
  `(sql-output ,@args))

)
