;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: wcof -*-
;;; $Id: lru-cache.lisp,v 1.17 2002/02/21 22:38:34 craig Exp $
;;;
;;; Copyright onShore, Inc. 2000
;;;

;;
;; Fixed-size cache with LRU replacement
;;

(in-package :maisql-sys)


(eval-when (:compile-toplevel :load-toplevel)

  (defmacro when-bind ((var expr) &body body)
    `(let ((,var ,expr))
      (when ,var
        ,@body)))
  
  )                                     ; eval-when

(defclass lru-cache ()
  ((capacity
    :initarg :capacity)
   (count
    :initform 0
    :reader cache-size)
   (test
    :initform #'eql
    :initarg :test)
   (entries-size
    :initarg :entries-size)
   (entries
    :initarg :entries)
   (buffer
    :initform nil)))

(defmethod print-object ((self lru-cache) stream)
  (print-unreadable-object
   (self stream :type t
         :identity t)
   (if (and (slot-boundp self 'count)
            (slot-boundp self 'capacity))
       (with-slots (capacity count)
         self
         (format stream
                 "~d/~d" count capacity))
       (write-string "invalid" stream))))

(defun make-cache (&key (size 100) (test #'eql))
  (let ((table-size (max 2 (floor (* size 1.5)))))
    (odcl:note-stat :lru-count :increment)
    (make-instance 'lru-cache :capacity size
                   :entries-size table-size
                   :test test
                   :entries (make-array table-size
                                        :initial-element nil))))

(defun destroy-cache (cache)
  ;(cache-reset cache)
  (with-slots (capacity entries-size entries buffer count)
    cache
    (slot-makunbound cache 'capacity)
    (setf entries-size 0
          count 0
          entries nil
          buffer nil))
  (odcl:note-stat :lru-count :decrement))

(defun cache-reset (cache)
  (with-slots (entries entries-size count buffer)
    cache
    (setf count 0
          buffer nil)
    (dotimes (x entries-size)
      (setf (aref entries x) nil)))
  (values))

(defun cache-get (cache key)
  (when-bind (entry (%cache-get cache key))
             (incf (lce-life entry))
             (lce-value entry)))

(defun cache-put (cache key value)
  (with-slots (buffer count capacity entries)
    cache
    (flet ((free-lru ()
             (when buffer
               (do ((buf buffer (lce-next buf)))
                   ((and (not (lce-modified buf))
                         (= 0 (lce-life buf)))
                    (let ((entry buf))
                      (setf buffer (lce-remove buf))
                      (%cache-delete cache entry)))
                 (decf (lce-life buf))))))
      (multiple-value-bind (entry index)
          (%cache-get cache key)
        (if entry
            (let ((old-value (lce-value entry)))
              (setf (lce-value entry) value)
              old-value)
            (let (en2)
              (if (= count capacity)
                  (free-lru))
              (if (null buffer)
                  (progn
                    (setf buffer (lce-seed index))
                    (setf en2 buffer))
                  (setf en2 (lce-insert buffer index)))
              (setf (lce-key en2) key)
              (setf (lce-value en2) value)
              (setf (lce-index en2) index)
              (setf (lce-collision en2) (aref entries index))
              (setf (aref entries index) en2)
              (incf count)
              nil))))))

(defun cache-update (cache key value)
  (with-slots (buffer count capacity entries)
    cache
    (multiple-value-bind (entry index)
        (%cache-get cache key)
      (declare (ignore index))
      (when entry
        (unless (lce-modified entry)
          (let ((old-value (lce-value entry)))
            (setf (lce-value entry) value)
            old-value))))))

(defun cache-mark (cache key)
  (if-bind (entry (%cache-get cache key))
           (setf (lce-modified entry) t)
           nil))

(defun cache-unmark (cache key)
  (if-bind (entry (%cache-get cache key))
           (not (setf (lce-modified entry) nil))
           nil))

(defun cache-delete (cache key)
  (when-bind (entry (%cache-get cache key))
    (with-slots (buffer)
      cache
      (if (eq buffer entry)
          (setf buffer (lce-remove entry))
          (lce-remove entry))
      (%cache-delete cache entry)
      (lce-value entry))))

(defun %cache-get (cache key)
  (unless (slot-boundp cache 'capacity)
    (error "invalid cache"))
  (with-slots (entries entries-size test)
    cache
    (let ((hash (mod (sxhash key) entries-size)))
      (do ((entry (aref entries hash) (lce-collision entry)))
          ((or (null entry)
               (funcall test (lce-key entry) key))
           (values entry hash))))))

(defun %cache-scanner (cache)
  "returns a function that iterates the contents of the cache, returning k v values"
  (with-slots (entries)
    cache
    (let ((max (1- (length entries)))
          (idx 0)
          (current-entry nil))
      (flet ((kv (entry)
               (when entry
                 (cons (lce-key entry) (lce-value entry)))))
        (lambda ()
          (if (and current-entry
                   (lce-collision current-entry))
              (progn
                (setf current-entry (lce-collision current-entry))
                (kv current-entry))
              (when (< idx max)
                (do ((entry (aref entries idx) (aref entries idx)))
                    ((or entry (<= max idx))
                     (progn
                       (setf current-entry entry)
                       (incf idx)
                       (kv entry)))
                  (incf idx)))))))))

(defun %cache-delete (cache entry)
  (with-slots (entries count)
    cache
    (let ((en2 (aref entries (lce-index entry))))
      (if (eq en2 entry)
          (setf (aref entries (lce-index entry))
                (lce-collision entry))
          (do ((en3 en2 (lce-collision en3)))
              ((or (null en3)
                   (and (eq (lce-collision en3) entry)
                        (setf (lce-collision en3)
                              (lce-collision (lce-collision en3)))))))))
    (decf count)))

(defstruct (lru-cache-entry (:conc-name lce-)
                            (:print-function %print-lru-cache-entry))
  index
  (life 0)
  key
  value
  collision
  next
  prev
  modified)

(defmethod %print-lru-cache-entry (self stream depth)
  (declare (ignore depth))
  (print-unreadable-object
   (self stream :type t)
   (with-slots (value)
     self
     (format stream
             "~s" value))))

(defun lce-seed (index)
  (let ((entry (make-lru-cache-entry :index index)))
    (setf (lce-next entry) entry
          (lce-prev entry) entry)
    entry))

(defun lce-insert (entry index)
  (let ((new-entry (make-lru-cache-entry :index index)))
    (setf (lce-prev new-entry)
          (lce-prev entry))
    (setf (lce-next (lce-prev new-entry))
          new-entry)
    (setf (lce-next new-entry)
          entry)
    new-entry))

(defun lce-remove (entry)
  (when (not (eq entry (lce-next entry)))
    (setf (lce-prev (lce-next entry))
          (lce-prev entry))
    (setf (lce-next (lce-prev entry))
          (lce-next entry))
    (lce-next entry)))
