;;;; MaiSQL --- Common Lisp Interface Layer to SQL Databases
;;;; This is copyrighted software.  See documentation for terms.
;;;; 
;;;; mysql-sql.cl --- SQL-Interface implementation for MySQL
;;;; 
;;;; Checkout Tag: $Name:  $
;;;; $Id: mysql-sql.lisp,v 1.3 2002/03/11 23:33:26 craig Exp $

(in-package :MAISQL-MYSQL)

;;;; %File Description:
;;;; 
;;;; 
;;;; 

(defmethod database-initialize-database-type ((database-type (eql :mysql)))
  t)

(defclass mysql-database (database)
  ((mysql-ptr :accessor database-mysql-ptr :initarg :mysql-ptr
	      :type (alien (* mysql-mysql)))))

(defmethod database-name-from-spec
    (connection-spec (database-type (eql :mysql)))
  (check-connection-spec connection-spec database-type (host db user password))
  (destructuring-bind (host db user password) connection-spec
    (concatenate 'string host "/" db "/" user)))

(defmethod database-connect (connection-spec (database-type (eql :mysql)))
  (check-connection-spec connection-spec database-type (host db user password))
  (destructuring-bind (host db user password)
      connection-spec
    (let ((mysql-ptr (make-alien mysql-mysql)))
      ;; When we fail inside, mysql-ptr is freed, unless we set it to nil
      (unwind-protect
	   (let ((result (mysql-connect mysql-ptr host user password)))
	     (when (null-alien result)
	       ;; Connect failed
	       (error 'maisql-connect-error
		      :database-type database-type
		      :connection-spec connection-spec
		      :errno (mysql-errno mysql-ptr)
		      :error (mysql-error-string mysql-ptr)))
	     ;; The connection is now open, so if we fail inside, we close
	     ;; it, unless we set mysql-ptr to nil to indicate success.
	     (unwind-protect
		  (if (not (zerop (mysql-select-db mysql-ptr db)))
		      (error 'maisql-connect-error
			     :database-type database-type
			     :connection-spec connection-spec
			     :errno (mysql-errno mysql-ptr)
			     :error (mysql-error-string mysql-ptr))
		      (prog1
			  ;; Success, make instance and clean mysql-ptr to
			  ;; prevent automatic clean-up.
			  (make-instance 'mysql-database
					 :name (database-name-from-spec
						connection-spec
						database-type)
					 :mysql-ptr mysql-ptr)
			(setq mysql-ptr nil)))
	       (when mysql-ptr (mysql-close mysql-ptr))))
	(when mysql-ptr (free-alien mysql-ptr))))))

(defmethod database-disconnect ((database mysql-database))
  (mysql-close (database-mysql-ptr database))
  (free-alien (database-mysql-ptr database))
  (setf (database-mysql-ptr database) nil)
  t)

(defmethod database-query (query-expression (database mysql-database))
  (with-slots (mysql-ptr) database
    (if (zerop (mysql-query mysql-ptr query-expression))
	(let ((res-ptr (mysql-use-result mysql-ptr)))
	  (if res-ptr
	      (loop for row = (mysql-fetch-row res-ptr)
		    until (null-alien row)
		    collect
		    (loop for i from 0 below (mysql-num-fields res-ptr)
			  collect
			  (deref row i)))
	      (error 'maisql-sql-error
		     :database database
		     :expression query-expression
		     :errno (mysql-errno mysql-ptr)
		     :error (mysql-error-string mysql-ptr))))
	(error 'maisql-sql-error
	       :database database
	       :expression query-expression
	       :errno (mysql-errno mysql-ptr)
	       :error (mysql-error-string mysql-ptr)))))

(defmethod database-execute-command (sql-expression (database mysql-database))
  (let ((mysql-ptr (database-mysql-ptr database)))
    (declare (type (alien (* mysql-mysql)) mysql-ptr))
    (if (zerop (mysql-query mysql-ptr sql-expression))
	t
	(error 'maisql-sql-error
	       :database database
	       :expression sql-expression
	       :errno (mysql-errno mysql-ptr)
	       :error (mysql-error-string mysql-ptr)))))

(defstruct mysql-result-set
  (res-ptr (sap-alien (int-sap 0) (* mysql-mysql-res))
	   :type (alien (* mysql-mysql-res)))
  (full-set nil))

(defmethod database-query-result-set
    (query-expression (database mysql-database) &optional full-set)
  (let ((mysql-ptr (database-mysql-ptr database)))
    (declare (type (alien (* mysql-mysql)) mysql-ptr))
    (if (zerop (mysql-query mysql-ptr query-expression))
	(let ((res-ptr (if full-set
			   (mysql-store-result mysql-ptr)
			   (mysql-use-result mysql-ptr))))
	  (declare (type (alien (* mysql-mysql-res)) res-ptr))
	  (if (not (null-alien res-ptr))
	      (if full-set
		  (values (make-mysql-result-set :res-ptr res-ptr :full-set t)
			  (mysql-num-fields res-ptr)
			  (mysql-num-rows res-ptr))
		  (values (make-mysql-result-set :res-ptr res-ptr)
			  (mysql-num-fields res-ptr)))
	      (error 'maisql-sql-error
		     :database database
		     :expression query-expression
		     :errno (mysql-errno mysql-ptr)
		     :error (mysql-error-string mysql-ptr))))
	(error 'maisql-sql-error
	       :database database
	       :expression query-expression
	       :errno (mysql-errno mysql-ptr)
	       :error (mysql-error-string mysql-ptr)))))

(defmethod database-dump-result-set (result-set (database mysql-database))
  (if (mysql-result-set-full-set result-set)
      (mysql-free-result (mysql-result-set-res-ptr result-set))
      (loop for row = (mysql-fetch-row (mysql-result-set-res-ptr result-set))
	    until (null-alien row)))
  t)

(defmethod database-store-next-row (result-set (database mysql-database) list)
  (let* ((res-ptr (mysql-result-set-res-ptr result-set))
	 (row (mysql-fetch-row res-ptr)))
    (declare (type (alien (* mysql-mysql-res)) res-ptr)
	     (type (alien mysql-row) row))
    (unless (null-alien row)
      (loop for i from 0 below (mysql-num-fields res-ptr)
	    for rest on list
	    do
	    (setf (car rest) (deref row i)))
      list)))

(defmethod database-pkey-constraint ((class view-metaclass) (database mysql-database))
  (when-bind (keylist (mapcar #'view-class-slot-column (keyslots-for-class class)))
      (format nil "CONSTRAINT ~APK PRIMARY KEY~A"
            (database-output-sql (view-table class) database)
            (database-output-sql keylist database))))

(defmethod database-generate-column-definition (class slotdef (database mysql-database))
    (declare (ignore class))
  (when (member (view-class-slot-db-kind slotdef) '(:base :key))
    (let ((cdef (list (sql-expression :attribute (view-class-slot-column slotdef))
                      (slot-definition-type slotdef)))
          (const (view-class-slot-db-constraints slotdef)))
      (if (equal (view-class-slot-db-kind slotdef) :key)
          (pushnew 'not-null const))
      (when const
        (setq cdef (append cdef (listify const))))
      cdef)))