;;; -*- Mode: LISP; Syntax: ANSI-Common-Lisp; Package: CL-USER; -*-
;;;
;;; Filename:    ssl-socketlisp
;;; Author:      Jochen Schmidt <jsc@dataheaven.de>
;;;              Wade Humeniuk <humeniuw@cadvision.com>
;;; Description: Handle connection stuff and library initialization
;;;              of ssl-sockets.
;;;

(in-package :ssl-internal)

;;;;;;;;;;;;;;;;;;;;;;
;;; Global context ;;;
;;;;;;;;;;;;;;;;;;;;;;

(defparameter *ssl-global-context* nil)
(defparameter *ssl-global-method* nil)

(defun ssl-initialized-p ()
  (and *ssl-global-context* *ssl-global-method*))

(defclass ssl-client-socket (ssl-socket-mixin simple-buffering-mixin) ())
(defclass ssl-server-socket (ssl-socket-mixin simple-buffering-mixin) ())

(defmethod initialize-instance :after ((s ssl-socket-mixin) &key host port &allow-other-keys)
  (declare (ignore host port))
   (let ((handle (ssl-new (or *ssl-global-context* (ssl-ctx-new (ssl-v23-method))))))
     (try-ssl-call (ssl-set-fd handle (ssl-socket-fd s)) handle)
     (setf (ssl-socket-method s) *ssl-global-method*)
     (setf (ssl-socket-ctx s) *ssl-global-context*)
     (setf (ssl-socket-handle s) handle)
     (when (and (slot-boundp s 'rsa-privatekey-file)
                (ssl-socket-rsa-privatekey-file s))
       (try-ssl-call (ssl-use-rsa-privatekey-file (ssl-socket-handle s) 
                                                  (ssl-socket-rsa-privatekey-file s)
                                                  +ssl-filetype-pem+)
                     handle))
     (when (and (slot-boundp s 'certificate-file)
                (ssl-socket-certificate-file s))
       (try-ssl-call (ssl-use-certificate-file (ssl-socket-handle s)
                                               (ssl-socket-certificate-file s)
                                               +ssl-filetype-pem+)
                     handle))))

(defmethod initialize-instance :after ((s ssl-client-socket) &key host port &allow-other-keys)
  (declare (ignore host port))
  (let ((handle (ssl-socket-handle s)))
    (ssl-set-connect-state handle)
    (ensure-ssl-funcall handle #'ssl-connect 
                        '(:sleep-time 0.25) handle)))
  
(defmethod initialize-instance :after ((s ssl-server-socket) &key host port &allow-other-keys)
  (declare (ignore host port))
  (let ((handle (ssl-socket-handle s)))
    (ssl-set-accept-state handle)
    (ensure-ssl-funcall handle #'ssl-accept
                        '(:sleep-time 0.25) handle)))

(defun close-ssl-socket (sock)
  (with-slots (handle input-buffer output-buffer) sock
    (ssl-shutdown handle)
    (ssl-socket-internal-close sock)
    (ssl-free handle)))
    
(defun initialize-ssl-library (&key (force nil))
  "Load all ciphers and errorstrings. Create a global context"
  (when (or force (not (ssl-initialized-p)))
    (format t "Initialize SSL...~%")
    (force-output t)
    (ssl-library-init)
    (ssl-load-error-strings)
    (ssl-socket-rand-seed)
    (setf *ssl-global-method* (ssl-v23-method))
    (setf *ssl-global-context* (ssl-ctx-new *ssl-global-method*))))

(defun shutdown-ssl-library ()
  (when (ssl-initialized-p)
    (ssl-ctx-free *ssl-global-context*)
    (setf *ssl-global-context* nil)
    (setf *ssl-global-method* nil)))